import MetaTrader5 as mt5
import pandas as pd
import numpy as np
from datetime import datetime, timedelta

def calculate_pearson_correlation(symbols, timeframe, lookback_days=250):
    """
    Calculate Pearson Correlation Coefficient for a list of symbols using MT5 historical data.

    Args:
        symbols (list): List of symbols (e.g., ["MU", "NVDA", "MPWR", "MCHP"]).
        timeframe (str): MT5 timeframe (e.g., "M15", "H1", "D1").
        lookback_days (int): Number of days for historical data (default: 90).

    Returns:
        pd.DataFrame: Correlation matrix (Pearson coefficients).
    """
    # Initialize MT5 connection
    if not mt5.initialize():
        print("MT5 initialization failed!")
        return None

    # Define time range (UTC)
    utc_from = datetime.strptime('2024-01-01 00:00', '%Y-%m-%d %H:%M')
    utc_to = datetime.strptime('2025-04-02 00:00', '%Y-%m-%d %H:%M')

    # Fetch close prices for all symbols
    close_prices = {}
    for symbol in symbols:
        rates = mt5.copy_rates_range(symbol, getattr(mt5, f"TIMEFRAME_{timeframe}"), utc_from, utc_to)
        if rates is None:
            print(f"Failed to fetch data for {symbol}")
            continue
        df = pd.DataFrame(rates)
        df['time'] = pd.to_datetime(df['time'], unit='s')
        close_prices[symbol] = df.set_index('time')['close']

    # Combine into a single DataFrame
    combined_df = pd.DataFrame(close_prices)
    if combined_df.empty:
        print("No data fetched for any symbol.")
        return None

    # Calculate Pearson correlation
    correlation_matrix = combined_df.corr(method='pearson')

    mt5.shutdown()
    return correlation_matrix

# Example Usage
if __name__ == "__main__":
    symbols = ['US500', 'JPN225']
    # symbols = ['NVDA', 'MCHP', 'MRVL', 'MPWR', 'MU','AMAT']
    timeframe = "D1"  # Daily timeframe
    lookback_days = 504  # ~ 1 year of data

    corr_matrix = calculate_pearson_correlation(symbols, timeframe, lookback_days)
    if corr_matrix is not None:
        print("Pearson Correlation Matrix:")
        print(corr_matrix)

# %pip install seaborn
import seaborn as sns
sns.heatmap(corr_matrix, annot=True, cmap="coolwarm")
# add symbols and dates to title
import matplotlib.pyplot as plt
# Define the symbols and date range used in the correlation calculation
symbols = ['US500', 'JPN225']
utc_from = datetime.strptime('2024-01-01 00:00', '%Y-%m-%d %H:%M')
utc_to = datetime.strptime('2025-04-02 00:00', '%Y-%m-%d %H:%M')
# utc_to = datetime.now()
plt.title(f"Pearson Correlation Matrix for {', '.join(symbols)}\nFrom {utc_from.date()} to {utc_to.date()}")
plt.show()