import os
import pandas as pd
import numpy as np
import statsmodels.api as sm
from statsmodels.regression.recursive_ls import RecursiveLS
import MetaTrader5 as mt5
import matplotlib.pyplot as plt
import warnings

# Suppress the specific ValueWarning from statsmodels regarding date frequencies
warnings.filterwarnings("ignore", message="A date index has been provided")

class CUSUMMonitor:
    def __init__(self, db_path="cointegration_data.db"):
        self.db_path = db_path

    def connect_mt5(self):
        if not mt5.initialize():
            raise ConnectionError("MT5 initialization failed")
        print("✅ MT5 Connected")

    def fetch_data(self, symbols, start_date, end_date, timeframe=mt5.TIMEFRAME_D1):
        data = {}
        utc_from = pd.to_datetime(start_date)
        utc_to = pd.to_datetime(end_date)

        for symbol in symbols:
            rates = mt5.copy_rates_range(symbol, timeframe, utc_from, utc_to)
            if rates is None or len(rates) == 0:
                continue
            df = pd.DataFrame(rates)
            df['time'] = pd.to_datetime(df['time'], unit='s')
            df.set_index('time', inplace=True)
            data[symbol] = df['close'].rename(symbol)
            
        return pd.concat(data, axis=1).dropna()

    def run_early_warning(self, data: pd.DataFrame):
        # Regression: Y = a + bX
        y = data.iloc[:, 0]
        x = sm.add_constant(data.iloc[:, 1])

        # We use simple OLS residuals for the CUSUM calculation to match plot logic
        res = RecursiveLS(y, x).fit()
        
        cusum_values = res.cusum
        nobs = len(cusum_values)
        
        # Manually calculate the 5% significance bound
        # The threshold for the CUSUM test at index 't'
        critical_val = 0.948
        limit = critical_val * (np.sqrt(nobs) + (2 * nobs / np.sqrt(nobs)))
        
        current_val = cusum_values[-1]
        is_breaking = np.abs(current_val) > limit
        
        return is_breaking, res

    def simulate(self, symbols, start_date, end_date, initial_window=60):
        self.connect_mt5()
        full_data = self.fetch_data(symbols, start_date, end_date)
        mt5.shutdown()

        if len(full_data) < initial_window:
            print("Data insufficient for initial window.")
            return

        print(f"Starting simulation for {symbols}...")
        print(f"Initial training window: {initial_window} bars.")

        for i in range(initial_window, len(full_data) + 1):
            current_slice = full_data.iloc[:i]
            current_date = current_slice.index[-1].date()

            # Skip monitoring on known low-liquidity periods
            if self.is_market_holiday(current_date):
                print(f"[{current_date}] Monitoring... ☕ Skipping (Holiday Filter Active)")
                continue
            
            is_breaking, res = self.run_early_warning(current_slice)
            
            # Print status every 10 days to show progress
            if i % 10 == 0 and not is_breaking:
                print(f"[{current_date}] Monitoring... ✅ Stable")

            if is_breaking:
                print(f"\n🛑 CRITICAL SIGNAL on {current_date}")
                print("The model has deviated beyond the 95% confidence interval.")
                
                # FIXED: statsmodels RecursiveLS plot_cusum uses 'fig' not 'ax'
                fig = plt.figure(figsize=(7, 4))
                res.plot_cusum(fig=fig) 
                plt.title(f"Early Warning Monitor (CUSUM): {symbols[0]} vs {symbols[1]}\nTrigger Date: {current_date}")
                plt.tight_layout()
                plt.show()
                return 

        print("\nSimulation complete. No structural break detected.")

    def is_market_holiday(self, date_obj):
        """
        Returns True if the date is a known low-liquidity period 
        (e.g., Christmas to New Year).
        """
        # Define a list of fixed holidays or a range
        month = date_obj.month
        day = date_obj.day

        # Example: Dec 24th through Jan 1st
        if month == 12 and day >= 24:
            return True
        if month == 1 and day == 1:
            return True

        return False

if __name__ == "__main__":
    # If the break on 2024-12-31 is too sensitive, 
    # increase initial_window to 80 or 90 to establish a stronger baseline.
    SYMBOLS = ['NVDA', 'INTC']
    START = '2025-06-01'
    END = '2025-12-30'
    
    monitor = CUSUMMonitor()
    monitor.simulate(SYMBOLS, START, END, initial_window=70)