
//+------------------------------------------------------------------+
//|                                                   supertrend.mq5 |
//|          Copyright 2026, MetaQuotes Ltd. Developer is Chacha Ian |
//|                          https://www.mql5.com/en/users/chachaian |
//+------------------------------------------------------------------+

#property copyright "Copyright 2026, MetaQuotes Ltd. Developer is Chacha Ian"
#property link      "https://www.mql5.com/en/users/chachaian"
#property version   "1.00"

#property indicator_chart_window
#property indicator_plots 3
#property indicator_buffers 8

#property indicator_color1 clrBlack, clrSeaGreen
#property indicator_label1 "SupertrendOpen;SupertrendHigh;SupertrendLow;SupertrendClose"


//+------------------------------------------------------------------+
//| User input variables                                             |
//+------------------------------------------------------------------+
input group "Information"
input int32_t atrPeriod     = 10;
input double  atrMultiplier = 1.5;            


//+------------------------------------------------------------------+
//| Global Variables                                                 |
//+------------------------------------------------------------------+
ENUM_TIMEFRAMES timeframe = PERIOD_CURRENT;

int atrHandle;
double atrValues [];

const double TREND_BULLISH =  1.0;
const double TREND_BEARISH =  0.0;
const double TREND_NEUTRAL = -1.0;


//+------------------------------------------------------------------+
//| Indicator buffers                                                |
//+------------------------------------------------------------------+
double bufCandleOpen[];
double bufCandleHigh[];
double bufCandleLow[];
double bufCandleClose[];
double bufCandleTrendColor[];

double bufSupertrendUpperBand[];
double bufSupertrendLowerBand[];

double bufTrendState[];


//+------------------------------------------------------------------+
//| Custom indicator initialization function                         |
//+------------------------------------------------------------------+
int OnInit(){

   //--- To configure the chart's appearance
   if(!ConfigureChartAppearance()){
      Print("Error while configuring chart appearance ", GetLastError());
      return INIT_FAILED;
   }
   
   //--- Initialize the ATR indicator
   atrHandle = iATR(_Symbol, timeframe, atrPeriod);
   if(atrHandle == INVALID_HANDLE){
      Print("Error while initializing the ATR indicator ", GetLastError());
      return(INIT_FAILED);
   }
   
   //--- Bind arrays to indicator buffers
   SetIndexBuffer(0, bufCandleOpen, INDICATOR_DATA);
   SetIndexBuffer(1, bufCandleHigh, INDICATOR_DATA);
   SetIndexBuffer(2, bufCandleLow, INDICATOR_DATA);
   SetIndexBuffer(3, bufCandleClose, INDICATOR_DATA);
   SetIndexBuffer(4, bufCandleTrendColor, INDICATOR_COLOR_INDEX);
   SetIndexBuffer(5, bufSupertrendUpperBand, INDICATOR_DATA);
   SetIndexBuffer(6, bufSupertrendLowerBand, INDICATOR_DATA);
   SetIndexBuffer(7, bufTrendState, INDICATOR_CALCULATIONS);
   
   //--- Configure Graphic Plots   
   PlotIndexSetInteger(0, PLOT_DRAW_TYPE, DRAW_COLOR_CANDLES);
   PlotIndexSetDouble (0, PLOT_EMPTY_VALUE, EMPTY_VALUE);
   
   PlotIndexSetInteger(1, PLOT_DRAW_TYPE, DRAW_LINE);
   PlotIndexSetInteger(1, PLOT_LINE_STYLE, STYLE_SOLID);
   PlotIndexSetInteger(1, PLOT_LINE_WIDTH, 1);
   PlotIndexSetInteger(1, PLOT_LINE_COLOR, clrBlack);
   PlotIndexSetDouble (1, PLOT_EMPTY_VALUE, EMPTY_VALUE);
   PlotIndexSetString (1, PLOT_LABEL, "downtrend");
   
   PlotIndexSetInteger(2, PLOT_DRAW_TYPE, DRAW_LINE);
   PlotIndexSetInteger(2, PLOT_LINE_STYLE, STYLE_SOLID);
   PlotIndexSetInteger(2, PLOT_LINE_WIDTH, 1);
   PlotIndexSetInteger(2, PLOT_LINE_COLOR, clrSeaGreen);
   PlotIndexSetDouble (2, PLOT_EMPTY_VALUE, EMPTY_VALUE);
   PlotIndexSetString (2, PLOT_LABEL, "uptrend");
   
   //--- General indicator configurations
   IndicatorSetString(INDICATOR_SHORTNAME, "Supertrend Indicator");
   IndicatorSetInteger(INDICATOR_DIGITS, Digits());    
   
   return INIT_SUCCEEDED;
}


//+------------------------------------------------------------------+
//| Custom indicator iteration function                              |
//+------------------------------------------------------------------+
int OnCalculate(const int32_t rates_total,
                const int32_t prev_calculated,
                const datetime &time[],
                const double &open[],
                const double &high[],
                const double &low[],
                const double &close[],
                const long &tick_volume[],
                const long &volume[],
                const int32_t &spread[])
{

   //--- Temporary buffers used to store price data for rendering custom colored candles
   double sOpen [];
   double sHigh [];
   double sLow  [];
   double sClose[];

   //--- This block is executed when the indicator is initially attached on a chart
   if(prev_calculated == 0){
   
      //--- Start with clean buffers at first calculation
      ArrayInitialize(bufCandleOpen, EMPTY_VALUE);
      ArrayInitialize(bufCandleHigh, EMPTY_VALUE);
      ArrayInitialize(bufCandleLow,  EMPTY_VALUE);
      ArrayInitialize(bufCandleClose, EMPTY_VALUE);
      ArrayInitialize(bufSupertrendUpperBand, EMPTY_VALUE);
      ArrayInitialize(bufSupertrendLowerBand, EMPTY_VALUE);
      ArrayInitialize(bufTrendState, TREND_NEUTRAL);
      
      //--- Copy price data into local working buffers to safely manipulate candle values without altering the original price arrays
      ArrayCopy(sOpen, open);
      ArrayCopy(sHigh, high);
      ArrayCopy(sLow, low);
      ArrayCopy(sClose, close);
      
      //--- Ensure sufficient historical data is available to calculate the ATR period.
      if(rates_total < atrPeriod){
         Print("Insufficient bars to initialize Supertrend: ATR period requires more historical data.");
         return rates_total;
      }
      
      //--- Get all ATR values
      int numberOfCopiedATRValues = CopyBuffer(atrHandle, 0, 0, rates_total, atrValues);
      if(numberOfCopiedATRValues == -1){
         Print("Error while copying ATR values: ", GetLastError());
         return rates_total;
      }
      
      //--- Iterate through historical price data to calculate Supertrend values starting from the first valid ATR window.
      for(int i = atrPeriod - 1; i < rates_total - 1; i++){
         
         //--- Calculate the current bar midpoint and derive the raw Supertrend upper and lower bands using the ATR multiplier
         double barMidpoint    = (high[i] + low[i]) / 2.0;
         double upperBandValue = barMidpoint + atrValues[i] * atrMultiplier;
         double lowerBandValue = barMidpoint - atrValues[i] * atrMultiplier;
         
         //--- Initialize the first valid Supertrend bands once enough ATR data is available and skip further processing for this bar
         if(i == (atrPeriod - 1)){
            bufSupertrendUpperBand[atrPeriod - 1] = upperBandValue;
            bufSupertrendLowerBand[atrPeriod - 1] = lowerBandValue;
            continue;
         }
         
         //--- Handle initial trend resolution when no prior trend has been established
         if(bufTrendState[i - 1] == TREND_NEUTRAL){
         
            if(close[i] > bufSupertrendUpperBand[i - 1] || close[i] < bufSupertrendLowerBand[i - 1]){
               
               if(close[i] > bufSupertrendUpperBand[i - 1]){
                  bufTrendState[i] = TREND_BULLISH;
                  SetBullishCandle(i, sOpen, sHigh, sLow, sClose);
                  bufSupertrendUpperBand[i] = EMPTY_VALUE;
                  bufSupertrendLowerBand[i] = lowerBandValue;
               }
               
               if(close[i] < bufSupertrendLowerBand[i - 1]){
                  bufTrendState[i] = TREND_BEARISH;
                  SetBearishCandle(i, sOpen, sHigh, sLow, sClose);
                  bufSupertrendLowerBand[i] = EMPTY_VALUE;
                  bufSupertrendUpperBand[i] = upperBandValue;
               }
               
            }else{
            
               if(upperBandValue < bufSupertrendUpperBand[i - 1]){
                  bufSupertrendUpperBand[i] = upperBandValue;
               }else{
                  bufSupertrendUpperBand[i] = bufSupertrendUpperBand[i - 1];
               }
               
               if(lowerBandValue > bufSupertrendLowerBand[i - 1]){
                  bufSupertrendLowerBand[i] = lowerBandValue;
               }else{
                  bufSupertrendLowerBand[i] = bufSupertrendLowerBand[i - 1];
               }   
            }          
         }
         
         //--- Maintain or invalidate the bullish trend based on price interaction with the lower Supertrend band
         if(bufTrendState[i - 1] == TREND_BULLISH){
            if(close[i] < bufSupertrendLowerBand[i - 1]){
               bufTrendState[i] = TREND_BEARISH;
               SetBearishCandle(i, sOpen, sHigh, sLow, sClose);
               bufSupertrendLowerBand[i] = EMPTY_VALUE;
               bufSupertrendUpperBand[i] = upperBandValue;
            }else{
               if(lowerBandValue > bufSupertrendLowerBand[i - 1]){
                  bufSupertrendLowerBand[i] = lowerBandValue;
                  SetBullishCandle(i, sOpen, sHigh, sLow, sClose);
                  bufTrendState[i] = TREND_BULLISH;
               }else{
                  bufSupertrendLowerBand[i] = bufSupertrendLowerBand[i - 1];
                  SetBullishCandle(i, sOpen, sHigh, sLow, sClose);
                  bufTrendState[i] = TREND_BULLISH;
               }
            }
         }
         
         //--- Maintain or invalidate the bearish trend based on price interaction with the upper Supertrend band
         if(bufTrendState[i - 1] == TREND_BEARISH){
            if(close[i] > bufSupertrendUpperBand[i - 1]){
               bufTrendState[i] = TREND_BULLISH;
               SetBullishCandle(i, sOpen, sHigh, sLow, sClose);
               bufSupertrendUpperBand[i] = EMPTY_VALUE;
               bufSupertrendLowerBand[i] = lowerBandValue;
            }else{
               if(upperBandValue < bufSupertrendUpperBand[i - 1]){
                  bufSupertrendUpperBand[i] = upperBandValue;
                  SetBearishCandle(i, sOpen, sHigh, sLow, sClose);
                  bufTrendState[i] = TREND_BEARISH;
               }else{
                  bufSupertrendUpperBand[i] = bufSupertrendUpperBand[i - 1];
                  SetBearishCandle(i, sOpen, sHigh, sLow, sClose);
                  bufTrendState[i] = TREND_BEARISH;
               }
            }
         }   
      }      
   }
   
   //--- This block is executed on every new bar open
   if(prev_calculated != rates_total && prev_calculated != 0){
   
      //--- Copy price data into local working buffers to safely manipulate candle values without altering the original price arrays
      ArrayCopy(sOpen, open);
      ArrayCopy(sHigh, high);
      ArrayCopy(sLow, low);
      ArrayCopy(sClose, close);
      
      //--- Start with clean buffers
      bufSupertrendLowerBand[rates_total - 1] = EMPTY_VALUE;
      bufSupertrendLowerBand[rates_total - 2] = EMPTY_VALUE;
      bufSupertrendUpperBand[rates_total - 1] = EMPTY_VALUE;
      bufSupertrendUpperBand[rates_total - 2] = EMPTY_VALUE;
      
      bufTrendState[rates_total - 1] = TREND_NEUTRAL;
      bufTrendState[rates_total - 2] = TREND_NEUTRAL;
   
      //--- Get all ATR values
      int numberOfCopiedATRValues = CopyBuffer(atrHandle, 0, 0, 3, atrValues);
      if(numberOfCopiedATRValues == -1){
         Print("Error while copying ATR values: ", GetLastError());
         return rates_total;
      }
      
      //--- Define indices for the most recently closed bar and the bar immediately preceding it
      int elapsedBarIndex      = rates_total - 2;
      int priorElapsedBarIndex = rates_total - 3;
      
      //--- Compute the Supertrend midpoint and derive the upper and lower bands using the ATR multiplier
      double barMidpoint    = (high[elapsedBarIndex] + low[elapsedBarIndex]) / 2.0;
      double upperBandValue = barMidpoint + atrValues[1] * atrMultiplier;
      double lowerBandValue = barMidpoint - atrValues[1] * atrMultiplier;
      
      //--- Handle Supertrend continuation or reversal logic when the previous bar was bullish
      if(bufTrendState[priorElapsedBarIndex] == TREND_BULLISH){
         if(close[elapsedBarIndex] < bufSupertrendLowerBand[priorElapsedBarIndex]){
            bufTrendState[elapsedBarIndex] = TREND_BEARISH;
            SetBearishCandle(elapsedBarIndex, sOpen, sHigh, sLow, sClose);
            bufSupertrendUpperBand[elapsedBarIndex] = upperBandValue;
            bufSupertrendLowerBand[elapsedBarIndex] = EMPTY_VALUE;
         }else{
            if(lowerBandValue > bufSupertrendLowerBand[priorElapsedBarIndex]){
               bufSupertrendLowerBand[elapsedBarIndex] = lowerBandValue;
               bufSupertrendUpperBand[elapsedBarIndex] = EMPTY_VALUE;
               SetBullishCandle(elapsedBarIndex, sOpen, sHigh, sLow, sClose);
               bufTrendState[elapsedBarIndex] = TREND_BULLISH;
            }else{
               bufSupertrendLowerBand[elapsedBarIndex] = bufSupertrendLowerBand[priorElapsedBarIndex];
               bufSupertrendUpperBand[elapsedBarIndex] = EMPTY_VALUE;
               SetBullishCandle(elapsedBarIndex, sOpen, sHigh, sLow, sClose);
               bufTrendState[elapsedBarIndex] = TREND_BULLISH;
            }
         }
      }
      
      //--- Handle Supertrend continuation or reversal logic when the previous bar was bearish
      if(bufTrendState[priorElapsedBarIndex] == TREND_BEARISH){
         if(close[elapsedBarIndex] > bufSupertrendUpperBand[priorElapsedBarIndex]){
            bufTrendState[elapsedBarIndex] = TREND_BULLISH;
            SetBullishCandle(elapsedBarIndex, sOpen, sHigh, sLow, sClose);
            bufSupertrendLowerBand[elapsedBarIndex] = lowerBandValue;
            bufSupertrendUpperBand[elapsedBarIndex] = EMPTY_VALUE;
         }else{
            if(upperBandValue < bufSupertrendUpperBand[priorElapsedBarIndex]){
               bufSupertrendUpperBand[elapsedBarIndex] = upperBandValue;
               bufSupertrendLowerBand[elapsedBarIndex] = EMPTY_VALUE;
               SetBearishCandle(elapsedBarIndex, sOpen, sHigh, sLow, sClose);
               bufTrendState[elapsedBarIndex] = TREND_BEARISH;
            }else{
               bufSupertrendUpperBand[elapsedBarIndex] = bufSupertrendUpperBand[priorElapsedBarIndex];
               bufSupertrendLowerBand[elapsedBarIndex] = EMPTY_VALUE;
               SetBearishCandle(elapsedBarIndex, sOpen, sHigh, sLow, sClose);
               bufTrendState[elapsedBarIndex] = TREND_BEARISH;
            }
         }
      }   
   }
   
   return rates_total;
}


//+------------------------------------------------------------------+
//| Deinitiatialization function                                     |
//+------------------------------------------------------------------+
void OnDeinit(const int reason){

   //--- Release ATR
   if(atrHandle != INVALID_HANDLE){
      IndicatorRelease(atrHandle);
   }
    
}


//--- CUSTOM UTILITY FUNCTIONS
//+------------------------------------------------------------------+
//| This function configures the chart's appearance.                 |
//+------------------------------------------------------------------+
bool ConfigureChartAppearance()
{
   if(!ChartSetInteger(0, CHART_COLOR_BACKGROUND, clrWhite)){
      Print("Error while setting chart background, ", GetLastError());
      return false;
   }
   
   if(!ChartSetInteger(0, CHART_SHOW_GRID, false)){
      Print("Error while setting chart grid, ", GetLastError());
      return false;
   }
   
   if(!ChartSetInteger(0, CHART_MODE, CHART_CANDLES)){
      Print("Error while setting chart mode, ", GetLastError());
      return false;
   }

   if(!ChartSetInteger(0, CHART_COLOR_FOREGROUND, clrBlack)){
      Print("Error while setting chart foreground, ", GetLastError());
      return false;
   }

   if(!ChartSetInteger(0, CHART_COLOR_CANDLE_BULL, clrWhite)){
      Print("Error while setting bullish candles color, ", GetLastError());
      return false;
   }
      
   if(!ChartSetInteger(0, CHART_COLOR_CANDLE_BEAR, clrWhite)){
      Print("Error while setting bearish candles color, ", GetLastError());
      return false;
   }
   
   if(!ChartSetInteger(0, CHART_COLOR_CHART_UP, clrSeaGreen)){
      Print("Error while setting bearish candles color, ", GetLastError());
      return false;
   }
   
   if(!ChartSetInteger(0, CHART_COLOR_CHART_DOWN, clrBlack)){
      Print("Error while setting bearish candles color, ", GetLastError());
      return false;
   }
   
   return true;
}


//+------------------------------------------------------------------+
//| Populates candle buffers for a bullish bar and assigns the bullish trend color at the given index |
//+------------------------------------------------------------------+
void SetBullishCandle(int index, double &open[], double &high[], double &low[], double &close[]){
   bufCandleOpen [index]  = open [index];
   bufCandleHigh [index]  = high [index];
   bufCandleLow  [index]  = low  [index];
   bufCandleClose[index]  = close[index];
   bufCandleTrendColor[index] = 1;
}


//+------------------------------------------------------------------+
//| Populates candle buffers for a bearish bar and assigns the bearish trend color at the given index |
//+------------------------------------------------------------------+
void SetBearishCandle(int index, double &open[], double &high[], double &low[], double &close[]){
   bufCandleOpen [index]  = open [index];
   bufCandleHigh [index]  = high [index];
   bufCandleLow  [index]  = low  [index];
   bufCandleClose[index]  = close[index];
   bufCandleTrendColor[index] = 0;
}

//+------------------------------------------------------------------+
