//+------------------------------------------------------------------+
//|                                   AccountDynamicsIndicator.mq5   |
//|                                  Copyright 2025, Persist FX      |
//|                                     Educational Analysis Tool    |
//+------------------------------------------------------------------+
#property copyright "Copyright 2025, Persist FX"
#property link      "https://www.mql5.com"
#property version   "2.0"
#property description "Visual analysis of account balance, equity, and floating P/L dynamics."
#property description "Displays historical account statistics for analytical purposes."
#property indicator_separate_window
#property indicator_buffers 4
#property indicator_plots   4

//--- plot Starting Balance (Reference Line)
#property indicator_label1  "Starting Balance"
#property indicator_type1   DRAW_LINE
#property indicator_color1  clrGray
#property indicator_style1  STYLE_DASH
#property indicator_width1  1

//--- plot Balance
#property indicator_label2  "Balance"
#property indicator_type2   DRAW_LINE
#property indicator_color2  clrOrange
#property indicator_style2  STYLE_SOLID
#property indicator_width2  2

//--- plot Equity
#property indicator_label3  "Equity"
#property indicator_type3   DRAW_LINE
#property indicator_color3  clrDodgerBlue
#property indicator_style3  STYLE_SOLID
#property indicator_width3  2

//--- plot Floating P/L
#property indicator_label4  "Floating P/L"
#property indicator_type4   DRAW_LINE
#property indicator_color4  clrLime
#property indicator_style4  STYLE_SOLID
#property indicator_width4  1

//--- input parameters
input int InpHistoryBars = 0;           // Number of bars to display (0 = all available)
input int InpUpdatePeriod = 300;        // History update period in seconds (default 5 min)

//--- indicator buffers
double StartingBalanceBuffer[];
double BalanceBuffer[];
double EquityBuffer[];
double FloatingBuffer[];

//--- structure to store balance changes over time
struct BalancePoint
{
   datetime time;
   double balance;
};

//--- structure for position tracking
struct PositionInfo
{
   ulong ticket;
   string symbol;
   datetime openTime;
   datetime closeTime;
   double openPrice;
   double volume;
   ENUM_POSITION_TYPE type;
   double tickSize;
   double tickValue;
};

//--- global variables
BalancePoint balanceHistory[];
PositionInfo positionCache[];
datetime lastHistoryUpdate = 0;
datetime earliestDealTime = 0;
double initialDeposit = 0;
int totalDealsProcessed = 0;

//--- price cache for multi-symbol efficiency
struct PriceCache
{
   string symbol;
   datetime time;
   double price;
};
PriceCache priceCache[];

//+------------------------------------------------------------------+
//| Custom indicator initialization function                         |
//+------------------------------------------------------------------+
int OnInit()
{
   //--- indicator buffers mapping
   SetIndexBuffer(0, StartingBalanceBuffer, INDICATOR_DATA);
   SetIndexBuffer(1, BalanceBuffer, INDICATOR_DATA);
   SetIndexBuffer(2, EquityBuffer, INDICATOR_DATA);
   SetIndexBuffer(3, FloatingBuffer, INDICATOR_DATA);
   
   //--- set arrays as series
   ArraySetAsSeries(StartingBalanceBuffer, true);
   ArraySetAsSeries(BalanceBuffer, true);
   ArraySetAsSeries(EquityBuffer, true);
   ArraySetAsSeries(FloatingBuffer, true);
   
   //--- set precision
   IndicatorSetInteger(INDICATOR_DIGITS, 2);
   
   //--- set indicator short name
   IndicatorSetString(INDICATOR_SHORTNAME, "Account Dynamics");
   
   //--- initialize buffers
   ArrayInitialize(StartingBalanceBuffer, EMPTY_VALUE);
   ArrayInitialize(BalanceBuffer, EMPTY_VALUE);
   ArrayInitialize(EquityBuffer, EMPTY_VALUE);
   ArrayInitialize(FloatingBuffer, EMPTY_VALUE);
   
   //--- build complete account history
   if(!BuildAccountHistory())
   {
      Print("Warning: Could not build complete account history");
   }
   
   lastHistoryUpdate = TimeCurrent();
   
   return(INIT_SUCCEEDED);
}

//+------------------------------------------------------------------+
//| Build complete account history from deals                        |
//+------------------------------------------------------------------+
bool BuildAccountHistory()
{
   datetime toDate = TimeCurrent();
   datetime fromDate = 0; // From account inception
   
   if(!HistorySelect(fromDate, toDate))
   {
      Print("Error: Failed to load history - ", GetLastError());
      return false;
   }
   
   //--- clear previous data
   ArrayResize(balanceHistory, 0);
   ArrayResize(positionCache, 0);
   ArrayResize(priceCache, 0);
   
   int totalDeals = HistoryDealsTotal();
   totalDealsProcessed = totalDeals;
   
   if(totalDeals == 0)
   {
      Print("No trading history found");
      initialDeposit = AccountInfoDouble(ACCOUNT_BALANCE);
      AddBalancePoint(TimeCurrent(), initialDeposit);
      earliestDealTime = TimeCurrent();
      return true;
   }
   
   Print("Processing ", totalDeals, " deals from account history...");
   
   //--- find earliest deal and initial deposit
   earliestDealTime = TimeCurrent();
   double runningBalance = 0;
   bool foundInitialDeposit = false;
   
   for(int i = 0; i < totalDeals; i++)
   {
      ulong dealTicket = HistoryDealGetTicket(i);
      if(dealTicket == 0) continue;
      
      datetime dealTime = (datetime)HistoryDealGetInteger(dealTicket, DEAL_TIME);
      ENUM_DEAL_TYPE dealType = (ENUM_DEAL_TYPE)HistoryDealGetInteger(dealTicket, DEAL_TYPE);
      
      //--- find first balance operation (deposit/credit)
      if(!foundInitialDeposit && (dealType == DEAL_TYPE_BALANCE || dealType == DEAL_TYPE_CREDIT))
      {
         initialDeposit = HistoryDealGetDouble(dealTicket, DEAL_PROFIT);
         runningBalance = initialDeposit;
         earliestDealTime = dealTime;
         foundInitialDeposit = true;
         AddBalancePoint(dealTime, runningBalance);
      }
      
      if(dealTime < earliestDealTime)
         earliestDealTime = dealTime;
   }
   
   //--- if no initial deposit found, use first deal as reference
   if(!foundInitialDeposit)
   {
      initialDeposit = AccountInfoDouble(ACCOUNT_BALANCE);
      runningBalance = initialDeposit;
      AddBalancePoint(earliestDealTime, initialDeposit);
   }
   
   //--- process all trading deals chronologically
   for(int i = 0; i < totalDeals; i++)
   {
      ulong dealTicket = HistoryDealGetTicket(i);
      if(dealTicket == 0) continue;
      
      datetime dealTime = (datetime)HistoryDealGetInteger(dealTicket, DEAL_TIME);
      ENUM_DEAL_ENTRY dealEntry = (ENUM_DEAL_ENTRY)HistoryDealGetInteger(dealTicket, DEAL_ENTRY);
      ENUM_DEAL_TYPE dealType = (ENUM_DEAL_TYPE)HistoryDealGetInteger(dealTicket, DEAL_TYPE);
      
      //--- skip non-trading deals in this loop
      if(dealType == DEAL_TYPE_BALANCE || dealType == DEAL_TYPE_CREDIT)
         continue;
      
      double dealProfit = HistoryDealGetDouble(dealTicket, DEAL_PROFIT);
      double dealSwap = HistoryDealGetDouble(dealTicket, DEAL_SWAP);
      double dealCommission = HistoryDealGetDouble(dealTicket, DEAL_COMMISSION);
      ulong positionId = HistoryDealGetInteger(dealTicket, DEAL_POSITION_ID);
      string symbol = HistoryDealGetString(dealTicket, DEAL_SYMBOL);
      
      //--- handle position entry
      if(dealEntry == DEAL_ENTRY_IN)
      {
         int size = ArraySize(positionCache);
         ArrayResize(positionCache, size + 1);
         
         positionCache[size].ticket = positionId;
         positionCache[size].symbol = symbol;
         positionCache[size].openTime = dealTime;
         positionCache[size].closeTime = 0;
         positionCache[size].openPrice = HistoryDealGetDouble(dealTicket, DEAL_PRICE);
         positionCache[size].volume = HistoryDealGetDouble(dealTicket, DEAL_VOLUME);
         positionCache[size].type = (dealType == DEAL_TYPE_BUY) ? POSITION_TYPE_BUY : POSITION_TYPE_SELL;
         
         //--- cache symbol specifications
         positionCache[size].tickSize = SymbolInfoDouble(symbol, SYMBOL_TRADE_TICK_SIZE);
         positionCache[size].tickValue = SymbolInfoDouble(symbol, SYMBOL_TRADE_TICK_VALUE);
         
         if(positionCache[size].tickSize == 0)
            positionCache[size].tickSize = SymbolInfoDouble(symbol, SYMBOL_POINT);
      }
      //--- handle position exit (balance change)
      else if(dealEntry == DEAL_ENTRY_OUT || dealEntry == DEAL_ENTRY_INOUT)
      {
         //--- mark position as closed
         for(int j = 0; j < ArraySize(positionCache); j++)
         {
            if(positionCache[j].ticket == positionId && positionCache[j].closeTime == 0)
            {
               positionCache[j].closeTime = dealTime;
               break;
            }
         }
         
         //--- update balance
         double balanceChange = dealProfit + dealSwap + dealCommission;
         if(balanceChange != 0)
         {
            runningBalance += balanceChange;
            AddBalancePoint(dealTime, runningBalance);
         }
      }
   }
   
   Print("History built: ", ArraySize(balanceHistory), " balance points, ", 
         ArraySize(positionCache), " positions tracked, Initial deposit: ", initialDeposit);
   
   return true;
}

//+------------------------------------------------------------------+
//| Add balance point to history                                     |
//+------------------------------------------------------------------+
void AddBalancePoint(datetime time, double balance)
{
   int size = ArraySize(balanceHistory);
   ArrayResize(balanceHistory, size + 1);
   balanceHistory[size].time = time;
   balanceHistory[size].balance = balance;
}

//+------------------------------------------------------------------+
//| Custom indicator iteration function                              |
//+------------------------------------------------------------------+
int OnCalculate(const int rates_total,
                const int prev_calculated,
                const datetime &time[],
                const double &open[],
                const double &high[],
                const double &low[],
                const double &close[],
                const long &tick_volume[],
                const long &volume[],
                const int &spread[])
{
   if(rates_total < 1)
      return 0;
   
   ArraySetAsSeries(time, true);
   
   //--- check if we need to update history
   datetime currentTime = TimeCurrent();
   if(currentTime - lastHistoryUpdate > InpUpdatePeriod)
   {
      //--- only rebuild if new deals appeared
      HistorySelect(0, currentTime);
      int currentDeals = HistoryDealsTotal();
      
      if(currentDeals != totalDealsProcessed)
      {
         Print("New deals detected, rebuilding history...");
         BuildAccountHistory();
      }
      
      lastHistoryUpdate = currentTime;
   }
   
   //--- calculate bars to process
   int limit = rates_total - prev_calculated;
   if(prev_calculated == 0)
      limit = rates_total;
   
   if(InpHistoryBars > 0 && limit > InpHistoryBars)
      limit = InpHistoryBars;
   
   //--- get current account values
   double currentBalance = AccountInfoDouble(ACCOUNT_BALANCE);
   double currentEquity = AccountInfoDouble(ACCOUNT_EQUITY);
   
   //--- process each bar
   for(int i = 0; i < limit; i++)
   {
      datetime barTime = time[i];
      
      //--- before earliest deal, no data available
      if(barTime < earliestDealTime)
      {
         StartingBalanceBuffer[i] = EMPTY_VALUE;
         BalanceBuffer[i] = EMPTY_VALUE;
         EquityBuffer[i] = EMPTY_VALUE;
         FloatingBuffer[i] = EMPTY_VALUE;
         continue;
      }
      
      //--- starting balance is constant horizontal line
      StartingBalanceBuffer[i] = initialDeposit;
      
      //--- for the most recent bar, use live account data
      if(i == 0)
      {
         BalanceBuffer[i] = currentBalance;
         FloatingBuffer[i] = currentEquity - currentBalance;
         EquityBuffer[i] = currentEquity;
      }
      else
      {
         //--- get historical balance at this bar
         double balanceAtTime = GetBalanceAtTime(barTime, currentBalance);
         
         //--- calculate historical floating P/L
         double floatingPL = CalculateFloatingPLAtTime(barTime);
         
         BalanceBuffer[i] = balanceAtTime;
         FloatingBuffer[i] = floatingPL;
         EquityBuffer[i] = balanceAtTime + floatingPL;
      }
   }
   
   return rates_total;
}

//+------------------------------------------------------------------+
//| Get balance at specific time from history                        |
//+------------------------------------------------------------------+
double GetBalanceAtTime(datetime targetTime, double currentBalance)
{
   int historySize = ArraySize(balanceHistory);
   
   if(historySize == 0)
      return currentBalance;
   
   //--- if before first balance point, return initial deposit
   if(targetTime < balanceHistory[0].time)
      return initialDeposit;
   
   //--- find the most recent balance point before target time
   double balance = initialDeposit;
   
   for(int i = 0; i < historySize; i++)
   {
      if(balanceHistory[i].time <= targetTime)
         balance = balanceHistory[i].balance;
      else
         break;
   }
   
   return balance;
}

//+------------------------------------------------------------------+
//| Calculate floating P/L at specific time                         |
//+------------------------------------------------------------------+
double CalculateFloatingPLAtTime(datetime barTime)
{
   double totalPL = 0;
   int positionsProcessed = 0;
   
   //--- iterate through all cached positions
   for(int i = 0; i < ArraySize(positionCache); i++)
   {
      //--- skip if position not yet opened at bar time
      if(positionCache[i].openTime > barTime)
         continue;
      
      //--- skip if position already closed at bar time
      if(positionCache[i].closeTime != 0 && positionCache[i].closeTime <= barTime)
         continue;
      
      //--- position was open at this bar time, calculate its P/L
      double priceAtTime = GetCachedPrice(positionCache[i].symbol, barTime);
      
      if(priceAtTime <= 0)
         continue;
      
      //--- calculate P/L using cached tick size and value
      if(positionCache[i].tickSize <= 0 || positionCache[i].tickValue <= 0)
         continue;
      
      double priceDiff = (positionCache[i].type == POSITION_TYPE_BUY) ?
                         priceAtTime - positionCache[i].openPrice :
                         positionCache[i].openPrice - priceAtTime;
      
      double positionPL = (priceDiff / positionCache[i].tickSize) * 
                          positionCache[i].tickValue * 
                          positionCache[i].volume;
      
      totalPL += positionPL;
      positionsProcessed++;
   }
   
   return totalPL;
}

//+------------------------------------------------------------------+
//| Get price with caching for performance                          |
//+------------------------------------------------------------------+
double GetCachedPrice(string symbol, datetime barTime)
{
   //--- check cache first
   for(int i = 0; i < ArraySize(priceCache); i++)
   {
      if(priceCache[i].symbol == symbol && priceCache[i].time == barTime)
         return priceCache[i].price;
   }
   
   //--- not in cache, fetch price
   double price = 0;
   
   if(symbol == _Symbol)
   {
      //--- current chart symbol (fastest)
      int shift = iBarShift(_Symbol, Period(), barTime, true);
      if(shift >= 0)
         price = iClose(_Symbol, Period(), shift);
   }
   else
   {
      //--- other symbol, use CopyRates
      MqlRates rates[];
      ArraySetAsSeries(rates, true);
      int copied = CopyRates(symbol, Period(), barTime, 1, rates);
      
      if(copied > 0)
         price = rates[0].close;
   }
   
   //--- add to cache
   if(price > 0)
   {
      int cacheSize = ArraySize(priceCache);
      
      //--- limit cache size to prevent memory issues
      if(cacheSize >= 1000)
      {
         //--- clear oldest 500 entries
         ArrayRemove(priceCache, 0, 500);
         cacheSize = ArraySize(priceCache);
      }
      
      ArrayResize(priceCache, cacheSize + 1);
      priceCache[cacheSize].symbol = symbol;
      priceCache[cacheSize].time = barTime;
      priceCache[cacheSize].price = price;
   }
   
   return price;
}

//+------------------------------------------------------------------+
//| Custom indicator deinitialization function                       |
//+------------------------------------------------------------------+
void OnDeinit(const int reason)
{
   Comment("");
   ArrayFree(balanceHistory);
   ArrayFree(positionCache);
   ArrayFree(priceCache);
}
//+------------------------------------------------------------------+