//+------------------------------------------------------------------+
//|                                                 TestDatabase.mq5 |
//|                        Copyright 2025, MetaQuotes Software Corp. |
//|                   https://www.mql5.com/en/users/swedishforexking |
//+------------------------------------------------------------------+
#property copyright "Copyright 2025, MetaQuotes Software Corp."
#property link      "https://www.mql5.com/en/users/swedishforexking"
#property version   "1.00"
//+------------------------------------------------------------------+
//| TestDatabase.mq5                                                 |
//+------------------------------------------------------------------+
#include "DatabaseORM v1.0.mqh"
#include "TradeReportModel v1.0.mqh"
#include "MyDBClassModel v1.0.mqh"


//+------------------------------------------------------------------+
class CBacktestDataGenerator
  {
private:
   CDatabaseORM      m_database;

public:
                     CBacktestDataGenerator(string db_name = "backtest_results.db") : m_database(db_name) {}

   bool              InitializeDatabase1()
     {
      Print("=== Backtest Data Generator by Native Base ===");

      Print("Initializing database...");

      uint db_flags = DATABASE_OPEN_READWRITE | DATABASE_OPEN_COMMON;
      if(!m_database.Connect(db_flags))
        {
         Print("Failed to connect to database");
         return false;
        }

      // Create tables
      report_model.AddField("id", FIELD_TYPE_INT, PRIMARY_KEY | AUTO_INCREMENT | REQUIRED);
      report_model.AddField("strategy_name", FIELD_TYPE_STRING, REQUIRED);
      report_model.AddField("report_date", FIELD_TYPE_DATETIME, REQUIRED);
      report_model.AddField("total_net_profit", FIELD_TYPE_DOUBLE);
      report_model.AddField("profit_factor", FIELD_TYPE_DOUBLE);
      report_model.AddField("max_drawdown_relative", FIELD_TYPE_DOUBLE);
      report_model.AddField("total_trades", FIELD_TYPE_INT);
      report_model.AddField("parameters", FIELD_TYPE_STRING);

      if(!m_database.CreateTable(report_model))
        {
         Print("Failed to create table");
         return false;
        }

      Print("Database initialized successfully");
      return true;
     }


   bool              InitializeDatabase2()
     {

      Print("=== Backtest Data Generator by Inherited Base ===");

      Print("Initializing database...");

      uint db_flags = DATABASE_OPEN_READWRITE | DATABASE_OPEN_COMMON;
      if(!m_database.Connect(db_flags))
        {
         Print("Failed to connect to database");
         return false;
        }

      // Create tables
      trade_model.DefineFields();
      if(!m_database.CreateTable(trade_model))
        {
         Print("Failed to create table");
         return false;
        }

      Print("Database initialized successfully");
      return true;
     }

   bool              InitializeDatabase3()
     {

      Print("=== Backtest Data Generator by Macro Base ===");

      Print("Initializing database...");

      uint db_flags = DATABASE_OPEN_READWRITE | DATABASE_OPEN_COMMON;
      if(!m_database.Connect(db_flags))
        {
         Print("Failed to connect to database");
         return false;
        }

      // Create tables
      macro_model.DefineFields();
      if(!m_database.CreateTable(macro_model))
        {
         Print("Failed to create table");
         return false;
        }

      Print("Database initialized successfully");
      return true;
     }

   void              GenerateSampleBacktestData1()
     {
      Print("Generating sample backtest data...");

      // Sample strategy configurations
      string strategies[] =
        {
         "MA_Crossover_10_20", "RSI_Strategy_14_30_70",
         "Bollinger_Bands_20_2", "MACD_Strategy_12_26_9",
         "Stochastic_14_3_3", "Parabolic_SAR_002_02",
         "Ichimoku_Cloud", "ADX_Strategy_14",
         "Price_Action_Breakout", "Volume_Weighted_MA"
        };

      string symbols[] = {"EURUSD", "GBPUSD", "USDJPY", "AUDUSD", "XAUUSD"};
      string timeframes[] = {"H1", "H4", "D1", "W1"};

      int totalRecords = 0;

      for(int i = 0; i < ArraySize(strategies); i++)
        {
         for(int j = 0; j < ArraySize(symbols); j++)
           {
            for(int k = 0; k < ArraySize(timeframes); k++)
              {
               // Generate realistic backtest results
               report_model["strategy_name"].SetValue(strategies[i] + "_" + symbols[j] + "_" + timeframes[k]);
               report_model["report_date"].SetValue(TimeCurrent() - (MathRand() % 2592000)); // Random date in last month
               report_model["total_net_profit"].SetValue(GenerateRealisticProfit());
               report_model["profit_factor"].SetValue(1.0 + (MathRand() % 200) / 100.0); // 1.0 to 3.0
               report_model["max_drawdown_relative"].SetValue(5.0 + (MathRand() % 250) / 10.0); // 5% to 30%
               report_model["total_trades"].SetValue(50 + (MathRand() % 450)); // 50 to 500 trades

               string parameters = StringFormat("Symbol=%s, Timeframe=%s, Strategy=%s",
                                                symbols[j], timeframes[k], strategies[i]);
               report_model["parameters"].SetValue(parameters);

               if(m_database.Insert(report_model))
                 {
                  totalRecords++;
                 }

               // Add some variation
               Sleep(10);
              }
           }
        }

      PrintFormat("Generated %d backtest records", totalRecords);
     }


   void              GenerateSampleBacktestData2()
     {
      Print("Generating sample backtest data...");

      // Sample strategy configurations
      string strategies[] =
        {
         "MA_Crossover_10_20", "RSI_Strategy_14_30_70",
         "Bollinger_Bands_20_2", "MACD_Strategy_12_26_9",
         "Stochastic_14_3_3", "Parabolic_SAR_002_02",
         "Ichimoku_Cloud", "ADX_Strategy_14",
         "Price_Action_Breakout", "Volume_Weighted_MA"
        };

      string symbols[] = {"EURUSD", "GBPUSD", "USDJPY", "AUDUSD", "XAUUSD"};
      string timeframes[] = {"H1", "H4", "D1", "W1"};

      int totalRecords = 0;

      for(int i = 0; i < ArraySize(strategies); i++)
        {
         for(int j = 0; j < ArraySize(symbols); j++)
           {
            for(int k = 0; k < ArraySize(timeframes); k++)
              {
               // Generate realistic backtest results
               trade_model.SetStrategyName(strategies[i] + "_" + symbols[j] + "_" + timeframes[k]);
               trade_model.SetReportDate(TimeCurrent() - (MathRand() % 2592000)); // Random date in last month
               trade_model.SetTotalNetProfit(GenerateRealisticProfit());
               trade_model.SetProfitFactor(1.0 + (MathRand() % 200) / 100.0); // 1.0 to 3.0
               trade_model.SetMaxDrawdown(5.0 + (MathRand() % 250) / 10.0); // 5% to 30%
               trade_model.SetTotalTrades(50 + (MathRand() % 450)); // 50 to 500 trades

               string parameters = StringFormat("Symbol=%s, Timeframe=%s, Strategy=%s",
                                                symbols[j], timeframes[k], strategies[i]);
               trade_model.SetParameters(parameters);

               if(m_database.Insert(trade_model))
                 {
                  totalRecords++;
                 }

               // Add some variation
               Sleep(10);
              }
           }
        }

      PrintFormat("Generated %d backtest records", totalRecords);
     }

   void              GenerateSampleBacktestData3()
     {
      Print("Generating sample backtest data...");

      // Sample strategy configurations
      string strategies[] =
        {
         "MA_Crossover_10_20", "RSI_Strategy_14_30_70",
         "Bollinger_Bands_20_2", "MACD_Strategy_12_26_9",
         "Stochastic_14_3_3", "Parabolic_SAR_002_02",
         "Ichimoku_Cloud", "ADX_Strategy_14",
         "Price_Action_Breakout", "Volume_Weighted_MA"
        };

      string symbols[] = {"EURUSD", "GBPUSD", "USDJPY", "AUDUSD", "XAUUSD"};
      string timeframes[] = {"H1", "H4", "D1", "W1"};

      int totalRecords = 0;

      for(int i = 0; i < ArraySize(strategies); i++)
        {
         for(int j = 0; j < ArraySize(symbols); j++)
           {
            for(int k = 0; k < ArraySize(timeframes); k++)
              {
               // Generate realistic backtest results
               macro_model.Set_strategy_name(strategies[i] + "_" + symbols[j] + "_" + timeframes[k]);
               macro_model.Set_report_date(TimeCurrent() - (MathRand() % 2592000)); // Random date in last month
               macro_model.Set_total_net_profit(GenerateRealisticProfit());
               macro_model.Set_profit_factor(1.0 + (MathRand() % 200) / 100.0); // 1.0 to 3.0
               macro_model.Set_max_drawdown_relative(5.0 + (MathRand() % 250) / 10.0); // 5% to 30%
               macro_model.Set_total_trades(50 + (MathRand() % 450)); // 50 to 500 trades

               string parameters = StringFormat("Symbol=%s, Timeframe=%s, Strategy=%s",
                                                symbols[j], timeframes[k], strategies[i]);
               macro_model.Set_parameters(parameters);

               if(m_database.Insert(macro_model))
                 {
                  totalRecords++;
                 }

               // Add some variation
               Sleep(10);
              }
           }
        }

      PrintFormat("Generated %d backtest records", totalRecords);
     }

private:
   double            GenerateRealisticProfit()
     {
      // Generate realistic profit/loss figures
      // 60% chance of profit, 40% chance of loss
      bool isProfitable = (MathRand() % 100) < 60;
      double baseAmount = 500 + (MathRand() % 9500); // $500 to $10,000

      if(isProfitable)
        {
         return baseAmount * (0.5 + (MathRand() % 150) / 100.0); // 0.5x to 2x
        }
      else
        {
         return -baseAmount * (0.1 + (MathRand() % 50) / 100.0); // -0.1x to -0.6x
        }
     }
  };

// Global instance
CBaseModel report_model;
CTradeReportModel trade_model;
CMyDBClassModel macro_model;

//+------------------------------------------------------------------+
CBacktestDataGenerator data_generator_native_base("backtest_results_native_base.db");
CBacktestDataGenerator data_generator_inherited_base("backtest_results_inherited_base.db");
CBacktestDataGenerator data_generator_macro_base("backtest_results_macro_base.db");

//+------------------------------------------------------------------+
//| Script program start function                                    |
//+------------------------------------------------------------------+
void OnStart()
  {

   if(data_generator_native_base.InitializeDatabase1() &&
      data_generator_inherited_base.InitializeDatabase2() &&
      data_generator_macro_base.InitializeDatabase3())
     {
      data_generator_native_base.GenerateSampleBacktestData1();
      data_generator_inherited_base.GenerateSampleBacktestData2();
      data_generator_macro_base.GenerateSampleBacktestData3();

      Print("Data generation completed!");

      // Demonstrate query capabilities
      RunDemoQueries();
     }
   else
     {
      Print("Failed to initialize database");
     }

  }


//+------------------------------------------------------------------+
void RunDemoQueries()
  {
   CDatabaseORM db_native_base("backtest_results_native_base.db");
   CDatabaseORM db_inherited_base("backtest_results_inherited_base.db");
   CDatabaseORM db_macro_base("backtest_results_macro_base.db");

      uint db_flags = DATABASE_OPEN_READWRITE | DATABASE_OPEN_COMMON;

   if(!db_native_base.Connect(db_flags))
     {
      Print("Failed to connect to database for queries");
      return;
     }
   if(!db_inherited_base.Connect(db_flags))
     {
      Print("Failed to connect to database for queries");
      return;
     }

   if(!db_macro_base.Connect(db_flags))
     {
      Print("Failed to connect to database for queries");
      return;
     }

   Print("\n=== QUERY DEMONSTRATIONS ===");

// Demonstration Select operations : retrieving data from db
   DemoSelectOperationsNativeBase(db_native_base);

// Demonstration Select operations : retrieving data from db
   DemoSelectOperationsInheritedBase(db_inherited_base);

// Demonstration Select operations : retrieving data from db
   DemoSelectOperationsMacroBase(db_macro_base);

  }

//+------------------------------------------------------------------+
void DemoSelectOperationsNativeBase(CDatabaseORM &db)
  {
   Print("\n--- 1. Select OPERATIONS by Native Base Model ---");

// Select example
   if(db.Select(report_model))
     {
      PrintFormat("Found record : %s", report_model["strategy_name"].GetValue());
     }

// SelectAll example
   CArrayObj found_array;

   int items = db.SelectAll(found_array,report_model,"profit_factor > 1.0");
   Print("items : ", items);
   if(items>0)
     {
      for(int i=0; i<found_array.Total(); i++)
        {
         CBaseModel* _temp = found_array.At(i);
         if(_temp!=NULL)
            PrintFormat("Found record : %s", _temp["strategy_name"].GetValue());
        }

     }


  }

//+------------------------------------------------------------------+
void DemoSelectOperationsInheritedBase(CDatabaseORM &db)
  {
   Print("\n--- 2. Select OPERATIONS by Inherited Base Model ---");

// Select example
   CTradeReportModel found_by_select;
   if(db.Select(found_by_select))
     {
      PrintFormat("Found record : %s", found_by_select.GetStrategyName());
     }

// SelectAll example
   CTradeReportModel found_by_selectAll;
   CArrayObj found_array;
   if(db.SelectAll(found_array,found_by_selectAll,"profit_factor > 1.0")>0)
     {
      for(int i=0; i<found_array.Total(); i++)
        {
         CTradeReportModel* _temp = found_array.At(i);
         PrintFormat("Found record : %s", _temp.GetStrategyName());
        }

     }

  }

//+------------------------------------------------------------------+
void DemoSelectOperationsMacroBase(CDatabaseORM &db)
  {
   Print("\n--- 3. Select OPERATIONS by Macro Base Model ---");

// Select example
//CMyDBClassModel found_by_select(macro_model);
   CMyDBClassModel found_by_select;
   if(db.Select(found_by_select))
     {
      PrintFormat("Found record : %s", found_by_select.Get_strategy_name());
     }

// SelectAll example
   CMyDBClassModel found_by_selectAll;
   CArrayObj found_array;
   if(db.SelectAll(found_array,found_by_selectAll,"profit_factor > 1.0")>0)
     {
      for(int i=0; i<found_array.Total(); i++)
        {
         CMyDBClassModel* _temp = found_array.At(i);
         PrintFormat("Found record : %s", _temp.Get_strategy_name());
        }

     }

  }

//+------------------------------------------------------------------+
