//+------------------------------------------------------------------+
//|                                             VolumeBoundary.mq5   |
//+------------------------------------------------------------------+
#property copyright "Wamek EA"
#property link      "https://www.mql5.com/en/users/wamek/news"
#property version   "1.00"

#property indicator_separate_window
#property indicator_buffers 3
#property indicator_plots   1

#property indicator_label1  "Volume Boundary"
#property indicator_type1   DRAW_HISTOGRAM
#property indicator_color1  clrBurlyWood
#property indicator_width1  2
#property indicator_style1  STYLE_SOLID

#property indicator_levelcolor clrGray
#property indicator_level1 0.8
#property indicator_level2 2.0
#property indicator_level3 0.0
#property indicator_level4 -0.8
#property indicator_level5 -2.0

// Input parameters
enum ifcn 
{
   scaledMethod = 1,  // Scaled Method
   logMethod = 2      // Log Method
};

enum sfcn 
{
   ButterflyCurve = 1,   // Butterfly Curve
   TripleSineCurve = 2   // Triple Sine Curve
};

input int VolumePeriod = 20;                    // Period for volume average and std dev
input double ScaleFactor = 1.0;                 // Scaling factor (m)
input ifcn InputMethod = 1;      // Input method
input sfcn SmoothingMethod = 1; // Smoothing method

// Buffers
double OscillatorBuffer[];
double AvgVolBuffer[];
double StdVolBuffer[];

// Global variables
double AvgVol[], StdVol[];
int min_rates;

//+------------------------------------------------------------------+
//| Custom indicator initialization function                         |
//+------------------------------------------------------------------+
int OnInit()
{
   // Set indicator properties
   IndicatorSetString(INDICATOR_SHORTNAME, "Volume Bound(" + string(VolumePeriod) + ")");
   
   // Set indicator buffers
   SetIndexBuffer(0, OscillatorBuffer, INDICATOR_DATA);
   SetIndexBuffer(1, AvgVolBuffer, INDICATOR_CALCULATIONS);
   SetIndexBuffer(2, StdVolBuffer, INDICATOR_CALCULATIONS);
   
   
   // Set indicator range based on smoothing method
   if(SmoothingMethod == ButterflyCurve)
   {
      IndicatorSetDouble(INDICATOR_MINIMUM, -3.2);
      IndicatorSetDouble(INDICATOR_MAXIMUM, 3.2);
   }
   else
   {
      IndicatorSetDouble(INDICATOR_MINIMUM, -1.2);
      IndicatorSetDouble(INDICATOR_MAXIMUM, 1.2);
   }
   
   // Initialize arrays
   ArraySetAsSeries(OscillatorBuffer, true);
   ArraySetAsSeries(AvgVolBuffer, true);
   ArraySetAsSeries(StdVolBuffer, true);
   
   min_rates = VolumePeriod +1;
   
   return(INIT_SUCCEEDED);
}

//+------------------------------------------------------------------+
//| Custom indicator deinitialization function                       |
//+------------------------------------------------------------------+
void OnDeinit(const int reason)
{
   // Clean up if needed
}

//+------------------------------------------------------------------+
//| Custom indicator iteration function                              |
//+------------------------------------------------------------------+
int OnCalculate(const int rates_total,
                const int prev_calculated,
                const datetime &time[],
                const double &open[],
                const double &high[],
                const double &low[],
                const double &close[],
                const long &tick_volume[],
                const long &volume[],
                const int &spread[])
{
   // Set array directions
   ArraySetAsSeries(tick_volume, true);
  // ArraySetAsSeries(volume, true);
   
   // Check for minimum bars
   if(rates_total < min_rates)
      return(0);
   
   int limit;
   if(prev_calculated == 0)
   {
      limit = rates_total - VolumePeriod - 1;
      // Initialize buffers
      ArrayInitialize(OscillatorBuffer, 0.0);
      ArrayInitialize(AvgVolBuffer, 0.0);
      ArrayInitialize(StdVolBuffer, 0.0);
   }
   else
      limit = rates_total - prev_calculated+1;
   
   // Main calculation loop
   for(int i = limit; i >= 0; i--)
   {
      // Calculate volume statistics
      CalculateVolumeStats(i, rates_total, tick_volume);
      
      // Calculate scaled input t
      double t = CalculateInputT(i, tick_volume);
      
      // Apply smoothing function
      if(SmoothingMethod == ButterflyCurve)
      {
         OscillatorBuffer[i] = ButterflyMethod(t);
      }
      else // Triple sine method
      {
         OscillatorBuffer[i] = TripleSineMethod(t);
      }
   }
   
   return(rates_total);
}

//+------------------------------------------------------------------+
//| Calculate average volume and standard deviation                  |
//+------------------------------------------------------------------+
void CalculateVolumeStats(int pos, int rates_total, const long &tick_volume[])
{
   double sum = 0.0;
   double sumSq = 0.0;
   int count = 0;
   
   for(int i = pos; i < pos + VolumePeriod && i < rates_total; i++)
   {
      double volume_val = (double)tick_volume[i];
      sum += volume_val;
      sumSq += volume_val * volume_val;
      count++;
   }
   
   if(count > 0)
   {
      AvgVolBuffer[pos] = sum / count;
      double variance = (sumSq / count) - (AvgVolBuffer[pos] * AvgVolBuffer[pos]);
      StdVolBuffer[pos] = MathSqrt(MathMax(variance, 0));
   }
   else
   {
      AvgVolBuffer[pos] = 0;
      StdVolBuffer[pos] = 1;
   }
}

//+------------------------------------------------------------------+
//| Butterfly method                                                 |
//+------------------------------------------------------------------+
double ButterflyMethod(double t)
{
   double expr = MathExp(MathCos(t)) - 2 * MathCos(4 * t) - MathPow(MathSin(t / 12.0), 5);
   return MathSin(t) * expr;
}

//+------------------------------------------------------------------+
//| Triple sine method                                               |
//+------------------------------------------------------------------+
double TripleSineMethod(double t)
{
   double sint = MathSin(t);
   return sint * sint * sint;
}

//+------------------------------------------------------------------+
//| Calculate scaled input t                                         |
//+------------------------------------------------------------------+
double CalculateInputT(int pos, const long &tick_volume[])
{
   double currentVolume = (double)tick_volume[pos];
   double t = 0;
   
   if(InputMethod == scaledMethod)
   {
      if(StdVolBuffer[pos] != 0)
         t = ScaleFactor * (currentVolume - AvgVolBuffer[pos]) / StdVolBuffer[pos];
      else
         t = 0;
   }
   else // logMethod
   {
      if(currentVolume > 0)
         t = ScaleFactor * MathLog(currentVolume);
      else
         t = 0;
   }
   
   return t;
}