//+------------------------------------------------------------------+
//|                                                  VolumeEA.mq5    |
//+------------------------------------------------------------------+
#property copyright "Wamek EA"
#property link      "https://www.mql5.com/en/users/wamek/news"
#property version   "1.00"

#include <Trade\Trade.mqh>

// Input parameters
enum ifcn 
{
    scaledMethod = 1,   // scaledMethod
    logMethod = 2       // logMethod
};

enum sfcn
{
    ButterflyCurve = 1,     // ButterflyCurve
    TripleSineCurve = 2     // TripleSineCurve
};

input int VolumePeriod = 20;           // Period for volume average and std dev
input double ScaleFactor = 1.0;        // Scaling factor (m)
input ifcn InputMethod = 1;             
input sfcn SmoothingMethod = 1;        

// Trading parameters
input double LotSize = 0.1;            // Lot size for trading
input double TakeProfit = 1000;        // Take profit in points
input double StopLoss = 500;           // Stop loss in points
input double Threshold = 1.0;          // Signal threshold level
input int MagicNumber = 978;           // Magic number
input int Slippage = 3;                // Maximum slippage

// Global variables
double AvgVol, StdVol;
int CrossType = 0;

// CTrade object
CTrade trade;

//+------------------------------------------------------------------+
//| Expert initialization function                                   |
//+------------------------------------------------------------------+
int OnInit()
{
    Print("Volume EA initialized");
    
    // Configure trade object
    trade.SetExpertMagicNumber(MagicNumber);
    trade.SetDeviationInPoints(Slippage);
    
    return(INIT_SUCCEEDED);
}

//+------------------------------------------------------------------+
//| Expert deinitialization function                                 |
//+------------------------------------------------------------------+
void OnDeinit(const int reason)
{
    Print("Volume EA deinitialized");
}

//+------------------------------------------------------------------+
//| Expert tick function                                             |
//+------------------------------------------------------------------+
void OnTick()
{
    // Check for new bar
    if(IsNewBar())
    {
        // Calculate signal for current and previous bar
        double currentSignal = CalculateSignal(1);
        double previousSignal = CalculateSignal(2);
           
        // Check trading conditions
        CheckForEntry(currentSignal, previousSignal);
    }
}

//+------------------------------------------------------------------+
//| Check for new bar                                                |
//+------------------------------------------------------------------+
bool IsNewBar()
{
    static datetime lastBarTime = 0;
    datetime currentBarTime = iTime(_Symbol, _Period, 0);
    
    if(currentBarTime != lastBarTime)
    {
        lastBarTime = currentBarTime;
        return true;
    }
    return false;
}

//+------------------------------------------------------------------+
//| Calculate volume oscillator signal                              |
//+------------------------------------------------------------------+
double CalculateSignal(int shift)
{
    // Calculate volume statistics
    if(InputMethod == 1)
        CalculateVolumeStats(shift);
    
    // Calculate scaled input t
    double t = CalculateInputT(shift);
    
    // Apply smoothing function
    double signal = 0;
    if(SmoothingMethod == 1) // Butterfly method
    {
        signal = ButterflyMethod(t);
    }
    else // Triple sine method
    {
        signal = TripleSineMethod(t);
    }
    
    return signal;
}

//+------------------------------------------------------------------+
//| Calculate average volume and standard deviation                  |
//+------------------------------------------------------------------+
void CalculateVolumeStats(int shift)
{
    double sum = 0.0;
    double sumSq = 0.0;
    int count = 0;
    
    for(int i = shift; i < shift + VolumePeriod && i < Bars(_Symbol, _Period); i++)
    {
        long volume = iTickVolume(_Symbol, _Period, i);
        sum += volume;
        sumSq += volume * volume;
        count++;
    }
    
    if(count > 0)
    {
        AvgVol = sum / count;
        double variance = (sumSq / count) - (AvgVol * AvgVol);
        StdVol = MathSqrt(MathMax(variance, 0));
    }
    else
    {
        AvgVol = 0;
        StdVol = 1;
    }
}

//+------------------------------------------------------------------+
//| Butterfly method                                                 |
//+------------------------------------------------------------------+
double ButterflyMethod(double t)
{
    double expr = MathExp(MathCos(t)) - 2 * MathCos(4 * t) - MathPow(MathSin(t / 12.0), 5);
    return MathSin(t) * expr;
}

//+------------------------------------------------------------------+
//| Triple sine method                                               |
//+------------------------------------------------------------------+
double TripleSineMethod(double t)
{
    double sint = MathSin(t);
    return sint * sint * sint;
}

//+------------------------------------------------------------------+
//| Calculate scaled input t                                         |
//+------------------------------------------------------------------+
double CalculateInputT(int shift)
{
    long currentVolume = iTickVolume(_Symbol, _Period, shift);
    double t = 0;
    
    if(InputMethod == 1) // Input 1: t = m*(volume - AvgVol)/stdVol
    {
        if(StdVol != 0)
            t = ScaleFactor * (currentVolume - AvgVol) / StdVol;
        else
            t = 0;
    }
    else // Input 2: t = m* log(Volume)
    {
        if(currentVolume > 0)
            t = ScaleFactor * MathLog(currentVolume);
        else
            t = 0;
    }
    
    return t;
}

//+------------------------------------------------------------------+
//| Check for entry conditions                                       |
//+------------------------------------------------------------------+
void CheckForEntry(double currentSignal, double previousSignal)
{
    // Count current positions
    int buyPositions = CountPositions(POSITION_TYPE_BUY);
    int sellPositions = CountPositions(POSITION_TYPE_SELL);
    
    // Check for cross above/below threshold
    bool crossAbove = (previousSignal <= Threshold && currentSignal > Threshold);
    bool crossBelow = (previousSignal >= -Threshold && currentSignal < -Threshold);
    
    // Get previous candle information
    double open = iOpen(_Symbol, _Period, 1);
    double close = iClose(_Symbol, _Period, 1);
    bool bullishCandle = (close > open);
    bool bearishCandle = (close < open);
 
    // --------------Buy conditions--------------------------------
    if(buyPositions < 2 && sellPositions == 0)
    {
        // First buy entry: cross below -ve/above +ve threshold with bullish candle
        if(buyPositions == 0 && (crossBelow || crossAbove) && bullishCandle)
        {   
            if(OpenBuyOrder())
            {
                CrossType = crossAbove ? 1 : -1;  // record cross type value for cross detection
                Print("First BUY entry ");
            }
        }
        // Second buy entry: cross above +ve threshold 
        else if(buyPositions != 0 && crossAbove && CrossType == -1)
        {
            if(OpenBuyOrder())
            {
                CrossType = 1; // update cross type value for cross detection
                Print("Second BUY entry - Cross above");
            }
        }
        
        // Second buy entry: cross below -ve threshold 
        else if(buyPositions != 0 && crossBelow && CrossType == 1)
        {
            if(OpenBuyOrder())
            {
                CrossType = -1;
                Print("Second BUY entry - Cross below ");
            }
        }                
    }
    
    // ---------------Sell conditions---------------------
    if(sellPositions < 2 && buyPositions == 0)
    {
        // First sell entry: cross above positive threshold with bearish candle
        if(sellPositions == 0 && (crossAbove || crossBelow) && bearishCandle)
        {
            if(OpenSellOrder())
            {
                CrossType = crossAbove ? 1 : -1;  // record cross type value for cross detection
                Print("First SELL entry " );
            }
        }
        // Second sell entry: cross below -ve threshold 
        else if(sellPositions != 0 && crossBelow && CrossType == 1)
        {
            if(OpenSellOrder())
            {
                CrossType = -1;  // Update cross type value for cross detection
                Print("Second SELL entry - Cross below ");
            }
        }
        
         // Second sell entry: cross above +ve threshold
        else if(sellPositions != 0 && crossAbove && CrossType == -1)
        {
            if(OpenSellOrder())
            {
                CrossType = 1;
                Print("Second SELL entry - Cross above ");
            }
        }      
    }
}

//+------------------------------------------------------------------+
//| Open buy order                                                   |
//+------------------------------------------------------------------+
bool OpenBuyOrder()
{
    double price = SymbolInfoDouble(_Symbol, SYMBOL_ASK);
    double sl = 0, tp = 0;
    double point = SymbolInfoDouble(_Symbol, SYMBOL_POINT);

    if(StopLoss > 0)
        sl = price - StopLoss * point;
    if(TakeProfit > 0)
        tp = price + TakeProfit * point;

    // Use CTrade to open buy position
    if(trade.Buy(LotSize, _Symbol, price, sl, tp, "Volume EA Buy"))
    {
        Print("BUY order opened successfully");
        return true;
    }
    else
    {
        Print("Error opening BUY order: ", trade.ResultRetcode(), " - ", trade.ResultRetcodeDescription());
        return false;
    }
}

//+------------------------------------------------------------------+
//| Open sell order                                                  |
//+------------------------------------------------------------------+
bool OpenSellOrder()
{
    double price = SymbolInfoDouble(_Symbol, SYMBOL_BID);
    double sl = 0, tp = 0;
    double point = SymbolInfoDouble(_Symbol, SYMBOL_POINT);

    if(StopLoss > 0)
        sl = price + StopLoss * point;
    if(TakeProfit > 0)
        tp = price - TakeProfit * point;

    // Use CTrade to open sell position
    if(trade.Sell(LotSize, _Symbol, price, sl, tp, "Volume EA Sell"))
    {
        Print("SELL order opened successfully");
        return true;
    }
    else
    {
        Print("Error opening SELL order: ", trade.ResultRetcode(), " - ", trade.ResultRetcodeDescription());
        return false;
    }
}

//+------------------------------------------------------------------+
//| Count positions by type                                          |
//+------------------------------------------------------------------+
int CountPositions(ENUM_POSITION_TYPE type)
{
    int count = 0;
    for(int i = 0; i < PositionsTotal(); i++)
    {
        ulong ticket = PositionGetTicket(i);
        if(PositionGetString(POSITION_SYMBOL) == _Symbol && 
           PositionGetInteger(POSITION_MAGIC) == MagicNumber &&
           PositionGetInteger(POSITION_TYPE) == type)
        {
            count++;
        }
    }
    return count;
}