//+------------------------------------------------------------------+
//|                                           Spread Sensitivity.mq5 |
//|                        GIT under Copyright 2025, MetaQuotes Ltd. |
//|                     https://www.mql5.com/en/users/johnhlomohang/ |
//+------------------------------------------------------------------+
#property copyright "GIT under Copyright 2025, MetaQuotes Ltd."
#property link      "https://www.mql5.com/en/users/johnhlomohang/"
#property version   "1.00"
#property description "Multi-Symbol EA with Adaptive Spread Sensitivity"
#property description "Dynamically switches between symbols based on spread efficiency"

#include <Trade/Trade.mqh>
#include <Trade/PositionInfo.mqh>

//+------------------------------------------------------------------+
//| Input Parameters                                                |
//+------------------------------------------------------------------+
input string   TradePairs = "EURUSD,GBPUSD,XAUUSD,US100,BTCUSD"; // Trading Pairs (comma separated)
input double   RiskPerTrade = 0.01;          // Risk % per trade
input int      MagicNumber = 98765;          // Magic Number

// Spread Sensitivity Settings
input group    "=== Spread Filter Settings ==="
input double   MaxAbsoluteSpread = 10.0;     // Max absolute spread (pips)
input double   MaxSpreadATRRatio = 0.25;     // Max spread/ATR ratio
input bool     UseAdaptiveFilter = true;     // Enable adaptive filtering
input int      DisableTimeoutSec = 60;       // Disable symbol timeout (seconds)
input bool     EnableSpreadRanking = true;   // Enable symbol ranking by spread
input int      MaxActiveSymbols = 3;         // Maximum active symbols at once

// Trading Strategy Settings
input group    "=== Trading Strategy Settings ==="
input ENUM_TIMEFRAMES TradingTimeframe = PERIOD_M5;  // Trading timeframe
input int      EMA_Fast_Period = 9;          // Fast EMA period
input int      EMA_Slow_Period = 21;         // Slow EMA period
input int      RSI_Period = 14;              // RSI period
input double   RSI_Overbought = 70;          // RSI overbought level
input double   RSI_Oversold = 30;            // RSI oversold level
input int      StopLoss_Pips = 30;           // Stop Loss in pips
input int      TakeProfit_Pips = 60;         // Take Profit in pips
input int      MaxOpenPositions = 1;         // Max positions per symbol
input int      TradeCooldownSeconds = 300;   // Cooldown between trades (seconds)

// ATR Settings for adaptive SL/TP
input group    "=== ATR Settings (Optional) ==="
input bool     UseATR_SL_TP = false;         // Use ATR for dynamic SL/TP
input double   ATR_SL_Multiplier = 1.5;      // ATR multiplier for SL
input double   ATR_TP_Multiplier = 2.0;      // ATR multiplier for TP

// Dashboard Settings
input group    "=== Dashboard Settings ==="
input bool     ShowDashboard = true;         // Show dashboard on chart
input color    DashboardBGColor = clrBlack;  // Dashboard background color
input color    DashboardTextColor = clrWhite;// Dashboard text color
input int      DashboardX = 20;              // Dashboard X position
input int      DashboardY = 20;              // Dashboard Y position
input int      FontSize = 8;                 // Dashboard font size

//+------------------------------------------------------------------+
//| Global Variables                                                 |
//+------------------------------------------------------------------+
string   SymbolList[];
int      TotalPairs;
CTrade   Trade;
CPositionInfo PositionInfo;
datetime LastDashboardUpdate = 0;

// Spread monitoring structure
struct SpreadData
{
   string      symbol;
   double      spreadInPips;
   double      atrValue;
   double      spreadATRRatio;
   double      spreadScore;
   datetime    disabledUntil;
   bool        isTradeable;
   bool        isActive;
   datetime    lastTradeTime;
   int         tradeAttempts;
   int         successfulTrades;
   color       statusColor;
};

SpreadData spreadData[];

// Dashboard messages
string DashboardMessages[10];

//+------------------------------------------------------------------+
//| Expert initialization function                                   |
//+------------------------------------------------------------------+
int OnInit()
{
   // Split trading pairs
   SplitString(TradePairs, ",", SymbolList);
   TotalPairs = ArraySize(SymbolList);
   
   if(TotalPairs == 0)
   {
      Print("Error: No symbols specified");
      return INIT_FAILED;
   }
   
   // Initialize spread data array
   ArrayResize(spreadData, TotalPairs);
   
   // Initialize dashboard messages
   for(int i = 0; i < 10; i++) DashboardMessages[i] = "";
   
   // Initialize each symbol
   for(int i = 0; i < TotalPairs; i++)
   {
      string symbol = SymbolList[i];
      
      // Validate symbol
      if(!SymbolInfoInteger(symbol, SYMBOL_TRADE_MODE))
      {
         Print("Warning: Symbol ", symbol, " is not available for trading");
         continue;
      }
      
      // Initialize spread data
      spreadData[i].symbol = symbol;
      spreadData[i].spreadInPips = 0;
      spreadData[i].atrValue = 0;
      spreadData[i].spreadATRRatio = 0;
      spreadData[i].spreadScore = 0;
      spreadData[i].disabledUntil = 0;
      spreadData[i].isTradeable = true;
      spreadData[i].isActive = true;
      spreadData[i].lastTradeTime = 0;
      spreadData[i].tradeAttempts = 0;
      spreadData[i].successfulTrades = 0;
      spreadData[i].statusColor = clrGreen;
      
      // Subscribe to symbol
      SymbolSelect(symbol, true);
   }
   
   // Set trade parameters
   Trade.SetExpertMagicNumber(MagicNumber);
   Trade.SetDeviationInPoints(10);
   
   // Initialize dashboard
   if(ShowDashboard) CreateDashboard();
   
   AddDashboardMessage("EA Initialized with " + IntegerToString(TotalPairs) + " symbols");
   Print("EA Initialized. Total pairs: ", TotalPairs);
   
   return INIT_SUCCEEDED;
}

//+------------------------------------------------------------------+
//| Expert deinitialization function                                 |
//+------------------------------------------------------------------+
void OnDeinit(const int reason)
{
   // Remove dashboard objects
   if(ShowDashboard) RemoveDashboard();
   
   // Print statistics
   Print("=== Trading Statistics ===");
   for(int i = 0; i < TotalPairs; i++)
   {
      Print(spreadData[i].symbol, ": ", 
            spreadData[i].tradeAttempts, " attempts, ", 
            spreadData[i].successfulTrades, " successful trades");
   }
   
   Print("EA Deinitialized");
}

//+------------------------------------------------------------------+
//| Expert tick function                                             |
//+------------------------------------------------------------------+
void OnTick()
{
   static int tickCounter = 0;
   tickCounter++;
   
   // Process one symbol per tick (prevents overloading)
   int symbolIndex = tickCounter % TotalPairs;
   string symbol = spreadData[symbolIndex].symbol;
   
   // Update spread data
   UpdateSpreadData(symbolIndex);
   
   // Check if symbol is tradeable
   if(!spreadData[symbolIndex].isTradeable || !spreadData[symbolIndex].isActive) 
      return;
   
   // Check cooldown period
   if(TimeCurrent() - spreadData[symbolIndex].lastTradeTime < TradeCooldownSeconds)
      return;
   
   // Execute trading logic
   ExecuteTradingLogic(symbolIndex);
   
   // Update dashboard every 10 ticks
   if(ShowDashboard && (tickCounter % 10 == 0) && (TimeCurrent() - LastDashboardUpdate >= 1))
   {
      UpdateDashboard();
      LastDashboardUpdate = TimeCurrent();
   }
}

//+------------------------------------------------------------------+
//| Timer function for spread updates                                |
//+------------------------------------------------------------------+
void OnTimer()
{
   // Update all spread data and rank symbols
   UpdateAllSpreadData();
   if(EnableSpreadRanking) RankSymbolsBySpread();
}

//+------------------------------------------------------------------+
//| Update spread data for all symbols                               |
//+------------------------------------------------------------------+
void UpdateAllSpreadData()
{
   for(int i = 0; i < TotalPairs; i++)
   {
      UpdateSpreadData(i);
   }
}

//+------------------------------------------------------------------+
//| Update spread data for specific symbol                           |
//+------------------------------------------------------------------+
void UpdateSpreadData(int index)
{
   string symbol = spreadData[index].symbol;
   
   // Get current bid and ask
   double bid = SymbolInfoDouble(symbol, SYMBOL_BID);
   double ask = SymbolInfoDouble(symbol, SYMBOL_ASK);
   
   if(bid == 0 || ask == 0) return;
   
   // Calculate spread in pips
   double point = SymbolInfoDouble(symbol, SYMBOL_POINT);
   double spreadPoints = (ask - bid) / point;
   spreadData[index].spreadInPips = NormalizeDouble(spreadPoints / 10, 1);
   
   // Calculate ATR for spread ratio
   spreadData[index].atrValue = CalculateATR(symbol, PERIOD_H1, 14);
   
   // Calculate spread/ATR ratio
   if(spreadData[index].atrValue > 0)
   {
      spreadData[index].spreadATRRatio = NormalizeDouble(spreadData[index].spreadInPips / spreadData[index].atrValue, 3);
   }
   
   // Evaluate tradeability
   EvaluateTradeability(index);
}

//+------------------------------------------------------------------+
//| Evaluate if symbol is tradeable based on spread                  |
//+------------------------------------------------------------------+
void EvaluateTradeability(int index)
{
   // Check if symbol is in timeout
   if(TimeCurrent() < spreadData[index].disabledUntil)
   {
      spreadData[index].isTradeable = false;
      spreadData[index].statusColor = clrOrange;
      return;
   }
   
   // Check absolute spread limit
   if(spreadData[index].spreadInPips > MaxAbsoluteSpread)
   {
      spreadData[index].isTradeable = false;
      spreadData[index].disabledUntil = TimeCurrent() + DisableTimeoutSec;
      AddDashboardMessage(spreadData[index].symbol + " disabled: High spread " + 
                          DoubleToString(spreadData[index].spreadInPips, 1));
      spreadData[index].statusColor = clrRed;
      return;
   }
   
   // Check ATR ratio if adaptive filtering is enabled
   if(UseAdaptiveFilter && spreadData[index].spreadATRRatio > MaxSpreadATRRatio)
   {
      spreadData[index].isTradeable = false;
      spreadData[index].statusColor = clrRed;
      return;
   }
   
   // All checks passed
   spreadData[index].isTradeable = true;
   spreadData[index].statusColor = clrGreen;
}

//+------------------------------------------------------------------+
//| Rank symbols by spread efficiency                                |
//+------------------------------------------------------------------+
void RankSymbolsBySpread()
{
   // Calculate spread score for each symbol
   for(int i = 0; i < TotalPairs; i++)
   {
      // Lower spread = better score
      double spreadComponent = 1.0 / (1.0 + spreadData[i].spreadInPips);
      
      // Lower ATR ratio = better score
      double atrComponent = 1.0 / (1.0 + spreadData[i].spreadATRRatio);
      
      // Combine components with weights
      spreadData[i].spreadScore = (0.6 * spreadComponent) + (0.4 * atrComponent);
   }
   
   // Simple bubble sort by score
   for(int i = 0; i < TotalPairs - 1; i++)
   {
      for(int j = i + 1; j < TotalPairs; j++)
      {
         if(spreadData[j].spreadScore > spreadData[i].spreadScore)
         {
            SpreadData temp = spreadData[i];
            spreadData[i] = spreadData[j];
            spreadData[j] = temp;
         }
      }
   }
   
   // Activate top N symbols
   for(int i = 0; i < TotalPairs; i++)
   {
      spreadData[i].isActive = (i < MaxActiveSymbols);
   }
}

//+------------------------------------------------------------------+
//| Execute trading logic for symbol                                 |
//+------------------------------------------------------------------+
void ExecuteTradingLogic(int index)
{
   string symbol = spreadData[index].symbol;
   
   // Check if symbol already has max positions
   if(CountOpenPositions(symbol) >= MaxOpenPositions) return;
   
   // Get indicator handles
   int handleEmaFast = iMA(symbol, TradingTimeframe, EMA_Fast_Period, 0, MODE_EMA, PRICE_CLOSE);
   int handleEmaSlow = iMA(symbol, TradingTimeframe, EMA_Slow_Period, 0, MODE_EMA, PRICE_CLOSE);
   int handleRSI = iRSI(symbol, TradingTimeframe, RSI_Period, PRICE_CLOSE);
   
   if(handleEmaFast == INVALID_HANDLE || handleEmaSlow == INVALID_HANDLE || handleRSI == INVALID_HANDLE)
   {
      Print("Error: Failed to create indicator handles for ", symbol);
      return;
   }
   
   // Get indicator values
   double emaFast[1], emaSlow[1], rsi[1];
   
   if(CopyBuffer(handleEmaFast, 0, 0, 1, emaFast) < 1) 
   {
      IndicatorRelease(handleEmaFast);
      IndicatorRelease(handleEmaSlow);
      IndicatorRelease(handleRSI);
      return;
   }
   
   if(CopyBuffer(handleEmaSlow, 0, 0, 1, emaSlow) < 1) 
   {
      IndicatorRelease(handleEmaFast);
      IndicatorRelease(handleEmaSlow);
      IndicatorRelease(handleRSI);
      return;
   }
   
   if(CopyBuffer(handleRSI, 0, 0, 1, rsi) < 1) 
   {
      IndicatorRelease(handleEmaFast);
      IndicatorRelease(handleEmaSlow);
      IndicatorRelease(handleRSI);
      return;
   }
   
   // Release indicator handles
   IndicatorRelease(handleEmaFast);
   IndicatorRelease(handleEmaSlow);
   IndicatorRelease(handleRSI);
   
   double emaF = emaFast[0];
   double emaS = emaSlow[0];
   double rsiV = rsi[0];
   
   // Generate trading signals
   if(emaF > emaS && rsiV < RSI_Oversold) // Buy signal: Fast EMA above Slow EMA and RSI oversold
   {
      spreadData[index].tradeAttempts++;
      double lotSize = CalculateLotSize(symbol);
      if(lotSize > 0)
      {
         if(ExecuteTrade(ORDER_TYPE_BUY, symbol, lotSize, StopLoss_Pips, TakeProfit_Pips))
         {
            spreadData[index].lastTradeTime = TimeCurrent();
            spreadData[index].successfulTrades++;
            AddDashboardMessage("BUY " + symbol + " | Spread: " + DoubleToString(spreadData[index].spreadInPips, 1));
         }
      }
   }
   else if(emaF < emaS && rsiV > RSI_Overbought) // Sell signal: Fast EMA below Slow EMA and RSI overbought
   {
      spreadData[index].tradeAttempts++;
      double lotSize = CalculateLotSize(symbol);
      if(lotSize > 0)
      {
         if(ExecuteTrade(ORDER_TYPE_SELL, symbol, lotSize, StopLoss_Pips, TakeProfit_Pips))
         {
            spreadData[index].lastTradeTime = TimeCurrent();
            spreadData[index].successfulTrades++;
            AddDashboardMessage("SELL " + symbol + " | Spread: " + DoubleToString(spreadData[index].spreadInPips, 1));
         }
      }
   }
}

//+------------------------------------------------------------------+
//| Execute trade with CTrade                                        |
//+------------------------------------------------------------------+
bool ExecuteTrade(ENUM_ORDER_TYPE tradeType, string symbol, double lotSize, int stopLossPips, int takeProfitPips)
{
   // Get symbol info
   double point = SymbolInfoDouble(symbol, SYMBOL_POINT);
   int digits = (int)SymbolInfoInteger(symbol, SYMBOL_DIGITS);
   
   // Get current price
   double ask = SymbolInfoDouble(symbol, SYMBOL_ASK);
   double bid = SymbolInfoDouble(symbol, SYMBOL_BID);
   double price = (tradeType == ORDER_TYPE_BUY) ? ask : bid;
   
   // Calculate pip size for different instruments
   double pipSize = CalculatePipSize(symbol, digits, point);
   
   // Use ATR for dynamic SL/TP if enabled
   double slDistance = 0, tpDistance = 0;
   
   if(UseATR_SL_TP)
   {
      double atr = iATR(symbol, TradingTimeframe, 14);
      if(atr > 0)
      {
         slDistance = atr * ATR_SL_Multiplier;
         tpDistance = atr * ATR_TP_Multiplier;
      }
   }
   
   // Fallback to fixed pips if ATR not used or failed
   if(slDistance == 0) slDistance = stopLossPips * pipSize;
   if(tpDistance == 0) tpDistance = takeProfitPips * pipSize;
   
   // Calculate SL and TP prices
   double sl = 0, tp = 0;
   
   if(slDistance > 0)
   {
      sl = (tradeType == ORDER_TYPE_BUY) ? price - slDistance : price + slDistance;
      sl = NormalizeDouble(sl, digits);
   }
   
   if(tpDistance > 0)
   {
      tp = (tradeType == ORDER_TYPE_BUY) ? price + tpDistance : price - tpDistance;
      tp = NormalizeDouble(tp, digits);
   }
   
   // Execute trade with CTrade
   bool success = false;
   
   if(tradeType == ORDER_TYPE_BUY)
   {
      success = Trade.Buy(lotSize, symbol, price, sl, tp, "Adaptive Spread EA");
   }
   else if(tradeType == ORDER_TYPE_SELL)
   {
      success = Trade.Sell(lotSize, symbol, price, sl, tp, "Adaptive Spread EA");
   }
   
   if(success)
   {
      PrintFormat("%s %s | Lot: %.2f | Price: %.5f | SL: %.5f | TP: %.5f | Spread: %.1f",
                  EnumToString(tradeType), symbol, lotSize, price, sl, tp, 
                  SymbolInfoDouble(symbol, SYMBOL_ASK) - SymbolInfoDouble(symbol, SYMBOL_BID));
      return true;
   }
   else
   {
      PrintFormat("Failed to open %s on %s | Error: %d", 
                  EnumToString(tradeType), symbol, GetLastError());
      return false;
   }
}

//+------------------------------------------------------------------+
//| Calculate pip size for different instruments                     |
//+------------------------------------------------------------------+
double CalculatePipSize(string symbol, int digits, double point)
{
   // Detect pip size automatically
   if(StringFind(symbol, "JPY") != -1)              // JPY pairs
      return (digits == 3) ? point * 10 : point;
   else if(StringFind(symbol, "XAU") != -1 || StringFind(symbol, "GOLD") != -1)  // Metals
      return 0.10;
   else if(StringFind(symbol, "BTC") != -1 || StringFind(symbol, "ETH") != -1)   // Cryptos
      return point * 100.0;
   else if(StringFind(symbol, "US") != -1 && digits <= 2)                         // Indices
      return point;
   else
      return (digits == 3 || digits == 5) ? point * 10 : point;                   // Default Forex
}

//+------------------------------------------------------------------+
//| Calculate position size based on risk                            |
//+------------------------------------------------------------------+
double CalculateLotSize(string symbol)
{
   double accountBalance = AccountInfoDouble(ACCOUNT_BALANCE);
   double minLot = SymbolInfoDouble(symbol, SYMBOL_VOLUME_MIN);
   double maxLot = SymbolInfoDouble(symbol, SYMBOL_VOLUME_MAX);
   double lotStep = SymbolInfoDouble(symbol, SYMBOL_VOLUME_STEP);
   
   if(accountBalance <= 0) return minLot;
   
   // Simple lot calculation based on risk percentage
   double riskAmount = accountBalance * (RiskPerTrade / 100.0);
   double tickValue = SymbolInfoDouble(symbol, SYMBOL_TRADE_TICK_VALUE);
   
   if(tickValue <= 0)
   {
      // Fallback calculation
      double contractSize = SymbolInfoDouble(symbol, SYMBOL_TRADE_CONTRACT_SIZE);
      tickValue = (SymbolInfoDouble(symbol, SYMBOL_TRADE_TICK_SIZE) * contractSize) / SymbolInfoDouble(symbol, SYMBOL_POINT);
   }
   
   double pipSize = CalculatePipSize(symbol, (int)SymbolInfoInteger(symbol, SYMBOL_DIGITS), 
                                    SymbolInfoDouble(symbol, SYMBOL_POINT));
   double stopLossPoints = StopLoss_Pips * 10; // Convert pips to points
  
   if(stopLossPoints > 0 && tickValue > 0)
   {
      double lotSize = riskAmount / (stopLossPoints * tickValue);
      lotSize = NormalizeDouble(lotSize, 2);
      
      // Apply lot size limits
      lotSize = MathMax(lotSize, minLot);
      lotSize = MathMin(lotSize, maxLot);
      lotSize = MathRound(lotSize / lotStep) * lotStep;
      
      return lotSize;
   }
   
   return minLot;
}

//+------------------------------------------------------------------+
//| Count open positions for a symbol                                |
//+------------------------------------------------------------------+
int CountOpenPositions(string symbol)
{
   int count = 0;
   
   for(int i = PositionsTotal() - 1; i >= 0; i--)
   {
      if(PositionGetSymbol(i) == symbol && PositionGetInteger(POSITION_MAGIC) == MagicNumber)
      {
         count++;
      }
   }
   
   return count;
}

//+------------------------------------------------------------------+
//| Calculate ATR value                                              |
//+------------------------------------------------------------------+
double CalculateATR(string symbol, ENUM_TIMEFRAMES timeframe, int period)
{
   int atrHandle = iATR(symbol, timeframe, period);
   
   if(atrHandle == INVALID_HANDLE) return 0;
   
   double atrBuffer[];
   ArraySetAsSeries(atrBuffer, true);
   
   if(CopyBuffer(atrHandle, 0, 0, 1, atrBuffer) < 1)
   {
      IndicatorRelease(atrHandle);
      return 0;
   }
   
   IndicatorRelease(atrHandle);
   return atrBuffer[0];
}

//+------------------------------------------------------------------+
//| Dashboard Functions                                              |
//+------------------------------------------------------------------+
void CreateDashboard()
{
   // Create main dashboard background
   ObjectCreate(0, "Dashboard_BG", OBJ_RECTANGLE_LABEL, 0, 0, 0);
   ObjectSetInteger(0, "Dashboard_BG", OBJPROP_XDISTANCE, DashboardX);
   ObjectSetInteger(0, "Dashboard_BG", OBJPROP_YDISTANCE, DashboardY);
   ObjectSetInteger(0, "Dashboard_BG", OBJPROP_XSIZE, 400);
   ObjectSetInteger(0, "Dashboard_BG", OBJPROP_YSIZE, 350);
   ObjectSetInteger(0, "Dashboard_BG", OBJPROP_BGCOLOR, DashboardBGColor);
   ObjectSetInteger(0, "Dashboard_BG", OBJPROP_BORDER_TYPE, BORDER_FLAT);
   ObjectSetInteger(0, "Dashboard_BG", OBJPROP_BORDER_COLOR, clrGray);
   ObjectSetInteger(0, "Dashboard_BG", OBJPROP_CORNER, CORNER_LEFT_UPPER);
   ObjectSetInteger(0, "Dashboard_BG", OBJPROP_HIDDEN, true);
   
   // Create title
   ObjectCreate(0, "Dashboard_Title", OBJ_LABEL, 0, 0, 0);
   ObjectSetString(0, "Dashboard_Title", OBJPROP_TEXT, "=== Adaptive Spread EA ===");
   ObjectSetInteger(0, "Dashboard_Title", OBJPROP_XDISTANCE, DashboardX + 10);
   ObjectSetInteger(0, "Dashboard_Title", OBJPROP_YDISTANCE, DashboardY + 10);
   ObjectSetInteger(0, "Dashboard_Title", OBJPROP_COLOR, clrYellow);
   ObjectSetInteger(0, "Dashboard_Title", OBJPROP_FONTSIZE, FontSize + 2);
   ObjectSetString(0, "Dashboard_Title", OBJPROP_FONT, "Consolas");
   ObjectSetInteger(0, "Dashboard_Title", OBJPROP_CORNER, CORNER_LEFT_UPPER);
   
   // Create ranking header
   ObjectCreate(0, "Dashboard_RankHeader", OBJ_LABEL, 0, 0, 0);
   ObjectSetString(0, "Dashboard_RankHeader", OBJPROP_TEXT, "=== Symbol Ranking ===");
   ObjectSetInteger(0, "Dashboard_RankHeader", OBJPROP_XDISTANCE, DashboardX + 10);
   ObjectSetInteger(0, "Dashboard_RankHeader", OBJPROP_YDISTANCE, DashboardY + 35);
   ObjectSetInteger(0, "Dashboard_RankHeader", OBJPROP_COLOR, clrYellow);
   ObjectSetInteger(0, "Dashboard_RankHeader", OBJPROP_FONTSIZE, FontSize);
   ObjectSetString(0, "Dashboard_RankHeader", OBJPROP_FONT, "Consolas");
   ObjectSetInteger(0, "Dashboard_RankHeader", OBJPROP_CORNER, CORNER_LEFT_UPPER);
   
   // Create symbol ranking labels
   for(int i = 0; i < MaxActiveSymbols + 2; i++)
   {
      string objName = "Dashboard_Rank_" + IntegerToString(i);
      ObjectCreate(0, objName, OBJ_LABEL, 0, 0, 0);
      ObjectSetInteger(0, objName, OBJPROP_XDISTANCE, DashboardX + 10);
      ObjectSetInteger(0, objName, OBJPROP_YDISTANCE, DashboardY + 55 + (i * 20));
      ObjectSetInteger(0, objName, OBJPROP_COLOR, DashboardTextColor);
      ObjectSetInteger(0, objName, OBJPROP_FONTSIZE, FontSize);
      ObjectSetString(0, objName, OBJPROP_FONT, "Consolas");
      ObjectSetInteger(0, objName, OBJPROP_CORNER, CORNER_LEFT_UPPER);
   }
   
   // Create messages header
   ObjectCreate(0, "Dashboard_MsgHeader", OBJ_LABEL, 0, 0, 0);
   ObjectSetString(0, "Dashboard_MsgHeader", OBJPROP_TEXT, "=== Messages ===");
   ObjectSetInteger(0, "Dashboard_MsgHeader", OBJPROP_XDISTANCE, DashboardX + 10);
   ObjectSetInteger(0, "Dashboard_MsgHeader", OBJPROP_YDISTANCE, DashboardY + 180);
   ObjectSetInteger(0, "Dashboard_MsgHeader", OBJPROP_COLOR, clrYellow);
   ObjectSetInteger(0, "Dashboard_MsgHeader", OBJPROP_FONTSIZE, FontSize);
   ObjectSetString(0, "Dashboard_MsgHeader", OBJPROP_FONT, "Consolas");
   ObjectSetInteger(0, "Dashboard_MsgHeader", OBJPROP_CORNER, CORNER_LEFT_UPPER);
   
   // Create message labels
   for(int i = 0; i < 10; i++)
   {
      string objName = "Dashboard_Msg_" + IntegerToString(i);
      ObjectCreate(0, objName, OBJ_LABEL, 0, 0, 0);
      ObjectSetInteger(0, objName, OBJPROP_XDISTANCE, DashboardX + 10);
      ObjectSetInteger(0, objName, OBJPROP_YDISTANCE, DashboardY + 200 + (i * 15));
      ObjectSetInteger(0, objName, OBJPROP_COLOR, DashboardTextColor);
      ObjectSetInteger(0, objName, OBJPROP_FONTSIZE, FontSize);
      ObjectSetString(0, objName, OBJPROP_FONT, "Consolas");
      ObjectSetInteger(0, objName, OBJPROP_CORNER, CORNER_LEFT_UPPER);
   }
}

void UpdateDashboard()
{
   if(!ShowDashboard) return;
   
   // Update ranking
   for(int i = 0; i < MathMin(MaxActiveSymbols + 2, TotalPairs); i++)
   {
      string objName = "Dashboard_Rank_" + IntegerToString(i);
      string status = spreadData[i].isActive ? "Yes" : "No";
      
      string text = IntegerToString(i+1) + ". " + spreadData[i].symbol + 
                    " | Spread: " + DoubleToString(spreadData[i].spreadInPips, 1) + 
                    " | Score: " + DoubleToString(spreadData[i].spreadScore, 3) + 
                    " | Active: " + status;
      
      ObjectSetString(0, objName, OBJPROP_TEXT, text);
      ObjectSetInteger(0, objName, OBJPROP_COLOR, spreadData[i].statusColor);
   }
   
   // Update messages
   for(int i = 0; i < 10; i++)
   {
      string objName = "Dashboard_Msg_" + IntegerToString(i);
      ObjectSetString(0, objName, OBJPROP_TEXT, DashboardMessages[i]);
   }
}

void RemoveDashboard()
{
   ObjectsDeleteAll(0, "Dashboard_");
}

void AddDashboardMessage(string message)
{
   // Shift messages up
   for(int i = 9; i > 0; i--)
   {
      DashboardMessages[i] = DashboardMessages[i-1];
   }
   
   // Add new message at the beginning
   DashboardMessages[0] = TimeToString(TimeCurrent(), TIME_SECONDS) + ": " + message;
}

//+------------------------------------------------------------------+
//| Helper Functions                                                 |
//+------------------------------------------------------------------+
void SplitString(const string inputString, const string separator, string& result[])
{
   string sep = separator;
   ushort u_sep = StringGetCharacter(sep, 0);
   int items = StringSplit(inputString, u_sep, result);
}

//+------------------------------------------------------------------+
//| Chart Event Handler                                              |
//+------------------------------------------------------------------+
void OnChartEvent(const int id, const long& lparam, const double& dparam, const string& sparam)
{
   if(ShowDashboard && (id == CHARTEVENT_CHART_CHANGE || id == CHARTEVENT_CLICK))
   {
      UpdateDashboard();
   }
}
//+------------------------------------------------------------------+
