//+------------------------------------------------------------------+
//|                                                  FlowerEA.mq5    |
//+------------------------------------------------------------------+
#property copyright "Wamek EA"
#property link      "https://www.mql5.com/en/users/wamek/news"
#property version   "1.00"

#include <Trade\Trade.mqh>

// Trading parameters
input double LotSize = 0.1;
input int StopLoss = 500;      // Stop Loss in points
input int TakeProfit = 1000;   // Take Profit in points
input int MinBarsBtwEntries = 10;  // minimum bars between entries
input int MagicNumber = 977;

// Flower parameters
input int n = 5;   // flower n
input int d = 1;    // flower d
input double k = 1.0;       // sensitivity multiplier for q
input int MAperiod = 14;   // MA period
input int ATRperiod = 14;   // ATR period
input bool Use_X_Component = true; // Use X component if true, Y component if false

// Global variables
double prevFVI = 0;
double curFVI = 0;
int barsSinceLastTrade = 0;

// Handles for indicators
int maHandle;
int atrHandle;
int aoHandle;

// CTrade object
CTrade trade;

//+------------------------------------------------------------------+
//| Expert initialization function                                   |
//+------------------------------------------------------------------+
int OnInit()
{
    Print("Flower EA initialized");
    
    // Initialize indicator handles
    maHandle = iMA(_Symbol, _Period, MAperiod, 0, MODE_SMA, PRICE_CLOSE);
    atrHandle = iATR(_Symbol, _Period, ATRperiod);
    aoHandle = iAO(_Symbol, _Period);
    
    // Check if handles are valid
    if(maHandle == INVALID_HANDLE || atrHandle == INVALID_HANDLE || aoHandle == INVALID_HANDLE)
    {
        Print("Error creating indicator handles");
        return(INIT_FAILED);
    }
    
    // Configure trade object
    trade.SetExpertMagicNumber(MagicNumber);
    trade.SetDeviationInPoints(10);
    
    return(INIT_SUCCEEDED);
}

//+------------------------------------------------------------------+
//| Expert deinitialization function                                 |
//+------------------------------------------------------------------+
void OnDeinit(const int reason)
{
    Print("Flower EA deinitialized");
    
    // Release indicator handles
    if(maHandle != INVALID_HANDLE) IndicatorRelease(maHandle);
    if(atrHandle != INVALID_HANDLE) IndicatorRelease(atrHandle);
    if(aoHandle != INVALID_HANDLE) IndicatorRelease(aoHandle);
}

//+------------------------------------------------------------------+
//| Expert tick function                                             |
//+------------------------------------------------------------------+
void OnTick()
{
    // Check for new bar
    if(IsNewBar())
    {
        barsSinceLastTrade++;
        CalculateFVI();
        CheckForEntry();
    }
}

//+------------------------------------------------------------------+
//| Check for new bar                                                |
//+------------------------------------------------------------------+
bool IsNewBar()
{
    static datetime lastBarTime = 0;
    datetime currentBarTime = iTime(_Symbol, _Period, 0);
    
    if(currentBarTime != lastBarTime)
    {
        lastBarTime = currentBarTime;
        return true;
    }
    return false;
}

//+------------------------------------------------------------------+
//| Update Flower Value Indicator                                    |
//+------------------------------------------------------------------+
void CalculateFVI()
{
    prevFVI = curFVI;

    // Get indicator values
    double ma[1], atr[1], ao[1];
    double close = iClose(_Symbol, _Period, 1);
    
    // Copy indicator values
    if(CopyBuffer(maHandle, 0, 1, 1, ma) < 1 || 
       CopyBuffer(atrHandle, 0, 1, 1, atr) < 1)
    {
        Print("Error copying indicator buffers");
        return;
    }

    // Calculate q
    double q = k * (close - ma[0]) / (atr[0] > 0 ? atr[0] : 1);

    // Calculate r(q)
    double r = MathSin(n * q / (2 * d));

    // Calculate FVI based on selected component
    if(Use_X_Component)
        curFVI = r * MathCos(q); // X component
    else
        curFVI = r * MathSin(q); // Y component
}

//+------------------------------------------------------------------+
//| Check for entry conditions                                       |
//+------------------------------------------------------------------+
void CheckForEntry()
{
    // Check minimum bars between entries
    if(barsSinceLastTrade < MinBarsBtwEntries)
        return;

    // Get Awesome Oscillator value
    double ao[1];
    if(CopyBuffer(aoHandle, 0, 1, 1, ao) < 1)
    {
        Print("Error copying AO buffer");
        return;
    }

    // Check for buy conditions
    if(ao[0] > 0 && prevFVI > -0.7 && curFVI < -0.7)
    {
        if(CountPositions(POSITION_TYPE_BUY) == 0)
        {
            OpenBuyOrder();
            barsSinceLastTrade = 0;
        }
    }

    // Check for sell conditions
    if(ao[0] < 0 && prevFVI < 0.7 && curFVI > 0.7)
    {
        if(CountPositions(POSITION_TYPE_SELL) == 0)
        {
            OpenSellOrder();
            barsSinceLastTrade = 0;
        }
    }
}

//+------------------------------------------------------------------+
//| Open buy order                                                   |
//+------------------------------------------------------------------+
void OpenBuyOrder()
{
    double price = SymbolInfoDouble(_Symbol, SYMBOL_ASK);
    double sl = 0, tp = 0;
    double point = SymbolInfoDouble(_Symbol, SYMBOL_POINT);

    if(StopLoss > 0)
        sl = price - StopLoss * point;
    if(TakeProfit > 0)
        tp = price + TakeProfit * point;

    // Use CTrade to open buy position
    if(trade.Buy(LotSize, _Symbol, price, sl, tp, "Flower EA Buy"))
    {
        Print("Buy order opened successfully");
    }
    else
    {
        Print("Error opening buy order: ", trade.ResultRetcode(), " - ", trade.ResultRetcodeDescription());
    }
}

//+------------------------------------------------------------------+
//| Open sell order                                                  |
//+------------------------------------------------------------------+
void OpenSellOrder()
{
    double price = SymbolInfoDouble(_Symbol, SYMBOL_BID);
    double sl = 0, tp = 0;
    double point = SymbolInfoDouble(_Symbol, SYMBOL_POINT);

    if(StopLoss > 0)
        sl = price + StopLoss * point;
    if(TakeProfit > 0)
        tp = price - TakeProfit * point;

    // Use CTrade to open sell position
    if(trade.Sell(LotSize, _Symbol, price, sl, tp, "Flower EA Sell"))
    {
        Print("Sell order opened successfully");
    }
    else
    {
        Print("Error opening sell order: ", trade.ResultRetcode(), " - ", trade.ResultRetcodeDescription());
    }
}

//+------------------------------------------------------------------+
//| Count positions by type                                          |
//+------------------------------------------------------------------+
int CountPositions(ENUM_POSITION_TYPE type)
{
    int count = 0;
    for(int i = 0; i < PositionsTotal(); i++)
    {
        ulong ticket = PositionGetTicket(i);
        if(PositionGetString(POSITION_SYMBOL) == _Symbol && 
           PositionGetInteger(POSITION_MAGIC) == MagicNumber &&
           PositionGetInteger(POSITION_TYPE) == type)
        {
            count++;
        }
    }
    return count;
}

////+------------------------------------------------------------------+
////| Count total positions                                            |
////+------------------------------------------------------------------+
//int CountTotalPositions()
//{
//    int count = 0;
//    for(int i = 0; i < PositionsTotal(); i++)
//    {
//        ulong ticket = PositionGetTicket(i);
//        if(PositionGetString(POSITION_SYMBOL) == _Symbol && 
//           PositionGetInteger(POSITION_MAGIC) == MagicNumber)
//        {
//            count++;
//        }
//    }
//    return count;
//}