//+------------------------------------------------------------------+
//|                                                  TSO_EA.mq5      |
//+------------------------------------------------------------------+
#property copyright "Wamek EA"
#property link      "https://www.mql5.com/en/users/wamek/news"
#property version   "1.00"

// Include the Trade class
#include <Trade\Trade.mqh>

// Input parameters
input double LotSize = 0.1;        // Lot size for each trade
input int StopLoss = 500;          // Stop Loss in points
input int TakeProfit = 1000;       // Take Profit in points
input int MAPeriod = 20;           // Moving Average Period
input double TSO_Threshold = 0.7;  // TSO Threshold Level
input int MinBarsBetweenEntries = 10; // Minimum bars between entries
input int MaxEntries = 5;          // Maximum number of positions

int MagicNumber = 4643;            // Magic Number for identification

// Global variables
double prevTSO = 0;
double curTSO = 0;
datetime lastBuyTime = 0;
datetime lastSellTime = 0;
int buyCount = 0;
int sellCount = 0;

// Handles for indicators
int ma_handle;
int std_handle;

// CTrade object
CTrade Trade;

//+------------------------------------------------------------------+
//| Expert initialization function                                   |
//+------------------------------------------------------------------+
int OnInit()
{
    // Set up the trade object
    Trade.SetExpertMagicNumber(MagicNumber);
    Trade.SetDeviationInPoints(10);
    
    // Create indicator handles
    ma_handle = iMA(_Symbol, _Period, MAPeriod, 0, MODE_SMA, PRICE_CLOSE);
    std_handle = iStdDev(_Symbol, _Period, MAPeriod, 0, MODE_SMA, PRICE_CLOSE);
    
    if(ma_handle == INVALID_HANDLE || std_handle == INVALID_HANDLE)
    {
        Print("Error creating indicator handles");
        return(INIT_FAILED);
    }
    
    Print("TSO EA initialized successfully");
    return(INIT_SUCCEEDED);
}

//+------------------------------------------------------------------+
//| Expert deinitialization function                                 |
//+------------------------------------------------------------------+
void OnDeinit(const int reason)
{
    // Release indicator handles
    if(ma_handle != INVALID_HANDLE)
        IndicatorRelease(ma_handle);
    if(std_handle != INVALID_HANDLE)
        IndicatorRelease(std_handle);
    
    Print("TSO EA deinitialized");
}

//+------------------------------------------------------------------+
//| Expert tick function                                             |
//+------------------------------------------------------------------+
void OnTick()
{
    if(!IsNewBar()) return;
    
    // Calculate TSO values
    CalculateTSO();
    
    // Check entry conditions
    CheckEntryConditions();
}

//+------------------------------------------------------------------+
//| Calculate TSO values                                             |
//+------------------------------------------------------------------+
void CalculateTSO()
{    
    // Store previous TSO
    prevTSO = curTSO;
    
    // Get MA and STD values
    double ma[1], std[1];
    
    if(CopyBuffer(ma_handle, 0, 1, 1, ma) < 1)
    {
        Print("Error copying MA buffer");
        return;
    }
    
    if(CopyBuffer(std_handle, 0, 1, 1, std) < 1)
    {
        Print("Error copying STD buffer");
        return;
    }
    
    // Get close price from previous bar
    double close_prev = iClose(_Symbol, _Period, 1);
    
    // Calculate x
    double x = (close_prev - ma[0]) / std[0];
    
    // Calculate current TSO
    curTSO = MathPow(MathSin(x), 3);
    
}

//+------------------------------------------------------------------+
//| Check entry conditions                                           |
//+------------------------------------------------------------------+
void CheckEntryConditions()
{
    // Count current positions
    CountPositions();
    
    // Check if we can open new positions
    if(buyCount + sellCount >= MaxEntries)
    {
        Print("Maximum entries reached: ", MaxEntries);
        return;
    }
    
    // Check Buy condition: PrevTSO > -0.7 && CurTSO < -0.7
    if(prevTSO > -TSO_Threshold && curTSO < -TSO_Threshold)
    {
        Print("Buy condition met");
        if(CanOpenBuy())
        {
            OpenBuyPosition();
        }
    }
    
    // Check Sell condition: PrevTSO < 0.7 && CurTSO > 0.7
    if(prevTSO < TSO_Threshold && curTSO > TSO_Threshold)
    {
        Print("Sell condition met");
        if(CanOpenSell())
        {
            OpenSellPosition();
        }
    }
}

//+------------------------------------------------------------------+
//| Check if we can open buy position                                |
//+------------------------------------------------------------------+
bool CanOpenBuy()
{
    // Don't open buy if we have sell positions
    if(sellCount > 0)
    {
        Print("Cannot open buy - sell positions exist");
        return false;
    }
    
    // Check minimum bars between entries
    if(TimeCurrent() - lastBuyTime < MinBarsBetweenEntries * PeriodSeconds())
    {
        Print("Cannot open buy - minimum bars not reached");
        return false;
    }
    
    // Check maximum entries
    if(buyCount >= MaxEntries)
    {
        Print("Cannot open buy - maximum buy entries reached");
        return false;
    }
    
    return true;
}

//+------------------------------------------------------------------+
//| Check if we can open sell position                               |
//+------------------------------------------------------------------+
bool CanOpenSell()
{
    // Don't open sell if we have buy positions
    if(buyCount > 0)
    {
        Print("Cannot open sell - buy positions exist");
        return false;
    }
    
    // Check minimum bars between entries
    if(TimeCurrent() - lastSellTime < MinBarsBetweenEntries * PeriodSeconds())
    {
        Print("Cannot open sell - minimum bars not reached");
        return false;
    }
    
    // Check maximum entries
    if(sellCount >= MaxEntries)
    {
        Print("Cannot open sell - maximum sell entries reached");
        return false;
    }
    
    return true;
}

//+------------------------------------------------------------------+
//| Open buy position                                                |
//+------------------------------------------------------------------+
void OpenBuyPosition()
{
    double ask = SymbolInfoDouble(_Symbol, SYMBOL_ASK);
    double point = SymbolInfoDouble(_Symbol, SYMBOL_POINT);
    double sl = 0, tp = 0;
    
    // Calculate Stop Loss and Take Profit
    if(StopLoss > 0)
        sl = ask - StopLoss * point;
    if(TakeProfit > 0)
        tp = ask + TakeProfit * point;
    
    // Open buy position using CTrade
    if(Trade.Buy(LotSize, _Symbol, ask, sl, tp, "TSO Buy"))
    {
        lastBuyTime = TimeCurrent();
        buyCount++;
        Print("Buy position opened successfully. Lot Size: ", LotSize);
    }
    else
    {
        Print("Error opening buy position: ", Trade.ResultRetcodeDescription());
    }
}

//+------------------------------------------------------------------+
//| Open sell position                                               |
//+------------------------------------------------------------------+
void OpenSellPosition()
{
    double bid = SymbolInfoDouble(_Symbol, SYMBOL_BID);
    double point = SymbolInfoDouble(_Symbol, SYMBOL_POINT);
    double sl = 0, tp = 0;
    
    // Calculate Stop Loss and Take Profit
    if(StopLoss > 0)
        sl = bid + StopLoss * point;
    if(TakeProfit > 0)
        tp = bid - TakeProfit * point;
    
    // Open sell position using CTrade
    if(Trade.Sell(LotSize, _Symbol, bid, sl, tp, "TSO Sell"))
    {
        lastSellTime = TimeCurrent();
        sellCount++;
        Print("Sell position opened successfully. Lot Size: ", LotSize);
    }
    else
    {
        Print("Error opening sell position: ", Trade.ResultRetcodeDescription());
    }
}

//+------------------------------------------------------------------+
//| Count current positions                                          |
//+------------------------------------------------------------------+
void CountPositions()
{
    buyCount = 0;
    sellCount = 0;
    
    // Reset counts and count current positions
    for(int i = 0; i < PositionsTotal(); i++)
    {
        ulong ticket = PositionGetTicket(i);
        if(ticket > 0)
        {
            string symbol = PositionGetString(POSITION_SYMBOL);
            long magic = PositionGetInteger(POSITION_MAGIC);
            
            if(symbol == _Symbol && magic == MagicNumber)
            {
                long type = PositionGetInteger(POSITION_TYPE);
                if(type == POSITION_TYPE_BUY)
                    buyCount++;
                else if(type == POSITION_TYPE_SELL)
                    sellCount++;
            }
        }
    }
    
    // Print position counts for debugging
    if(buyCount > 0 || sellCount > 0)
    {
        Print(StringFormat("Current positions - Buy: %d, Sell: %d", buyCount, sellCount));
    }
}

//+------------------------------------------------------------------+
//| Check for new bar                                                |
//+------------------------------------------------------------------+
bool IsNewBar()
{
    static datetime lastBar = 0;
    datetime currentBar = iTime(_Symbol, _Period, 0);
    
    if(currentBar == lastBar) 
        return false;
    
    lastBar = currentBar;
    return true;
}

