//+------------------------------------------------------------------+
//|                     Butterfly Strategy EA.mq5                    |
//+------------------------------------------------------------------+
#property copyright "Wamek EA"
#property link      "https://www.mql5.com/en/users/wamek/news"
#property version   "1.00"

//--- Include CTrade class
#include <Trade/Trade.mqh>

//--- input parameters
input double Lots        = 0.01;
input double StopLoss    = 500;
input double TakeProfit  = 750;
input int    Slippage    = 10;
input double tmStep      = 0.05; // Step size
input int    FastMAPeriod = 7;
input int    SlowMAPeriod = 20;

//--- Strategy Selection
input bool UsePriceStep = false;      //Use ClosePrice 
input bool EnableStrategy1 = true;   // Enable Strategy 1 Peak
input bool EnableStrategy2 = false;  // Enable Strategy 2 Trough

string IndicatorName = "ButterFlyOscillator";

//--- global variables
double prevValue, currValue;
double maFast, maSlow;
datetime lastBar;

//--- handles
int butterflyHandle = INVALID_HANDLE;
int fastMAHandle = INVALID_HANDLE;
int slowMAHandle = INVALID_HANDLE;

//--- CTrade object
CTrade trade;

//+------------------------------------------------------------------+
//| Expert initialization function                                   |
//+------------------------------------------------------------------+
int OnInit()
{   
   //--- Initialize trade object
   trade.SetExpertMagicNumber(897);
   trade.SetDeviationInPoints(Slippage);
   
   //--- Initialize indicator handles
   butterflyHandle = iCustom(_Symbol, _Period, IndicatorName, UsePriceStep, tmStep);
   fastMAHandle = iMA(_Symbol, _Period, FastMAPeriod, 0, MODE_SMA, PRICE_CLOSE);
   slowMAHandle = iMA(_Symbol, _Period, SlowMAPeriod, 0, MODE_SMA, PRICE_CLOSE);
   
   if(butterflyHandle == INVALID_HANDLE || fastMAHandle == INVALID_HANDLE || slowMAHandle == INVALID_HANDLE)
   {
      Print("Error creating indicator handles");
      return(INIT_FAILED);
   }
   
   return(INIT_SUCCEEDED);
}

//+------------------------------------------------------------------+
//| Expert deinitialization function                                 |
//+------------------------------------------------------------------+
void OnDeinit(const int reason)
{
   //--- Release indicator handles
   if(butterflyHandle != INVALID_HANDLE)
      IndicatorRelease(butterflyHandle);
   if(fastMAHandle != INVALID_HANDLE)
      IndicatorRelease(fastMAHandle);
   if(slowMAHandle != INVALID_HANDLE)
      IndicatorRelease(slowMAHandle);
}

//+------------------------------------------------------------------+
//| Expert tick function                                             |
//+------------------------------------------------------------------+
void OnTick()
{
   //--- Check for new bar
   if(!IsNewBar()) return;

   int shift = 1;

   //--- Get indicator values using CopyBuffer
   double butterflyBuffer[2];
   if(CopyBuffer(butterflyHandle, 0, shift, 2, butterflyBuffer) < 2)
      return;
   
   currValue = butterflyBuffer[1];
   prevValue = butterflyBuffer[0];
 
  
   if(currValue == EMPTY_VALUE || prevValue == EMPTY_VALUE)
      return;

   //--- Get Moving Averages
   double fastMABuffer[1], slowMABuffer[1];
   if(CopyBuffer(fastMAHandle, 0, shift, 1, fastMABuffer) < 1)
      return;
   if(CopyBuffer(slowMAHandle, 0, shift, 1, slowMABuffer) < 1)
      return;
   
   maFast = fastMABuffer[0];
   maSlow = slowMABuffer[0];

   //--- Check existing positions
   if(PositionsTotal() > 0) return;

   //--- Strategy 1: Peak
   if(EnableStrategy1 && prevValue < 2.5 && currValue > 2.5)
   {
      if(maFast > maSlow)
         OpenBuy();
      else if(maFast < maSlow)
         OpenSell();
   }

   //--- Strategy 2: Trough
   if(EnableStrategy2 && prevValue > -2.5 && currValue < -2.5)
   {
      if(maFast > maSlow)
         OpenBuy();
      else if(maFast < maSlow)
         OpenSell();
   }
}

//+------------------------------------------------------------------+
//| Open Buy Position                                                |
//+------------------------------------------------------------------+
void OpenBuy()
{
   double ask = SymbolInfoDouble(_Symbol, SYMBOL_ASK);
   double point = SymbolInfoDouble(_Symbol, SYMBOL_POINT);
   
   double sl = ask - StopLoss * point;
   double tp = ask + TakeProfit * point;
   
   //--- Open buy position using CTrade
   if(trade.Buy(Lots, _Symbol, 0, sl, tp, "Buy Signal"))
      Print("BUY position opened successfully");
   else
      Print("BUY position failed. Error: ", trade.ResultRetcode(), " - ", trade.ResultRetcodeDescription());
}

//+------------------------------------------------------------------+
//| Open Sell Position                                               |
//+------------------------------------------------------------------+
void OpenSell()
{
   double bid = SymbolInfoDouble(_Symbol, SYMBOL_BID);
   double point = SymbolInfoDouble(_Symbol, SYMBOL_POINT);
   
   double sl = bid + StopLoss * point;
   double tp = bid - TakeProfit * point;
   
   //--- Open sell position using CTrade
   if(trade.Sell(Lots, _Symbol, 0, sl, tp, "Sell Signal"))
      Print("SELL position opened successfully");
   else
      Print("SELL position failed. Error: ", trade.ResultRetcode(), " - ", trade.ResultRetcodeDescription());
}

//+------------------------------------------------------------------+
//| Check for new bar                                                |
//+------------------------------------------------------------------+
bool IsNewBar()
{
   datetime currentBar = iTime(_Symbol, _Period, 0);
   
   if(currentBar == lastBar) 
      return false;
   
   lastBar = currentBar;
   return true;
}