//+------------------------------------------------------------------+
//|                                         PPr_Strategy_EA.mq5      |
//|    DeM_MFI_RSI_corr (Pseudo Pearson Correlation) Indicator       |
//+------------------------------------------------------------------+
#property copyright "Wamek EA"
#property link      "https://www.mql5.com/en/users/wamek/news"
#property version   "1.00"

#include <Trade/Trade.mqh>
CTrade Trade;

//--- Input parameters
input double Lots = 0.01;
input double StopLoss = 300;
input double TakeProfit = 700;
input int Slippage = 3;

input int CorrPeriod = 21;
input int RSIPeriod = 14;
input int MFIPeriod = 14;
input int DeMPeriod = 14;
input double PPr = 0.5;    // PseudoCorrelatedValue (0.1 to 1)

input int FastMAPeriod = 2;
input int SlowMAPeriod = 20;

//--- Strategy Selection
input bool EnableStrategy1 = true;   // Enable Strategy 1 Correlated
input bool EnableStrategy2 = false;  // Enable Strategy 2 NotCorrelated

string IndicatorName = "PseudoPC";

//--- Global variables
datetime lastBar;
int indicatorHandle = INVALID_HANDLE;
int fastMaHandle = INVALID_HANDLE;
int slowMaHandle = INVALID_HANDLE;

double corrBuff[];
//+------------------------------------------------------------------+
int OnInit()
{
   Print("PPr Strategy EA initialized");
   
   //--- Create indicator handles
   indicatorHandle = iCustom(Symbol(), Period(), IndicatorName, 
                            CorrPeriod, RSIPeriod, MFIPeriod, DeMPeriod);
   fastMaHandle = iMA(Symbol(), Period(), FastMAPeriod, 0, MODE_EMA, PRICE_CLOSE);
   slowMaHandle = iMA(Symbol(), Period(), SlowMAPeriod, 0, MODE_EMA, PRICE_CLOSE);
   
   if(indicatorHandle == INVALID_HANDLE || fastMaHandle == INVALID_HANDLE || slowMaHandle == INVALID_HANDLE)
   {
      Print("Error creating indicator handles");
      return(INIT_FAILED);
   }
   
   //---Set CorrBuff Array as TimeSeries
   ArraySetAsSeries(corrBuff,true);
   
   //--- Configure trade settings
   Trade.SetExpertMagicNumber(7809);
   Trade.SetDeviationInPoints(Slippage);
   
   return(INIT_SUCCEEDED);
}
//+------------------------------------------------------------------+
void OnDeinit(const int reason)
{
   if(indicatorHandle != INVALID_HANDLE) IndicatorRelease(indicatorHandle);
   if(fastMaHandle != INVALID_HANDLE) IndicatorRelease(fastMaHandle);
   if(slowMaHandle != INVALID_HANDLE) IndicatorRelease(slowMaHandle);
}
//+------------------------------------------------------------------+
void OnTick()
{
   //--- Check for new bar
   if(!IsNewBar()) return;

   //--- Get indicator values
     double corr_curr, corr_prev;

   if(CopyBuffer(indicatorHandle, 0, 1, 2, corrBuff) <= 0)
   {
      Print("Error copying indicator buffer");
      return;
   }
   
   corr_curr = corrBuff[0]; corr_prev=corrBuff[1];
   
   //--- Get Moving Average values
   double maFast[1], maSlow[1];
   
   if(CopyBuffer(fastMaHandle, 0, 1, 1, maFast) <= 0 ||
      CopyBuffer(slowMaHandle, 0, 1, 1, maSlow) <= 0)
   {
      Print("Error copying MA buffers");
      return;
   }

   //--- Check open positions
   if(PositionsTotal() > 0) return; // one trade at a time

   //--- STRATEGY 1 (Correlated) ---
   if(EnableStrategy1 && corr_prev < PPr && corr_curr > PPr)
   {
      if(maFast[0] > maSlow[0])
         OpenTrade(ORDER_TYPE_BUY, "Strategy1 BUY");
      else if(maFast[0] < maSlow[0])
         OpenTrade(ORDER_TYPE_SELL, "Strategy1 SELL");
   }

   //--- STRATEGY 2 (Not Correlated) ---
   if(EnableStrategy2 && corr_prev > -PPr && corr_curr < -PPr)
   {
      if(maFast[0] > maSlow[0])
         OpenTrade(ORDER_TYPE_BUY, "Strategy2 BUY");
      else if(maFast[0] < maSlow[0])
         OpenTrade(ORDER_TYPE_SELL, "Strategy2 SELL");
   }
}
//+------------------------------------------------------------------+
//+------------------------------------------------------------------+
//|     Send Trade Orders                                            |
//+------------------------------------------------------------------+

void OpenTrade(ENUM_ORDER_TYPE type, string comment)
{
   double price, sl, tp;
   double point = SymbolInfoDouble(Symbol(), SYMBOL_POINT);
   
   if(type == ORDER_TYPE_BUY)
   {
      price = SymbolInfoDouble(Symbol(), SYMBOL_ASK);
      sl = price - StopLoss * point;
      tp = price + TakeProfit * point;
   }
   else
   {
      price = SymbolInfoDouble(Symbol(), SYMBOL_BID);
      sl = price + StopLoss * point;
      tp = price - TakeProfit * point;
   }

   if(Trade.PositionOpen(Symbol(), type, Lots, price, sl, tp, comment))
      Print(comment, " position opened successfully.");
   else
      Print("PositionOpen failed: ", Trade.ResultRetcodeDescription());
}
//+------------------------------------------------------------------+

//+------------------------------------------------------------------+
//| Check for new bar                                                |
//+------------------------------------------------------------------+
bool IsNewBar()
{
   datetime currentBar = iTime(Symbol(), Period(), 0);
   
   if(currentBar == lastBar) 
      return false;
   
   lastBar = currentBar;
   return true;
}
//+------------------------------------------------------------------+