//+------------------------------------------------------------------+
//|                                                  CointNasdaq.mq5 |
//|                                  Copyright 2025, MetaQuotes Ltd. |
//|                                             https://www.mql5.com |
//+------------------------------------------------------------------+
//+------------------------------------------------------------------+
//| Expert Advisor - Cointegration Statistical Arbitrage             |
//| Assets: Dynamic assets allocation                                |
//| Strategy: Mean-reversion on Johansen cointegration portfolio     |
//+------------------------------------------------------------------+

#include <Math\Stat\Math.mqh>
#include <StatArb\CointNasdaq.mqh>

// Input parameters
input int    InpUpdateFreq  = 1;     // Update frequency in minutes
input string InpDbFilename = "StatArb\\statarb-0.4.db"; // SQLite database filename
input string InpStrategyName = "CointNasdaq_H4_60"; // Strategy name
input double InpEntryThreshold = 2.0;    // Entry threshold (std dev)
input double InpExitThreshold = 0.3;     // Exit threshold (std dev)
input double InpLotSize = 10.0;           // Lot size per leg
input int InpMaxSlippage = 3;               // Max allowed slippage

// Global variables
string symbols[] = {}; // Asset symbols
double weights[] = {}; // Portfolio weights
ENUM_TIMEFRAMES timeframe = PERIOD_CURRENT; // Strategy cointegrated timeframe
int lookback_period;       // Lookback for moving average/standard deviation

double spreadBuffer[];                // Array to store historical spread values
double spreadMean = 0.0;              // Mean of the spread
double spreadStdDev = 0.0;            // Std dev of the spread
bool tradeOpen = false;               // Flag to track if a trade is open
double currentSpread = 0.0;           // Current spread

//+------------------------------------------------------------------+
//| Expert initialization function                                   |
//+------------------------------------------------------------------+
int OnInit()
  {
   ResetLastError();
// Check if all symbols are available
   for(int i = 0; i < ArraySize(symbols); i++)
     {
      if(!SymbolSelect(symbols[i], true))
        {
         Print("Error: Symbol ", symbols[i], " not found!");
         return(INIT_FAILED);
        }
     }
// Initialize spread buffer
   ArrayResize(spreadBuffer, lookback_period);
// Set a timer for spread, mean, stdev calculations
// and strategy parameters update (check DB)
   EventSetTimer(InpUpdateFreq * 60); // min one minute
// check if we are backtesting
   if(MQLInfoInteger(MQL_TESTER))
     {
      Print("Running on tester");
      ArrayResize(symbols, 4);
      ArrayResize(weights, 4);
      //{"NVDA", "INTC", "AVGO", "ASX"};
      symbols[0] = "NVDA";
      symbols[1] = "INTC";
      symbols[2] = "AVGO";
      symbols[3] = "ASX";
      // {1.0, -1.9598590335874817, -0.36649674991957104, 21.608207065113874};
      weights[0] = 1.0;
      weights[1] = -1.9598590335874817;
      weights[2] = -0.36649674991957104;
      weights[3] = 21.608207065113874;
      timeframe = PERIOD_H4;
      lookback_period = 60;
     }
   else
     {
      // Load strategy parameters from database
      if(!LoadStrategyFromDB(InpDbFilename,
                             InpStrategyName,
                             symbols,
                             weights,
                             timeframe,
                             lookback_period))
        {
         // Handle error - maybe use default values
         printf("Error at " + __FUNCTION__ + " %s ",
                getUninitReasonText(GetLastError()));
         return INIT_FAILED;
        }
     }
   return(INIT_SUCCEEDED);
  }
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
void OnTimer(void)
  {
   ResetLastError();
   if(!MQLInfoInteger(MQL_TESTER))
     {
      // Wrapper around LoadStrategyFromDB: for clarity
      if(!UpdateModelParams(InpDbFilename,
                            InpStrategyName,
                            symbols,
                            weights,
                            timeframe,
                            lookback_period))
        {
         printf("%s failed: %s", __FUNCTION__, GetLastError());
        }
     }
// Initialize spread buffer
   ArrayResize(spreadBuffer, lookback_period);
// Calculate current spread value
   currentSpread = CalculateSpread(symbols, timeframe, weights);
// Update spread buffer (rolling window)
   static int barCount = 0;
   if(barCount < lookback_period)
     {
      spreadBuffer[barCount]
         = currentSpread;
      barCount++;
      return; // Wait until buffer is filled
     }
// Shift buffer (remove oldest value, add newest)
   for(int i = 0; i < lookback_period - 1; i++)
      spreadBuffer[i] = spreadBuffer[i + 1];
   spreadBuffer[lookback_period - 1] = currentSpread;
// Calculate mean and standard deviation using custom functions
   spreadMean = CalculateMA(spreadBuffer, lookback_period);
   spreadStdDev = CalculateStdDev(spreadBuffer, lookback_period, spreadMean);
  }
//+------------------------------------------------------------------+
//| Expert tick function                                             |
//+------------------------------------------------------------------+
void OnTick()
  {
// Trading logic
   if(!tradeOpen)
     {
      // Check for entry signal (spread deviates from mean)
      if(currentSpread > spreadMean + InpEntryThreshold * spreadStdDev)
        {
         // Short spread (sell MU/NVDA, buy MPWR/MCHP)
         ExecuteTrade(ORDER_TYPE_SELL, symbols, weights, InpLotSize);
         tradeOpen = true;
        }
      else
         if(currentSpread < spreadMean - InpEntryThreshold * spreadStdDev)
           {
            // Buy spread (buy MU/NVDA, sell MPWR/MCHP)
            ExecuteTrade(ORDER_TYPE_BUY, symbols, weights, InpLotSize);
            tradeOpen = true;
           }
     }
   else
     {
      // Check for exit signal (spread reverts to mean)
      if((currentSpread <= spreadMean + InpExitThreshold * spreadStdDev) &&
         (currentSpread >= spreadMean - InpExitThreshold * spreadStdDev))
        {
         CloseAllTrades(InpMaxSlippage);
         tradeOpen = false;
        }
     }
// Optional: Display spread in chart comment
   Comment(StringFormat("Spread: %.2f | Mean: %.2f | StdDev: %.2f",
                        currentSpread,
                        spreadMean,
                        spreadStdDev));
  }
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
void OnDeinit(const int32_t reason)
  {
   EventKillTimer();
//--- The first way to get a deinitialization reason code
   Print(__FUNCTION__, " Deinitialization reason code = ", reason);
//--- The second way to get a deinitialization reason code
   Print(__FUNCTION__, " _UninitReason = ", getUninitReasonText(_UninitReason));
//--- The third way to get a deinitialization reason code
   Print(__FUNCTION__, " UninitializeReason() = ", getUninitReasonText(UninitializeReason()));
  }
//+------------------------------------------------------------------+
// from MQL5 ref
//+------------------------------------------------------------------+
//| Return a textual description of the deinitialization reason code |
//+------------------------------------------------------------------+
string getUninitReasonText(int reasonCode)
  {
   string text = "";
//---
   switch(reasonCode)
     {
      case REASON_ACCOUNT:
         text = "Account was changed";
         break;
      case REASON_CHARTCHANGE:
         text = "Symbol or timeframe was changed";
         break;
      case REASON_CHARTCLOSE:
         text = "Chart was closed";
         break;
      case REASON_PARAMETERS:
         text = "Input-parameter was changed";
         break;
      case REASON_RECOMPILE:
         text = "Program " + __FILE__ + " was recompiled";
         break;
      case REASON_REMOVE:
         text = "Program " + __FILE__ + " was removed from chart";
         break;
      case REASON_TEMPLATE:
         text = "New template was applied to chart";
         break;
      default:
         text = "Another reason";
     }
//---
   return text;
  }
//+------------------------------------------------------------------+

//+------------------------------------------------------------------+
