//+------------------------------------------------------------------+
//| SessionVisualizer.mqh                                            |
//| Modular class for Forex session OHLC visualization               |
//| Copyright 2025: Clemence Benjamin                                |
//| Version: 1.0                                                     |
//+------------------------------------------------------------------+
#property strict

// ARGB Macro: Creates a color with alpha (transparency) - A:0-255, R/G/B:0-255
#define ARGB(a,r,g,b) ((color)(((uchar)(a))<<24)|(((uchar)(r))<<16)|(((uchar)(g))<<8)|((uchar)(b)))

#include <Object.mqh>  // For object management

// Session enum and struct
enum SESSION_TYPE {
   SESSION_SYDNEY = 0,  // 22:00-07:00 GMT
   SESSION_TOKYO   = 1,  // 00:00-09:00 GMT
   SESSION_LONDON  = 2,  // 08:00-17:00 GMT
   SESSION_NEWYORK = 3   // 13:00-22:00 GMT
};

struct SessionInfo {
   SESSION_TYPE type;
   string       name;       // e.g., "LON"
   int          open_hour;  // GMT open hour
   int          close_hour; // GMT close hour
   color        sess_color; // Line/fill color (renamed to fix prior compilation)
   bool         enabled;
};

// Runtime globals (will integrate with EA's g_*)
input int    InpSessionLookback = 10;  // Days of historical sessions
input bool   InpShowSessionWicks = false;
input int    InpWickAlpha = 120;
input color  InpFillBull = clrLime;
input color  InpFillBear = clrPink;

// Class definition
class CSessionVisualizer : public CObject {
private:
   SessionInfo m_sessions[4];  // Array of sessions
   int         m_gmt_offset;   // Broker GMT offset (hours)
   string      m_prefix;       // Object prefix, e.g., "SESS_"

public:
   CSessionVisualizer(string prefix = "SESS_") : m_prefix(prefix) {
      // Initialize sessions (GMT times; adjust for DST if needed)
      m_sessions[SESSION_SYDNEY].type = SESSION_SYDNEY; m_sessions[SESSION_SYDNEY].name = "SYD"; m_sessions[SESSION_SYDNEY].open_hour = 22; m_sessions[SESSION_SYDNEY].close_hour = 7; m_sessions[SESSION_SYDNEY].sess_color = clrAqua; m_sessions[SESSION_SYDNEY].enabled = true;
      m_sessions[SESSION_TOKYO].type = SESSION_TOKYO;   m_sessions[SESSION_TOKYO].name = "TOK";   m_sessions[SESSION_TOKYO].open_hour = 0;  m_sessions[SESSION_TOKYO].close_hour = 9; m_sessions[SESSION_TOKYO].sess_color = clrYellow; m_sessions[SESSION_TOKYO].enabled = true;
      m_sessions[SESSION_LONDON].type = SESSION_LONDON; m_sessions[SESSION_LONDON].name = "LON";  m_sessions[SESSION_LONDON].open_hour = 8;  m_sessions[SESSION_LONDON].close_hour = 17; m_sessions[SESSION_LONDON].sess_color = clrRed; m_sessions[SESSION_LONDON].enabled = true;
      m_sessions[SESSION_NEWYORK].type = SESSION_NEWYORK; m_sessions[SESSION_NEWYORK].name = "NY"; m_sessions[SESSION_NEWYORK].open_hour = 13; m_sessions[SESSION_NEWYORK].close_hour = 22; m_sessions[SESSION_NEWYORK].sess_color = clrBlue; m_sessions[SESSION_NEWYORK].enabled = true;
      
      // Detect GMT offset (simple: compare current time to GMT)
      MqlDateTime dt;
      TimeToStruct(TimeCurrent(), dt);
      m_gmt_offset = -dt.hour % 24;  // Placeholder; refine with server time query
   }
   
   ~CSessionVisualizer() {
      DeleteAllSessionObjects();
   }
   
   // Main method: Refresh sessions for lookback days
   void RefreshSessions(int lookback_days = 10) {
      DeleteAllSessionObjects();  // Clear old
      datetime end_time = TimeCurrent();
      datetime start_time = end_time - (lookback_days * 86400);  // Days in seconds
      
      for (int day = 0; day < lookback_days; day++) {
         datetime day_start = start_time + (day * 86400);
         for (int s = 0; s < 4; s++) {
            if (!m_sessions[s].enabled) continue;
            DrawSessionForDay(m_sessions[s], day_start);
         }
      }
      ChartRedraw();
   }
   
   // Toggle session
   void SetSessionEnabled(SESSION_TYPE type, bool enabled) {
      for (int i = 0; i < 4; i++) {
         if (m_sessions[i].type == type) {
            m_sessions[i].enabled = enabled;
            break;
         }
      }
   }
   
   // Set color for session
   void SetSessionColor(SESSION_TYPE type, color col) {
      for (int i = 0; i < 4; i++) {
         if (m_sessions[i].type == type) {
            m_sessions[i].sess_color = col;
            break;
         }
      }
   }

private:
   // Draw one session for a specific day
   void DrawSessionForDay(const SessionInfo &sess, datetime day_start) {
      MqlDateTime dt_open, dt_close;
      TimeToStruct(day_start, dt_open);
      dt_open.hour = sess.open_hour + m_gmt_offset;
      dt_open.min = 0; dt_open.sec = 0;
      datetime t_open = StructToTime(dt_open);
      
      TimeToStruct(day_start, dt_close);
      dt_close.hour = sess.close_hour + m_gmt_offset;
      dt_close.min = 0; dt_close.sec = 0;
      datetime t_close = StructToTime(dt_close);
      
      // Handle overnight sessions (e.g., Sydney wraps to next day)
      if (sess.close_hour < sess.open_hour) {
         t_close += 86400;  // Add one day
      }
      
      // Skip if outside lookback
      if (t_close < TimeCurrent() - (InpSessionLookback * 86400)) return;
      
      // Get OHLC for session window
      double ohlc_open, ohlc_high, ohlc_low, ohlc_close;
      if (!GetOHLCInTimeRange(t_open, t_close, ohlc_open, ohlc_high, ohlc_low, ohlc_close)) return;
      
      // Draw vertical lines at open/close
      string vline_open = m_prefix + sess.name + "_O_" + IntegerToString((int)t_open);
      ObjectCreate(0, vline_open, OBJ_VLINE, 0, t_open, 0);
      ObjectSetInteger(0, vline_open, OBJPROP_COLOR, sess.sess_color);
      ObjectSetInteger(0, vline_open, OBJPROP_WIDTH, 1);
      ObjectSetInteger(0, vline_open, OBJPROP_STYLE, STYLE_DASH);
      ObjectSetInteger(0, vline_open, OBJPROP_BACK, true);
      
      string vline_close = m_prefix + sess.name + "_C_" + IntegerToString((int)t_close);
      ObjectCreate(0, vline_close, OBJ_VLINE, 0, t_close, 0);
      ObjectSetInteger(0, vline_close, OBJPROP_COLOR, sess.sess_color);
      ObjectSetInteger(0, vline_close, OBJPROP_WIDTH, 1);
      ObjectSetInteger(0, vline_close, OBJPROP_STYLE, STYLE_DASH);
      ObjectSetInteger(0, vline_close, OBJPROP_BACK, true);
      
      // Body fill (candlestick body)
      color body_color = (ohlc_close > ohlc_open) ? InpFillBull : InpFillBear;
      string rect_body = m_prefix + sess.name + "_B_" + IntegerToString((int)t_open);
      ObjectCreate(0, rect_body, OBJ_RECTANGLE, 0, t_open, MathMin(ohlc_open, ohlc_close), t_close, MathMax(ohlc_open, ohlc_close));
      ObjectSetInteger(0, rect_body, OBJPROP_BGCOLOR, body_color);
      ObjectSetInteger(0, rect_body, OBJPROP_COLOR, body_color);
      ObjectSetInteger(0, rect_body, OBJPROP_FILL, true);
      ObjectSetInteger(0, rect_body, OBJPROP_BACK, true);
      ObjectSetInteger(0, rect_body, OBJPROP_SELECTABLE, false);
      
      // Wicks (optional) - Now compiles cleanly with ARGB defined
      if (InpShowSessionWicks) {
         uint wick_col = ARGB(InpWickAlpha, 128, 128, 128);  // Gray, transparent (removed unnecessary cast)
         
         // Upper wick
         if (ohlc_high > MathMax(ohlc_open, ohlc_close)) {
            string wick_upper = m_prefix + sess.name + "_WU_" + IntegerToString((int)t_open);
            ObjectCreate(0, wick_upper, OBJ_RECTANGLE, 0, t_open, MathMax(ohlc_open, ohlc_close), t_close, ohlc_high);
            ObjectSetInteger(0, wick_upper, OBJPROP_BGCOLOR, wick_col);
            ObjectSetInteger(0, wick_upper, OBJPROP_FILL, true);
            ObjectSetInteger(0, wick_upper, OBJPROP_BACK, true);
         }
         
         // Lower wick
         if (ohlc_low < MathMin(ohlc_open, ohlc_close)) {
            string wick_lower = m_prefix + sess.name + "_WL_" + IntegerToString((int)t_open);
            ObjectCreate(0, wick_lower, OBJ_RECTANGLE, 0, t_open, ohlc_low, t_close, MathMin(ohlc_open, ohlc_close));
            ObjectSetInteger(0, wick_lower, OBJPROP_BGCOLOR, wick_col);
            ObjectSetInteger(0, wick_lower, OBJPROP_FILL, true);
            ObjectSetInteger(0, wick_lower, OBJPROP_BACK, true);
         }
      }
      
      // Labels (open/close)
      string lbl_open = m_prefix + sess.name + "_OL_" + IntegerToString((int)t_open);
      ObjectCreate(0, lbl_open, OBJ_TEXT, 0, t_open, ohlc_open);
      ObjectSetString(0, lbl_open, OBJPROP_TEXT, sess.name + " Open");
      ObjectSetInteger(0, lbl_open, OBJPROP_COLOR, sess.sess_color);
      ObjectSetInteger(0, lbl_open, OBJPROP_FONTSIZE, 8);
      
      string lbl_close = m_prefix + sess.name + "_CL_" + IntegerToString((int)t_close);
      ObjectCreate(0, lbl_close, OBJ_TEXT, 0, t_close, ohlc_close);
      ObjectSetString(0, lbl_close, OBJPROP_TEXT, sess.name + " Close");
      ObjectSetInteger(0, lbl_close, OBJPROP_COLOR, sess.sess_color);
      ObjectSetInteger(0, lbl_close, OBJPROP_FONTSIZE, 8);
   }
   
   // Helper: Get OHLC in time range [start, end]
   bool GetOHLCInTimeRange(datetime start, datetime end, double &open, double &high, double &low, double &close) {
      // Fixed: Use iBarShift for accurate bar count in range
      int shift_start = iBarShift(_Symbol, _Period, start, false);
      int shift_end = iBarShift(_Symbol, _Period, end, false);
      if (shift_start < 0 || shift_end < 0) return false;
      int bars_in_range = shift_start - shift_end + 1;
      if (bars_in_range <= 0) return false;
      
      double opens[], highs[], lows[], closes[];
      ArraySetAsSeries(opens, true); ArraySetAsSeries(highs, true); ArraySetAsSeries(lows, true); ArraySetAsSeries(closes, true);
      
      // Copy from shift_end to shift_start (recent to old)
      if (CopyOpen(_Symbol, _Period, shift_end, bars_in_range, opens) != bars_in_range) return false;
      if (CopyHigh(_Symbol, _Period, shift_end, bars_in_range, highs) != bars_in_range) return false;
      if (CopyLow(_Symbol, _Period, shift_end, bars_in_range, lows) != bars_in_range) return false;
      if (CopyClose(_Symbol, _Period, shift_end, bars_in_range, closes) != bars_in_range) return false;
      
      open = opens[bars_in_range - 1];  // Oldest open (session start)
      high = highs[ArrayMaximum(highs, 0, bars_in_range)];
      low = lows[ArrayMinimum(lows, 0, bars_in_range)];
      close = closes[0];  // Newest close (session end)
      
      return true;
   }
   
   // Cleanup
   void DeleteAllSessionObjects() {
      int total = ObjectsTotal(0);
      for (int i = total - 1; i >= 0; i--) {
         string name = ObjectName(0, i);
         if (StringFind(name, m_prefix) == 0) {
            ObjectDelete(0, name);
         }
      }
   }
};