//+------------------------------------------------------------------+
//|                                                   dependence.mqh |
//|                                  Copyright 2025, MetaQuotes Ltd. |
//|                                             https://www.mql5.com |
//+------------------------------------------------------------------+
#property copyright "Copyright 2025, MetaQuotes Ltd."
#property link      "https://www.mql5.com"
#ifndef _NP_
  #include<np.mqh>
#endif 
//+------------------------------------------------------------------+
//| pair selection method enumeration                                |
//+------------------------------------------------------------------+
enum ENUM_DEPENDENCE_MEASURE
  {
   KENDAL_TAU = 0,//kendals tau
   SPEAR_RHO//spearmans rho
  };
//+------------------------------------------------------------------+
//| Tail dependence calculation method                               |
//+------------------------------------------------------------------+
enum ENUM_TAIL_DEPENDENCE_METHOD
 {
  TAIL_SCHMIDT=0,//Schmid Schmidt
  TAIL_STADTMUELLER//Schmidt Stadtmueller
 };
//+------------------------------------------------------------------+
//| dependence matrix                                                |
//+------------------------------------------------------------------+
matrix dependence(matrix &data, ENUM_DEPENDENCE_MEASURE method)
 {
  matrix out;
  if(!out.Diag(vector::Ones(data.Cols())))
   {
    Print(__FUNCTION__, " error ", GetLastError());
    return matrix::Zeros(0,0);
   }
   
  for(ulong i = 1; i<data.Cols(); ++i)
    {
     for(ulong j = 0; j<(i); ++j)
       out[i,j] = dependence(data.Col(i),data.Col(j),method);
    }
    
  matrix outT = out.Transpose();
  matrix mask = matrix::Ones(out.Rows(),out.Cols());
  mask = mask.TriU(1);
  matrix smask = np::whereMatrixIsLt(mask,0.5);
  out = outT.TriU(1) + (smask*out);
  return out;
 }
//+------------------------------------------------------------------+
//| dependence calculation                                           |
//+------------------------------------------------------------------+
double dependence(vector &A, vector &B, ENUM_DEPENDENCE_MEASURE method)
  {
   double stat = EMPTY_VALUE;
   switch(method)
     {
      case KENDAL_TAU:
         stat = np::kendalTau(A,B);
         break;
      case SPEAR_RHO:
         stat =  np::spearmanRho(A,B);
         break;
     }
   return stat;
  }
//+------------------------------------------------------------------+
//| tail dependence                                                  |
//+------------------------------------------------------------------+
matrix lambda(matrix &pseudo_obs,double cutOff,ENUM_TAIL_DEPENDENCE_METHOD calculation_method,bool lowerTail=true)
 {
  ulong d = pseudo_obs.Cols();
  if(pseudo_obs.Max()>1.0 || pseudo_obs.Min()<0.0 ||
     cutOff<0.0 || cutOff>1.0 || d<2)
    {
     Print(__FUNCTION__, " Invalid parameter(s) ");
     matrix::Zeros(0,0);
    }
    
  matrix u = (lowerTail)?pseudo_obs:1.0 - pseudo_obs;
  
  switch(calculation_method)
   {
    case TAIL_SCHMIDT:
     return SchmidAlgo(u,cutOff);
    case TAIL_STADTMUELLER:
     return StadtmuellerAlgo(u,cutOff);
   }
   
  return matrix::Zeros(0,0);
 }
//+------------------------------------------------------------------+
//| schimdt tail dependence method                                   |
//+------------------------------------------------------------------+
matrix SchmidAlgo(matrix &obs, double cutoff)
 {
  matrix lam;
  if(!lam.Diag(vector::Ones(obs.Cols())))
   {
    Print(__FUNCTION__, " error ", GetLastError());
    return matrix::Zeros(0,0);
   }
  
  matrix pmu = cutoff - obs;
  
  matrix mk = np::whereMatrixIsGt(pmu,0.0);
  
  pmu*=mk;
 
  for(ulong i = 1; i<obs.Cols(); ++i)
   {
    for(ulong j = 0; j<(i); ++j)
     {
      lam[i,j] = (pmu.Col(i)*pmu.Col(j)).Mean();
     }
   }
   
  double op = pow((cutoff*cutoff)/2.0,2.0);
  double om = pow(cutoff,3.0)/3.0;
  
  lam = (lam-op)/(om - op);
  
  lam.Clip(0.0,1.0);
  
  matrix lamT = lam.Transpose();
  matrix mask = matrix::Ones(lam.Rows(),lam.Cols());
  mask = mask.TriU(1);
  matrix smask = np::whereMatrixIsLt(mask,0.5);
  lam = lamT.TriU(1) + (smask*lam);
  return lam;
 }
//+------------------------------------------------------------------+
//| Schmidt Stadtmueller tail dependence method                      |
//+------------------------------------------------------------------+
matrix StadtmuellerAlgo(matrix& obs, double cutoff)
 {
  matrix lam;
  if(!lam.Diag(vector::Ones(obs.Cols())))
   {
    Print(__FUNCTION__, " error ", GetLastError());
    return matrix::Zeros(0,0);
   }
   
  matrix up = np::whereMatrixIsLte(obs,cutoff);
  
  for(ulong i = 1; i<obs.Cols(); ++i)
   {
    for(ulong j = 0; j<i; ++j)
     {
      vector sum = up.Col(i)+up.Col(j);
      vector sum2mask = np::whereVectorIsEq(sum,2.0,1);
      lam[i,j] = ((sum2mask.Sum())/double(sum2mask.Size()))/cutoff;
     }
   }
  
  lam.Clip(0.0,1.0);
  
  matrix lamT = lam.Transpose();
  matrix mask = matrix::Ones(lam.Rows(),lam.Cols());
  mask = mask.TriU(1);
  matrix smask = np::whereMatrixIsLt(mask,0.5);
  lam = lamT.TriU(1) + (smask*lam);
  return lam;
 
 }