//+------------------------------------------------------------------+
//|                                                        mixed.mqh |
//|                                  Copyright 2025, MetaQuotes Ltd. |
//|                                             https://www.mql5.com |
//+------------------------------------------------------------------+
#property copyright "Copyright 2025, MetaQuotes Ltd."
#property link      "https://www.mql5.com"
#include"bivariate_copula.mqh"
//+------------------------------------------------------------------+
//| mixed copula type                                                |
//+------------------------------------------------------------------+
enum ENUM_COPULA_MIX
  {
   CFG_MIX=0,//Clayton-Frank-Gumbel
   CFJ_MIX,//Clayton-Frank-Joe
   CTG_MIX,//Clayton-Student-Gumbel
   JFG_MIX//Joe-Frank-Gumbel
  };
//+------------------------------------------------------------------+
//| utility function                                                 |
//+------------------------------------------------------------------+
double     away_from_0(double x, double lower_limit = -1.e-5, double upper_limit = 1.e-5)
  {
   bool small_pos_bool = ((x>=0.0)&&(x<upper_limit));
   bool small_neg_bool = ((x>lower_limit)&&(x<0.0));
   bool small_bool = (small_pos_bool||small_neg_bool);
   double remapped_param = (x * double(!small_bool) + upper_limit * double(small_pos_bool) + lower_limit * double(small_neg_bool));
   return remapped_param;
  }
//+------------------------------------------------------------------+
//|base class for mixed copula                                       |
//+------------------------------------------------------------------+
class CMixedCopula:public CObject
  {
protected:
   uint              m_num_cops;
   CBivariateCopula* m_copulas[];
   vector            m_weights;
   vector            m_cop_params;
   ENUM_COPULA_MIX   m_mixture;
public:
                     CMixedCopula(void):m_weights(vector::Zeros(0)),
                     m_cop_params(vector::Zeros(0)),
                     m_num_cops(0),
                     m_mixture(WRONG_VALUE)
     {
     }
                    ~CMixedCopula(void)
     {
      for(uint i = 0; i<m_copulas.Size(); ++i)
         if(CheckPointer(m_copulas[i]) == POINTER_DYNAMIC)
            delete m_copulas[i];
     }
   virtual double    Fit(matrix& qdata, ulong maxiter = 25, double gamma_scad = 0.0, double a_scad = 0.0,double weight_margin=1.e-2)
     {
      return EMPTY_VALUE;
     }

   double            Copula_PDF(double u, double v, double _eps = 1.e-5)
     {
      u = fmin(fmax(_eps,u),1. - _eps);
      v = fmin(fmax(_eps,v),1. - _eps);
      vector pdf_ = vector::Zeros(m_weights.Size());
      for(uint i = 0; i<m_copulas.Size(); ++i)
         pdf_[i] = m_copulas[i]?m_weights[i]*m_copulas[i].Copula_PDF(u,v,false):0.0;
      return pdf_.Sum();
     }
   double            Copula_CDF(double u, double v, double _eps = 1.e-5)
     {
      u = fmin(fmax(_eps,u),1. - _eps);
      v = fmin(fmax(_eps,v),1. - _eps);
      vector cdf_ = vector::Zeros(m_weights.Size());
      for(uint i = 0; i<m_copulas.Size(); ++i)
         cdf_[i] = m_copulas[i]?m_weights[i]*m_copulas[i].Copula_CDF(u,v,false):0.0;
      return cdf_.Sum();
     }
   double            Conditional_Probability(double u, double v, double _eps = 1.e-5)
     {
      u = fmin(fmax(_eps,u),1. - _eps);
      v = fmin(fmax(_eps,v),1. - _eps);
      vector result_ = vector::Zeros(m_weights.Size());
      for(uint i = 0; i<m_copulas.Size(); ++i)
         result_[i] = m_copulas[i]?m_weights[i]*m_copulas[i].Conditional_Probability(u,v,false):0.0;
      return result_.Sum();
     }
   vector            Copula_PDF(vector& u, vector& v, double _eps = 1.e-5)
     {
      vector pdf_ = vector::Zeros(u.Size());
      for(uint i = 0; i<m_copulas.Size(); ++i)
        {
         if(!m_copulas[i])
            continue;
         m_copulas[i].Set_eps(_eps);
         pdf_ += m_weights[i]*m_copulas[i].Copula_PDF(u,v);
        }
      return pdf_;
     }
   vector            Copula_CDF(vector& u, vector& v, double _eps = 1.e-5)
     {
      vector cdf_ = vector::Zeros(u.Size());
      for(uint i = 0; i<m_copulas.Size(); ++i)
        {
         if(!m_copulas[i])
            continue;
         m_copulas[i].Set_eps(_eps);
         cdf_+=m_weights[i]*m_copulas[i].Copula_CDF(u,v);
        }
      return cdf_;
     }
   vector            Conditional_Probability(vector& u, vector& v, double _eps = 1.e-5)
     {
      vector result_ = vector::Zeros(u.Size());
      for(uint i = 0; i<m_copulas.Size(); ++i)
        {
         if(!m_copulas[i])
            continue;
         m_copulas[i].Set_eps(_eps);
         result_+=m_weights[i]*m_copulas[i].Conditional_Probability(u,v);
        }
      return result_;
     }
   virtual double    Penalized_Log_Likelihood(matrix &data,double g_scad, double a_scad)
     {
      return EMPTY_VALUE;
     }
   matrix            Sample(ulong num)
     {
      vector r = np::arange(m_weights.Size());
      vector cop_identities = np::choice(r,m_weights,num);
      matrix sample_pairs = matrix::Zeros(num,2);
      matrix temp;
      for(ulong i = 0; i<cop_identities.Size(); ++i)
        {
         temp = m_copulas[uint(cop_identities[i])].Sample(1);
         sample_pairs.Row(temp.Row(0),i);
        }
      return sample_pairs;
     }
   uint              Mixture_Size(void)
     {
      return m_num_cops;
     }
   vector            Params(void)
     {
      return m_cop_params;
     }
   vector            Weights(void)
     {
      return m_weights;
     }
   virtual int       Type(void)
     {
      return int(m_mixture);
     }
   virtual bool      Save(const int file_handle)
     {
      if(file_handle!=INVALID_HANDLE)
        {
         if(FileWriteLong(file_handle,-1)==sizeof(long))
           {
            if(FileWriteInteger(file_handle,int(m_copulas.Size()),INT_VALUE)!=INT_VALUE)
               return(false);
            if(FileWriteInteger(file_handle,int(m_mixture),INT_VALUE)!=INT_VALUE)
               return(false);
            if(FileWriteLong(file_handle,long(m_weights.Size()))!=sizeof(long))
               return(false);
            if(m_weights.Size())
              {
               for(ulong i = 0; i<m_weights.Size(); ++i)
                  if(FileWriteDouble(file_handle,m_weights[i])!=sizeof(double))
                     return(false);
              }
           }
        }
      else
         return false;

      for(ulong i = 0; i<m_copulas.Size(); ++i)
        {
         if(CheckPointer(m_copulas[i])!=POINTER_INVALID)
           {
            if(!m_copulas[i].Save(file_handle))
               return false;
           }
        }
      return true;
     }
   virtual bool      Load(const int file_handle)
     {
      if(file_handle!=INVALID_HANDLE)
        {
         long ff = FileReadLong(file_handle);
         if(ff == -1)
           {
            int size = FileReadInteger(file_handle,INT_VALUE);
            ENUM_COPULA_MIX mixture = ENUM_COPULA_MIX(FileReadInteger(file_handle,INT_VALUE));
            if(ArraySize(m_copulas)!=size || mixture!=m_mixture)
              {
               Print(__FUNCTION__, " You are attempting to load a model incompatible with this class ");
               return false;
              }
            ulong size_weights = (ulong)FileReadLong(file_handle);
            if(size_weights)
              {
               m_weights = vector::Zeros(size_weights);
               switch(m_mixture)
                 {
                  case CFG_MIX:
                     m_cop_params = vector::Zeros(size_weights);
                     break;
                  case CTG_MIX:
                     m_cop_params = vector::Zeros(size_weights+1);
                     break;
                 }
               for(ulong i = 0; i<size_weights; ++i)
                  m_weights[i] = FileReadDouble(file_handle);
              }
           }
         else
            return false;
        }
      else
         return false;
      //---
      for(ulong i = 0,k = 0; i<m_copulas.Size(); ++i)
        {
         m_copulas[i] = bivariate_copula(file_handle);
         if(m_copulas[i]==NULL)
           {
            Print(__FUNCTION__, " failed to load a bivariate copula from file ", GetLastError());
            return false;
           }
         if(m_weights.Size())
           {
            switch(ENUM_COPULA_TYPE(m_copulas[i].Type()))
              {
               case CLAYTON_COPULA:
               case GUMBEL_COPULA:
               case JOE_COPULA:
               case N13_COPULA:
               case N14_COPULA:
               case FRANK_COPULA:
                  m_cop_params[k++] = m_copulas[i].Get_theta();
                  break;
               case STUDENT_COPULA:
                  m_cop_params[k++] = m_copulas[i].Get_nu();
                  m_cop_params[k++] = m_copulas[i].Get_rho();
                  break;
              }
           }
        }
      return true;
     }
  };
//+------------------------------------------------------------------+
//|Implements the 5-fold cross-validation tuning process of the SCAD |
//|parameters                                                        |
//+------------------------------------------------------------------+
bool tune_scad_parameters(matrix &data, double gama_start, double gama_stop, ulong gamma_count, double a_start, double a_stop,ulong a_count, ENUM_COPULA_MIX mix, bool shuffle, int seed, bool show_details, double& out_gamma, double& out_a)
  {
   np::Folds folds[];
   if(!np::kfold(data.Rows(),folds,5,shuffle,seed))
      return false;

   CMixedCopula* mixedcop = NULL;

   switch(mix)
     {
      case CFG_MIX:
         mixedcop = new CFGMixCop();
         break;
      case CTG_MIX:
         mixedcop = new CTGMixCop();
         break;
      case CFJ_MIX:
         mixedcop = new CFJMixCop();
         break;
      case JFG_MIX:
         mixedcop = new JFGMixCop();
         break;
     }

   if(CheckPointer(mixedcop) == POINTER_INVALID)
     {
      Print(__FUNCTION__, " failed to initialize ", EnumToString(mix), " copula ");
      return false;
     }

   matrix traindata,testdata;
   double ll, best_score;
   vector scores = vector::Zeros(folds.Size());
   vector gamma_grid = np::linspace(gama_start,gama_stop,gamma_count);
   vector a_grid = np::linspace(a_start,a_stop,a_count);
   best_score = -DBL_MAX;
   bool use_mle = false;
   double mle = 0;
   double p_mle = 0;
   for(ulong i = 0; i<gamma_grid.Size(); ++i)
     {
      for(ulong j = 0; j<a_grid.Size(); ++j)
        {
         for(uint k = 0; k<folds.Size() && !IsStopped(); ++k)
           {
            traindata = np::selectMatrixRows(data,folds[k].train_indices);
            testdata = np::selectMatrixRows(data,folds[k].test_indices);
            ll = mixedcop.Fit(traindata,25,gamma_grid[i],a_grid[j]);

            if(ll == EMPTY_VALUE)
              {
               Print(__FUNCTION__, " error fitting data ", "| sparcity ", gamma_grid[i], " bias ", a_grid[j]);
               continue;
              }

            scores[k] = mixedcop.Penalized_Log_Likelihood(testdata,gamma_grid[i],a_grid[j]);

           }
         ll = scores.Sum();
         if(show_details)
            Print(__FUNCTION__, "| sparcity: ", gamma_grid[i], "| bias: ", a_grid[j], "| likelihood  ", ll, " \nweights ", mixedcop.Weights(), "| params ", mixedcop.Params());
         if(ll>best_score)
           {
            best_score = ll;
            out_a = a_grid[j];
            out_gamma = gamma_grid[i];
           }
        }
     }

   delete mixedcop;
   return true;
  }

//+------------------------------------------------------------------+
//| Joe, Frank and Gumbel mixed copula                               |
//+------------------------------------------------------------------+
class JFGMixCop:public CMixedCopula
  {
protected:
   //+------------------------------------------------------------------+
   //| Derived class from CNDimensional_Func                            |
   //+------------------------------------------------------------------+
   class CNDimensional_Func1 : public CNDimensional_Func
     {
   protected:
      vector            m_u1,m_u2,m_w;
      JFGMixCop*        m_obj;
      bool              m_centered,m_individual,m_ifpenalty;
      double            m_epsilon,m_gscad,m_ascad,m_multiplier;
   public:
      //--- constructor, destructor
                     CNDimensional_Func1(void) {}
                    ~CNDimensional_Func1(void)
        {
         m_obj = NULL;
        }
      void           set_params(JFGMixCop &obj, vector& u1, vector& u2,vector& weights, double gamma_scad, double a_scad, bool if_penalty = true, double multiplier= 1.,bool center = false, double ep=EMPTY_VALUE)
        {
         m_u1  = u1;
         m_u2 = u2;
         m_w = weights;
         m_obj = GetPointer(obj);
         m_centered = center;
         m_epsilon = ep;
         m_gscad = gamma_scad;
         m_ascad = a_scad;
         m_ifpenalty = if_penalty;
         m_multiplier = multiplier;
        }
      virtual void      Func(double &x[],double &func,CObject &obj)
        {
         vector _x;
         _x.Assign(x);
         func = m_obj.objective(_x,m_u1,m_u2,m_w,m_gscad,m_ascad,m_ifpenalty,m_multiplier);
        }

      virtual void      Func(CRowDouble &x,double &func,CObject &obj)
        {
         vector _x = x.ToVector();
         func = m_obj.objective(_x,m_u1,m_u2,m_w,m_gscad,m_ascad,m_ifpenalty,m_multiplier);
        }
     };

   double            _ml_qfunction(vector& u1, vector& u2, vector& cop_params,vector& weights, double gamma_scad, double a_scad, bool if_penalty = true, double multiplier= 1.)
     {
      ulong num = u1.Size();


      m_copulas[0].Set_theta(cop_params[0]);
      m_copulas[1].Set_theta(cop_params[1]);
      m_copulas[2].Set_theta(cop_params[2]);

      vector lljoe,llfrank,llgumbel,llmix;

      lljoe = m_copulas[0].Copula_PDF(u1,u2,true,true);
      llfrank = m_copulas[1].Copula_PDF(u1,u2,true,true);
      llgumbel = m_copulas[2].Copula_PDF(u1,u2,true,true);

      llmix = weights[0]*lljoe+weights[1]*llfrank+(1.0-weights[0]-weights[1])*llgumbel;
      llmix.Clip(1.e-15,DBL_MAX);
      double llsum = log(llmix).Sum();
      double sum = 0;
      if(gamma_scad>0.0 && a_scad>2.)
         for(ulong i = 0; i<weights.Size(); sum+=scad_penalty(weights[i],gamma_scad,a_scad), ++i);
      double penalty = double(num)*sum;

      return (llsum-penalty*double(if_penalty))*multiplier;
     }
   vector            _maximization_step(matrix&qd, double gamma_scad, double a_scad, vector& cop_params, vector& weights)
     {
      vector u1 = qd.Col(0);
      vector u2 = qd.Col(1);

      double eps = 1.e-3;

      double _x_[3];
      for(uint i = 0; i<_x_.Size(); _x_[i] = cop_params[i], ++i);

      double bndl[3] = {1.,-50.,1.};
      double bndu[3] = {100.,50.,100.};

      CObject             obj;
      CNDimensional_Func1 ffunc;
      CNDimensional_Rep   frep;

      ffunc.set_params(this,u1,u2,weights,gamma_scad,a_scad,bool(gamma_scad>0.0&&a_scad>2.0),-1);

      CMinBLEICStateShell state;

      CMinBLEICReportShell rep;

      double epsg=eps;

      double epsf=0;

      double epsx=0;

      double epso=eps;

      double epsi=eps;

      double diffstep=1.0e-6;

      CAlglib::MinBLEICCreateF(_x_,diffstep,state);

      CAlglib::MinBLEICSetBC(state,bndl,bndu);

      CAlglib::MinBLEICSetInnerCond(state,epsg,epsf,epsx);

      CAlglib::MinBLEICSetOuterCond(state,epso,epsi);

      CAlglib::MinBLEICOptimize(state,ffunc,frep,false,obj);

      CAlglib::MinBLEICResults(state,_x_,rep);

      vector out = vector::Zeros(0);

      int termination_reason = rep.GetTerminationType();

      if(termination_reason<0)
        {
         Print(__FUNCTION__, " termination reason ", termination_reason);
         return out;
        }

      out.Assign(_x_);

      return out;
     }
   vector            _expectation_step(matrix &qd,double gamma_scad, double a_scad,vector& cop_params,vector &weights)
     {
      //---
      ulong num = qd.Rows();
      vector u1 = qd.Col(0);
      vector u2 = qd.Col(1);
      //---
      double dif = 1;
      double tol_weight = 1.e-2;
      long iteration = 0;
      //---
      for(ulong i = 0; i<3; ++i)
         m_copulas[i].Set_theta(cop_params[i]);
      //---
      vector nweights;
      //---
      if(gamma_scad>0.0 && a_scad>2.0)
        {
         while(dif>tol_weight && iteration<10)
           {
            nweights = vector::Zeros(3);
            nweights.Fill(double("nan"));
            iteration+=1;
            for(ulong i = 0; i<3; ++i)
              {
               vector sum_ml_1st = vector::Zeros(u1.Size());
               double sum,sum_ml,numerator,denominator;
               sum = sum_ml = numerator = denominator = 1.e-12;
               for(ulong t = 0; t<num; ++t)
                 {
                  sum = 1.e-12;
                  for(ulong j = 0; j<3; ++j)
                     sum+=weights[j]*m_copulas[j].Copula_PDF(u1[t],u2[t],true,true);
                  sum_ml_1st[t] = weights[i]*m_copulas[i].Copula_PDF(u1[t],u2[t],true,true)/sum;
                 }
               sum_ml = sum_ml_1st.Sum();
               numerator = weights[i]*scad_derivative(weights[i],gamma_scad,a_scad)-sum_ml/double(num);
               for(ulong j = 0; j<3; ++j)
                  denominator+=weights[j]*scad_derivative(weights[j],gamma_scad,a_scad);
               denominator-=1.;
               nweights[i] = fabs(numerator/denominator);
              }
            dif = MathAbs(weights-nweights).Sum();
            weights = nweights;
           }
        }
      else
        {
         matrix wd = matrix::Zeros(weights.Size(),num);
         vector td, temp;
         while(dif>tol_weight && iteration<10)
           {
            nweights = vector::Zeros(weights.Size());
            iteration+=1;

            for(ulong i = 0; i<wd.Rows(); ++i)
               if(!wd.Row(weights[i]*m_copulas[i].Copula_PDF(u1,u2,true,true),i))
                 {
                  Print(__FUNCTION__, " row insertion error ", GetLastError());
                  return vector::Zeros(0);
                 }
            td = wd.Sum(0);
            td.Clip(1.e-12,DBL_MAX);
            for(ulong i = 0; i<weights.Size(); ++i)
              {
               temp = (wd.Row(i)/td);
               nweights[i] = fabs(temp.Sum()/double(num));
              }
            dif = MathAbs(weights-nweights).Sum();
            weights = nweights;
           }
        }
      //---
      return weights;
     }
   matrix            _fit_quantile_em(matrix &qd, ulong max_iter,double gamma_scad, double a_scad)
     {
      vector init_weights = {0.33, 0.33, 1. - 0.33 - 0.33};
      vector init_cop_params = {3,4,5};
      vector weights = _expectation_step(qd,gamma_scad,a_scad,init_cop_params,init_weights);
      if(!weights.Size())
         return matrix::Zeros(0,0);
      vector cop_params = _maximization_step(qd,gamma_scad,a_scad,init_cop_params,weights);
      if(!cop_params.Size())
         return matrix::Zeros(0,0);
      vector oldparams,newparams;
      oldparams = newparams = vector::Zeros(6);

      np::vectorCopy(oldparams,init_weights,0,3);
      np::vectorCopy(oldparams,init_cop_params,3);

      np::vectorCopy(newparams,weights,0,3);
      np::vectorCopy(newparams,cop_params,3);

      double ll_diff = MathAbs(oldparams-newparams).Sum();
      ulong i = 1;
      while(i<max_iter && ll_diff>(5.*1.e-2))
        {
         np::vectorCopy(oldparams,weights,0,3);
         np::vectorCopy(oldparams,cop_params,3);

         weights = _expectation_step(qd,gamma_scad,a_scad,cop_params,weights);
         if(!weights.Size())
           {
            Print(__FUNCTION__, " invalid weights ", i);
            return matrix::Zeros(0,0);
           }

         cop_params = _maximization_step(qd,gamma_scad,a_scad,cop_params,weights);
         if(!cop_params.Size())
           {
            Print(__FUNCTION__, " failed convergence at iteration ", i);
            return matrix::Zeros(0,0);
           }

         np::vectorCopy(newparams,weights,0,3);
         np::vectorCopy(newparams,cop_params,3);

         ll_diff = MathAbs(oldparams-newparams).Sum();
         i+=1;
        }

      matrix out = matrix::Zeros(3,3);
      out.Row(weights,0);
      out.Row(cop_params,1);
      return out;
     }
public:
                     JFGMixCop(void)
     {
      m_mixture = JFG_MIX;
      m_num_cops = 3;
      ArrayResize(m_copulas,int(m_num_cops));
      for(uint i = 0; i<m_num_cops; ++i)
         m_copulas[i] = NULL;
     }

                     JFGMixCop(vector &params, vector& weights)
     {
      m_mixture = JFG_MIX;
      m_num_cops = 3;
      ArrayResize(m_copulas,int(m_num_cops));
      if(params.Size()<3 || weights.Size()<3)
         Print(__FUNCTION__, " invalid parameters ");
      else
        {
         m_weights = weights;
         m_cop_params = params;
         for(ulong i = 0; i<3; ++i)
           {
            if(!i)
               m_copulas[i] = new CJoe();
            else
               if(i==1)
                  m_copulas[i] = new CFrank();
               else
                  m_copulas[i] = new CGumbel();
            m_copulas[i].Set_theta(m_cop_params[i]);
           }
        }
     }
                    ~JFGMixCop(void)
     {

     }

   double                   objective(vector& x,vector& u1, vector& u2,vector& weights, double gamma_scad, double a_scad, bool if_penalty = true, double multiplier= 1.)
     {
      return  _ml_qfunction(u1,u2,x,weights,gamma_scad,a_scad,if_penalty,multiplier);
     }
   virtual double            Fit(matrix& qdata, ulong maxiter = 50, double gamma_scad = 0.0,double a_scad = 0.0, double weight_margin = 1.e-2) override
     {
      if(CheckPointer(m_copulas[0])==POINTER_INVALID)
         m_copulas[0] = new CJoe();
      if(CheckPointer(m_copulas[1])==POINTER_INVALID)
         m_copulas[1] = new CFrank();
      if(CheckPointer(m_copulas[2])==POINTER_INVALID)
         m_copulas[2] = new CGumbel();

      matrix wc = _fit_quantile_em(qdata,maxiter,gamma_scad,a_scad);
      if(!wc.Rows())
         return EMPTY_VALUE;
      wc.Row(adjust_weights(wc.Row(0),weight_margin),0);
      m_weights = wc.Row(0);
      m_cop_params = wc.Row(1);

      m_copulas[0].Set_theta(m_cop_params[0]);
      m_copulas[1].Set_theta(m_cop_params[1]);
      m_copulas[2].Set_theta(m_cop_params[2]);

      vector u1 = qdata.Col(0);
      vector u2 = qdata.Col(1);

      return _ml_qfunction(u1,u2,wc.Row(1),wc.Row(0),gamma_scad,a_scad,false);
     }
   virtual double    Penalized_Log_Likelihood(matrix &qdata,double g_scad, double a_scad)
     {
      return _ml_qfunction(qdata.Col(0),qdata.Col(1),m_cop_params,m_weights,g_scad,a_scad,true);
     }
   CJoe*             JoeCopula(void)
     {
      if(m_copulas[0] == NULL)
        {
         m_copulas[0] = new CJoe();
         m_copulas[0].Set_theta(m_cop_params[0]);
        }

      return (CJoe*)m_copulas[0];
     }
   CFrank*           FrankCopula(void)
     {
      if(m_copulas[1]==NULL)
        {
         m_copulas[1] = new CFrank();
         m_copulas[1].Set_theta(m_cop_params[1]);
        }

      return (CFrank*)m_copulas[1];
     }
   CGumbel*          GumbelCopula(void)
     {
      if(m_copulas[2]==NULL)
        {
         m_copulas[2] = new CGumbel();
         m_copulas[2].Set_theta(m_cop_params[2]);
        }

      return (CGumbel*)m_copulas[2];
     }

  };
//+------------------------------------------------------------------+
//|Clayton, Frank and Gumbel mixed copula.                           |
//+------------------------------------------------------------------+
class CFGMixCop:public CMixedCopula
  {
protected:
   //+------------------------------------------------------------------+
   //| Derived class from CNDimensional_Func                            |
   //+------------------------------------------------------------------+
   class CNDimensional_Func1 : public CNDimensional_Func
     {
   protected:
      vector            m_u1,m_u2,m_w;
      CFGMixCop*        m_obj;
      bool              m_centered,m_individual,m_ifpenalty;
      double            m_epsilon,m_gscad,m_ascad,m_multiplier;
   public:
      //--- constructor, destructor
                     CNDimensional_Func1(void) {}
                    ~CNDimensional_Func1(void)
        {
         m_obj = NULL;
        }
      void           set_params(CFGMixCop &obj, vector& u1, vector& u2,vector& weights, double gamma_scad, double a_scad, bool if_penalty = true, double multiplier= 1.,bool center = false, double ep=EMPTY_VALUE)
        {
         m_u1  = u1;
         m_u2 = u2;
         m_w = weights;
         m_obj = GetPointer(obj);
         m_centered = center;
         m_epsilon = ep;
         m_gscad = gamma_scad;
         m_ascad = a_scad;
         m_ifpenalty = if_penalty;
         m_multiplier = multiplier;
        }
      virtual void      Func(double &x[],double &func,CObject &obj)
        {
         vector _x;
         _x.Assign(x);
         func = m_obj.objective(_x,m_u1,m_u2,m_w,m_gscad,m_ascad,m_ifpenalty,m_multiplier);
        }

      virtual void      Func(CRowDouble &x,double &func,CObject &obj)
        {
         vector _x = x.ToVector();
         func = m_obj.objective(_x,m_u1,m_u2,m_w,m_gscad,m_ascad,m_ifpenalty,m_multiplier);
        }
     };

   double            _ml_qfunction(vector& u1, vector& u2, vector& cop_params,vector& weights, double gamma_scad, double a_scad, bool if_penalty = true, double multiplier= 1.)
     {
      ulong num = u1.Size();


      m_copulas[0].Set_theta(cop_params[0]);
      m_copulas[1].Set_theta(cop_params[1]);
      m_copulas[2].Set_theta(cop_params[2]);

      vector llclayton,llfrank,llgumbel,llmix;

      llclayton = m_copulas[0].Copula_PDF(u1,u2,true,true);
      llfrank = m_copulas[1].Copula_PDF(u1,u2,true,true);
      llgumbel = m_copulas[2].Copula_PDF(u1,u2,true,true);

      llmix = weights[0]*llclayton+weights[1]*llfrank+(1.0-weights[0]-weights[1])*llgumbel;
      llmix.Clip(1.e-15,DBL_MAX);
      double llsum = log(llmix).Sum();
      double sum = 0;
      if(gamma_scad>0.0 && a_scad>2.)
         for(ulong i = 0; i<weights.Size(); sum+=scad_penalty(weights[i],gamma_scad,a_scad), ++i);
      double penalty = double(num)*sum;

      return (llsum-penalty*double(if_penalty))*multiplier;
     }
   vector            _maximization_step(matrix&qd, double gamma_scad, double a_scad, vector& cop_params, vector& weights)
     {
      vector u1 = qd.Col(0);
      vector u2 = qd.Col(1);

      double eps = 1.e-3;

      double _x_[3];
      for(uint i = 0; i<_x_.Size(); _x_[i] = cop_params[i], ++i);

      double bndl[3] = {-1.,-50.,1.};
      double bndu[3] = {100.,50.,100.};

      CObject             obj;
      CNDimensional_Func1 ffunc;
      CNDimensional_Rep   frep;

      ffunc.set_params(this,u1,u2,weights,gamma_scad,a_scad,bool(gamma_scad>0.0&&a_scad>2.0),-1);

      CMinBLEICStateShell state;

      CMinBLEICReportShell rep;

      double epsg=eps;

      double epsf=0;

      double epsx=0;

      double epso=eps;

      double epsi=eps;

      double diffstep=1.0e-6;

      CAlglib::MinBLEICCreateF(_x_,diffstep,state);

      CAlglib::MinBLEICSetBC(state,bndl,bndu);

      CAlglib::MinBLEICSetInnerCond(state,epsg,epsf,epsx);

      CAlglib::MinBLEICSetOuterCond(state,epso,epsi);

      CAlglib::MinBLEICOptimize(state,ffunc,frep,false,obj);

      CAlglib::MinBLEICResults(state,_x_,rep);

      vector out = vector::Zeros(0);

      int termination_reason = rep.GetTerminationType();

      if(termination_reason<0)
        {
         Print(__FUNCTION__, " termination reason ", termination_reason);
         return out;
        }

      out.Assign(_x_);

      return out;
     }
   vector            _expectation_step(matrix &qd,double gamma_scad, double a_scad,vector& cop_params,vector &weights)
     {
      //---
      ulong num = qd.Rows();
      vector u1 = qd.Col(0);
      vector u2 = qd.Col(1);
      //---
      double dif = 1;
      double tol_weight = 1.e-2;
      long iteration = 0;
      //---
      for(ulong i = 0; i<3; ++i)
         m_copulas[i].Set_theta(cop_params[i]);
      //---
      vector nweights;
      //---
      if(gamma_scad>0.0 && a_scad>2.0)
        {
         while(dif>tol_weight && iteration<10)
           {
            nweights = vector::Zeros(3);
            nweights.Fill(double("nan"));
            iteration+=1;
            for(ulong i = 0; i<3; ++i)
              {
               vector sum_ml_1st = vector::Zeros(u1.Size());
               double sum,sum_ml,numerator,denominator;
               sum = sum_ml = numerator = denominator = 1.e-12;
               for(ulong t = 0; t<num; ++t)
                 {
                  sum = 1.e-12;
                  for(ulong j = 0; j<3; ++j)
                     sum+=weights[j]*m_copulas[j].Copula_PDF(u1[t],u2[t],true,true);
                  sum_ml_1st[t] = weights[i]*m_copulas[i].Copula_PDF(u1[t],u2[t],true,true)/sum;
                 }
               sum_ml = sum_ml_1st.Sum();
               numerator = weights[i]*scad_derivative(weights[i],gamma_scad,a_scad)-sum_ml/double(num);
               for(ulong j = 0; j<3; ++j)
                  denominator+=weights[j]*scad_derivative(weights[j],gamma_scad,a_scad);
               denominator-=1.;
               nweights[i] = fabs(numerator/denominator);
              }
            dif = MathAbs(weights-nweights).Sum();
            weights = nweights;
           }
        }
      else
        {
         matrix wd = matrix::Zeros(weights.Size(),num);
         vector td, temp;
         while(dif>tol_weight && iteration<10)
           {
            nweights = vector::Zeros(weights.Size());
            iteration+=1;

            for(ulong i = 0; i<wd.Rows(); ++i)
               if(!wd.Row(weights[i]*m_copulas[i].Copula_PDF(u1,u2,true,true),i))
                 {
                  Print(__FUNCTION__, " row insertion error ", GetLastError());
                  return vector::Zeros(0);
                 }
            td = wd.Sum(0);
            td.Clip(1.e-12,DBL_MAX);
            for(ulong i = 0; i<weights.Size(); ++i)
              {
               temp = (wd.Row(i)/td);
               nweights[i] = fabs(temp.Sum()/double(num));
              }
            dif = MathAbs(weights-nweights).Sum();
            weights = nweights;
           }
        }
      //---
      return weights;
     }
   matrix            _fit_quantile_em(matrix &qd, ulong max_iter,double gamma_scad, double a_scad)
     {
      vector init_weights = {0.33, 0.33, 1. - 0.33 - 0.33};
      vector init_cop_params = {3,4,5};
      vector weights = _expectation_step(qd,gamma_scad,a_scad,init_cop_params,init_weights);
      if(!weights.Size())
         return matrix::Zeros(0,0);
      vector cop_params = _maximization_step(qd,gamma_scad,a_scad,init_cop_params,weights);
      if(!cop_params.Size())
         return matrix::Zeros(0,0);
      vector oldparams,newparams;
      oldparams = newparams = vector::Zeros(6);

      np::vectorCopy(oldparams,init_weights,0,3);
      np::vectorCopy(oldparams,init_cop_params,3);

      np::vectorCopy(newparams,weights,0,3);
      np::vectorCopy(newparams,cop_params,3);

      double ll_diff = MathAbs(oldparams-newparams).Sum();
      ulong i = 1;
      while(i<max_iter && ll_diff>(5.*1.e-2))
        {
         np::vectorCopy(oldparams,weights,0,3);
         np::vectorCopy(oldparams,cop_params,3);

         weights = _expectation_step(qd,gamma_scad,a_scad,cop_params,weights);
         if(!weights.Size())
           {
            Print(__FUNCTION__, " invalid weights ", i);
            return matrix::Zeros(0,0);
           }

         cop_params = _maximization_step(qd,gamma_scad,a_scad,cop_params,weights);
         if(!cop_params.Size())
           {
            Print(__FUNCTION__, " failed convergence at iteration ", i);
            return matrix::Zeros(0,0);
           }

         np::vectorCopy(newparams,weights,0,3);
         np::vectorCopy(newparams,cop_params,3);

         ll_diff = MathAbs(oldparams-newparams).Sum();
         i+=1;
        }

      matrix out = matrix::Zeros(3,3);
      out.Row(weights,0);
      out.Row(cop_params,1);
      return out;
     }
public:
                     CFGMixCop(void)
     {
      m_mixture = CFG_MIX;
      m_num_cops = 3;
      ArrayResize(m_copulas,int(m_num_cops));
      for(uint i = 0; i<m_num_cops; ++i)
         m_copulas[i] = NULL;
     }

                     CFGMixCop(vector &params, vector& weights)
     {
      m_mixture = CFG_MIX;
      m_num_cops = 3;
      ArrayResize(m_copulas,int(m_num_cops));
      if(params.Size()<3 || weights.Size()<3)
         Print(__FUNCTION__, " invalid parameters ");
      else
        {
         m_weights = weights;
         m_cop_params = params;
         for(ulong i = 0; i<3; ++i)
           {
            if(!i)
               m_copulas[i] = new CClayton();
            else
               if(i==1)
                  m_copulas[i] = new CFrank();
               else
                  m_copulas[i] = new CGumbel();
            m_copulas[i].Set_theta(m_cop_params[i]);
           }
        }
     }
                    ~CFGMixCop(void)
     {

     }

   double                   objective(vector& x,vector& u1, vector& u2,vector& weights, double gamma_scad, double a_scad, bool if_penalty = true, double multiplier= 1.)
     {
      return  _ml_qfunction(u1,u2,x,weights,gamma_scad,a_scad,if_penalty,multiplier);
     }
   virtual double            Fit(matrix& qdata, ulong maxiter = 50, double gamma_scad = 0.0,double a_scad = 0.0, double weight_margin = 1.e-2) override
     {
      if(CheckPointer(m_copulas[0])==POINTER_INVALID)
         m_copulas[0] = new CClayton();
      if(CheckPointer(m_copulas[1])==POINTER_INVALID)
         m_copulas[1] = new CFrank();
      if(CheckPointer(m_copulas[2])==POINTER_INVALID)
         m_copulas[2] = new CGumbel();

      matrix wc = _fit_quantile_em(qdata,maxiter,gamma_scad,a_scad);
      if(!wc.Rows())
         return EMPTY_VALUE;
      wc.Row(adjust_weights(wc.Row(0),weight_margin),0);
      m_weights = wc.Row(0);
      m_cop_params = wc.Row(1);

      m_copulas[0].Set_theta(m_cop_params[0]);
      m_copulas[1].Set_theta(m_cop_params[1]);
      m_copulas[2].Set_theta(m_cop_params[2]);

      vector u1 = qdata.Col(0);
      vector u2 = qdata.Col(1);

      return _ml_qfunction(u1,u2,wc.Row(1),wc.Row(0),gamma_scad,a_scad,false);
     }
   virtual double    Penalized_Log_Likelihood(matrix &qdata,double g_scad, double a_scad)
     {
      return _ml_qfunction(qdata.Col(0),qdata.Col(1),m_cop_params,m_weights,g_scad,a_scad,true);
     }
   CClayton*         ClaytonCopula(void)
     {
      if(m_copulas[0] == NULL)
        {
         m_copulas[0] = new CClayton();
         m_copulas[0].Set_theta(m_cop_params[0]);
        }

      return (CClayton*)m_copulas[0];
     }
   CFrank*           FrankCopula(void)
     {
      if(m_copulas[1]==NULL)
        {
         m_copulas[1] = new CFrank();
         m_copulas[1].Set_theta(m_cop_params[1]);
        }

      return (CFrank*)m_copulas[1];
     }
   CGumbel*          GumbelCopula(void)
     {
      if(m_copulas[2]==NULL)
        {
         m_copulas[2] = new CGumbel();
         m_copulas[2].Set_theta(m_cop_params[2]);
        }

      return (CGumbel*)m_copulas[2];
     }

  };
//+------------------------------------------------------------------+
//| Clayton, Student-t and Gumbel mixed copula.                      |
//+------------------------------------------------------------------+
class CTGMixCop:public CMixedCopula
  {
protected:
   vector            _get_epsilon(vector &x, double s, ulong n, double epsilon = EMPTY_VALUE)
     {
      vector out;
      if(epsilon == EMPTY_VALUE)
         out = pow(DBL_EPSILON,1./s)*np::maximum(MathAbs(x),0.1);
      else
        {
         out = vector::Zeros(n);
         out.Fill(epsilon);
        }
      return out;
     }

   //+------------------------------------------------------------------+
   //| Derived class from CNDimensional_Func                            |
   //+------------------------------------------------------------------+
   class CNDimensional_Func1 : public CNDimensional_Func
     {
   protected:
      vector            m_u1,m_u2,m_w;
      CTGMixCop*        m_obj;
      bool              m_centered,m_individual,m_ifpenalty;
      double            m_epsilon,m_gscad,m_ascad,m_multiplier;
   public:
      //--- constructor, destructor
                     CNDimensional_Func1(void) {}
                    ~CNDimensional_Func1(void)
        {
         m_obj = NULL;
        }
      void           set_params(CTGMixCop &obj, vector& u1, vector& u2,vector& weights, double gamma_scad, double a_scad, bool if_penalty = true, double multiplier= 1.,bool center = false, double ep=EMPTY_VALUE)
        {
         m_u1  = u1;
         m_u2 = u2;
         m_w = weights;
         m_obj = GetPointer(obj);
         m_centered = center;
         m_epsilon = ep;
         m_gscad = gamma_scad;
         m_ascad = a_scad;
         m_ifpenalty = if_penalty;
         m_multiplier = multiplier;
        }
      virtual void      Func(double &x[],double &func,CObject &obj)
        {
         vector _x;
         _x.Assign(x);
         func = m_obj.objective_1(_x,m_u1,m_u2,m_w,m_gscad,m_ascad,m_ifpenalty,m_multiplier);
        }

      virtual void      Func(CRowDouble &x,double &func,CObject &obj)
        {
         vector _x = x.ToVector();
         func = m_obj.objective_1(_x,m_u1,m_u2,m_w,m_gscad,m_ascad,m_ifpenalty,m_multiplier);
        }
     };
   //+------------------------------------------------------------------+
   //| Derived class from CNDimensional_Func                            |
   //+------------------------------------------------------------------+
   class CNDimensional_Func2 : public CNDimensional_Func
     {
   protected:
      vector            m_u1,m_u2,m_w;
      CTGMixCop*        m_obj;
      bool              m_centered,m_individual,m_ifpenalty;
      double            m_epsilon,m_gscad,m_ascad,m_multiplier;
   public:
      //--- constructor, destructor
                     CNDimensional_Func2(void) {}
                    ~CNDimensional_Func2(void)
        {
         m_obj = NULL;
        }
      void           set_params(CTGMixCop &obj, vector& u1, vector& u2,vector& weights, double gamma_scad, double a_scad, bool if_penalty = true, double multiplier= 1.,bool center = false, double ep=EMPTY_VALUE)
        {
         m_u1  = u1;
         m_u2 = u2;
         m_w = weights;
         m_obj = GetPointer(obj);
         m_centered = center;
         m_epsilon = ep;
         m_gscad = gamma_scad;
         m_ascad = a_scad;
         m_ifpenalty = if_penalty;
         m_multiplier = multiplier;
        }
      virtual void      Func(double &x[],double &func,CObject &obj)
        {
         vector _x;
         _x.Assign(x);
         func = m_obj.objective_2(_x,m_u1,m_u2,m_w,m_gscad,m_ascad,m_ifpenalty,m_multiplier);
        }

      virtual void      Func(CRowDouble &x,double &func,CObject &obj)
        {
         vector _x = x.ToVector();
         func = m_obj.objective_2(_x,m_u1,m_u2,m_w,m_gscad,m_ascad,m_ifpenalty,m_multiplier);
        }
     };

   void              _fit_quantile_em(matrix& qd, ulong max_iter, double gamma_scad, double a_scad, vector& out[])
     {
      vector init_weights = {0.33,0.33,1.-0.66};
      vector init_cop_params = {3,0.5,4,5};

      vector weights = _expectation_step(qd,gamma_scad,a_scad,init_cop_params,init_weights);
      vector cop_params = _maximization_step(qd,gamma_scad,a_scad,init_cop_params,weights);
      if(!cop_params.Size())
        {
         Print(__FUNCTION__, " failed convergence at 2 step initialization ", weights);
         return;
        }

      vector oldparams,newparams;
      oldparams=newparams=vector::Zeros(7);
      np::vectorCopy(oldparams,init_weights,0,3);
      np::vectorCopy(oldparams,init_cop_params,3);

      np::vectorCopy(newparams,weights,0,3);
      np::vectorCopy(newparams,cop_params,3);

      double ll_diff = MathAbs(oldparams-newparams).Sum();
      ulong i = 1;
      bool no_student = false;
      while(i<max_iter && ll_diff>(1.e-2))
        {
         no_student = false;
         np::vectorCopy(oldparams,weights,0,3);
         np::vectorCopy(oldparams,cop_params,3);

         weights = _expectation_step(qd,gamma_scad,a_scad,cop_params,weights);

         if(!weights.Size())
           {
            Print(__FUNCTION__, " invalid weights ");
            return;
           }

         if(weights[1]<1.e-2)
           {
            weights = adjust_weights(weights,1.e-2);
            cop_params = _maximization_step_no_t(qd,gamma_scad,a_scad,cop_params,weights);
            no_student = true;
           }
         else
            cop_params = _maximization_step(qd,gamma_scad,a_scad,cop_params,weights);
         //---
         if(!cop_params.Size())
           {
            Print(__FUNCTION__, " failed convergence ", weights);
            return;
           }

         np::vectorCopy(newparams,weights,0,3);
         np::vectorCopy(newparams,cop_params,3);

         ll_diff = MathAbs(oldparams-newparams).Sum();
         i+=1;
        }

      out[0] = weights;
      out[1] = cop_params;

      return;
     }
   vector            _expectation_step(matrix &qd, double gamma_scad,double a_scad, vector& cop_params, vector& weights)
     {
      ulong num = qd.Rows();

      vector u1 = qd.Col(0);
      vector u2 = qd.Col(1);

      double dif = 1;

      double tol_weight = 1.e-2;

      ulong iter = 0;

      matrix cov = {{1.,cop_params[1]},{cop_params[1],1.}};

      m_copulas[0].Set_theta(cop_params[0]);
      m_copulas[1].Set_covariance(cov);
      m_copulas[1].Set_nu(cop_params[2]);
      m_copulas[2].Set_theta(cop_params[3]);

      if(gamma_scad>0.0 && a_scad>2.)
        {
         vector nweights;
         while(dif>tol_weight && iter<10)
           {
            nweights = vector::Zeros(3);

            iter+=1;
            for(ulong i = 0; i<3; ++i)
              {
               vector sum_ml_st = u1*0.0;
               double sum;
               for(ulong t = 0; t<num; ++t)
                 {
                  sum = 1.e-12;
                  for(ulong j = 0; j<3; ++j)
                     sum+=weights[j]*m_copulas[j].Copula_PDF(u1[t],u2[t],true,true);
                  sum_ml_st[t] = weights[i]*m_copulas[i].Copula_PDF(u1[t],u2[t],true,true)/sum;
                 }
               double numerator = weights[i]*scad_derivative(weights[i],gamma_scad,a_scad)-sum_ml_st.Sum()/double(num);
               double denominator = 0.0;
               for(ulong k = 0; k<weights.Size(); ++k)
                  denominator+=weights[k]*scad_derivative(weights[k],gamma_scad,a_scad);
               denominator-=1.;
               nweights[i] = denominator?fabs(numerator/denominator):0.0;
              }
            dif  = MathAbs(weights-nweights).Sum();
            weights = nweights;
           }
        }
      else
        {
         matrix wd = matrix::Zeros(weights.Size(),num);
         vector td, temp, nweights;
         while(dif>tol_weight && iter<10)
           {
            nweights = vector::Zeros(weights.Size());
            iter+=1;

            for(ulong i = 0; i<wd.Rows(); ++i)
               if(!wd.Row(weights[i]*m_copulas[i].Copula_PDF(u1,u2,true,true),i))
                 {
                  Print(__FUNCTION__, " row insertion error ", GetLastError());
                  return vector::Zeros(0);
                 }
            td = wd.Sum(0);
            td.Clip(1.e-12,DBL_MAX);
            for(ulong i = 0; i<weights.Size(); ++i)
              {
               temp = (wd.Row(i)/td);
               nweights[i] = fabs(temp.Sum()/double(num));
              }
            dif = MathAbs(weights-nweights).Sum();
            weights = nweights;
           }
        }
      return weights;
     }
   vector            _maximization_step(matrix& qd, double gamma_scad, double a_scad, vector& cop_params, vector& weights)
     {
      vector u1 = qd.Col(0);
      vector u2 = qd.Col(1);

      double eps = 1.e-3;

      double _x_[4];
      for(uint i = 0; i<_x_.Size(); _x_[i] = cop_params[i], ++i);

      double bndl[4] = {-1.,eps,2.,1.};
      double bndu[4] = {100.,1.0 - eps,7.9999,100.};

      CObject             obj;
      CNDimensional_Func1 ffunc;
      CNDimensional_Rep   frep;

      ffunc.set_params(this,u1,u2,weights,gamma_scad,a_scad,bool(gamma_scad>0.0 && a_scad>2.0),-1);

      CMinBLEICStateShell state;

      CMinBLEICReportShell rep;

      double epsg=eps;

      double epsf=0;

      double epsx=0;

      double epso=eps;

      double epsi=eps;

      double diffstep=1.0e-6;

      CAlglib::MinBLEICCreateF(_x_,diffstep,state);

      CAlglib::MinBLEICSetBC(state,bndl,bndu);

      CAlglib::MinBLEICSetInnerCond(state,epsg,epsf,epsx);

      CAlglib::MinBLEICSetOuterCond(state,epso,epsi);

      CAlglib::MinBLEICOptimize(state,ffunc,frep,false,obj);

      CAlglib::MinBLEICResults(state,_x_,rep);

      vector out = vector::Zeros(0);

      int termination_reason = rep.GetTerminationType();

      if(termination_reason<0)
        {
         Print(__FUNCTION__, " termination reason ", termination_reason);
         return out;
        }

      out.Assign(_x_);

      return out;


     }
   double            _ml_qfunc(vector& u1, vector& u2, vector& cop_params, vector& weights, double gamma_scad, double a_scad, bool if_penalty = true, double multiplier=1.)
     {
      double num = double(u1.Size());
      matrix cov = {{1.,cop_params[1]},{cop_params[1],1.}};

      m_copulas[0].Set_theta(cop_params[0]);

      m_copulas[1].Set_covariance(cov);
      m_copulas[1].Set_nu(cop_params[2]);

      m_copulas[2].Set_theta(cop_params[3]);

      vector ll_student,ll_clayton,ll_gumbel,ll_mix;
      ll_student = m_copulas[1].Copula_PDF(u1,u2,true,true);
      ll_clayton = m_copulas[0].Copula_PDF(u1,u2,true,true);
      ll_gumbel = m_copulas[2].Copula_PDF(u1,u2,true,true);

      ll_mix = weights[0]*ll_clayton+weights[1]*ll_student+(1. - weights[0] - weights[1])*ll_gumbel;
      ll_mix.Clip(1.e-15,DBL_MAX);
      double ll_sum = log(ll_mix).Sum();

      double pen = 0.0;
      if(gamma_scad>0.0 && a_scad>2.)
         for(ulong i = 0; i<weights.Size(); ++i)
            pen+=scad_penalty(weights[i],gamma_scad,a_scad);
      pen*=num;
      return (ll_sum-pen*double(if_penalty))*multiplier;
     }
   vector            _maximization_step_no_t(matrix& qd, double gamma_scad, double a_scad, vector& cop_params, vector& weights)
     {
      vector u1 = qd.Col(0);
      vector u2 = qd.Col(1);

      double _x_[2];

      _x_[0] = cop_params[0];
      _x_[1] = cop_params[2];

      double bndl[2] = {-1.,1.};
      double bndu[2] = {100.,100.};

      double eps = 1.e-3;

      CObject             obj;
      CNDimensional_Func2 ffunc;
      CNDimensional_Rep   frep;

      ffunc.set_params(this,u1,u2,weights,gamma_scad,a_scad,bool(gamma_scad>0. && a_scad>2.),-1);

      CMinBLEICStateShell state;

      CMinBLEICReportShell rep;

      double epsg=eps;

      double epsf=0;

      double epsx=0;

      double epso=eps;

      double epsi=eps;

      double diffstep=1.0e-6;

      CAlglib::MinBLEICCreateF(_x_,diffstep,state);

      CAlglib::MinBLEICSetBC(state,bndl,bndu);

      CAlglib::MinBLEICSetInnerCond(state,epsg,epsf,epsx);

      CAlglib::MinBLEICSetOuterCond(state,epso,epsi);

      CAlglib::MinBLEICOptimize(state,ffunc,frep,false,obj);

      CAlglib::MinBLEICResults(state,_x_,rep);

      vector out = vector::Zeros(0);

      int termination_reason = rep.GetTerminationType();

      if(termination_reason<0)
        {
         Print(__FUNCTION__, " termination reason ", termination_reason);
         return out;
        }

      out.Assign(_x_);

      vector params_wou(4);

      params_wou[0] = out[0];
      params_wou[1] = cop_params[1];
      params_wou[2] = cop_params[2];
      params_wou[3] = out[1];

      return params_wou;
     }
   double            _ml_qfunc_no_t(vector& u1, vector& u2, vector& cop_params, vector& weights, double gamma_scad, double a_scad, bool if_penalty=true, double multiplier=1.)
     {
      double num = double(u1.Size());

      m_copulas[0].Set_theta(cop_params[0]);
      m_copulas[2].Set_theta(cop_params[1]);

      vector ll_clayton,ll_gumbel,ll_mix;
      ll_clayton = m_copulas[0].Copula_PDF(u1,u2,true,true);
      ll_gumbel = m_copulas[2].Copula_PDF(u1,u2,true,true);

      ll_mix = weights[0]*ll_clayton+(1. - weights[0])*ll_gumbel;
      ll_mix.Clip(1.e-15,DBL_MAX);
      double ll_sum = log(ll_mix).Sum();
      double pen = 0.0;
      if(gamma_scad>0.0 && a_scad>2.)
         for(ulong i = 0; i<weights.Size(); ++i)
            pen+=scad_penalty(weights[i],gamma_scad,a_scad);
      pen*=num;
      return (ll_sum-pen*double(if_penalty))*multiplier;

     }
public:
                     CTGMixCop(void)
     {
      m_mixture = CTG_MIX;
      m_num_cops = 3;
      ArrayResize(m_copulas,int(m_num_cops));
      for(uint i = 0; i<m_num_cops; ++i)
         m_copulas[i] = NULL;
     }
                     CTGMixCop(vector& params, vector& weights)
     {
      m_mixture = CTG_MIX;
      m_num_cops = 3;
      ArrayResize(m_copulas,int(m_num_cops));
      if(params.Size()<4 || weights.Size()<3)
         Print(__FUNCTION__, " invalid parameters ");
      else
        {
         m_weights = weights;
         m_cop_params = params;
         for(ulong i = 0; i<3; ++i)
           {
            if(i == 0 || i == 2)
              {
               if(!i)
                 {
                  m_copulas[i] = new CClayton();
                  m_copulas[i].Set_theta(m_cop_params[0]);
                 }
               else
                 {
                  m_copulas[i] = new CGumbel();
                  m_copulas[i].Set_theta(m_cop_params[3]);
                 }
              }
            else
              {
               m_copulas[i] = new CStudent();
               matrix corr = {{1.,m_cop_params[1]},{m_cop_params[1],1.}};
               m_copulas[i].Set_covariance(corr);
               m_copulas[i].Set_nu(m_cop_params[2]);
              }
           }
        }
     }
                    ~CTGMixCop(void)
     {

     }
   double                   objective_1(vector& x,vector& u1, vector& u2,vector& weights, double gamma_scad, double a_scad, bool if_penalty = true, double multiplier= 1.)
     {
      return _ml_qfunc(u1,u2,x,weights,gamma_scad,a_scad,if_penalty,multiplier);
     }
   double                   objective_2(vector& x,vector& u1, vector& u2,vector& weights, double gamma_scad, double a_scad, bool if_penalty = true, double multiplier= 1.)
     {
      return _ml_qfunc_no_t(u1,u2,x,weights,gamma_scad,a_scad,if_penalty,multiplier);
     }
   virtual double            Fit(matrix& qdata, ulong maxiter = 25, double gamma_scad = 0.0, double a_scad = 0.0,double weight_margin=1.e-2) override
     {

      if(CheckPointer(m_copulas[0])==POINTER_INVALID)
         m_copulas[0] = new CClayton();
      if(CheckPointer(m_copulas[1])==POINTER_INVALID)
         m_copulas[1] = new CStudent();
      if(CheckPointer(m_copulas[2])==POINTER_INVALID)
         m_copulas[2] = new CGumbel();

      vector wc[2];
      _fit_quantile_em(qdata,maxiter,gamma_scad,a_scad,wc);

      if(wc[0].Size()==0)
         return EMPTY_VALUE;

      wc[0] = adjust_weights(wc[0],weight_margin);

      m_weights = wc[0];
      m_cop_params = wc[1];
      m_cop_params[2] = floor(m_cop_params[2]);

      matrix cov = {{1.,m_cop_params[1]},{m_cop_params[1],1.}};
      m_copulas[0].Set_theta(m_cop_params[0]);
      m_copulas[1].Set_covariance(cov);
      m_copulas[1].Set_nu(m_cop_params[2]);
      m_copulas[2].Set_theta(m_cop_params[3]);

      vector u1 = qdata.Col(0);
      vector u2 = qdata.Col(1);

      return _ml_qfunc(u1,u2,m_cop_params,m_weights,gamma_scad,a_scad,false);
     }
   virtual  virtual double    Penalized_Log_Likelihood(matrix &qdata,double g_scad, double a_scad)
     {
      return _ml_qfunc(qdata.Col(0),qdata.Col(1),m_cop_params,m_weights,g_scad,a_scad,true);
     }
   CClayton*          ClaytonCopula(void)
     {
      if(m_copulas[0]==NULL)
        {
         m_copulas[0] = new CClayton();
         m_copulas[0].Set_theta(m_cop_params[0]);
        }
      return (CClayton*)m_copulas[0];
     }
   CStudent*           StudentCopula(void)
     {
      if(m_copulas[1]==NULL)
        {
         m_copulas[1] = new CStudent();
         matrix cov = {{1.,m_cop_params[1]},{m_cop_params[1],1.}};
         m_copulas[1].Set_covariance(cov);
         m_copulas[1].Set_nu(m_cop_params[2]);
        }

      return (CStudent*)m_copulas[1];
     }
   CGumbel*          GumbelCopula(void)
     {
      if(m_copulas[2]==NULL)
        {
         m_copulas[2] = new CGumbel();
         m_copulas[2].Set_theta(m_cop_params[3]);
        }

      return (CGumbel*)m_copulas[2];
     }

  };
//+------------------------------------------------------------------+
//|Clayton, Frank and Joe mixed copula.                              |
//+------------------------------------------------------------------+
class CFJMixCop:public CMixedCopula
  {
protected:
   //+------------------------------------------------------------------+
   //| Derived class from CNDimensional_Func                            |
   //+------------------------------------------------------------------+
   class CNDimensional_Func1 : public CNDimensional_Func
     {
   protected:
      vector            m_u1,m_u2,m_w;
      CFJMixCop*        m_obj;
      bool              m_centered,m_individual,m_ifpenalty;
      double            m_epsilon,m_gscad,m_ascad,m_multiplier;
   public:
      //--- constructor, destructor
                     CNDimensional_Func1(void) {}
                    ~CNDimensional_Func1(void)
        {
         m_obj = NULL;
        }
      void           set_params(CFJMixCop &obj, vector& u1, vector& u2,vector& weights, double gamma_scad, double a_scad, bool if_penalty = true, double multiplier= 1.,bool center = false, double ep=EMPTY_VALUE)
        {
         m_u1  = u1;
         m_u2 = u2;
         m_w = weights;
         m_obj = GetPointer(obj);
         m_centered = center;
         m_epsilon = ep;
         m_gscad = gamma_scad;
         m_ascad = a_scad;
         m_ifpenalty = if_penalty;
         m_multiplier = multiplier;
        }
      virtual void      Func(double &x[],double &func,CObject &obj)
        {
         vector _x;
         _x.Assign(x);
         func = m_obj.objective(_x,m_u1,m_u2,m_w,m_gscad,m_ascad,m_ifpenalty,m_multiplier);
        }

      virtual void      Func(CRowDouble &x,double &func,CObject &obj)
        {
         vector _x = x.ToVector();
         func = m_obj.objective(_x,m_u1,m_u2,m_w,m_gscad,m_ascad,m_ifpenalty,m_multiplier);
        }
     };

   double            _ml_qfunction(vector& u1, vector& u2, vector& cop_params,vector& weights, double gamma_scad, double a_scad, bool if_penalty = true, double multiplier= 1.)
     {
      ulong num = u1.Size();


      m_copulas[0].Set_theta(cop_params[0]);
      m_copulas[1].Set_theta(cop_params[1]);
      m_copulas[2].Set_theta(cop_params[2]);

      vector llclayton,llfrank,lljoe,llmix;

      llclayton = m_copulas[0].Copula_PDF(u1,u2,true,true);
      llfrank = m_copulas[1].Copula_PDF(u1,u2,true,true);
      lljoe = m_copulas[2].Copula_PDF(u1,u2,true,true);

      llmix = weights[0]*llclayton+weights[1]*llfrank+(1.0-weights[0]-weights[1])*lljoe;
      llmix.Clip(1.e-15,DBL_MAX);
      double llsum = log(llmix).Sum();
      double sum = 0;
      if(gamma_scad>0.0 && a_scad>2.)
         for(ulong i = 0; i<weights.Size(); sum+=scad_penalty(weights[i],gamma_scad,a_scad), ++i);
      double penalty = double(num)*sum;

      return (llsum-penalty*double(if_penalty))*multiplier;
     }
   vector            _maximization_step(matrix&qd, double gamma_scad, double a_scad, vector& cop_params, vector& weights)
     {
      vector u1 = qd.Col(0);
      vector u2 = qd.Col(1);

      double eps = 1.e-3;

      double _x_[3];
      for(uint i = 0; i<_x_.Size(); _x_[i] = cop_params[i], ++i);

      double bndl[3] = {-1.,-50.,1.};
      double bndu[3] = {100.,50.,100.};

      CObject             obj;
      CNDimensional_Func1 ffunc;
      CNDimensional_Rep   frep;

      ffunc.set_params(this,u1,u2,weights,gamma_scad,a_scad,bool(gamma_scad>0.0&&a_scad>2.0),-1);

      CMinBLEICStateShell state;

      CMinBLEICReportShell rep;

      double epsg=eps;

      double epsf=0;

      double epsx=0;

      double epso=eps;

      double epsi=eps;

      double diffstep=1.0e-6;

      CAlglib::MinBLEICCreateF(_x_,diffstep,state);

      CAlglib::MinBLEICSetBC(state,bndl,bndu);

      CAlglib::MinBLEICSetInnerCond(state,epsg,epsf,epsx);

      CAlglib::MinBLEICSetOuterCond(state,epso,epsi);

      CAlglib::MinBLEICOptimize(state,ffunc,frep,false,obj);

      CAlglib::MinBLEICResults(state,_x_,rep);

      vector out = vector::Zeros(0);

      int termination_reason = rep.GetTerminationType();

      if(termination_reason<0)
        {
         Print(__FUNCTION__, " termination reason ", termination_reason);
         return out;
        }

      out.Assign(_x_);

      return out;
     }
   vector            _expectation_step(matrix &qd,double gamma_scad, double a_scad,vector& cop_params,vector &weights)
     {
      //---
      ulong num = qd.Rows();
      vector u1 = qd.Col(0);
      vector u2 = qd.Col(1);
      //---
      double dif = 1;
      double tol_weight = 1.e-2;
      long iteration = 0;
      //---
      for(ulong i = 0; i<3; ++i)
         m_copulas[i].Set_theta(cop_params[i]);
      //---
      vector nweights;
      //---
      if(gamma_scad>0.0 && a_scad>2.0)
        {
         while(dif>tol_weight && iteration<10)
           {
            nweights = vector::Zeros(3);
            nweights.Fill(double("nan"));
            iteration+=1;
            for(ulong i = 0; i<3; ++i)
              {
               vector sum_ml_1st = vector::Zeros(u1.Size());
               double sum,sum_ml,numerator,denominator;
               sum = sum_ml = numerator = denominator = 1.e-12;
               for(ulong t = 0; t<num; ++t)
                 {
                  sum = 1.e-12;
                  for(ulong j = 0; j<3; ++j)
                     sum+=weights[j]*m_copulas[j].Copula_PDF(u1[t],u2[t],true,true);
                  sum_ml_1st[t] = weights[i]*m_copulas[i].Copula_PDF(u1[t],u2[t],true,true)/sum;
                 }
               sum_ml = sum_ml_1st.Sum();
               numerator = weights[i]*scad_derivative(weights[i],gamma_scad,a_scad)-sum_ml/double(num);
               for(ulong j = 0; j<3; ++j)
                  denominator+=weights[j]*scad_derivative(weights[j],gamma_scad,a_scad);
               denominator-=1.;
               nweights[i] = fabs(numerator/denominator);
              }
            dif = MathAbs(weights-nweights).Sum();
            weights = nweights;
           }
        }
      else
        {
         matrix wd = matrix::Zeros(weights.Size(),num);
         vector td, temp;
         while(dif>tol_weight && iteration<10)
           {
            nweights = vector::Zeros(weights.Size());
            iteration+=1;

            for(ulong i = 0; i<wd.Rows(); ++i)
               if(!wd.Row(weights[i]*m_copulas[i].Copula_PDF(u1,u2,true,true),i))
                 {
                  Print(__FUNCTION__, " row insertion error ", GetLastError());
                  return vector::Zeros(0);
                 }
            td = wd.Sum(0);
            td.Clip(1.e-12,DBL_MAX);
            for(ulong i = 0; i<weights.Size(); ++i)
              {
               temp = (wd.Row(i)/td);
               nweights[i] = fabs(temp.Sum()/double(num));
              }
            dif = MathAbs(weights-nweights).Sum();
            weights = nweights;
           }
        }
      //---
      return weights;
     }
   matrix            _fit_quantile_em(matrix &qd, ulong max_iter,double gamma_scad, double a_scad)
     {
      vector init_weights = {0.33, 0.33, 1. - 0.33 - 0.33};
      vector init_cop_params = {3,4,5};
      vector weights = _expectation_step(qd,gamma_scad,a_scad,init_cop_params,init_weights);
      if(!weights.Size())
         return matrix::Zeros(0,0);
      vector cop_params = _maximization_step(qd,gamma_scad,a_scad,init_cop_params,weights);
      if(!cop_params.Size())
         return matrix::Zeros(0,0);
      vector oldparams,newparams;
      oldparams = newparams = vector::Zeros(6);

      np::vectorCopy(oldparams,init_weights,0,3);
      np::vectorCopy(oldparams,init_cop_params,3);

      np::vectorCopy(newparams,weights,0,3);
      np::vectorCopy(newparams,cop_params,3);

      double ll_diff = MathAbs(oldparams-newparams).Sum();
      ulong i = 1;
      while(i<max_iter && ll_diff>(5.*1.e-2))
        {
         np::vectorCopy(oldparams,weights,0,3);
         np::vectorCopy(oldparams,cop_params,3);

         weights = _expectation_step(qd,gamma_scad,a_scad,cop_params,weights);
         if(!weights.Size())
           {
            Print(__FUNCTION__, " invalid weights ", i);
            return matrix::Zeros(0,0);
           }

         cop_params = _maximization_step(qd,gamma_scad,a_scad,cop_params,weights);
         if(!cop_params.Size())
           {
            Print(__FUNCTION__, " failed convergence at iteration ", i);
            return matrix::Zeros(0,0);
           }

         np::vectorCopy(newparams,weights,0,3);
         np::vectorCopy(newparams,cop_params,3);

         ll_diff = MathAbs(oldparams-newparams).Sum();
         i+=1;
        }

      matrix out = matrix::Zeros(3,3);
      out.Row(weights,0);
      out.Row(cop_params,1);
      return out;
     }
public:
                     CFJMixCop(void)
     {
      m_num_cops = 3;
      m_mixture = CFJ_MIX;
      ArrayResize(m_copulas,int(m_num_cops));
      for(uint i = 0; i<m_num_cops; ++i)
         m_copulas[i] = NULL;
     }

                     CFJMixCop(vector &params, vector& weights)
     {
      m_num_cops = 3;
      m_mixture = CFJ_MIX;
      ArrayResize(m_copulas,int(m_num_cops));
      if(params.Size()<3 || weights.Size()<3)
         Print(__FUNCTION__, " invalid parameters ");
      else
        {
         m_weights = weights;
         m_cop_params = params;
         for(ulong i = 0; i<3; ++i)
           {
            if(!i)
               m_copulas[i] = new CClayton();
            else
               if(i==1)
                  m_copulas[i] = new CFrank();
               else
                  m_copulas[i] = new CJoe();
            m_copulas[i].Set_theta(m_cop_params[i]);
           }
        }
     }
                    ~CFJMixCop(void)
     {

     }

   double                   objective(vector& x,vector& u1, vector& u2,vector& weights, double gamma_scad, double a_scad, bool if_penalty = true, double multiplier= 1.)
     {
      return  _ml_qfunction(u1,u2,x,weights,gamma_scad,a_scad,if_penalty,multiplier);
     }
   virtual double            Fit(matrix& qdata, ulong maxiter = 50, double gamma_scad = 0.0,double a_scad = 0.0, double weight_margin = 1.e-2) override
     {
      if(CheckPointer(m_copulas[0])==POINTER_INVALID)
         m_copulas[0] = new CClayton();
      if(CheckPointer(m_copulas[1])==POINTER_INVALID)
         m_copulas[1] = new CFrank();
      if(CheckPointer(m_copulas[2])==POINTER_INVALID)
         m_copulas[2] = new CJoe();

      matrix wc = _fit_quantile_em(qdata,maxiter,gamma_scad,a_scad);
      if(!wc.Rows())
         return EMPTY_VALUE;
      wc.Row(adjust_weights(wc.Row(0),weight_margin),0);
      m_weights = wc.Row(0);
      m_cop_params = wc.Row(1);

      m_copulas[0].Set_theta(m_cop_params[0]);
      m_copulas[1].Set_theta(m_cop_params[1]);
      m_copulas[2].Set_theta(m_cop_params[2]);

      vector u1 = qdata.Col(0);
      vector u2 = qdata.Col(1);

      return _ml_qfunction(u1,u2,wc.Row(1),wc.Row(0),gamma_scad,a_scad,false);
     }
   virtual double    Penalized_Log_Likelihood(matrix &qdata,double g_scad, double a_scad)
     {
      return _ml_qfunction(qdata.Col(0),qdata.Col(1),m_cop_params,m_weights,g_scad,a_scad,true);
     }
   CClayton*         ClaytonCopula(void)
     {
      if(m_copulas[0] == NULL)
        {
         m_copulas[0] = new CClayton();
         m_copulas[0].Set_theta(m_cop_params[0]);
        }

      return (CClayton*)m_copulas[0];
     }
   CFrank*           FrankCopula(void)
     {
      if(m_copulas[1]==NULL)
        {
         m_copulas[1] = new CFrank();
         m_copulas[1].Set_theta(m_cop_params[1]);
        }

      return (CFrank*)m_copulas[1];
     }
   CJoe*             JoeCopula(void)
     {
      if(m_copulas[2]==NULL)
        {
         m_copulas[2] = new CJoe();
         m_copulas[2].Set_theta(m_cop_params[2]);
        }

      return (CJoe*)m_copulas[2];
     }

  };
//+------------------------------------------------------------------+
//|Load a mixed copula model from file and return the correct pointer|
//+------------------------------------------------------------------+
CMixedCopula * mixed_copula(const int file_handle)
  {
   vector weights, cop_params;
   int size = 0;
   weights = cop_params = vector::Zeros(0);
   ENUM_COPULA_MIX mixture = WRONG_VALUE;
   if(file_handle!=INVALID_HANDLE)
     {
      long ff = FileReadLong(file_handle);
      if(ff == -1)
        {
         size = FileReadInteger(file_handle,INT_VALUE);
         mixture = ENUM_COPULA_MIX(FileReadInteger(file_handle,INT_VALUE));
         ulong size_weights = (ulong)FileReadLong(file_handle);
         if(size_weights)
           {
            weights = vector::Zeros(size_weights);
            switch(mixture)
              {
               case CFG_MIX:
               case CFJ_MIX:
               case JFG_MIX:
                  cop_params = vector::Zeros(size_weights);
                  break;
               case CTG_MIX:
                  cop_params = vector::Zeros(size_weights+1);
                  break;
              }
            for(ulong i = 0; i<size_weights; ++i)
               weights[i] = FileReadDouble(file_handle);
           }
        }
      else
         return NULL;
     }
   else
      return NULL;
//---
   for(ulong i = 0,k = 0; i<ulong(size); ++i)
     {
      CBivariateCopula* copula = bivariate_copula(file_handle);
      if(copula==NULL)
        {
         Print(__FUNCTION__, " failed to load a bivariate copula from file ", GetLastError());
         return NULL;
        }
      if(weights.Size())
        {
         switch(ENUM_COPULA_TYPE(copula.Type()))
           {
            case CLAYTON_COPULA:
            case GUMBEL_COPULA:
            case FRANK_COPULA:
            case JOE_COPULA:
               cop_params[k++] = copula.Get_theta();
               break;
            case STUDENT_COPULA:
               cop_params[k++] = copula.Get_rho();
               cop_params[k++] = copula.Get_nu();
               break;
           }
        }

      delete copula;
      copula = NULL;
     }

   if(!weights.Size() || !cop_params.Size())
      return NULL;

   switch(mixture)
     {
      case CTG_MIX:
         return new CTGMixCop(cop_params,weights);
      case CFG_MIX:
         return new CFGMixCop(cop_params,weights);
      case CFJ_MIX:
         return new CFJMixCop(cop_params,weights);
      case JFG_MIX:
         return new JFGMixCop(cop_params,weights);
      default:
         Print(__FUNCTION__, " unrecognized mixed copula ");
         return NULL;
     }

   return NULL;
  }
//+------------------------------------------------------------------+
