//+------------------------------------------------------------------+
//| FlexibleMultiTFEngulfingEA.mq5 |
//| Expert Advisor for MetaTrader 5 |
//| Description: Trades based on H1/H4/D1 highs/lows purges, waits |
//| for engulfing candles on H1 or M15, with flexible execution. |
//+------------------------------------------------------------------+
#property copyright "Your Name"
#property link "https://www.example.com"
#property version "1.06"
//--- Input parameters
input double LotSize = 0.1; // Lot size for trades
input int LookbackBars = 5; // Number of bars to check for highs/lows
input double StopLossPips = 30.0; // Stop loss in pips (adjusted for GOLD#)
input double TakeProfitPips = 90.0; // Take profit in pips (adjusted for GOLD#)
input double TrailingStopPips = 40.0;// Trailing stop in pips
input double BreakevenPips = 30.0; // Move to BE after 30 pips profit
input double EngulfingMinBodyRatio = 0.5; // Min body ratio for engulfing candle
input ENUM_TIMEFRAMES TradeTimeframe = PERIOD_M15; // Primary timeframe for trading
input ENUM_TIMEFRAMES ConfirmationTimeframe = PERIOD_M5; // Lower timeframe for engulfing
input ENUM_TIMEFRAMES TrendTimeframe = PERIOD_H1; // Trend from H1 SMA
input int MaxCandlesPostPurge = 5; // Max H1 candles to wait for engulfing
input double VolumeThreshold = 1.2; // Volume multiplier for liquidity confirmation
input bool UseTrendFilter = true; // Use SMA trend filter
input int SMAPeriod = 50; // SMA period for trend filter
input int MaxPositions = 3; // Max simultaneous positions
//--- Global variables
double lastHigh, lastLow;
datetime lastCandleTime;
double contractSize;
datetime lastPurgeTime = 0;
bool purgeDetected = false;
bool highPurge = false;
//+------------------------------------------------------------------+
//| Expert initialization function |
//+------------------------------------------------------------------+
int OnInit()
{
   if(TradeTimeframe != PERIOD_M15 && TradeTimeframe != PERIOD_M30)
   {
      Print("Invalid trade timeframe. Use M15 or M30.");
      return(INIT_PARAMETERS_INCORRECT);
   }
   if(ConfirmationTimeframe != PERIOD_M5 && ConfirmationTimeframe != PERIOD_M15)
   {
      Print("Invalid confirmation timeframe. Use M5 or M15.");
      return(INIT_PARAMETERS_INCORRECT);
   }
  
   contractSize = SymbolInfoDouble(_Symbol, SYMBOL_TRADE_CONTRACT_SIZE);
   if(contractSize == 0)
   {
      Print("Failed to get contract size for ", _Symbol);
      return(INIT_PARAMETERS_INCORRECT);
   }
  
   lastHigh = 0; lastLow = 0;
   lastCandleTime = 0;
   purgeDetected = false;
   lastPurgeTime = 0;
  
   Print("EA Initialized on ", EnumToString(TradeTimeframe), ", Confirmation TF: ", EnumToString(ConfirmationTimeframe), ", Contract Size: ", contractSize);
   return(INIT_SUCCEEDED);
}
//+------------------------------------------------------------------+
//| Expert tick function |
//+------------------------------------------------------------------+
void OnTick()
{
   // Check if a new candle has formed on the trade timeframe
   if(!IsNewCandle()) return;
  
   // Manage breakeven and trailing stop for open positions
   ManageBreakevenAndTrailing();
  
   // Get current and previous candle data for trade timeframe
   MqlRates ratesTrade[];
   ArraySetAsSeries(ratesTrade, true);
   if(CopyRates(_Symbol, TradeTimeframe, 0, 4, ratesTrade) < 4)
   {
      Print("Failed to load trade timeframe rates data");
      return;
   }
  
   // Get candle data for confirmation timeframe
   MqlRates ratesConf[];
   ArraySetAsSeries(ratesConf, true);
   if(CopyRates(_Symbol, ConfirmationTimeframe, 0, 4, ratesConf) < 4)
   {
      Print("Failed to load confirmation timeframe rates data");
      return;
   }
  
   // Current trade timeframe candle (index 0)
   double currentOpen = ratesTrade[0].open;
   double currentClose = ratesTrade[0].close;
   double currentHigh = ratesTrade[0].high;
   double currentLow = ratesTrade[0].low;
   long currentVolume = ratesTrade[0].tick_volume;
   datetime currentTime = ratesTrade[0].time;
  
   // Update highs and lows for trade timeframe
   UpdateHighsLows();
  
   // Check for purge (liquidity sweep) on trade timeframe
   bool highPurged = (currentHigh > lastHigh && lastHigh > 0);
   bool lowPurged = (currentLow < lastLow && lastLow > 0);
  
   // Update purge status
   if(highPurged || lowPurged)
   {
      purgeDetected = true;
      lastPurgeTime = currentTime;
      highPurge = highPurged;
   }
  
   // Check if within the post-purge window
   bool withinPurgeWindow = false;
   if(purgeDetected)
   {
      int candlesSincePurge = iBarShift(_Symbol, TradeTimeframe, lastPurgeTime, true);
      withinPurgeWindow = candlesSincePurge <= MaxCandlesPostPurge;
      if(!withinPurgeWindow)
      {
         purgeDetected = false; // Reset if window expires
         Print("Purge window expired: ", candlesSincePurge, " candles since last purge");
      }
   }
  
   // Check volume for liquidity confirmation
   bool volumeConfirmed = IsVolumeSpike(currentVolume);
   if(currentVolume <= 1)
   {
      Print("Warning: Tick volume is ", currentVolume, ". Possible data issue. Bypassing volume check.");
      volumeConfirmed = true;
   }
  
   // Check trend with SMA
   bool isBullishTrend = UseTrendFilter ? IsBullishTrend() : true;
  
   // Check for engulfing candles on trade timeframe (current + previous 2 candles)
   bool bullishEngulfingTrade = false, bearishEngulfingTrade = false;
   for(int i = 0; i < 3; i++)
   {
      if(IsBullishEngulfing(ratesTrade[i], ratesTrade[i+1]))
         bullishEngulfingTrade = true;
      if(IsBearishEngulfing(ratesTrade[i], ratesTrade[i+1]))
         bearishEngulfingTrade = true;
   }
  
   // Check for engulfing candles on confirmation timeframe (current + previous 2 candles)
   bool bullishEngulfingConf = false, bearishEngulfingConf = false;
   for(int i = 0; i < 3; i++)
   {
      if(IsBullishEngulfing(ratesConf[i], ratesConf[i+1]))
         bullishEngulfingConf = true;
      if(IsBearishEngulfing(ratesConf[i], ratesConf[i+1]))
         bearishEngulfingConf = true;
   }
  
   // Debug logs
   Print("Trade TF OHLC: Open=", currentOpen, ", High=", currentHigh, ", Low=", currentLow, ", Close=", currentClose);
   Print("Recent High/Low: ", lastHigh, "/", lastLow);
   Print("Current High/Low: ", currentHigh, "/", currentLow);
   Print("High Purge Check: ", currentHigh - lastHigh);
   Print("Low Purge Check: ", lastLow - currentLow);
   if(highPurged) Print("High purged: ", currentHigh, " > ", lastHigh);
   if(lowPurged) Print("Low purged: ", currentLow, " < ", lastLow);
   if(!highPurged && !lowPurged) Print("No purge detected");
   if(purgeDetected) Print("Purge detected at ", TimeToString(lastPurgeTime), ", Candles since purge: ", iBarShift(_Symbol, TradeTimeframe, lastPurgeTime, true));
   if(volumeConfirmed) Print("Volume spike confirmed: ", currentVolume);
   else Print("No volume spike detected: ", currentVolume);
   if(UseTrendFilter) Print("Trend: ", isBullishTrend ? "Bullish (Price above SMA)" : "Bearish (Price below SMA)");
   else Print("Trend filter disabled");
   if(bullishEngulfingTrade) Print("Bullish engulfing detected on Trade TF");
   if(bearishEngulfingTrade) Print("Bearish engulfing detected on Trade TF");
   if(bullishEngulfingConf) Print("Bullish engulfing detected on Conf TF");
   if(bearishEngulfingConf) Print("Bearish engulfing detected on Conf TF");
   if(!bullishEngulfingTrade && !bearishEngulfingTrade && !bullishEngulfingConf && !bearishEngulfingConf)
      Print("No engulfing candle detected on Trade or Conf TF");
   if(currentHigh == currentLow) Print("Warning: Zero-range candle detected on Trade TF");
  
   // Trade logic
   if(purgeDetected && withinPurgeWindow && PositionsTotalForSymbol() < MaxPositions)
   {
      if(highPurge && (bullishEngulfingTrade || bullishEngulfingConf) && volumeConfirmed && isBullishTrend)
      {
         Print("Buy signal: High purged, bullish engulfing on ", bullishEngulfingTrade ? "Trade TF" : "Conf TF", ", volume confirmed, bullish trend");
         PlaceTrade(ORDER_TYPE_BUY, currentClose);
      }
      else if(!highPurge && (bearishEngulfingTrade || bearishEngulfingConf) && volumeConfirmed && !isBullishTrend)
      {
         Print("Sell signal: Low purged, bearish engulfing on ", bearishEngulfingTrade ? "Trade TF" : "Conf TF", ", volume confirmed, bearish trend");
         PlaceTrade(ORDER_TYPE_SELL, currentClose);
      }
   }
}
//+------------------------------------------------------------------+
//| Check if a new candle has formed |
//+------------------------------------------------------------------+
bool IsNewCandle()
{
   datetime currentCandleTime = iTime(_Symbol, TradeTimeframe, 0);
   if(currentCandleTime != lastCandleTime)
   {
      lastCandleTime = currentCandleTime;
      return true;
   }
   return false;
}
//+------------------------------------------------------------------+
//| Update highest high and lowest low for TradeTimeframe |
//+------------------------------------------------------------------+
void UpdateHighsLows()
{
   MqlRates rates[];
   ArraySetAsSeries(rates, true);
   if(CopyRates(_Symbol, TradeTimeframe, 1, LookbackBars, rates) >= LookbackBars)
   {
      lastHigh = rates[0].high;
      lastLow = rates[0].low;
      for(int i = 1; i < LookbackBars; i++)
      {
         if(rates[i].high > lastHigh) lastHigh = rates[i].high;
         if(rates[i].low < lastLow) lastLow = rates[i].low;
      }
   }
   else Print("Failed to load TradeTimeframe rates");
}
//+------------------------------------------------------------------+
//| Check for volume spike for liquidity confirmation |
//+------------------------------------------------------------------+
bool IsVolumeSpike(long currentVolume)
{
   double avgVolume = 0;
   MqlRates rates[];
   ArraySetAsSeries(rates, true);
   if(CopyRates(_Symbol, TradeTimeframe, 1, LookbackBars, rates) < LookbackBars)
   {
      Print("Failed to load rates for volume check");
      return true;
   }
  
   int validBars = 0;
   for(int i = 0; i < LookbackBars; i++)
   {
      if(rates[i].tick_volume > 1)
      {
         avgVolume += rates[i].tick_volume;
         validBars++;
      }
   }
   if(validBars == 0)
   {
      Print("Warning: No valid volume data. Average volume is 0. Bypassing volume check.");
      return true;
   }
   avgVolume /= validBars;
  
   Print("Current volume: ", currentVolume, ", Average volume: ", avgVolume, ", Valid bars: ", validBars);
   return currentVolume >= VolumeThreshold * avgVolume;
}
//+------------------------------------------------------------------+
//| Check trend using SMA on TrendTimeframe |
//+------------------------------------------------------------------+
bool IsBullishTrend()
{
   int smaHandle = iMA(_Symbol, TrendTimeframe, SMAPeriod, 0, MODE_SMA, PRICE_CLOSE);
   double sma[];
   ArraySetAsSeries(sma, true);
   if(CopyBuffer(smaHandle, 0, 0, 1, sma) < 1)
   {
      Print("Failed to load SMA data");
      return false;
   }
   double currentPrice = iClose(_Symbol, TrendTimeframe, 0);
   Print("Current price: ", currentPrice, ", SMA: ", sma[0]);
   return currentPrice > sma[0];
}
//+------------------------------------------------------------------+
//| Check for bullish engulfing candle |
//+------------------------------------------------------------------+
bool IsBullishEngulfing(MqlRates &current, MqlRates &previous)
{
   double currentBody = MathAbs(current.close - current.open);
   double prevBody = MathAbs(previous.close - previous.open);
  
   if(current.close > current.open &&
      previous.close < previous.open &&
      current.open <= previous.close &&
      current.close >= previous.open &&
      currentBody >= EngulfingMinBodyRatio * prevBody)
   {
      return true;
   }
   return false;
}
//+------------------------------------------------------------------+
//| Check for bearish engulfing candle |
//+------------------------------------------------------------------+
bool IsBearishEngulfing(MqlRates &current, MqlRates &previous)
{
   double currentBody = MathAbs(current.close - current.open);
   double prevBody = MathAbs(previous.close - previous.open);
  
   if(current.close < current.open &&
      previous.close > previous.open &&
      current.open >= previous.close &&
      current.close <= previous.open &&
      currentBody >= EngulfingMinBodyRatio * prevBody)
   {
      return true;
   }
   return false;
}
//+------------------------------------------------------------------+
//| Place a trade |
//+------------------------------------------------------------------+
void PlaceTrade(ENUM_ORDER_TYPE orderType, double price)
{
   MqlTradeRequest request = {};
   MqlTradeResult result = {};
  
   request.action = TRADE_ACTION_DEAL;
   request.symbol = _Symbol;
   request.volume = LotSize;
   request.type = orderType;
   request.price = price;
   request.sl = (orderType == ORDER_TYPE_BUY) ? price - StopLossPips * _Point * 100 : price + StopLossPips * _Point * 100;
   request.tp = (orderType == ORDER_TYPE_BUY) ? price + TakeProfitPips * _Point * 100 : price - TakeProfitPips * _Point * 100;
   request.type_filling = ORDER_FILLING_IOC;
  
   if(OrderSend(request, result))
   {
      Print("Trade placed successfully: ", orderType == ORDER_TYPE_BUY ? "BUY" : "SELL", " at ", price, " SL: ", request.sl, " TP: ", request.tp);
   }
   else
   {
      Print("Trade failed: ", result.retcode);
   }
}
//+------------------------------------------------------------------+
//| Manage breakeven and trailing stop for open positions |
//+------------------------------------------------------------------+
void ManageBreakevenAndTrailing()
{
   if(!PositionSelect(_Symbol)) return;
  
   double openPrice = PositionGetDouble(POSITION_PRICE_OPEN);
   double currentSL = PositionGetDouble(POSITION_SL);
   double profitPips = 0;
  
   if(PositionGetInteger(POSITION_TYPE) == POSITION_TYPE_BUY)
   {
      double bid = SymbolInfoDouble(_Symbol, SYMBOL_BID);
      profitPips = (bid - openPrice) / (_Point * 100);
  
      // Breakeven
      if(profitPips >= BreakevenPips && (currentSL < openPrice || currentSL == 0))
      {
         double newSL = openPrice + 5 * _Point; // Slight buffer
         ModifySL(newSL);
      }
      // Trailing
      double newSL = bid - TrailingStopPips * _Point * 100;
      if(newSL > currentSL && newSL > openPrice)
         ModifySL(newSL);
   }
   else // POSITION_TYPE_SELL
   {
      double ask = SymbolInfoDouble(_Symbol, SYMBOL_ASK);
      profitPips = (openPrice - ask) / (_Point * 100);
  
      // Breakeven
      if(profitPips >= BreakevenPips && (currentSL > openPrice || currentSL == 0))
      {
         double newSL = openPrice - 5 * _Point; // Slight buffer
         ModifySL(newSL);
      }
      // Trailing
      double newSL = ask + TrailingStopPips * _Point * 100;
      if((newSL < currentSL || currentSL == 0) && newSL < openPrice)
         ModifySL(newSL);
   }
}
//+------------------------------------------------------------------+
//| Modify SL helper |
//+------------------------------------------------------------------+
void ModifySL(double newSL)
{
   MqlTradeRequest request = {};
   MqlTradeResult result = {};
   request.action = TRADE_ACTION_SLTP;
   request.position = PositionGetInteger(POSITION_TICKET);
   request.symbol = _Symbol;
   request.sl = NormalizeDouble(newSL, _Digits);
   request.tp = PositionGetDouble(POSITION_TP);
   if(OrderSend(request, result))
      Print("SL updated: ", newSL);
   else
      Print("Failed to update SL: ", result.retcode);
}
//+------------------------------------------------------------------+
//| Count positions for symbol |
//+------------------------------------------------------------------+
int PositionsTotalForSymbol()
{
   int count = 0;
   for(int i = PositionsTotal() - 1; i >= 0; i--)
   {
      if(PositionGetSymbol(i) == _Symbol)
         count++;
   }
   return count;
}
//+------------------------------------------------------------------+
//| Expert deinitialization function |
//+------------------------------------------------------------------+
void OnDeinit(const int reason)
{
   Print("EA Deinitialized, reason: ", reason);
}
//+------------------------------------------------------------------+