
#include <Trade\Trade.mqh>

#property copyright "Your Name"
#property link      "https://x.ai"
#property version   "2.26"

input double RiskPct = 2.0; // Base risk per trade %
input double MaxLossUSD = 110.0; // Maximum loss per trade in USD
input double RecTgt = 7000.0; // Equity recovery target
input int ATR_Prd = 14; // ATR period
input int Brk_Prd = 10; // Breakout period
input int EMA_Prd = 20; // EMA period
input string GS_Url = ""; // Google Sheets webhook URL
input bool NewsFilt = true; // News filter
input int NewsPause = 15; // Pause minutes
input double MinBrkStr = 0.1; // Min breakout strength (x ATR)
input int Vol_Prd = 1; // Volume period
input bool Bypass = true; // Bypass volume, breakout, HTF trend filters for testing
input bool useHTF = false; // Use D1 or H4 EMA trend filter
input string NewsAPI_Url = "https://www.alphavantage.co/query?function=NEWS_SENTIMENT&apikey="; // Alpha Vantage API URL
input string NewsAPI_Key = "pub_3f54bba977384ac19b6839a744444aba"; // Alpha Vantage API key
input double DailyDDLimit = 2.5; // Daily drawdown limit (%)
input double OverallDDLimit = 5.5; // Overall drawdown limit (%)
input double TargetBalanceOrEquity = 6600.0; // Target balance or equity to pass challenge ($)
input bool ResetProfitTarget = false; // Reset target to resume trading

double CurRisk = RiskPct;
double OrigRisk = RiskPct;
double LastEqHigh = 0;
double StartingBalance = 0; // Balance at initialization
double DailyBalance = 0; // Balance at start of day
datetime LastDay = 0; // Last day for daily balance reset
bool ProfitTargetReached = false; // Flag for target reached
bool DailyDDReached = false; // Flag for daily drawdown reached
CTrade trade;
int h_ema_d1 = INVALID_HANDLE;
int h_ema_h4 = INVALID_HANDLE;
int winStreak = 0;
int lossStreak = 0;
string SymbolName = _Symbol;

struct TimeframeData {
   ENUM_TIMEFRAMES tf;
   int h_atr;
   int h_vol;
   int h_vol_ma;
   datetime lastSig;
   datetime lastBar;
};

TimeframeData tfs[];
struct NewsEvt { 
   datetime time; 
   string evt; 
   int impact; 
};
NewsEvt newsCal[];
int newsCnt = 0;

struct TradeLog {
   ulong ticket;
   bool isWin;
   double profit;
   double brkStr;
   double vol;
   double risk;
   ENUM_TIMEFRAMES tf;
};
TradeLog tradeHistory[];
int tradeCnt = 0;
double dynBrkStr = MinBrkStr;

int OnInit() {
   if(AccountInfoDouble(ACCOUNT_BALANCE) < 10.0) {
      Print("Low balance: ", AccountInfoDouble(ACCOUNT_BALANCE));
      return(INIT_FAILED);
   }
   string sym = Symbol();
   bool selected;
   if(!SymbolSelect(sym, true)) {
      Print("Error: Symbol ", sym, " not found in Market Watch. Available: ", _Symbol);
      SymbolName = _Symbol;
   } else {
      SymbolName = sym;
      Print("Symbol validated: ", SymbolName, ", Selected in Market Watch: ", selected);
   }
   if(!SymbolSelect(SymbolName, true)) {
      Print("Error: Failed to select ", SymbolName, " in Market Watch");
      return(INIT_FAILED);
   }

   Print("Please ensure ", NewsAPI_Url, " is added to allowed WebRequest URLs in MT5 settings");

   StartingBalance = AccountInfoDouble(ACCOUNT_BALANCE);
   LastEqHigh = AccountInfoDouble(ACCOUNT_EQUITY);
   DailyBalance = StartingBalance;
   LastDay = TimeCurrent() / 86400 * 86400; // Start of current day
   ProfitTargetReached = ResetProfitTarget ? false : ProfitTargetReached; // Allow manual reset
   ArrayResize(newsCal, 100);
   ArrayResize(tradeHistory, 100);
   ArrayResize(tfs, 2);
   tfs[0].tf = PERIOD_M15;
   tfs[1].tf = PERIOD_H1;
   for(int i = 0; i < 2; i++) {
      tfs[i].h_atr = iATR(SymbolName, tfs[i].tf, ATR_Prd);
      tfs[i].h_vol = iVolumes(SymbolName, tfs[i].tf, VOLUME_TICK);
      tfs[i].h_vol_ma = iMA(SymbolName, tfs[i].tf, Vol_Prd, 0, MODE_SMA, PRICE_CLOSE);
      tfs[i].lastSig = 0;
      tfs[i].lastBar = 0;
      if(tfs[i].h_atr == INVALID_HANDLE || tfs[i].h_vol == INVALID_HANDLE || tfs[i].h_vol_ma == INVALID_HANDLE) {
         Print("Indicator init failed for ", EnumToString(tfs[i].tf));
         return(INIT_FAILED);
      }
   }
   h_ema_d1 = iMA(SymbolName, PERIOD_D1, EMA_Prd, 0, MODE_EMA, PRICE_CLOSE);
   h_ema_h4 = iMA(SymbolName, PERIOD_H4, EMA_Prd, 0, MODE_EMA, PRICE_CLOSE);
   if(h_ema_d1 == INVALID_HANDLE || h_ema_h4 == INVALID_HANDLE) {
      Print("EMA init failed");
      return(INIT_FAILED);
   }
   if(NewsFilt) FetchNewsCalendar();
   Print("EA initialized. Timeframes: M15, H1, News events: ", newsCnt, ", Bypass: ", Bypass, ", UseHTF: ", useHTF, 
         ", Timezone: EAT (UTC+3), Server: ", TerminalInfoString(TERMINAL_NAME), 
         ", Starting Balance: ", StartingBalance, ", Target Balance/Equity: ", TargetBalanceOrEquity);
   return(INIT_SUCCEEDED);
}

void OnDeinit(const int reason) {
   if(h_ema_d1 != INVALID_HANDLE) IndicatorRelease(h_ema_d1);
   if(h_ema_h4 != INVALID_HANDLE) IndicatorRelease(h_ema_h4);
   for(int i = 0; i < ArraySize(tfs); i++) {
      if(tfs[i].h_atr != INVALID_HANDLE) IndicatorRelease(tfs[i].h_atr);
      if(tfs[i].h_vol != INVALID_HANDLE) IndicatorRelease(tfs[i].h_vol);
      if(tfs[i].h_vol_ma != INVALID_HANDLE) IndicatorRelease(tfs[i].h_vol_ma);
   }
   Print("EA stopped: ", reason);
}

void CloseAllPositions() {
   for(int i = PositionsTotal() - 1; i >= 0; i--) {
      ulong ticket = PositionGetTicket(i);
      if(!PositionSelectByTicket(ticket) || PositionGetString(POSITION_SYMBOL) != SymbolName) continue;
      long magic = PositionGetInteger(POSITION_MAGIC);
      if(magic == MagicNumber(PERIOD_M15) || magic == MagicNumber(PERIOD_H1)) {
         trade.PositionClose(ticket);
         Print("Closed position: Ticket=", ticket, ", Symbol=", SymbolName, ", Magic=", magic);
      }
   }
}

void OnTick() {
   datetime currentDay = TimeCurrent() / 86400 * 86400;
   if(currentDay > LastDay) {
      DailyBalance = AccountInfoDouble(ACCOUNT_BALANCE);
      LastDay = currentDay;
      DailyDDReached = false; // Reset daily drawdown flag
      Print("Daily balance reset: ", DailyBalance, " at ", TimeToString(currentDay, TIME_DATE));
   }

   double equity = AccountInfoDouble(ACCOUNT_EQUITY);
   double balance = AccountInfoDouble(ACCOUNT_BALANCE);
   if(balance >= TargetBalanceOrEquity || equity >= TargetBalanceOrEquity) {
      CloseAllPositions();
      ProfitTargetReached = true;
      Print("Trading paused: Target balance or equity reached. Balance=", balance, ", Equity=", equity, 
            ", Target=", TargetBalanceOrEquity, ". All positions closed. Set ResetProfitTarget=true or restart EA to resume trading.");
   }
   if(ProfitTargetReached) {
      Print("Trading paused: Target previously reached. Balance=", balance, ", Equity=", equity, ", Target=", TargetBalanceOrEquity);
      return;
   }

   double dailyDD = (DailyBalance - equity) / DailyBalance * 100;
   double overallDD = (StartingBalance - equity) / StartingBalance * 100;
   if(dailyDD >= DailyDDLimit) {
      CloseAllPositions();
      DailyDDReached = true;
      Print("Trading paused and positions closed: Daily DD=", StringFormat("%.2f", dailyDD), "% (Limit: ", DailyDDLimit, 
            "%), Equity=", equity, ", Daily Balance=", DailyBalance);
      return;
   }
   if(overallDD >= OverallDDLimit) {
      CloseAllPositions();
      Print("Trading paused and positions closed: Overall DD=", StringFormat("%.2f", overallDD), "% (Limit: ", OverallDDLimit, 
            "%), Equity=", equity, ", Starting Balance=", StartingBalance);
      return;
   }
   if(DailyDDReached) {
      Print("Trading paused: Daily drawdown limit previously reached today. Balance=", balance, ", Equity=", equity);
      return;
   }

   static datetime lastNewsFetch = 0;
   if(NewsFilt && TimeCurrent() >= lastNewsFetch + 4 * 3600) {
      FetchNewsCalendar();
      lastNewsFetch = TimeCurrent();
   }

   for(int i = 0; i < ArraySize(tfs); i++) {
      if(!NewBar(tfs[i].tf)) continue;

      bool hasPosition = false;
      for(int j = PositionsTotal() - 1; j >= 0; j--) {
         ulong ticket = PositionGetTicket(j);
         if(PositionSelectByTicket(ticket) && PositionGetString(POSITION_SYMBOL) == SymbolName && PositionGetInteger(POSITION_MAGIC) == MagicNumber(tfs[i].tf)) {
            hasPosition = true;
            break;
         }
      }
      if(hasPosition) {
         ManageTrades(tfs[i].tf);
         continue;
      }

      if(TimeCurrent() <= tfs[i].lastSig + PeriodSeconds(PERIOD_H1)) {
         Print(tfs[i].tf, " Trade skipped: Within 1-hour cooldown");
         continue;
      }

      if(NewsFilt && IsNews()) {
         Print(tfs[i].tf, " Trade skipped: News pause active");
         continue;
      }

      double eq = AccountInfoDouble(ACCOUNT_EQUITY);
      if(eq > LastEqHigh) LastEqHigh = eq;
      if(eq < LastEqHigh && lossStreak >= 2) CurRisk = MathMax(OrigRisk * 0.25, 0.1);
      else if(winStreak >= 3) CurRisk = MathMin(OrigRisk * 1.25, 5.0);
      else CurRisk = OrigRisk;
      if(eq >= RecTgt) {
         CurRisk = OrigRisk;
         winStreak = 0;
         lossStreak = 0;
      }

      bool bullHTF = !useHTF || (BullTrend(PERIOD_D1) || BullTrend(PERIOD_H4));
      bool bearHTF = !useHTF || (BearTrend(PERIOD_D1) || BearTrend(PERIOD_H4));
      bool buyBrk = BuyBrk(tfs[i].tf);
      bool sellBrk = SellBrk(tfs[i].tf);
      Print(tfs[i].tf, " Signal check: BullHTF=", bullHTF, ", BearHTF=", bearHTF, ", BuyBrk=", buyBrk, ", SellBrk=", sellBrk, ", Bid=", SymbolInfoDouble(SymbolName, SYMBOL_BID), ", Ask=", SymbolInfoDouble(SymbolName, SYMBOL_ASK));

      double atr[1], vol[2], vol_ma[1];
      if(CopyBuffer(tfs[i].h_atr, 0, 0, 1, atr) < 1 || 
         CopyBuffer(tfs[i].h_vol, 0, 0, 2, vol) < 2 || 
         CopyBuffer(tfs[i].h_vol_ma, 0, 1, 1, vol_ma) < 1) {
         Print(tfs[i].tf, " Trade skipped: Indicator copy failed");
         continue;
      }

      double slPips = atr[0] * 1.5 / _Point;
      double lots = CalcLots(eq, CurRisk, slPips);
      double margReq = SymbolInfoDouble(SymbolName, SYMBOL_MARGIN_INITIAL) * lots;
      double freeMarg = AccountInfoDouble(ACCOUNT_MARGIN_FREE);
      Print(tfs[i].tf, " Lot size: ", lots, ", Margin required: ", margReq, ", Free margin: ", freeMarg, ", SL Pips: ", slPips, ", Contract size: ", SymbolInfoDouble(SymbolName, SYMBOL_TRADE_CONTRACT_SIZE));

      if(freeMarg < margReq * 2) {
         Print(tfs[i].tf, " Trade skipped: Margin low (", freeMarg, " < ", margReq * 1.2, ")");
         continue;
      }

      double brkStr = MathAbs(buyBrk ? SymbolInfoDouble(SymbolName, SYMBOL_ASK) - iHigh(SymbolName, tfs[i].tf, iHighest(SymbolName, tfs[i].tf, MODE_HIGH, Brk_Prd, 1)) :
                                     iLow(SymbolName, tfs[i].tf, iLowest(SymbolName, tfs[i].tf, MODE_LOW, Brk_Prd, 1)) - SymbolInfoDouble(SymbolName, SYMBOL_BID)) / atr[0];
      if(!Bypass && brkStr < dynBrkStr) {
         Print(tfs[i].tf, " Trade skipped: Breakout strength too low (", brkStr, " < ", dynBrkStr, ")");
         continue;
      }

      if(bullHTF && buyBrk) {
         double price = SymbolInfoDouble(SymbolName, SYMBOL_ASK);
         double sl = price - slPips * _Point;
         double tp = price + slPips * 2 * _Point;
         Print(tfs[i].tf, " Attempting Buy: Price=", price, ", SL=", sl, ", TP=", tp, ", Lots=", lots);
         trade.SetExpertMagicNumber(MagicNumber(tfs[i].tf));
         if(trade.Buy(lots, SymbolName, price, sl, tp)) {
            tfs[i].lastSig = TimeCurrent();
            LogTrd(trade.ResultOrder(), SymbolName, price, sl, tp, "Open", brkStr, vol[1], CurRisk, tfs[i].tf);
            Print(tfs[i].tf, " Buy opened: Ticket=", trade.ResultOrder());
         } else {
            Print(tfs[i].tf, " Buy failed: Retcode=", trade.ResultRetcode(), ", Error=", GetLastError(), ", Comment=", trade.ResultComment());
         }
      }
      else if(bearHTF && sellBrk) {
         double price = SymbolInfoDouble(SymbolName, SYMBOL_BID);
         double sl = price + slPips * _Point;
         double tp = price - slPips * 2 * _Point;
         Print(tfs[i].tf, " Attempting Sell: Price=", price, ", SL=", sl, ", TP=", tp, ", Lots=", lots);
         trade.SetExpertMagicNumber(MagicNumber(tfs[i].tf));
         if(trade.Sell(lots, SymbolName, price, sl, tp)) {
            tfs[i].lastSig = TimeCurrent();
            LogTrd(trade.ResultOrder(), SymbolName, price, sl, tp, "Open", brkStr, vol[1], CurRisk, tfs[i].tf);
            Print(tfs[i].tf, " Sell opened: Ticket=", trade.ResultOrder());
         } else {
            Print(tfs[i].tf, " Sell failed: Retcode=", trade.ResultRetcode(), ", Error=", GetLastError(), ", Comment=", trade.ResultComment());
         }
      }
      else {
         Print(tfs[i].tf, " Trade skipped: No valid signal");
      }
   }
}

void ManageTrades(ENUM_TIMEFRAMES tf) {
   for(int i = PositionsTotal() - 1; i >= 0; i--) {
      ulong ticket = PositionGetTicket(i);
      if(!PositionSelectByTicket(ticket) || PositionGetString(POSITION_SYMBOL) != SymbolName || PositionGetInteger(POSITION_MAGIC) != MagicNumber(tf)) continue;
      double openPrice, sl, tp, currPrice, lots, profit;
      if(!PositionGetDouble(POSITION_PRICE_OPEN, openPrice) ||
         !PositionGetDouble(POSITION_SL, sl) ||
         !PositionGetDouble(POSITION_TP, tp) ||
         !PositionGetDouble(POSITION_VOLUME, lots) ||
         !PositionGetDouble(POSITION_PROFIT, profit)) continue;
      currPrice = PositionGetInteger(POSITION_TYPE) == POSITION_TYPE_BUY ? SymbolInfoDouble(SymbolName, SYMBOL_BID) : SymbolInfoDouble(SymbolName, SYMBOL_ASK);
      int idx = TimeframeIndex(tf);
      if(idx < 0) continue;
      double atr[1];
      if(CopyBuffer(tfs[idx].h_atr, 0, 0, 1, atr) < 1) continue;

      // Close trade if opposite breakout occurs (market structure shift)
      if(PositionGetInteger(POSITION_TYPE) == POSITION_TYPE_BUY && SellBrk(tf)) {
         trade.PositionClose(ticket);
         Print(tf, " Buy closed due to sell breakout: Ticket=", ticket, ", Profit=", profit);
         LogTrd(ticket, SymbolName, openPrice, sl, tp, "Close", 0, 0, CurRisk, tf);
         continue;
      }
      else if(PositionGetInteger(POSITION_TYPE) == POSITION_TYPE_SELL && BuyBrk(tf)) {
         trade.PositionClose(ticket);
         Print(tf, " Sell closed due to buy breakout: Ticket=", ticket, ", Profit=", profit);
         LogTrd(ticket, SymbolName, openPrice, sl, tp, "Close", 0, 0, CurRisk, tf);
         continue;
      }

      // Partial close at 1:2 RR
      if((PositionGetInteger(POSITION_TYPE) == POSITION_TYPE_BUY && currPrice >= openPrice + (openPrice - sl) * 2) ||
         (PositionGetInteger(POSITION_TYPE) == POSITION_TYPE_SELL && currPrice <= openPrice - (sl - openPrice) * 2)) {
         if(lots > SymbolInfoDouble(SymbolName, SYMBOL_VOLUME_MIN) * 2) {
            trade.PositionClosePartial(ticket, lots / 2);
            trade.PositionModify(ticket, openPrice, tp);
            Print(tf, " Partial close at 1:2 RR: Ticket=", ticket, ", Lots=", lots / 2);
         }
      }

      // Trailing stop
      double trail = atr[0] * 1.5 / _Point;
      if(PositionGetInteger(POSITION_TYPE) == POSITION_TYPE_BUY && currPrice > openPrice + trail * _Point && sl < currPrice - trail * _Point) {
         trade.PositionModify(ticket, currPrice - trail * _Point, tp);
         Print(tf, " Trailing stop updated: Ticket=", ticket, ", New SL=", currPrice - trail * _Point);
      }
      else if(PositionGetInteger(POSITION_TYPE) == POSITION_TYPE_SELL && currPrice < openPrice - trail * _Point && sl > currPrice + trail * _Point) {
         trade.PositionModify(ticket, currPrice + trail * _Point, tp);
         Print(tf, " Trailing stop updated: Ticket=", ticket, ", New SL=", currPrice + trail * _Point);
      }

      // Log closed trades
      if(profit != 0 && !PositionSelectByTicket(ticket)) {
         LogTrd(ticket, SymbolName, openPrice, sl, tp, "Close", 0, 0, CurRisk, tf);
         Print(tf, " Position closed: Ticket=", ticket, ", Profit=", profit);
      }
   }
}

void FetchNewsCalendar() {
   string url = NewsAPI_Url + NewsAPI_Key;
   string headers = "";
   char post[], result[];
   string result_headers;
   int timeout = 5000;
   int res = WebRequest("GET", url, headers, timeout, post, result, result_headers);
   if(res != 200) {
      Print("News API request failed: HTTP ", res, ", Error=", GetLastError());
      // Fallback to manual calendar
      newsCal[0].time = StringToTime("2025.07.15 14:30"); // CPI
      newsCal[0].evt = "CPI";
      newsCal[0].impact = 90;
      newsCal[1].time = StringToTime("2025.07.23 20:00"); // FOMC
      newsCal[1].evt = "FOMC";
      newsCal[1].impact = 90;
      newsCnt = 2;
      Print("Using fallback news calendar with ", newsCnt, " events");
      return;
   }

   string response = CharArrayToString(result);
   newsCnt = 0;
   ArrayResize(newsCal, 100);

   // Parse Alpha Vantage NEWS_SENTIMENT response
   int pos = 0;
   while(pos >= 0 && newsCnt < 100) {
      pos = StringFind(response, "\"items\":", pos);
      if(pos < 0) break;
      pos = StringFind(response, "{", pos);
      if(pos < 0) break;

      int end = StringFind(response, "}", pos);
      if(end < 0) break;
      string item = StringSubstr(response, pos, end - pos + 1);

      // Extract fields
      string evtName = ExtractJsonField(item, "\"title\":");
      string evtTime = ExtractJsonField(item, "\"time_published\":");
      string relevance = ExtractJsonField(item, "\"relevance_score\":");

      if(evtName != "" && evtTime != "") {
         // Convert time format (e.g., "20250715T143000" to datetime)
         string dt = StringSubstr(evtTime, 0, 4) + "." + StringSubstr(evtTime, 4, 2) + "." + StringSubstr(evtTime, 6, 2) + " " + 
                     StringSubstr(evtTime, 9, 2) + ":" + StringSubstr(evtTime, 11, 2);
         newsCal[newsCnt].time = StringToTime(dt);
         newsCal[newsCnt].evt = evtName;
         newsCal[newsCnt].impact = (relevance != "") ? (int)(StringToDouble(relevance) * 100) : 80;
         if(newsCal[newsCnt].impact > 80 && newsCal[newsCnt].time > TimeCurrent() - 7 * 86400) {
            newsCnt++;
            Print("News event loaded: ", evtName, " at ", TimeToString(newsCal[newsCnt-1].time), ", Impact=", newsCal[newsCnt-1].impact);
         }
      }
      pos = end + 1;
   }
   Print("Loaded ", newsCnt, " high-impact news events from API");
}

string ExtractJsonField(string json, string field) {
   int pos = StringFind(json, field);
   if(pos < 0) return "";
   pos += StringLen(field);
   if(StringFind(json, "\"", pos) == pos) pos++; // Skip opening quote
   int end = StringFind(json, "\"", pos);
   if(end < 0) return "";
   return StringSubstr(json, pos, end - pos);
}

bool NewBar(ENUM_TIMEFRAMES tf) {
   int idx = TimeframeIndex(tf);
   if(idx < 0) return false;
   datetime cur = iTime(SymbolName, tf, 0);
   if(cur != tfs[idx].lastBar) {
      tfs[idx].lastBar = cur;
      return true;
   }
   return false;
}

bool BullTrend(ENUM_TIMEFRAMES tf) {
   int handle = (tf == PERIOD_D1) ? h_ema_d1 : h_ema_h4;
   double ema[2];
   if(CopyBuffer(handle, 0, 1, 2, ema) < 2) {
      Print("EMA copy failed for ", EnumToString(tf));
      return false;
   }
   Print("EMA ", EnumToString(tf), ": ", ema[1], " vs ", ema[0]);
   return ema[1] > ema[0];
}

bool BearTrend(ENUM_TIMEFRAMES tf) {
   int handle = (tf == PERIOD_D1) ? h_ema_d1 : h_ema_h4;
   double ema[2];
   if(CopyBuffer(handle, 0, 1, 2, ema) < 2) {
      Print("EMA copy failed for ", EnumToString(tf));
      return false;
   }
   Print("EMA ", EnumToString(tf), ": ", ema[1], " vs ", ema[0]);
   return ema[1] < ema[0];
}

bool BuyBrk(ENUM_TIMEFRAMES tf) {
   double high = iHigh(SymbolName, tf, iHighest(SymbolName, tf, MODE_HIGH, Brk_Prd, 1));
   double price = SymbolInfoDouble(SymbolName, SYMBOL_ASK);
   Print(tf, " BuyBrk check: Price=", price, ", High=", high);
   return price > high;
}

bool SellBrk(ENUM_TIMEFRAMES tf) {
   double low = iLow(SymbolName, tf, iLowest(SymbolName, tf, MODE_LOW, Brk_Prd, 1));
   double price = SymbolInfoDouble(SymbolName, SYMBOL_BID);
   Print(tf, " SellBrk check: Price=", price, ", Low=", low);
   return price < low;
}

double CalcLots(double eq, double riskPct, double slPips) {
   double pipVal = SymbolInfoDouble(SymbolName, SYMBOL_TRADE_TICK_VALUE);
   double riskAmt = MathMin(eq * (riskPct / 100), MaxLossUSD);
   double lots = riskAmt / (slPips * pipVal);
   double minLot = SymbolInfoDouble(SymbolName, SYMBOL_VOLUME_MIN);
   double maxLot = SymbolInfoDouble(SymbolName, SYMBOL_VOLUME_MAX);
   lots = NormalizeDouble(MathMax(minLot, MathMin(maxLot, lots)), 2);
   Print("CalcLots: Equity=", eq, ", Risk%=", riskPct, ", SL Pips=", slPips, ", PipVal=", pipVal, ", Lots=", lots, ", MinLot=", minLot, ", MaxLot=", maxLot);
   return lots;
}

bool IsNews() {
   if(newsCnt == 0 && NewsFilt) {
      Print("No news events loaded, bypassing news filter");
      return false;
   }
   datetime now = TimeCurrent();
   Print("News check: Current time=", TimeToString(now, TIME_DATE|TIME_MINUTES));
   for(int i = 0; i < newsCnt; i++) {
      if(now >= newsCal[i].time - NewsPause * 60 && now <= newsCal[i].time + NewsPause * 60 && newsCal[i].impact > 80) {
         Print("News event active: ", newsCal[i].evt, " at ", TimeToString(newsCal[i].time, TIME_DATE|TIME_MINUTES));
         return true;
      }
   }
   Print("No active news events");
   return false;
}

void LogTrd(ulong ticket, string sym, double price, double sl, double tp, string stat, double brkStr, double vol, double risk, ENUM_TIMEFRAMES tf) {
   string data = StringFormat("T=%I64u,S=%s,Tm=%s,P=%f,SL=%f,TP=%f,St=%s,BrkStr=%f,Vol=%f,Risk=%f,TF=%s",
      ticket, sym, TimeToString(TimeCurrent(), TIME_DATE|TIME_MINUTES), price, sl, tp, stat, brkStr, vol, risk, EnumToString(tf));
   if(StringLen(GS_Url) > 0) Print("Webhook pending: ", data);
   Print("Trd: ", data);

   if(stat == "Close" && tradeCnt < ArraySize(tradeHistory)) {
      double profit;
      if(PositionSelectByTicket(ticket) && PositionGetDouble(POSITION_PROFIT, profit)) {
         tradeHistory[tradeCnt].ticket = ticket;
         tradeHistory[tradeCnt].isWin = profit > 0;
         tradeHistory[tradeCnt].profit = profit;
         tradeHistory[tradeCnt].brkStr = brkStr;
         tradeHistory[tradeCnt].vol = vol;
         tradeHistory[tradeCnt].risk = risk;
         tradeHistory[tradeCnt].tf = tf;
         tradeCnt++;
         UpdateWinLossStreak();
         AdjustBreakoutStrength();
      }
   }
}

void UpdateWinLossStreak() {
   if(tradeCnt > 0) {
      if(tradeHistory[tradeCnt-1].isWin) {
         winStreak++;
         lossStreak = 0;
      } else {
         lossStreak++;
         winStreak = 0;
      }
   }
}

void AdjustBreakoutStrength() {
   if(tradeCnt < 10) return;
   int lossCnt = 0;
   double avgBrkStr = 0;
   for(int i = tradeCnt - 10; i < tradeCnt; i++) {
      if(!tradeHistory[i].isWin) lossCnt++;
      avgBrkStr += tradeHistory[i].brkStr;
   }
   avgBrkStr /= 10;
   if(lossCnt >= 5 && avgBrkStr < MinBrkStr * 1.5) dynBrkStr = MinBrkStr * 1.5;
   else if(lossCnt <= 2) dynBrkStr = MinBrkStr;
   Print("Breakout strength adjusted: dynBrkStr=", dynBrkStr);
}

long MagicNumber(ENUM_TIMEFRAMES tf) {
   if(tf == PERIOD_M15) return 1015;
   if(tf == PERIOD_H1) return 1060;
   return 0;
}

int TimeframeIndex(ENUM_TIMEFRAMES tf) {
   for(int i = 0; i < ArraySize(tfs); i++) {
      if(tfs[i].tf == tf) return i;
   }
   return -1;
}