//+------------------------------------------------------------------+
//|                                                  zenBreakout.mq5 |
//|          Copyright 2025, MetaQuotes Ltd. Developer is Chacha Ian |
//|                          https://www.mql5.com/en/users/chachaian |
//+------------------------------------------------------------------+
#property copyright "Copyright 2025, MetaQuotes Ltd. Developer is Chacha Ian"
#property link      "https://www.mql5.com/en/users/chachaian"
#property version   "1.10"
#resource "\\Indicators\\heikinAshiIndicator.ex5"

//--- Macros
#define zenBreakout "zenBreakout"

//--- Custom enumerations
enum ENUM_RISK_REWARD_RATIO   { ONE_TO_ONE, ONE_TO_ONEandHALF, ONE_TO_TWO, ONE_TO_THREE, ONE_TO_FOUR, ONE_TO_FIVE, ONE_TO_SIX };
enum ENUM_LOT_SIZE_INPUT_MODE { MODE_MANUAL, MODE_AUTO };

//--- Libraries
#include <Trade\Trade.mqh>
#include <ChartObjects\ChartObjectsLines.mqh>

//--- Input parameters
input group "Information"
input ulong magicNumber         = 254700680002;
input ENUM_TIMEFRAMES timeframe = PERIOD_CURRENT;

input group "Risk Management"
input ENUM_LOT_SIZE_INPUT_MODE lotSizeMode = MODE_AUTO;
input double riskPerTradePercent           = 1.0;
input double lotSize                       = 0.1;
input ENUM_RISK_REWARD_RATIO RRr           = ONE_TO_ONEandHALF;

//--- Data Structures
struct MqlTradeInfo
{
   ulong orderTicket;                 
   ENUM_ORDER_TYPE type;
   ENUM_POSITION_TYPE posType;
   double entryPrice;
   double takeProfitLevel;
   double stopLossLevel;
   datetime openTime;
   double lotSize;   
};

struct MqlAppData 
{
   double bidPrice;
   double askPrice;
   double currentBalance;
   double currentEquity;
   datetime currentGmtTime;
   datetime lastDailyCheckTime;
   datetime lastBarOpenTime;
   double contractSize;
   long digitValue;
   double amountAtRisk;
   MqlTradeInfo tradeInfo;
};

//--- Global variables
CTrade Trade;
CChartObjectTrend TrendLine;
MqlAppData AppData;

int    heikinAshiIndicatorHandle;
double heikinAshiOpen     [];
double heikinAshiHigh     [];
double heikinAshiLow      [];
double heikinAshiClose    [];

int    fractalsIndicatorHandle;
double swingHighs [];
double swingLows  [];

//+------------------------------------------------------------------+
//| Expert initialization function                                   |
//+------------------------------------------------------------------+
int OnInit(){

   //--- Configure chart appearance
   if(!ConfigureChartAppearance()){
      return(INIT_FAILED);
   }

   //--- Set Expert Magic Number
   Trade.SetExpertMagicNumber(magicNumber);
   
   //--- Initialize global variables
   AppData.currentBalance        = AccountInfoDouble(ACCOUNT_BALANCE);
   AppData.currentEquity         = AccountInfoDouble(ACCOUNT_EQUITY);
   AppData.lastDailyCheckTime    = iTime(_Symbol, PERIOD_D1, 0);
   AppData.lastBarOpenTime       = 0;
   AppData.digitValue            = SymbolInfoInteger(_Symbol, SYMBOL_DIGITS);
   AppData.contractSize          = SymbolInfoDouble (_Symbol, SYMBOL_TRADE_CONTRACT_SIZE);
   
   //--- Initialize the Heikin Ashi indicator
   heikinAshiIndicatorHandle     = iCustom(_Symbol, timeframe, "::Indicators\\heikinAshiIndicator.ex5");
   if(heikinAshiIndicatorHandle  == INVALID_HANDLE){
      Print("Error while initializing The Heikin Ashi Indicator: ", GetLastError());
      return INIT_FAILED;
   }
   
   //--- Initialize the Fractals indicator
   fractalsIndicatorHandle = iFractals(_Symbol, timeframe);
   if(fractalsIndicatorHandle == INVALID_HANDLE){
      Print("Error while initializing The Fractals Indicator: ", GetLastError());
      return INIT_FAILED;
   }
   
   //--- Set Arrays as series
   ArraySetAsSeries(heikinAshiOpen,  true);
   ArraySetAsSeries(heikinAshiHigh,  true);
   ArraySetAsSeries(heikinAshiLow,   true);
   ArraySetAsSeries(heikinAshiClose, true);
   ArraySetAsSeries(swingHighs,      true);
   ArraySetAsSeries(swingLows,       true); 
   
   return(INIT_SUCCEEDED);
}

//+------------------------------------------------------------------+
//| Expert deinitialization function                                 |
//+------------------------------------------------------------------+
void OnDeinit(const int reason){
   
   //--- Delete all graphical objects
   ObjectsDeleteAll(0);
   
   //--- Free up memory used by indicators
   if(heikinAshiIndicatorHandle != INVALID_HANDLE){
      IndicatorRelease(heikinAshiIndicatorHandle);
   }
   
   if(fractalsIndicatorHandle != INVALID_HANDLE){
      IndicatorRelease(fractalsIndicatorHandle);
   }
   
}

//+------------------------------------------------------------------+
//| Expert tick function                                             |
//+------------------------------------------------------------------+
void OnTick(){

   //--- Scope variables
   AppData.bidPrice           = SymbolInfoDouble (_Symbol, SYMBOL_BID);
   AppData.askPrice           = SymbolInfoDouble (_Symbol, SYMBOL_ASK);
   AppData.currentBalance     = AccountInfoDouble(ACCOUNT_BALANCE);
   AppData.currentEquity      = AccountInfoDouble(ACCOUNT_EQUITY);
   AppData.amountAtRisk       = (riskPerTradePercent/100.0) * AppData.currentBalance;  

   //--- Get a few Heikin Ashi values
   int copiedHeikinAshiOpen = CopyBuffer(heikinAshiIndicatorHandle, 0, 0, 10, heikinAshiOpen);
   if(copiedHeikinAshiOpen  == -1){
      Print("Error while copying Heikin Ashi Open prices: ", GetLastError());
      return;
   }
   
   int copiedHeikinAshiHigh = CopyBuffer(heikinAshiIndicatorHandle, 1, 0, 10, heikinAshiHigh);
   if(copiedHeikinAshiHigh  == -1){
      Print("Error while copying Heikin Ashi High prices: ", GetLastError());
      return;
   }
   
   int copiedHeikinAshiLow = CopyBuffer(heikinAshiIndicatorHandle, 2, 0, 10, heikinAshiLow);
   if(copiedHeikinAshiLow  == -1){
      Print("Error while copying Heikin Ashi Low prices: ", GetLastError());
      return;
   }
   
   int copiedHeikinAshiClose = CopyBuffer(heikinAshiIndicatorHandle, 3, 0, 10, heikinAshiClose);
   if(copiedHeikinAshiClose  == -1){
      Print("Error while copying Heikin Ashi Close prices: ", GetLastError());
      return;
   }
   
   //--- Get the latest Fractals indicator values
   int copiedSwingHighs = CopyBuffer(fractalsIndicatorHandle, 0, 0, 200, swingHighs);
   if(copiedSwingHighs == -1){
      Print("Error while copying fractal's indicator swing highs: ", GetLastError());
   }
   
   int copiedSwingLows = CopyBuffer(fractalsIndicatorHandle, 1, 0, 200, swingLows);
   if(copiedSwingLows == -1){
      Print("Error while copying fractal's indicator swing lows: ", GetLastError());
   }
   
   //--- Run this block on new bar open
   if(IsNewBar(_Symbol, timeframe, AppData.lastBarOpenTime)){
      
      datetime timeStart  = 0;
      int      indexStart = 0;
      datetime timeEnd    = 0;
      int      indexEnd   = 0;
      
      //--- Handle Bullish Signals
      if(IsBullishSignal(timeStart, indexStart, timeEnd, indexEnd)){
         if(!IsThereAnActiveBuyPosition(magicNumber) && !IsThereAnActiveSellPosition(magicNumber)){
            OpenBuy();
         }
         double high = iHigh(_Symbol, timeframe, indexStart);
         TrendLine.Create(0, GenerateUniqueName(zenBreakout), 0, timeStart, high, timeEnd, high);
      }
      
      //--- Handle Bearish Signals
      if(IsBearishSignal(timeStart, indexStart, timeEnd, indexEnd)){
         if(!IsThereAnActiveBuyPosition(magicNumber) && !IsThereAnActiveSellPosition(magicNumber)){
            OpenSel();
         }
         double low  = iLow (_Symbol, timeframe, indexStart);
         TrendLine.Create(0, GenerateUniqueName(zenBreakout), 0, timeStart, low, timeEnd, low);
      }
   }

}

//--- Utility functions
//+------------------------------------------------------------------+
//| This function configures the chart's appearance.                 |                                   |
//+------------------------------------------------------------------+
bool ConfigureChartAppearance()
{
   if(!ChartSetInteger(0, CHART_COLOR_BACKGROUND, clrWhite)){
      Print("Error while setting chart background, ", GetLastError());
      return false;
   }
   
   if(!ChartSetInteger(0, CHART_SHOW_GRID, false)){
      Print("Error while setting chart grid, ", GetLastError());
      return false;
   }
   
   if(!ChartSetInteger(0, CHART_MODE, CHART_LINE)){
      Print("Error while setting chart mode, ", GetLastError());
      return false;
   }

   if(!ChartSetInteger(0, CHART_COLOR_FOREGROUND, clrBlack)){
      Print("Error while setting chart foreground, ", GetLastError());
      return false;
   }
   
   return true; 
}

//+-------------------------------------------------------------------------+
//| Function to generate a unique graphical object name with a given prefix |                                   |
//+-------------------------------------------------------------------------+
string GenerateUniqueName(string prefix){
   int attempt = 0;
   string uniqueName;
   while(true)
   {
      uniqueName = prefix + IntegerToString(MathRand() + attempt);
      if(ObjectFind(0, uniqueName) < 0)
         break;
      attempt++;
   }
   return uniqueName;
}
//+-------------------------------------------------------------------------+
//| Returns true if Heikin Ashi candle is bullish and has no lower wick     |                                   |
//+-------------------------------------------------------------------------+
bool IsBullishBreakoutCandle(int index)
{
   if(index < 0 || index >= ArraySize(heikinAshiOpen)) return false;

   double open  = heikinAshiOpen[index];
   double close = heikinAshiClose[index];
   double low   = heikinAshiLow[index];

   //--- Candle must be bullish and have no lower wick
   return (close > open && low >= MathMin(open, close));
}

//+-------------------------------------------------------------------------+
//| Returns true if Heikin Ashi candle is bearish and has no upper wick     |                                   |
//+-------------------------------------------------------------------------+
bool IsBearishBreakoutCandle(int index)
{
   if(index < 0 || index >= ArraySize(heikinAshiOpen)) return false;

   double open  = heikinAshiOpen[index];
   double close = heikinAshiClose[index];
   double high  = heikinAshiHigh[index];

   //--- Candle must be bearish and have no upper wick
   return (close < open && high <= MathMax(open, close));
}

//+----------------------------------------------------------------------------------------------+
//| Returns the index of the most recent swing high before 'fromIndex'. Returns -1 if not found  |                                   |
//+----------------------------------------------------------------------------------------------+
int FindMostRecentSwingHighIndex(int fromIndex)
{
   if(fromIndex <= 0 || fromIndex >= ArraySize(swingHighs))
      fromIndex = 1;

   for(int i = fromIndex; i < ArraySize(swingHighs); i++)
   {
      if(swingHighs[i] != EMPTY_VALUE)
         return i;
   }

   return -1; //--- No swing high found
}

//+----------------------------------------------------------------------------------------------+
//| Returns the index of the most recent swing low before 'fromIndex'. Returns -1 if not found   |                                   |
//+----------------------------------------------------------------------------------------------+
int FindMostRecentSwingLowIndex(int fromIndex)
{
   if(fromIndex <= 0 || fromIndex >= ArraySize(swingLows))
      fromIndex = 1;

   for(int i = fromIndex; i < ArraySize(swingLows); i++)
   {
      if(swingLows[i] != EMPTY_VALUE)
         return i;
   }

   return -1; // No swing low found
}

//+------------------------------------------------------------------+
//| This function detects a bullish signal                           |
//+------------------------------------------------------------------+
bool IsBullishSignal(datetime &timeStart, int &indexStart, datetime &timeEnd, int &indexEnd)
{
   indexStart = FindMostRecentSwingHighIndex(1);
   double recentSwingHigh               = iHigh(_Symbol, timeframe, indexStart);
   double previousHeikinAshiCandleClose = heikinAshiClose[1];
   double previousHeikinAshiCandleOpen  = heikinAshiOpen[1];
   
   if(IsBullishBreakoutCandle(1)){
      if(previousHeikinAshiCandleClose > recentSwingHigh && previousHeikinAshiCandleOpen < recentSwingHigh){
         timeStart = iTime(_Symbol, timeframe, indexStart);
         indexEnd  = 0;
         timeEnd   = iTime(_Symbol, timeframe, indexEnd);
         return true;
      }
   }
   return false;
}

//+------------------------------------------------------------------+
//| This function detects a bearish signal                           |
//+------------------------------------------------------------------+
bool IsBearishSignal(datetime &timeStart, int &indexStart, datetime &timeEnd, int &indexEnd)
{
   indexStart = FindMostRecentSwingLowIndex(1);
   double recentSwingLow                = iLow(_Symbol, timeframe, indexStart);
   double previousHeikinAshiCandleClose = heikinAshiClose[1];
   double previousHeikinAshiCandleOpen  = heikinAshiOpen[1];
   
   if(IsBearishBreakoutCandle(1)){
      if(previousHeikinAshiCandleClose < recentSwingLow && previousHeikinAshiCandleOpen > recentSwingLow){
         timeStart = iTime(_Symbol, timeframe, indexStart);
         indexEnd  = 0;
         timeEnd   = iTime(_Symbol, timeframe, indexEnd);
         return true;
      }
   }
   return false;
}

//+-------------------------------------------------------------------+
//| Function to check if there's a new bar on a given chart timeframe |                           |
//+-------------------------------------------------------------------+
bool IsNewBar(string symbol, ENUM_TIMEFRAMES tf, datetime &lastTm)
{

   datetime currentTime = iTime(symbol, tf, 0);
   if(currentTime != lastTm){
      lastTm       = currentTime;
      return true;
   }  
   return false;
   
}

//+------------------------------------------------------------------+
//| To check if there is an active buy position opened by this EA    |                                 |
//+------------------------------------------------------------------+
bool IsThereAnActiveBuyPosition(ulong magicNm){
   for(int i = PositionsTotal() - 1; i >= 0; i--){
      ulong ticket = PositionGetTicket(i);
      if(ticket == 0){
         Print("Error while fetching position ticket ", _LastError);
         continue;
      }else{
         if(PositionGetInteger(POSITION_MAGIC) == magicNm && PositionGetInteger(POSITION_TYPE) == POSITION_TYPE_BUY){
            return true;
         }
      }
   }
   return false;
}

//+------------------------------------------------------------------+
//| To check if there is an active sell position opened by this EA   |                                 |
//+------------------------------------------------------------------+
bool IsThereAnActiveSellPosition(ulong mgcNumber){
   for(int i = PositionsTotal() - 1; i >= 0; i--){
      ulong ticket = PositionGetTicket(i);
      if(ticket == 0){
         Print("Error while fetching position ticket ", _LastError);
         continue;
      }else{
         if(PositionGetInteger(POSITION_MAGIC) == mgcNumber && PositionGetInteger(POSITION_TYPE) == POSITION_TYPE_SELL){
            return true;
         }
      }
   }
   return false;
}

//+------------------------------------------------------------------+
//| To open a buy position                                           |
//+------------------------------------------------------------------+
bool OpenBuy(){
   double rewardValue = 1.0;
   switch(RRr){
      case ONE_TO_ONE: 
         rewardValue = 1.0;
         break;
      case ONE_TO_ONEandHALF:
         rewardValue = 1.5;
         break;
      case ONE_TO_TWO: 
         rewardValue = 2.0;
         break;
      case ONE_TO_THREE: 
         rewardValue = 3.0;
         break;
      case ONE_TO_FOUR: 
         rewardValue = 4.0;
         break;
      case ONE_TO_FIVE: 
         rewardValue = 5.0;
         break;
      case ONE_TO_SIX: 
         rewardValue = 6.0;
         break;
      default:
         rewardValue = 1.0;
         break;
   }
   ENUM_POSITION_TYPE positionType    = POSITION_TYPE_BUY;
   ENUM_ORDER_TYPE   action           = ORDER_TYPE_BUY;
   double stopLevel                   = iLow(_Symbol, timeframe, 1);
   double askPrice                    = AppData.askPrice;
   double bidPrice                    = AppData.bidPrice;
   double stopDistance                = askPrice - stopLevel;
   double targetLevel                 = askPrice + (stopDistance * rewardValue);  
   double lotSz                       = AppData.amountAtRisk / (AppData.contractSize * stopDistance);
   
   if(lotSizeMode == MODE_AUTO){
      lotSz                              = NormalizeDouble(lotSz, 2);
   }else{
      lotSz                              = NormalizeDouble(lotSize, 2);
   }
      
   if(!Trade.Buy(lotSz, _Symbol, askPrice, stopLevel, targetLevel)){
      Print("Error while opening a long position, ", GetLastError());
      Print(Trade.ResultRetcode());
      Print(Trade.ResultComment());
      return false;
   }else{
      MqlTradeResult result = {};
      Trade.Result(result);
      AppData.tradeInfo.orderTicket                 = result.order;
      AppData.tradeInfo.type                        = action;
      AppData.tradeInfo.posType                     = positionType;
      AppData.tradeInfo.entryPrice                  = result.price;
      AppData.tradeInfo.takeProfitLevel             = targetLevel;
      AppData.tradeInfo.stopLossLevel               = stopLevel;
      AppData.tradeInfo.openTime                    = AppData.currentGmtTime;
      AppData.tradeInfo.lotSize                     = lotSz;
      return true;
   }
   
   return false; 
}

//+------------------------------------------------------------------+
//| To open a sell position                                          |
//+------------------------------------------------------------------+
bool OpenSel(){
   double rewardValue = 1.0;
   switch(RRr){
      case ONE_TO_ONE: 
         rewardValue = 1.0;
         break;
      case ONE_TO_ONEandHALF:
         rewardValue = 1.5;
         break;
      case ONE_TO_TWO: 
         rewardValue = 2.0;
         break;
      case ONE_TO_THREE: 
         rewardValue = 3.0;
         break;
      case ONE_TO_FOUR: 
         rewardValue = 4.0;
         break;
      case ONE_TO_FIVE: 
         rewardValue = 5.0;
         break;
      case ONE_TO_SIX: 
         rewardValue = 6.0;
         break;
      default:
         rewardValue = 1.0;
         break;
   }
   ENUM_POSITION_TYPE positionType    = POSITION_TYPE_SELL;
   ENUM_ORDER_TYPE   action           = ORDER_TYPE_SELL;
   double stopLevel                   = iHigh(_Symbol, timeframe, 1);
   double bidPrice                    = AppData.bidPrice;
   double askPrice                    = AppData.askPrice;
   double stopDistance                = stopLevel - bidPrice;
   double targetLevel                 = bidPrice - (stopDistance * rewardValue);
   double lotSz                       = AppData.amountAtRisk / (AppData.contractSize * stopDistance);
   
   if(lotSizeMode == MODE_AUTO){
      lotSz                              = NormalizeDouble(lotSz, 2);
   }else{
      lotSz                              = NormalizeDouble(lotSize, 2);
   }
   
   if(!Trade.Sell(lotSz, _Symbol, bidPrice, stopLevel, targetLevel)){
      Print("Error while opening a short position, ", GetLastError());
      Print(Trade.ResultRetcode());
      Print(Trade.ResultComment());
      return false;
   }else{ 
      MqlTradeResult result = {};
      Trade.Result(result);
      AppData.tradeInfo.orderTicket                 = result.order;
      AppData.tradeInfo.type                        = action;
      AppData.tradeInfo.posType                     = positionType;
      AppData.tradeInfo.entryPrice                  = result.price;
      AppData.tradeInfo.takeProfitLevel             = targetLevel;
      AppData.tradeInfo.stopLossLevel               = stopLevel;
      AppData.tradeInfo.openTime                    = AppData.currentGmtTime;
      AppData.tradeInfo.lotSize                     = lotSz;
      return true;
   }
   return false; 
}
//+------------------------------------------------------------------+