//+------------------------------------------------------------------+
//|                                          heikinAshiIndicator.mq5 |
//|                                  Copyright 2025, MetaQuotes Ltd. |
//|                          https://www.mql5.com/en/users/chachaian |
//+------------------------------------------------------------------+
#property copyright "Copyright 2025, MetaQuotes Ltd."
#property link      "https://www.mql5.com/en/users/chachaian"
#property version   "1.00"
#property indicator_chart_window
#property indicator_buffers 5
#property indicator_plots   1
#property indicator_color1 C'38,166,154', C'239,83,80', clrYellow
#property indicator_label1 "HeikinAshiOpen;HeikinAshiHigh;HeikinAshiLow;HeikinAshiClose"

//Global variables
double haOpen      [];
double haHigh      [];
double haLow       [];
double haClose     [];
double colorBuffer [];

//+------------------------------------------------------------------+
//| Custom indicator initialization function                         |
//+------------------------------------------------------------------+
int OnInit()
{

   if(!ConfigureChartAppearance()){
      Print("Error while configuring chart appearance: ", GetLastError());
      return INIT_FAILED;
   }
   
   //--- Registration of indicator buffers
   if(!SetIndexBuffer(0, haOpen, INDICATOR_DATA)){
      Print("Error while registering an indicator buffer: ", GetLastError());
      return INIT_FAILED;
   }
   
   if(!SetIndexBuffer(1, haHigh, INDICATOR_DATA)){
      Print("Error while registering an indicator buffer: ", GetLastError());
      return INIT_FAILED;
   }
   
   if(!SetIndexBuffer(2, haLow,   INDICATOR_DATA)){
      Print("Error while registering an indicator buffer: ", GetLastError());
      return INIT_FAILED;
   }
   
   if(!SetIndexBuffer(3, haClose, INDICATOR_DATA)){
      Print("Error while registering an indicator buffer: ", GetLastError());
      return INIT_FAILED;
   }
   
   if(!SetIndexBuffer(4, colorBuffer, INDICATOR_COLOR_INDEX)){
      Print("Error while registering an indicator buffer: ", GetLastError());
      return INIT_FAILED;
   }
   
   //--- Configuration of graphic plots
   if(!PlotIndexSetInteger(0, PLOT_DRAW_TYPE, DRAW_COLOR_CANDLES)){
      Print("Error while configuring graphic plots: ", GetLastError());
      return INIT_FAILED;
   }
   
   if(!PlotIndexSetInteger(0, PLOT_SHOW_DATA, true)){
      Print("Error while configuring graphic plots: ", GetLastError());
      return INIT_FAILED;
   }
   
   //--- General indicator configurations
   if(!IndicatorSetInteger(INDICATOR_DIGITS, Digits())){
      Print("Error while setting indicator values accuracy: ", GetLastError());
      return INIT_FAILED;
   }
   
   if(!IndicatorSetString(INDICATOR_SHORTNAME, "HeikinAshi")){
      Print("Error while setting indicator shortname: ", GetLastError());
      return INIT_FAILED;
   }
   
   return INIT_SUCCEEDED;
}

//+------------------------------------------------------------------+
//| Custom indicator iteration function                              |
//+------------------------------------------------------------------+
int OnCalculate(const int32_t  rates_total,
                const int32_t  prev_calculated,
                const datetime &time       [],
                const double   &open       [],
                const double   &high       [],
                const double   &low        [],
                const double   &close      [],
                const long     &tick_volume[],
                const long     &volume     [],
                const int32_t  &spread     []) 
{
   //--- This block is executed when the indicator is initially attached on a chart
   if(prev_calculated == 0){
      GetHeikinAshiValues(open, high, low, close, rates_total);
      GetHeikinAshiColors(rates_total);
   }
   
   //--- This block is executed on every new bar open
   if(prev_calculated != rates_total && prev_calculated != 0){
      GetHeikinAshiValues(open, high, low, close, rates_total);
      GetHeikinAshiColors(rates_total);
   }
   
   //--- This block is executed on arrival of new price (tick) data
   if(prev_calculated == rates_total){
      GetCurrentHeikinAshiValue(open, high, low, close, rates_total);
      GetCurrentHeikinAshiColor(rates_total);
   }
   
   return(rates_total);
}

//--- Utility functions
//+----------------------------------------------------------------------------------+
//| Calculates Heikin Ashi values for all historical candles using price data arrays.|
//+----------------------------------------------------------------------------------+

void GetHeikinAshiValues(const double &open[], const double &high[], const double &low[], const double &close[], const int32_t rates_total)
{ 

   if(ArraySize(open) < rates_total){
      return;
   }
    
   //--- Run a loop through all historical bars
   for(int i=0; i<rates_total; i++){      
      if(i == 0){
         haOpen [i] = (open[i] + close[i]) / 2.0;
         haClose[i] = (open[i] + high[i] + low[i] + close[i]) / 4.0;
         haHigh [i] = MathMax(high[i], MathMax(open[i], close[i]));
         haLow  [i] = MathMin(low [i], MathMin(open[i], close[i]));
      }else{
         haOpen [i] = (haOpen[i-1] + haClose[i-1]) / 2.0;
         haClose[i] = (open[i] + high[i] + low[i] + close[i]) / 4.0;
         haHigh [i] = MathMax(high[i], MathMax(haOpen[i], haClose[i]));
         haLow  [i] = MathMin(low [i], MathMin(haOpen[i], haClose[i]));  
      }
   }
   
}

//+---------------------------------------------------------------------------------------+
//| Calculates Heikin Ashi values for the most recent candle only (for real-time updates).|
//+---------------------------------------------------------------------------------------+

void GetCurrentHeikinAshiValue(const double &open[], const double &high[], const double &low[], const double &close[], const int32_t rates_total)
{
   haOpen [rates_total - 1] = (haOpen[rates_total-2] + haClose[rates_total-2]) / 2.0;
   haClose[rates_total - 1] = (open[rates_total - 1] + high[rates_total - 1] + low[rates_total - 1] + close[rates_total - 1]) / 4.0;
   haHigh [rates_total - 1] = MathMax(high[rates_total - 1], MathMax(haOpen[rates_total - 1], haClose[rates_total - 1]));
   haLow  [rates_total - 1] = MathMin(low [rates_total - 1], MathMin(haOpen[rates_total - 1], haClose[rates_total - 1])); 
}


//+------------------------------------------------------------------------------------------------------------------+
//| Assigns a color code to each historical Heikin Ashi candle based on its direction (bullish, bearish, or neutral).|
//+------------------------------------------------------------------------------------------------------------------+

void GetHeikinAshiColors(const int32_t rates_total)
{
   
   for(int i=0; i<rates_total; i++){
      if(haOpen[i] < haClose[i]){
         colorBuffer[i] = 0;
      }
      
      if(haOpen[i] > haClose[i]){
         colorBuffer[i] = 1;
      }
      
      if(haOpen[i] == haClose[i]){
         colorBuffer[i] = 2;
      }
   }
   
}

//+-----------------------------------------------------------------------------------------------+
//| Assigns a color code to the latest Heikin Ashi candle only (used for real-time color updates).|
//+-----------------------------------------------------------------------------------------------+

void GetCurrentHeikinAshiColor(const int32_t rates_total)
{
      if(haOpen[rates_total - 1] < haClose[rates_total - 1]){
         colorBuffer[rates_total - 1] = 0;
      }
      
      else if(haOpen[rates_total - 1] > haClose[rates_total - 1]){
         colorBuffer[rates_total - 1] = 1;
      }
      
      else {
         colorBuffer[rates_total - 1] = 2;
      }
}


//+-------------------------------------------------+
//| This function configures the chart's appearance.|
//+-------------------------------------------------+
bool ConfigureChartAppearance()
{
   if(!ChartSetInteger(0, CHART_COLOR_BACKGROUND, clrWhite)){
      Print("Error while setting chart background, ", GetLastError());
      return false;
   }
   
   if(!ChartSetInteger(0, CHART_SHOW_GRID, false)){
      Print("Error while setting chart grid, ", GetLastError());
      return false;
   }
   
   if(!ChartSetInteger(0, CHART_MODE, CHART_LINE)){
      Print("Error while setting chart mode, ", GetLastError());
      return false;
   }

   if(!ChartSetInteger(0, CHART_COLOR_FOREGROUND, clrBlack)){
      Print("Error while setting chart foreground, ", GetLastError());
      return false;
   }
   
   return true;
}
//+------------------------------------------------------------------+
