//+------------------------------------------------------------------+
//|                                 ARIMA_Trading_System.mq5         |
//|                                  Arima Forecast System           |
//|                                                                  |
//+------------------------------------------------------------------+
#property copyright "Arima Forecast System "
#property link      "https://www.mql5.com"
#property version   "1.00"
#property indicator_chart_window
#property indicator_buffers 3
#property indicator_plots   1

#property indicator_label1  "ARIMA Forecast"
#property indicator_type1   DRAW_LINE
#property indicator_color1  clrDodgerBlue
#property indicator_style1  STYLE_SOLID
#property indicator_width1  2

//--- Input parameters
input int lookback = 200;        // Lookback period for ARIMA
input int forecast_bars = 20;    // Number of bars to forecast
input int p = 3;                 // ARIMA p parameter (AR order)
input int d = 1;                 // ARIMA d parameter (differencing)
input int q = 2;                 // ARIMA q parameter (MA order)
input double learning_rate = 0.01; // Learning rate for gradient descent
input int max_iterations = 100;   // Max iterations for optimization

//--- Indicator buffers
double ForecastBuffer[];
double PriceBuffer[];
double ErrorBuffer[];

//--- Global variables
double prices[];
double differenced[];
double ar_coeffs[];
double ma_coeffs[];
double errors[];

//+------------------------------------------------------------------+
//| Custom indicator initialization function                           |
//+------------------------------------------------------------------+
int OnInit()
{
   // Set indicator buffers
   SetIndexBuffer(0, ForecastBuffer, INDICATOR_DATA);
   SetIndexBuffer(1, PriceBuffer, INDICATOR_CALCULATIONS);
   SetIndexBuffer(2, ErrorBuffer, INDICATOR_CALCULATIONS);
   
   ArraySetAsSeries(ForecastBuffer, true);
   ArraySetAsSeries(PriceBuffer, true);
   ArraySetAsSeries(ErrorBuffer, true);
   
   PlotIndexSetString(0, PLOT_LABEL, "ARIMA Forecast");
   PlotIndexSetInteger(0, PLOT_SHIFT, forecast_bars); // Shift forecast to future
   
   // Initialize arrays
   ArrayResize(prices, lookback);
   ArrayResize(differenced, lookback);
   ArrayResize(ar_coeffs, p);
   ArrayResize(ma_coeffs, q);
   ArrayResize(errors, lookback);
   
   // Initialize coefficients
   for(int i = 0; i < p; i++) ar_coeffs[i] = 0.1;
   for(int i = 0; i < q; i++) ma_coeffs[i] = 0.1;
   
   return(INIT_SUCCEEDED);
}

//+------------------------------------------------------------------+
//| Compute log-likelihood for ARIMA                                  |
//+------------------------------------------------------------------+
double ComputeLogLikelihood(double &data[], double &ar[], double &ma[])
{
   double ll = 0.0;
   double residuals[];
   ArrayResize(residuals, lookback);
   ArrayInitialize(residuals, 0.0);
   
   // Compute residuals
   for(int i = p; i < lookback; i++)
   {
      double ar_part = 0.0;
      double ma_part = 0.0;
      
      for(int j = 0; j < p && i - j - 1 >= 0; j++)
         ar_part += ar[j] * data[i - j - 1];
      
      for(int j = 0; j < q && i - j - 1 >= 0; j++)
         ma_part += ma[j] * errors[i - j - 1];
      
      residuals[i] = data[i] - (ar_part + ma_part);
      errors[i] = residuals[i];
      
      // Assume residuals are normally distributed
      ll -= 0.5 * MathLog(2 * M_PI) + 0.5 * residuals[i] * residuals[i];
   }
   
   return ll;
}

//+------------------------------------------------------------------+
//| Optimize AR and MA coefficients using gradient descent            |
//+------------------------------------------------------------------+
void OptimizeCoefficients(double &data[])
{
   double temp_ar[], temp_ma[];
   ArrayCopy(temp_ar, ar_coeffs);
   ArrayCopy(temp_ma, ma_coeffs);
   
   double best_ll = -DBL_MAX;
   double grad_ar[], grad_ma[];
   ArrayResize(grad_ar, p);
   ArrayResize(grad_ma, q);
   
   for(int iter = 0; iter < max_iterations; iter++)
   {
      ArrayInitialize(grad_ar, 0.0);
      ArrayInitialize(grad_ma, 0.0);
      
      // Compute gradients
      for(int i = p; i < lookback; i++)
      {
         double ar_part = 0.0, ma_part = 0.0;
         for(int j = 0; j < p && i - j - 1 >= 0; j++)
            ar_part += temp_ar[j] * data[i - j - 1];
         for(int j = 0; j < q && i - j - 1 >= 0; j++)
            ma_part += temp_ma[j] * errors[i - j - 1];
         
         double residual = data[i] - (ar_part + ma_part);
         
         for(int j = 0; j < p && i - j - 1 >= 0; j++)
            grad_ar[j] += -residual * data[i - j - 1];
         for(int j = 0; j < q && i - j - 1 >= 0; j++)
            grad_ma[j] += -residual * errors[i - j - 1];
      }
      
      // Update coefficients
      for(int j = 0; j < p; j++)
         temp_ar[j] += learning_rate * grad_ar[j] / lookback;
      for(int j = 0; j < q; j++)
         temp_ma[j] += learning_rate * grad_ma[j] / lookback;
      
      // Compute log-likelihood
      double current_ll = ComputeLogLikelihood(data, temp_ar, temp_ma);
      
      // Update if better
      if(current_ll > best_ll)
      {
         best_ll = current_ll;
         ArrayCopy(ar_coeffs, temp_ar);
         ArrayCopy(ma_coeffs, temp_ma);
      }
      else
      {
         // Early stopping if no improvement
         break;
      }
   }
}

//+------------------------------------------------------------------+
//| Custom indicator iteration function                                |
//+------------------------------------------------------------------+
int OnCalculate(const int rates_total,
                const int prev_calculated,
                const datetime &time[],
                const double &open[],
                const double &high[],
                const double &low[],
                const double &close[],
                const long &tick_volume[],
                const long &volume[],
                const int &spread[])
{
   ArraySetAsSeries(close, true);
   ArraySetAsSeries(time, true);
   
   // Check if enough data
   if(rates_total < lookback + forecast_bars) return(0);
   
   // Copy close prices
   for(int i = 0; i < lookback; i++)
   {
      prices[i] = close[i];
      PriceBuffer[i] = close[i];
   }
   
   // Apply differencing
   for(int i = 0; i < lookback - d; i++)
   {
      if(d == 1)
         differenced[i] = prices[i] - prices[i + 1];
      else if(d == 2)
         differenced[i] = (prices[i] - prices[i + 1]) - (prices[i + 1] - prices[i + 2]);
      else
         differenced[i] = prices[i];
   }
   
   // Optimize AR and MA coefficients
   ArrayInitialize(errors, 0.0);
   OptimizeCoefficients(differenced);
   
   // ARIMA forecast
   double forecast[];
   ArrayResize(forecast, forecast_bars);
   double undiff[];
   ArrayResize(undiff, forecast_bars);
   
   // Initialize forecast
   forecast[0] = prices[0];
   undiff[0] = prices[0];
   
   // Forecast loop
   for(int i = 1; i < forecast_bars; i++)
   {
      double ar_part = 0.0;
      double ma_part = 0.0;
      
      // AR component
      for(int j = 0; j < p && i - j - 1 >= 0; j++)
         ar_part += ar_coeffs[j] * (j < lookback ? differenced[j] : forecast[i - j - 1]);
      
      // MA component
      for(int j = 0; j < q && i - j - 1 >= 0; j++)
         ma_part += ma_coeffs[j] * errors[j];
      
      // Update forecast
      forecast[i] = ar_part + ma_part;
      
      // Reverse differencing
      if(d == 1)
         undiff[i] = undiff[i - 1] + forecast[i];
      else if(d == 2)
         undiff[i] = undiff[i - 1] + (undiff[i - 1] - (i >= 2 ? undiff[i - 2] : prices[1])) + forecast[i];
      else
         undiff[i] = forecast[i];
      
      // Update errors
      errors[i % lookback] = forecast[i] - (i < lookback ? differenced[i] : 0.0);
   }
   
   // Fill ForecastBuffer
   for(int i = 0; i < forecast_bars; i++)
   {
      ForecastBuffer[i] = undiff[i];
   }
   
   // Extend historical prices
   for(int i = forecast_bars; i < lookback; i++)
   {
      ForecastBuffer[i] = prices[i - forecast_bars];
   }
   
   return(rates_total);
}

//+------------------------------------------------------------------+
//| Custom indicator deinitialization function                         |
//+------------------------------------------------------------------+
void OnDeinit(const int reason)
{
   // Clean up
}