//+------------------------------------------------------------------+
//|                                    MQL5 NEWS CALENDAR PART 8.mq5 |
//|      Copyright 2025, ALLAN MUNENE MUTIIRIA. #@Forex Algo-Trader. |
//|                                     https://forexalgo-trader.com |
//+------------------------------------------------------------------+
#property copyright "Copyright 2025, ALLAN MUNENE MUTIIRIA. #@Forex Algo-Trader"
#property link      "https://forexalgo-trader.com"
#property description "MQL5 NEWS CALENDAR PART 8 - Strategy Tester CSV Trading Support"
#property version   "1.00"

//---- Include trading library
#include <Trade\Trade.mqh>
CTrade trade;

//---- Define resource for CSV (from PART 7)
#resource "\\Files\\Database\\EconomicCalendar.csv" as string EconomicCalendarData

//---- UI element definitions
#define MAIN_REC "MAIN_REC"
#define SUB_REC1 "SUB_REC1"
#define SUB_REC2 "SUB_REC2"
#define HEADER_LABEL "HEADER_LABEL"
#define ARRAY_CALENDAR "ARRAY_CALENDAR"
#define ARRAY_NEWS "ARRAY_NEWS"
#define DATA_HOLDERS "DATA_HOLDERS"
#define TIME_LABEL "TIME_LABEL"
#define IMPACT_LABEL "IMPACT_LABEL"
#define FILTER_LABEL "FILTER_LABEL"
#define FILTER_CURR_BTN "FILTER_CURR_BTN"
#define FILTER_IMP_BTN "FILTER_IMP_BTN"
#define FILTER_TIME_BTN "FILTER_TIME_BTN"
#define CANCEL_BTN "CANCEL_BTN"
#define CURRENCY_BTNS "CURRENCY_BTNS"

//---- Calendar columns and button sizes
string array_calendar[] = {"Date","Time","Cur.","Imp.","Event","Actual","Forecast","Previous"};
int buttons[] = {80,50,50,40,281,60,70,70};

//---- Filter arrays
string curr_filter[] = {"AUD","CAD","CHF","EUR","GBP","JPY","NZD","USD"};
string curr_filter_selected[];
string impact_labels[] = {"None","Low","Medium","High"};
string impact_filter_selected[];
ENUM_CALENDAR_EVENT_IMPORTANCE allowed_importance_levels[] = {
   CALENDAR_IMPORTANCE_NONE, CALENDAR_IMPORTANCE_LOW, CALENDAR_IMPORTANCE_MODERATE, CALENDAR_IMPORTANCE_HIGH
};
ENUM_CALENDAR_EVENT_IMPORTANCE imp_filter_selected[];

//---- Event name tracking
string current_eventNames_data[];
string previous_eventNames_data[];
string last_dashboard_eventNames[]; // Added: Cache for last dashboard event names in tester mode
datetime last_dashboard_update = 0; // Added: Track last dashboard update time in tester mode

//---- Filter flags
bool enableCurrencyFilter = true;
bool enableImportanceFilter = true;
bool enableTimeFilter = true;
bool isDashboardUpdate = true;
bool filters_changed = true; // Added: Flag to detect filter changes in tester mode

//---- Event counters
int totalEvents_Considered = 0;
int totalEvents_Filtered = 0;
int totalEvents_Displayable = 0;

//---- Input parameters (PART 6)
sinput group "General Calendar Settings"
input ENUM_TIMEFRAMES start_time = PERIOD_H12;
input ENUM_TIMEFRAMES end_time = PERIOD_H12;
input ENUM_TIMEFRAMES range_time = PERIOD_H8;
input bool updateServerTime = true; // Enable/Disable Server Time Update in Panel
input bool debugLogging = false; // Added: Enable debug logging in tester mode

//---- Input parameters for tester mode (from PART 7, minimal)
sinput group "Strategy Tester CSV Settings"
input datetime StartDate = D'2025.03.01'; // Download Start Date
input datetime EndDate = D'2025.03.21';   // Download End Date

//---- Structure for CSV events (from PART 7)
struct EconomicEvent {
   string eventDate;      // Date of the event
   string eventTime;      // Time of the event
   string currency;       // Currency affected
   string event;          // Event description
   string importance;     // Importance level
   double actual;         // Actual value
   double forecast;       // Forecast value
   double previous;       // Previous value
   datetime eventDateTime; // Added: Store precomputed datetime for efficiency
};

//---- Global array for tester mode events
EconomicEvent allEvents[];
EconomicEvent filteredEvents[]; // Added: Filtered events for tester mode optimization

//---- Trade settings
enum ETradeMode {
   TRADE_BEFORE,
   TRADE_AFTER,
   NO_TRADE,
   PAUSE_TRADING
};
input ETradeMode tradeMode = TRADE_BEFORE;
input int tradeOffsetHours = 12;
input int tradeOffsetMinutes = 5;
input int tradeOffsetSeconds = 0;
input double tradeLotSize = 0.01;

//---- Trade control
bool tradeExecuted = false;
datetime tradedNewsTime = 0;
int triggeredNewsEvents[];

//+------------------------------------------------------------------+
//| Filter events for tester mode                                     | // Added: Function to pre-filter events by date range
//+------------------------------------------------------------------+
void FilterEventsForTester() {
   ArrayResize(filteredEvents, 0);
   int eventIndex = 0;
   for (int i = 0; i < ArraySize(allEvents); i++) {
      datetime eventDateTime = allEvents[i].eventDateTime;
      if (eventDateTime < StartDate || eventDateTime > EndDate) {
         if (debugLogging) Print("Event ", allEvents[i].event, " skipped in filter due to date range: ", TimeToString(eventDateTime)); // Modified: Conditional logging
         continue;
      }
      ArrayResize(filteredEvents, eventIndex + 1);
      filteredEvents[eventIndex] = allEvents[i];
      eventIndex++;
   }
   if (debugLogging) Print("Tester mode: Filtered ", eventIndex, " events."); // Modified: Conditional logging
   filters_changed = false;
}

//+------------------------------------------------------------------+
//| Expert initialization function                                    |
//+------------------------------------------------------------------+
int OnInit() {
   //---- Create dashboard UI
   createRecLabel(MAIN_REC,50,50,740,410,clrSeaGreen,1);
   createRecLabel(SUB_REC1,50+3,50+30,740-3-3,410-30-3,clrWhite,1);
   createRecLabel(SUB_REC2,50+3+5,50+30+50+27,740-3-3-5-5,410-30-3-50-27-10,clrGreen,1);
   createLabel(HEADER_LABEL,50+3+5,50+5,"MQL5 Economic Calendar",clrWhite,15);

   //---- Create calendar buttons
   int startX = 59;
   for (int i = 0; i < ArraySize(array_calendar); i++) {
      createButton(ARRAY_CALENDAR+IntegerToString(i),startX,132,buttons[i],25,
                   array_calendar[i],clrWhite,13,clrGreen,clrNONE,"Calibri Bold");
      startX += buttons[i]+3;
   }

   //---- Initialize for live mode (unchanged)
   int totalNews = 0;
   bool isNews = false;
   MqlCalendarValue values[];
   datetime startTime = TimeTradeServer() - PeriodSeconds(start_time);
   datetime endTime = TimeTradeServer() + PeriodSeconds(end_time);
   string country_code = "US";
   string currency_base = SymbolInfoString(_Symbol,SYMBOL_CURRENCY_BASE);
   int allValues = CalendarValueHistory(values,startTime,endTime,NULL,NULL);

   //---- Load CSV events for tester mode
   if (MQLInfoInteger(MQL_TESTER)) {
      if (!LoadEventsFromResource()) {
         Print("Failed to load events from CSV resource.");
         return(INIT_FAILED);
      }
      Print("Tester mode: Loaded ", ArraySize(allEvents), " events from CSV.");
      FilterEventsForTester(); // Added: Pre-filter events for tester mode
   }

   //---- Create UI elements
   createLabel(TIME_LABEL,70,85,"Server Time: "+TimeToString(TimeCurrent(),TIME_DATE|TIME_SECONDS)+
               "   |||   Total News: "+IntegerToString(allValues),clrBlack,14,"Times new roman bold");
   createLabel(IMPACT_LABEL,70,105,"Impact: ",clrBlack,14,"Times new roman bold");
   createLabel(FILTER_LABEL,370,55,"Filters:",clrYellow,16,"Impact");

   //---- Create filter buttons
   string filter_curr_text = enableCurrencyFilter ? ShortToString(0x2714)+"Currency" : ShortToString(0x274C)+"Currency";
   color filter_curr_txt_color = enableCurrencyFilter ? clrLime : clrRed;
   bool filter_curr_state = enableCurrencyFilter;
   createButton(FILTER_CURR_BTN,430,55,110,26,filter_curr_text,filter_curr_txt_color,12,clrBlack);
   ObjectSetInteger(0,FILTER_CURR_BTN,OBJPROP_STATE,filter_curr_state);

   string filter_imp_text = enableImportanceFilter ? ShortToString(0x2714)+"Importance" : ShortToString(0x274C)+"Importance";
   color filter_imp_txt_color = enableImportanceFilter ? clrLime : clrRed;
   bool filter_imp_state = enableImportanceFilter;
   createButton(FILTER_IMP_BTN,430+110,55,120,26,filter_imp_text,filter_imp_txt_color,12,clrBlack);
   ObjectSetInteger(0,FILTER_IMP_BTN,OBJPROP_STATE,filter_imp_state);

   string filter_time_text = enableTimeFilter ? ShortToString(0x2714)+"Time" : ShortToString(0x274C)+"Time";
   color filter_time_txt_color = enableTimeFilter ? clrLime : clrRed;
   bool filter_time_state = enableTimeFilter;
   createButton(FILTER_TIME_BTN,430+110+120,55,70,26,filter_time_text,filter_time_txt_color,12,clrBlack);
   ObjectSetInteger(0,FILTER_TIME_BTN,OBJPROP_STATE,filter_time_state);

   createButton(CANCEL_BTN,430+110+120+79,51,50,30,"X",clrWhite,17,clrRed,clrNONE);

   //---- Create impact buttons
   int impact_size = 100;
   for (int i = 0; i < ArraySize(impact_labels); i++) {
      color impact_color = clrBlack, label_color = clrBlack;
      if (impact_labels[i] == "None") label_color = clrWhite;
      else if (impact_labels[i] == "Low") impact_color = clrYellow;
      else if (impact_labels[i] == "Medium") impact_color = clrOrange;
      else if (impact_labels[i] == "High") impact_color = clrRed;
      createButton(IMPACT_LABEL+string(i),140+impact_size*i,105,impact_size,25,
                   impact_labels[i],label_color,12,impact_color,clrBlack);
   }

   //---- Create currency buttons
   int curr_size = 51, button_height = 22, spacing_x = 0, spacing_y = 3, max_columns = 4;
   for (int i = 0; i < ArraySize(curr_filter); i++) {
      int row = i / max_columns;
      int col = i % max_columns;
      int x_pos = 575 + col * (curr_size + spacing_x);
      int y_pos = 83 + row * (button_height + spacing_y);
      createButton(CURRENCY_BTNS+IntegerToString(i),x_pos,y_pos,curr_size,button_height,curr_filter[i],clrBlack);
   }

   //---- Initialize filters
   if (enableCurrencyFilter) {
      ArrayFree(curr_filter_selected);
      ArrayCopy(curr_filter_selected, curr_filter);
      Print("CURRENCY FILTER ENABLED");
      ArrayPrint(curr_filter_selected);
      for (int i = 0; i < ArraySize(curr_filter_selected); i++) {
         ObjectSetInteger(0, CURRENCY_BTNS+IntegerToString(i), OBJPROP_STATE, true);
      }
   }

   if (enableImportanceFilter) {
      ArrayFree(imp_filter_selected);
      ArrayCopy(imp_filter_selected, allowed_importance_levels);
      ArrayFree(impact_filter_selected);
      ArrayCopy(impact_filter_selected, impact_labels);
      Print("IMPORTANCE FILTER ENABLED");
      ArrayPrint(imp_filter_selected);
      ArrayPrint(impact_filter_selected);
      for (int i = 0; i < ArraySize(imp_filter_selected); i++) {
         string btn_name = IMPACT_LABEL+string(i);
         ObjectSetInteger(0, btn_name, OBJPROP_STATE, true);
         ObjectSetInteger(0, btn_name, OBJPROP_BORDER_COLOR, clrNONE);
      }
   }

   //---- Update dashboard
   update_dashboard_values(curr_filter_selected, imp_filter_selected);
   ChartRedraw(0);
   return(INIT_SUCCEEDED);
}

//+------------------------------------------------------------------+
//| Expert deinitialization function                                  |
//+------------------------------------------------------------------+
void OnDeinit(const int reason) {
   destroy_Dashboard();
   deleteTradeObjects();
}

//+------------------------------------------------------------------+
//| Expert tick function                                              |
//+------------------------------------------------------------------+
void OnTick() {
   UpdateFilterInfo();
   CheckForNewsTrade();
   if (isDashboardUpdate) {
      if (MQLInfoInteger(MQL_TESTER)) {
         datetime currentTime = TimeTradeServer();
         datetime timeRange = PeriodSeconds(range_time);
         datetime timeAfter = currentTime + timeRange;
         if (filters_changed || last_dashboard_update < timeAfter) { // Modified: Update on filter change or time range shift
            update_dashboard_values(curr_filter_selected, imp_filter_selected);
            ArrayFree(last_dashboard_eventNames);
            ArrayCopy(last_dashboard_eventNames, current_eventNames_data);
            last_dashboard_update = currentTime;
         }
      } else {
         update_dashboard_values(curr_filter_selected, imp_filter_selected);
      }
   }
}

//+------------------------------------------------------------------+
//| Chart event handler                                               |
//+------------------------------------------------------------------+
void OnChartEvent(const int id, const long& lparam, const double& dparam, const string& sparam) {
   if (id == CHARTEVENT_OBJECT_CLICK) {
      UpdateFilterInfo();
      CheckForNewsTrade();
      if (sparam == CANCEL_BTN) {
         isDashboardUpdate = false;
         destroy_Dashboard();
      }
      if (sparam == FILTER_CURR_BTN) {
         bool btn_state = ObjectGetInteger(0,sparam,OBJPROP_STATE);
         enableCurrencyFilter = btn_state;
         if (!MQLInfoInteger(MQL_TESTER) || debugLogging) Print(sparam+" STATE = "+(string)btn_state+", FLAG = "+(string)enableCurrencyFilter); // Modified: Conditional logging
         string filter_curr_text = enableCurrencyFilter ? ShortToString(0x2714)+"Currency" : ShortToString(0x274C)+"Currency";
         color filter_curr_txt_color = enableCurrencyFilter ? clrLime : clrRed;
         ObjectSetString(0,FILTER_CURR_BTN,OBJPROP_TEXT,filter_curr_text);
         ObjectSetInteger(0,FILTER_CURR_BTN,OBJPROP_COLOR,filter_curr_txt_color);
         if (MQLInfoInteger(MQL_TESTER)) filters_changed = true; // Added: Mark filters changed
         update_dashboard_values(curr_filter_selected,imp_filter_selected);
         if (!MQLInfoInteger(MQL_TESTER) || debugLogging) Print("Success. Changes updated! State: "+(string)enableCurrencyFilter); // Modified: Conditional logging
         ChartRedraw(0);
      }
      if (sparam == FILTER_IMP_BTN) {
         bool btn_state = ObjectGetInteger(0,sparam,OBJPROP_STATE);
         enableImportanceFilter = btn_state;
         if (!MQLInfoInteger(MQL_TESTER) || debugLogging) Print(sparam+" STATE = "+(string)btn_state+", FLAG = "+(string)enableImportanceFilter); // Modified: Conditional logging
         string filter_imp_text = enableImportanceFilter ? ShortToString(0x2714)+"Importance" : ShortToString(0x274C)+"Importance";
         color filter_imp_txt_color = enableImportanceFilter ? clrLime : clrRed;
         ObjectSetString(0,FILTER_IMP_BTN,OBJPROP_TEXT,filter_imp_text);
         ObjectSetInteger(0,FILTER_IMP_BTN,OBJPROP_COLOR,filter_imp_txt_color);
         if (MQLInfoInteger(MQL_TESTER)) filters_changed = true; // Added: Mark filters changed
         update_dashboard_values(curr_filter_selected,imp_filter_selected);
         if (!MQLInfoInteger(MQL_TESTER) || debugLogging) Print("Success. Changes updated! State: "+(string)enableImportanceFilter); // Modified: Conditional logging
         ChartRedraw(0);
      }
      if (sparam == FILTER_TIME_BTN) {
         bool btn_state = ObjectGetInteger(0,sparam,OBJPROP_STATE);
         enableTimeFilter = btn_state;
         if (!MQLInfoInteger(MQL_TESTER) || debugLogging) Print(sparam+" STATE = "+(string)btn_state+", FLAG = "+(string)enableTimeFilter); // Modified: Conditional logging
         string filter_time_text = enableTimeFilter ? ShortToString(0x2714)+"Time" : ShortToString(0x274C)+"Time";
         color filter_time_txt_color = enableTimeFilter ? clrLime : clrRed;
         ObjectSetString(0,FILTER_TIME_BTN,OBJPROP_TEXT,filter_time_text);
         ObjectSetInteger(0,FILTER_TIME_BTN,OBJPROP_COLOR,filter_time_txt_color);
         if (MQLInfoInteger(MQL_TESTER)) filters_changed = true; // Added: Mark filters changed
         update_dashboard_values(curr_filter_selected,imp_filter_selected);
         if (!MQLInfoInteger(MQL_TESTER) || debugLogging) Print("Success. Changes updated! State: "+(string)enableTimeFilter); // Modified: Conditional logging
         ChartRedraw(0);
      }
      if (StringFind(sparam,CURRENCY_BTNS) >= 0) {
         string selected_curr = ObjectGetString(0,sparam,OBJPROP_TEXT);
         if (!MQLInfoInteger(MQL_TESTER) || debugLogging) Print("BTN NAME = ",sparam,", CURRENCY = ",selected_curr); // Modified: Conditional logging
         bool btn_state = ObjectGetInteger(0,sparam,OBJPROP_STATE);
         if (btn_state == false) {
            if (!MQLInfoInteger(MQL_TESTER) || debugLogging) Print("BUTTON IS IN UN-SELECTED MODE."); // Modified: Conditional logging
            for (int i = 0; i < ArraySize(curr_filter_selected); i++) {
               if (curr_filter_selected[i] == selected_curr) {
                  for (int j = i; j < ArraySize(curr_filter_selected) - 1; j++) {
                     curr_filter_selected[j] = curr_filter_selected[j + 1];
                  }
                  ArrayResize(curr_filter_selected, ArraySize(curr_filter_selected) - 1);
                  if (!MQLInfoInteger(MQL_TESTER) || debugLogging) Print("Removed from selected filters: ", selected_curr); // Modified: Conditional logging
                  break;
               }
            }
         } else {
            if (!MQLInfoInteger(MQL_TESTER) || debugLogging) Print("BUTTON IS IN SELECTED MODE. TAKE ACTION"); // Modified: Conditional logging
            bool already_selected = false;
            for (int j = 0; j < ArraySize(curr_filter_selected); j++) {
               if (curr_filter_selected[j] == selected_curr) {
                  already_selected = true;
                  break;
               }
            }
            if (!already_selected) {
               ArrayResize(curr_filter_selected, ArraySize(curr_filter_selected) + 1);
               curr_filter_selected[ArraySize(curr_filter_selected) - 1] = selected_curr;
               if (!MQLInfoInteger(MQL_TESTER) || debugLogging) Print("Added to selected filters: ", selected_curr); // Modified: Conditional logging
            } else {
               if (!MQLInfoInteger(MQL_TESTER) || debugLogging) Print("Currency already selected: ", selected_curr); // Modified: Conditional logging
            }
         }
         if (!MQLInfoInteger(MQL_TESTER) || debugLogging) Print("SELECTED ARRAY SIZE = ",ArraySize(curr_filter_selected)); // Modified: Conditional logging
         if (!MQLInfoInteger(MQL_TESTER) || debugLogging) ArrayPrint(curr_filter_selected); // Modified: Conditional logging
         if (MQLInfoInteger(MQL_TESTER)) filters_changed = true; // Added: Mark filters changed
         update_dashboard_values(curr_filter_selected,imp_filter_selected);
         if (!MQLInfoInteger(MQL_TESTER) || debugLogging) Print("SUCCESS. DASHBOARD UPDATED"); // Modified: Conditional logging
         ChartRedraw(0);
      }
      if (StringFind(sparam, IMPACT_LABEL) >= 0) {
         string selected_imp = ObjectGetString(0, sparam, OBJPROP_TEXT);
         ENUM_CALENDAR_EVENT_IMPORTANCE selected_importance_lvl = get_importance_level(impact_labels,allowed_importance_levels,selected_imp);
         if (!MQLInfoInteger(MQL_TESTER) || debugLogging) Print("BTN NAME = ", sparam, ", IMPORTANCE LEVEL = ", selected_imp,"(",selected_importance_lvl,")"); // Modified: Conditional logging
         bool btn_state = ObjectGetInteger(0, sparam, OBJPROP_STATE);
         color color_border = btn_state ? clrNONE : clrBlack;
         if (btn_state == false) {
            if (!MQLInfoInteger(MQL_TESTER) || debugLogging) Print("BUTTON IS IN UN-SELECTED MODE."); // Modified: Conditional logging
            for (int i = 0; i < ArraySize(imp_filter_selected); i++) {
               if (impact_filter_selected[i] == selected_imp) {
                  for (int j = i; j < ArraySize(imp_filter_selected) - 1; j++) {
                     imp_filter_selected[j] = imp_filter_selected[j + 1];
                     impact_filter_selected[j] = impact_filter_selected[j + 1];
                  }
                  ArrayResize(imp_filter_selected, ArraySize(imp_filter_selected) - 1);
                  ArrayResize(impact_filter_selected, ArraySize(impact_filter_selected) - 1);
                  if (!MQLInfoInteger(MQL_TESTER) || debugLogging) Print("Removed from selected importance filters: ", selected_imp,"(",selected_importance_lvl,")"); // Modified: Conditional logging
                  break;
               }
            }
         } else {
            if (!MQLInfoInteger(MQL_TESTER) || debugLogging) Print("BUTTON IS IN SELECTED MODE. TAKE ACTION"); // Modified: Conditional logging
            bool already_selected = false;
            for (int j = 0; j < ArraySize(imp_filter_selected); j++) {
               if (impact_filter_selected[j] == selected_imp) {
                  already_selected = true;
                  break;
               }
            }
            if (!already_selected) {
               ArrayResize(imp_filter_selected, ArraySize(imp_filter_selected) + 1);
               imp_filter_selected[ArraySize(imp_filter_selected) - 1] = selected_importance_lvl;
               ArrayResize(impact_filter_selected, ArraySize(impact_filter_selected) + 1);
               impact_filter_selected[ArraySize(impact_filter_selected) - 1] = selected_imp;
               if (!MQLInfoInteger(MQL_TESTER) || debugLogging) Print("Added to selected importance filters: ", selected_imp,"(",selected_importance_lvl,")"); // Modified: Conditional logging
            } else {
               if (!MQLInfoInteger(MQL_TESTER) || debugLogging) Print("Importance level already selected: ", selected_imp,"(",selected_importance_lvl,")"); // Modified: Conditional logging
            }
         }
         if (!MQLInfoInteger(MQL_TESTER) || debugLogging) Print("SELECTED ARRAY SIZE = ", ArraySize(imp_filter_selected)," >< ",ArraySize(impact_filter_selected)); // Modified: Conditional logging
         if (!MQLInfoInteger(MQL_TESTER) || debugLogging) ArrayPrint(imp_filter_selected); // Modified: Conditional logging
         if (!MQLInfoInteger(MQL_TESTER) || debugLogging) ArrayPrint(impact_filter_selected); // Modified: Conditional logging
         if (MQLInfoInteger(MQL_TESTER)) filters_changed = true; // Added: Mark filters changed
         update_dashboard_values(curr_filter_selected,imp_filter_selected);
         ObjectSetInteger(0,sparam,OBJPROP_BORDER_COLOR,color_border);
         if (!MQLInfoInteger(MQL_TESTER) || debugLogging) Print("SUCCESS. DASHBOARD UPDATED"); // Modified: Conditional logging
         ChartRedraw(0);
      }
   }
}

//+------------------------------------------------------------------+
//| Load events from CSV resource                                    |
//+------------------------------------------------------------------+
bool LoadEventsFromResource() {
   string fileData = EconomicCalendarData;
   Print("Raw resource content (size: ", StringLen(fileData), " bytes):\n", fileData);
   string lines[];
   int lineCount = StringSplit(fileData, '\n', lines);
   if (lineCount <= 1) {
      Print("Error: No data lines found in resource! Raw data: ", fileData);
      return false;
   }
   ArrayResize(allEvents, 0);
   int eventIndex = 0;
   for (int i = 1; i < lineCount; i++) {
      if (StringLen(lines[i]) == 0) {
         if (debugLogging) Print("Skipping empty line ", i); // Modified: Conditional logging
         continue;
      }
      string fields[];
      int fieldCount = StringSplit(lines[i], ',', fields);
      if (debugLogging) Print("Line ", i, ": ", lines[i], " (field count: ", fieldCount, ")"); // Modified: Conditional logging
      if (fieldCount < 8) {
         Print("Malformed line ", i, ": ", lines[i], " (field count: ", fieldCount, ")");
         continue;
      }
      string dateStr = fields[0];
      string timeStr = fields[1];
      string currency = fields[2];
      string event = fields[3];
      for (int j = 4; j < fieldCount - 4; j++) {
         event += "," + fields[j];
      }
      string importance = fields[fieldCount - 4];
      string actualStr = fields[fieldCount - 3];
      string forecastStr = fields[fieldCount - 2];
      string previousStr = fields[fieldCount - 1];
      datetime eventDateTime = StringToTime(dateStr + " " + timeStr);
      if (eventDateTime == 0) {
         Print("Error: Invalid datetime conversion for line ", i, ": ", dateStr, " ", timeStr);
         continue;
      }
      ArrayResize(allEvents, eventIndex + 1);
      allEvents[eventIndex].eventDate = dateStr;
      allEvents[eventIndex].eventTime = timeStr;
      allEvents[eventIndex].currency = currency;
      allEvents[eventIndex].event = event;
      allEvents[eventIndex].importance = importance;
      allEvents[eventIndex].actual = StringToDouble(actualStr);
      allEvents[eventIndex].forecast = StringToDouble(forecastStr);
      allEvents[eventIndex].previous = StringToDouble(previousStr);
      allEvents[eventIndex].eventDateTime = eventDateTime; // Added: Store precomputed datetime
      if (debugLogging) Print("Loaded event ", eventIndex, ": ", dateStr, " ", timeStr, ", ", currency, ", ", event); // Modified: Conditional logging
      eventIndex++;
   }
   Print("Loaded ", eventIndex, " events from resource into array.");
   return eventIndex > 0;
}

//+------------------------------------------------------------------+
//| Update dashboard values                                           |
//+------------------------------------------------------------------+
void update_dashboard_values(string &curr_filter_array[], ENUM_CALENDAR_EVENT_IMPORTANCE &imp_filter_array[]) {
   totalEvents_Considered = 0;
   totalEvents_Filtered = 0;
   totalEvents_Displayable = 0;
   ArrayFree(current_eventNames_data);

   datetime timeRange = PeriodSeconds(range_time);
   datetime timeBefore = TimeTradeServer() - timeRange;
   datetime timeAfter = TimeTradeServer() + timeRange;

   int startY = 162;

   if (MQLInfoInteger(MQL_TESTER)) {
      if (filters_changed) FilterEventsForTester(); // Added: Re-filter events if filters changed
      //---- Tester mode: Process filtered events
      for (int i = 0; i < ArraySize(filteredEvents); i++) {
         totalEvents_Considered++;
         datetime eventDateTime = filteredEvents[i].eventDateTime;
         if (eventDateTime < StartDate || eventDateTime > EndDate) {
            if (debugLogging) Print("Event ", filteredEvents[i].event, " skipped due to date range."); // Modified: Conditional logging
            continue;
         }

         bool timeMatch = !enableTimeFilter;
         if (enableTimeFilter) {
            if (eventDateTime <= TimeTradeServer() && eventDateTime >= timeBefore) timeMatch = true;
            else if (eventDateTime >= TimeTradeServer() && eventDateTime <= timeAfter) timeMatch = true;
         }
         if (!timeMatch) {
            if (debugLogging) Print("Event ", filteredEvents[i].event, " skipped due to time filter."); // Modified: Conditional logging
            continue;
         }

         bool currencyMatch = !enableCurrencyFilter;
         if (enableCurrencyFilter) {
            for (int j = 0; j < ArraySize(curr_filter_array); j++) {
               if (filteredEvents[i].currency == curr_filter_array[j]) {
                  currencyMatch = true;
                  break;
               }
            }
         }
         if (!currencyMatch) {
            if (debugLogging) Print("Event ", filteredEvents[i].event, " skipped due to currency filter."); // Modified: Conditional logging
            continue;
         }

         bool importanceMatch = !enableImportanceFilter;
         if (enableImportanceFilter) {
            string imp_str = filteredEvents[i].importance;
            ENUM_CALENDAR_EVENT_IMPORTANCE event_imp = (imp_str == "None") ? CALENDAR_IMPORTANCE_NONE :
                                                      (imp_str == "Low") ? CALENDAR_IMPORTANCE_LOW :
                                                      (imp_str == "Medium") ? CALENDAR_IMPORTANCE_MODERATE :
                                                      CALENDAR_IMPORTANCE_HIGH;
            for (int k = 0; k < ArraySize(imp_filter_array); k++) {
               if (event_imp == imp_filter_array[k]) {
                  importanceMatch = true;
                  break;
               }
            }
         }
         if (!importanceMatch) {
            if (debugLogging) Print("Event ", filteredEvents[i].event, " skipped due to importance filter."); // Modified: Conditional logging
            continue;
         }

         totalEvents_Filtered++;
         if (totalEvents_Displayable >= 11) continue;
         totalEvents_Displayable++;

         color holder_color = (totalEvents_Displayable % 2 == 0) ? C'213,227,207' : clrWhite;
         createRecLabel(DATA_HOLDERS+string(totalEvents_Displayable),62,startY-1,716,26+1,holder_color,1,clrNONE);

         int startX = 65;
         string news_data[ArraySize(array_calendar)];
         news_data[0] = filteredEvents[i].eventDate;
         news_data[1] = filteredEvents[i].eventTime;
         news_data[2] = filteredEvents[i].currency;
         color importance_color = clrBlack;
         if (filteredEvents[i].importance == "Low") importance_color = clrYellow;
         else if (filteredEvents[i].importance == "Medium") importance_color = clrOrange;
         else if (filteredEvents[i].importance == "High") importance_color = clrRed;
         news_data[3] = ShortToString(0x25CF);
         news_data[4] = filteredEvents[i].event;
         news_data[5] = DoubleToString(filteredEvents[i].actual, 3);
         news_data[6] = DoubleToString(filteredEvents[i].forecast, 3);
         news_data[7] = DoubleToString(filteredEvents[i].previous, 3);

         for (int k = 0; k < ArraySize(array_calendar); k++) {
            if (k == 3) {
               createLabel(ARRAY_NEWS+IntegerToString(i)+" "+array_calendar[k],startX,startY-(22-12),news_data[k],importance_color,22,"Calibri");
            } else {
               createLabel(ARRAY_NEWS+IntegerToString(i)+" "+array_calendar[k],startX,startY,news_data[k],clrBlack,12,"Calibri");
            }
            startX += buttons[k]+3;
         }

         ArrayResize(current_eventNames_data, ArraySize(current_eventNames_data)+1);
         current_eventNames_data[ArraySize(current_eventNames_data)-1] = filteredEvents[i].event;
         startY += 25;
      }
   } else {
      //---- Live mode: Unchanged from PART 6
      MqlCalendarValue values[];
      datetime startTime = TimeTradeServer() - PeriodSeconds(start_time);
      datetime endTime = TimeTradeServer() + PeriodSeconds(end_time);
      int allValues = CalendarValueHistory(values,startTime,endTime,NULL,NULL);

      for (int i = 0; i < allValues; i++) {
         MqlCalendarEvent event;
         CalendarEventById(values[i].event_id, event);
         MqlCalendarCountry country;
         CalendarCountryById(event.country_id, country);
         MqlCalendarValue value;
         CalendarValueById(values[i].id, value);
         totalEvents_Considered++;

         bool currencyMatch = false;
         if (enableCurrencyFilter) {
            for (int j = 0; j < ArraySize(curr_filter_array); j++) {
               if (country.currency == curr_filter_array[j]) {
                  currencyMatch = true;
                  break;
               }
            }
            if (!currencyMatch) continue;
         }

         bool importanceMatch = false;
         if (enableImportanceFilter) {
            for (int k = 0; k < ArraySize(imp_filter_array); k++) {
               if (event.importance == imp_filter_array[k]) {
                  importanceMatch = true;
                  break;
               }
            }
            if (!importanceMatch) continue;
         }

         bool timeMatch = false;
         if (enableTimeFilter) {
            datetime eventTime = values[i].time;
            if (eventTime <= TimeTradeServer() && eventTime >= timeBefore) timeMatch = true;
            else if (eventTime >= TimeTradeServer() && eventTime <= timeAfter) timeMatch = true;
            if (!timeMatch) continue;
         }

         totalEvents_Filtered++;
         if (totalEvents_Displayable >= 11) continue;
         totalEvents_Displayable++;

         color holder_color = (totalEvents_Displayable % 2 == 0) ? C'213,227,207' : clrWhite;
         createRecLabel(DATA_HOLDERS+string(totalEvents_Displayable),62,startY-1,716,26+1,holder_color,1,clrNONE);

         int startX = 65;
         string news_data[ArraySize(array_calendar)];
         news_data[0] = TimeToString(values[i].time,TIME_DATE);
         news_data[1] = TimeToString(values[i].time,TIME_MINUTES);
         news_data[2] = country.currency;
         color importance_color = clrBlack;
         if (event.importance == CALENDAR_IMPORTANCE_LOW) importance_color = clrYellow;
         else if (event.importance == CALENDAR_IMPORTANCE_MODERATE) importance_color = clrOrange;
         else if (event.importance == CALENDAR_IMPORTANCE_HIGH) importance_color = clrRed;
         news_data[3] = ShortToString(0x25CF);
         news_data[4] = event.name;
         news_data[5] = DoubleToString(value.GetActualValue(),3);
         news_data[6] = DoubleToString(value.GetForecastValue(),3);
         news_data[7] = DoubleToString(value.GetPreviousValue(),3);

         for (int k = 0; k < ArraySize(array_calendar); k++) {
            if (k == 3) {
               createLabel(ARRAY_NEWS+IntegerToString(i)+" "+array_calendar[k],startX,startY-(22-12),news_data[k],importance_color,22,"Calibri");
            } else {
               createLabel(ARRAY_NEWS+IntegerToString(i)+" "+array_calendar[k],startX,startY,news_data[k],clrBlack,12,"Calibri");
            }
            startX += buttons[k]+3;
         }

         ArrayResize(current_eventNames_data, ArraySize(current_eventNames_data)+1);
         current_eventNames_data[ArraySize(current_eventNames_data)-1] = event.name;
         startY += 25;
      }
   }

   string timeText = updateServerTime ? "Server Time: "+TimeToString(TimeCurrent(),TIME_DATE|TIME_SECONDS) : "Server Time: Static";
   updateLabel(TIME_LABEL,timeText+"   |||   Total News: "+IntegerToString(totalEvents_Displayable)+"/"+
               IntegerToString(totalEvents_Filtered)+"/"+IntegerToString(totalEvents_Considered));

   if (isChangeInStringArrays(previous_eventNames_data, current_eventNames_data)) {
      Print("CHANGES IN EVENT NAMES DETECTED. UPDATE THE DASHBOARD VALUES");
      ObjectsDeleteAll(0,DATA_HOLDERS);
      ObjectsDeleteAll(0,ARRAY_NEWS);
      ArrayFree(current_eventNames_data);
      totalEvents_Displayable = 0;
      totalEvents_Filtered = 0;
      totalEvents_Considered = 0;
      startY = 162;

      if (MQLInfoInteger(MQL_TESTER)) {
         for (int i = 0; i < ArraySize(filteredEvents); i++) {
            totalEvents_Considered++;
            datetime eventDateTime = filteredEvents[i].eventDateTime;
            if (eventDateTime < StartDate || eventDateTime > EndDate) {
               if (debugLogging) Print("Event ", filteredEvents[i].event, " skipped due to date range."); // Modified: Conditional logging
               continue;
            }

            bool timeMatch = !enableTimeFilter;
            if (enableTimeFilter) {
               if (eventDateTime <= TimeTradeServer() && eventDateTime >= timeBefore) timeMatch = true;
               else if (eventDateTime >= TimeTradeServer() && eventDateTime <= timeAfter) timeMatch = true;
            }
            if (!timeMatch) {
               if (debugLogging) Print("Event ", filteredEvents[i].event, " skipped due to time filter."); // Modified: Conditional logging
               continue;
            }

            bool currencyMatch = !enableCurrencyFilter;
            if (enableCurrencyFilter) {
               for (int j = 0; j < ArraySize(curr_filter_array); j++) {
                  if (filteredEvents[i].currency == curr_filter_array[j]) {
                     currencyMatch = true;
                     break;
                  }
               }
            }
            if (!currencyMatch) {
               if (debugLogging) Print("Event ", filteredEvents[i].event, " skipped due to currency filter."); // Modified: Conditional logging
               continue;
            }

            bool importanceMatch = !enableImportanceFilter;
            if (enableImportanceFilter) {
               string imp_str = filteredEvents[i].importance;
               ENUM_CALENDAR_EVENT_IMPORTANCE event_imp = (imp_str == "None") ? CALENDAR_IMPORTANCE_NONE :
                                                         (imp_str == "Low") ? CALENDAR_IMPORTANCE_LOW :
                                                         (imp_str == "Medium") ? CALENDAR_IMPORTANCE_MODERATE :
                                                         CALENDAR_IMPORTANCE_HIGH;
               for (int k = 0; k < ArraySize(imp_filter_array); k++) {
                  if (event_imp == imp_filter_array[k]) {
                     importanceMatch = true;
                     break;
                  }
               }
            }
            if (!importanceMatch) {
               if (debugLogging) Print("Event ", filteredEvents[i].event, " skipped due to importance filter."); // Modified: Conditional logging
               continue;
            }

            totalEvents_Filtered++;
            if (totalEvents_Displayable >= 11) continue;
            totalEvents_Displayable++;

            color holder_color = (totalEvents_Displayable % 2 == 0) ? C'213,227,207' : clrWhite;
            createRecLabel(DATA_HOLDERS+string(totalEvents_Displayable),62,startY-1,716,26+1,holder_color,1,clrNONE);

            int startX = 65;
            string news_data[ArraySize(array_calendar)];
            news_data[0] = filteredEvents[i].eventDate;
            news_data[1] = filteredEvents[i].eventTime;
            news_data[2] = filteredEvents[i].currency;
            color importance_color = clrBlack;
            if (filteredEvents[i].importance == "Low") importance_color = clrYellow;
            else if (filteredEvents[i].importance == "Medium") importance_color = clrOrange;
            else if (filteredEvents[i].importance == "High") importance_color = clrRed;
            news_data[3] = ShortToString(0x25CF);
            news_data[4] = filteredEvents[i].event;
            news_data[5] = DoubleToString(filteredEvents[i].actual, 3);
            news_data[6] = DoubleToString(filteredEvents[i].forecast, 3);
            news_data[7] = DoubleToString(filteredEvents[i].previous, 3);

            for (int k = 0; k < ArraySize(array_calendar); k++) {
               if (k == 3) {
                  createLabel(ARRAY_NEWS+IntegerToString(i)+" "+array_calendar[k],startX,startY-(22-12),news_data[k],importance_color,22,"Calibri");
               } else {
                  createLabel(ARRAY_NEWS+IntegerToString(i)+" "+array_calendar[k],startX,startY,news_data[k],clrBlack,12,"Calibri");
               }
               startX += buttons[k]+3;
            }

            ArrayResize(current_eventNames_data, ArraySize(current_eventNames_data)+1);
            current_eventNames_data[ArraySize(current_eventNames_data)-1] = filteredEvents[i].event;
            startY += 25;
         }
      } else {
         MqlCalendarValue values[];
         datetime startTime = TimeTradeServer() - PeriodSeconds(start_time);
         datetime endTime = TimeTradeServer() + PeriodSeconds(end_time);
         int allValues = CalendarValueHistory(values,startTime,endTime,NULL,NULL);

         for (int i = 0; i < allValues; i++) {
            MqlCalendarEvent event;
            CalendarEventById(values[i].event_id, event);
            MqlCalendarCountry country;
            CalendarCountryById(event.country_id, country);
            MqlCalendarValue value;
            CalendarValueById(values[i].id, value);
            totalEvents_Considered++;

            bool currencyMatch = false;
            if (enableCurrencyFilter) {
               for (int j = 0; j < ArraySize(curr_filter_array); j++) {
                  if (country.currency == curr_filter_array[j]) {
                     currencyMatch = true;
                     break;
                  }
               }
               if (!currencyMatch) continue;
            }

            bool importanceMatch = false;
            if (enableImportanceFilter) {
               for (int k = 0; k < ArraySize(imp_filter_array); k++) {
                  if (event.importance == imp_filter_array[k]) {
                     importanceMatch = true;
                     break;
                  }
               }
               if (!importanceMatch) continue;
            }

            bool timeMatch = false;
            if (enableTimeFilter) {
               datetime eventTime = values[i].time;
               if (eventTime <= TimeTradeServer() && eventTime >= timeBefore) timeMatch = true;
               else if (eventTime >= TimeTradeServer() && eventTime <= timeAfter) timeMatch = true;
               if (!timeMatch) continue;
            }

            totalEvents_Filtered++;
            if (totalEvents_Displayable >= 11) continue;
            totalEvents_Displayable++;

            color holder_color = (totalEvents_Displayable % 2 == 0) ? C'213,227,207' : clrWhite;
            createRecLabel(DATA_HOLDERS+string(totalEvents_Displayable),62,startY-1,716,26+1,holder_color,1,clrNONE);

            int startX = 65;
            string news_data[ArraySize(array_calendar)];
            news_data[0] = TimeToString(values[i].time,TIME_DATE);
            news_data[1] = TimeToString(values[i].time,TIME_MINUTES);
            news_data[2] = country.currency;
            color importance_color = clrBlack;
            if (event.importance == CALENDAR_IMPORTANCE_LOW) importance_color = clrYellow;
            else if (event.importance == CALENDAR_IMPORTANCE_MODERATE) importance_color = clrOrange;
            else if (event.importance == CALENDAR_IMPORTANCE_HIGH) importance_color = clrRed;
            news_data[3] = ShortToString(0x25CF);
            news_data[4] = event.name;
            news_data[5] = DoubleToString(value.GetActualValue(),3);
            news_data[6] = DoubleToString(value.GetForecastValue(),3);
            news_data[7] = DoubleToString(value.GetPreviousValue(),3);

            for (int k = 0; k < ArraySize(array_calendar); k++) {
               if (k == 3) {
                  createLabel(ARRAY_NEWS+IntegerToString(i)+" "+array_calendar[k],startX,startY-(22-12),news_data[k],importance_color,22,"Calibri");
               } else {
                  createLabel(ARRAY_NEWS+IntegerToString(i)+" "+array_calendar[k],startX,startY,news_data[k],clrBlack,12,"Calibri");
               }
               startX += buttons[k]+3;
            }

            ArrayResize(current_eventNames_data, ArraySize(current_eventNames_data)+1);
            current_eventNames_data[ArraySize(current_eventNames_data)-1] = event.name;
            startY += 25;
         }
      }
      if (!MQLInfoInteger(MQL_TESTER) || debugLogging) Print("CURRENT EVENT NAMES DATA SIZE = ",ArraySize(current_eventNames_data)); // Modified: Conditional logging
      if (!MQLInfoInteger(MQL_TESTER) || debugLogging) ArrayPrint(current_eventNames_data); // Modified: Conditional logging
      if (!MQLInfoInteger(MQL_TESTER) || debugLogging) Print("PREVIOUS EVENT NAMES DATA SIZE (Before) = ",ArraySize(previous_eventNames_data)); // Modified: Conditional logging
      if (!MQLInfoInteger(MQL_TESTER) || debugLogging) ArrayPrint(previous_eventNames_data); // Modified: Conditional logging
      ArrayFree(previous_eventNames_data);
      ArrayCopy(previous_eventNames_data, current_eventNames_data);
      if (!MQLInfoInteger(MQL_TESTER) || debugLogging) Print("PREVIOUS EVENT NAMES DATA SIZE (After) = ",ArraySize(previous_eventNames_data)); // Modified: Conditional logging
      if (!MQLInfoInteger(MQL_TESTER) || debugLogging) ArrayPrint(previous_eventNames_data); // Modified: Conditional logging
   }
}

//+------------------------------------------------------------------+
//| Check for news trade (adapted for tester mode trading)            |
//+------------------------------------------------------------------+
void CheckForNewsTrade() {
   if (!MQLInfoInteger(MQL_TESTER) || debugLogging) Print("CheckForNewsTrade called at: ", TimeToString(TimeTradeServer(), TIME_SECONDS)); // Modified: Conditional logging
   if (tradeMode == NO_TRADE || tradeMode == PAUSE_TRADING) {
      if (ObjectFind(0, "NewsCountdown") >= 0) {
         ObjectDelete(0, "NewsCountdown");
         Print("Trading disabled. Countdown removed.");
      }
      return;
   }

   datetime currentTime = TimeTradeServer();
   int offsetSeconds = tradeOffsetHours * 3600 + tradeOffsetMinutes * 60 + tradeOffsetSeconds;

   if (tradeExecuted) {
      if (currentTime < tradedNewsTime) {
         int remainingSeconds = (int)(tradedNewsTime - currentTime);
         int hrs = remainingSeconds / 3600;
         int mins = (remainingSeconds % 3600) / 60;
         int secs = remainingSeconds % 60;
         string countdownText = "News in: " + IntegerToString(hrs) + "h " +
                               IntegerToString(mins) + "m " + IntegerToString(secs) + "s";
         if (ObjectFind(0, "NewsCountdown") < 0) {
            createButton1("NewsCountdown", 50, 17, 300, 30, countdownText, clrWhite, 12, clrBlue, clrBlack);
            Print("Post-trade countdown created: ", countdownText);
         } else {
            updateLabel1("NewsCountdown", countdownText);
            Print("Post-trade countdown updated: ", countdownText);
         }
      } else {
         int elapsed = (int)(currentTime - tradedNewsTime);
         if (elapsed < 15) {
            int remainingDelay = 15 - elapsed;
            string countdownText = "News Released, resetting in: " + IntegerToString(remainingDelay) + "s";
            if (ObjectFind(0, "NewsCountdown") < 0) {
               createButton1("NewsCountdown", 50, 17, 300, 30, countdownText, clrWhite, 12, clrRed, clrBlack);
               ObjectSetInteger(0,"NewsCountdown",OBJPROP_BGCOLOR,clrRed);
               Print("Post-trade reset countdown created: ", countdownText);
            } else {
               updateLabel1("NewsCountdown", countdownText);
               ObjectSetInteger(0,"NewsCountdown",OBJPROP_BGCOLOR,clrRed);
               Print("Post-trade reset countdown updated: ", countdownText);
            }
         } else {
            Print("News Released. Resetting trade status after 15 seconds.");
            if (ObjectFind(0, "NewsCountdown") >= 0) ObjectDelete(0, "NewsCountdown");
            tradeExecuted = false;
         }
      }
      return;
   }

   datetime lowerBound = currentTime - PeriodSeconds(start_time);
   datetime upperBound = currentTime + PeriodSeconds(end_time);
   if (debugLogging) Print("Event time range: ", TimeToString(lowerBound, TIME_SECONDS), " to ", TimeToString(upperBound, TIME_SECONDS)); // Modified: Conditional logging

   datetime candidateEventTime = 0;
   string candidateEventName = "";
   string candidateTradeSide = "";
   int candidateEventID = -1;

   if (MQLInfoInteger(MQL_TESTER)) {
      //---- Tester mode: Process filtered events
      int totalValues = ArraySize(filteredEvents);
      if (debugLogging) Print("Total events found: ", totalValues); // Modified: Conditional logging
      if (totalValues <= 0) {
         if (ObjectFind(0, "NewsCountdown") >= 0) ObjectDelete(0, "NewsCountdown");
         return;
      }

      for (int i = 0; i < totalValues; i++) {
         datetime eventTime = filteredEvents[i].eventDateTime;
         if (eventTime < lowerBound || eventTime > upperBound || eventTime < StartDate || eventTime > EndDate) {
            if (debugLogging) Print("Event ", filteredEvents[i].event, " skipped due to date range."); // Modified: Conditional logging
            continue;
         }

         bool currencyMatch = !enableCurrencyFilter;
         if (enableCurrencyFilter) {
            for (int k = 0; k < ArraySize(curr_filter_selected); k++) {
               if (filteredEvents[i].currency == curr_filter_selected[k]) {
                  currencyMatch = true;
                  break;
               }
            }
            if (!currencyMatch) {
               if (debugLogging) Print("Event ", filteredEvents[i].event, " skipped due to currency filter."); // Modified: Conditional logging
               continue;
            }
         }

         bool impactMatch = !enableImportanceFilter;
         if (enableImportanceFilter) {
            string imp_str = filteredEvents[i].importance;
            ENUM_CALENDAR_EVENT_IMPORTANCE event_imp = (imp_str == "None") ? CALENDAR_IMPORTANCE_NONE :
                                                      (imp_str == "Low") ? CALENDAR_IMPORTANCE_LOW :
                                                      (imp_str == "Medium") ? CALENDAR_IMPORTANCE_MODERATE :
                                                      CALENDAR_IMPORTANCE_HIGH;
            for (int k = 0; k < ArraySize(imp_filter_selected); k++) {
               if (event_imp == imp_filter_selected[k]) {
                  impactMatch = true;
                  break;
               }
            }
            if (!impactMatch) {
               if (debugLogging) Print("Event ", filteredEvents[i].event, " skipped due to impact filter."); // Modified: Conditional logging
               continue;
            }
         }

         bool alreadyTriggered = false;
         for (int j = 0; j < ArraySize(triggeredNewsEvents); j++) {
            if (triggeredNewsEvents[j] == i) {
               alreadyTriggered = true;
               break;
            }
         }
         if (alreadyTriggered) {
            if (debugLogging) Print("Event ", filteredEvents[i].event, " already triggered a trade. Skipping."); // Modified: Conditional logging
            continue;
         }

         if (tradeMode == TRADE_BEFORE) {
            if (currentTime >= (eventTime - offsetSeconds) && currentTime < eventTime) {
               double forecast = filteredEvents[i].forecast;
               double previous = filteredEvents[i].previous;
               if (forecast == 0.0 || previous == 0.0) {
                  if (debugLogging) Print("Skipping event ", filteredEvents[i].event, " because forecast or previous value is empty."); // Modified: Conditional logging
                  continue;
               }
               if (forecast == previous) {
                  if (debugLogging) Print("Skipping event ", filteredEvents[i].event, " because forecast equals previous."); // Modified: Conditional logging
                  continue;
               }
               if (candidateEventTime == 0 || eventTime < candidateEventTime) {
                  candidateEventTime = eventTime;
                  candidateEventName = filteredEvents[i].event;
                  candidateEventID = i;
                  candidateTradeSide = (forecast > previous) ? "BUY" : "SELL";
                  if (debugLogging) Print("Candidate event: ", filteredEvents[i].event, " with event time: ", TimeToString(eventTime, TIME_SECONDS), " Side: ", candidateTradeSide); // Modified: Conditional logging
               }
            }
         }
      }
   } else {
      //---- Live mode: Unchanged from PART 6
      MqlCalendarValue values[];
      int totalValues = CalendarValueHistory(values, lowerBound, upperBound, NULL, NULL);
      Print("Total events found: ", totalValues);
      if (totalValues <= 0) {
         if (ObjectFind(0, "NewsCountdown") >= 0) ObjectDelete(0, "NewsCountdown");
         return;
      }

      for (int i = 0; i < totalValues; i++) {
         MqlCalendarEvent event;
         if (!CalendarEventById(values[i].event_id, event)) continue;
         MqlCalendarCountry country;
         CalendarCountryById(event.country_id, country);
         bool currencyMatch = false;
         if (enableCurrencyFilter) {
            for (int k = 0; k < ArraySize(curr_filter_selected); k++) {
               if (country.currency == curr_filter_selected[k]) {
                  currencyMatch = true;
                  break;
               }
            }
            if (!currencyMatch) {
               Print("Event ", event.name, " skipped due to currency filter.");
               continue;
            }
         }
         bool impactMatch = false;
         if (enableImportanceFilter) {
            for (int k = 0; k < ArraySize(imp_filter_selected); k++) {
               if (event.importance == imp_filter_selected[k]) {
                  impactMatch = true;
                  break;
               }
            }
            if (!impactMatch) {
               Print("Event ", event.name, " skipped due to impact filter.");
               continue;
            }
         }
         if (enableTimeFilter && values[i].time > upperBound) {
            Print("Event ", event.name, " skipped due to time filter.");
            continue;
         }
         bool alreadyTriggered = false;
         for (int j = 0; j < ArraySize(triggeredNewsEvents); j++) {
            if (triggeredNewsEvents[j] == values[i].event_id) {
               alreadyTriggered = true;
               break;
            }
         }
         if (alreadyTriggered) {
            Print("Event ", event.name, " already triggered a trade. Skipping.");
            continue;
         }
         if (tradeMode == TRADE_BEFORE) {
            if (currentTime >= (values[i].time - offsetSeconds) && currentTime < values[i].time) {
               MqlCalendarValue calValue;
               if (!CalendarValueById(values[i].id, calValue)) {
                  Print("Error retrieving calendar value for event: ", event.name);
                  continue;
               }
               double forecast = calValue.GetForecastValue();
               double previous = calValue.GetPreviousValue();
               if (forecast == 0.0 || previous == 0.0) {
                  Print("Skipping event ", event.name, " because forecast or previous value is empty.");
                  continue;
               }
               if (forecast == previous) {
                  Print("Skipping event ", event.name, " because forecast equals previous.");
                  continue;
               }
               if (candidateEventTime == 0 || values[i].time < candidateEventTime) {
                  candidateEventTime = values[i].time;
                  candidateEventName = event.name;
                  candidateEventID = (int)values[i].event_id;
                  candidateTradeSide = (forecast > previous) ? "BUY" : "SELL";
                  Print("Candidate event: ", event.name, " with event time: ", TimeToString(values[i].time, TIME_SECONDS), " Side: ", candidateTradeSide);
               }
            }
         }
      }
   }

   if (tradeMode == TRADE_BEFORE && candidateEventTime > 0) {
      datetime targetTime = candidateEventTime - offsetSeconds;
      if (debugLogging) Print("Candidate target time: ", TimeToString(targetTime, TIME_SECONDS)); // Modified: Conditional logging
      if (currentTime >= targetTime && currentTime < candidateEventTime) {
         if (MQLInfoInteger(MQL_TESTER)) {
            //---- Tester mode: Execute trade for filtered event
            for (int i = 0; i < ArraySize(filteredEvents); i++) {
               datetime eventTime = filteredEvents[i].eventDateTime;
               if (eventTime == candidateEventTime) {
                  if (currentTime >= eventTime) {
                     if (debugLogging) Print("Skipping candidate ", filteredEvents[i].event, " because current time is past event time."); // Modified: Conditional logging
                     continue;
                  }
                  double forecast = filteredEvents[i].forecast;
                  double previous = filteredEvents[i].previous;
                  if (forecast == 0.0 || previous == 0.0 || forecast == previous) {
                     if (debugLogging) Print("Skipping candidate ", filteredEvents[i].event, " due to invalid forecast/previous values."); // Modified: Conditional logging
                     continue;
                  }
                  string newsInfo = "Trading on news: " + filteredEvents[i].event +
                                    " ("+TimeToString(eventTime, TIME_MINUTES)+")";
                  Print(newsInfo);
                  createLabel1("NewsTradeInfo", 355, 22, newsInfo, clrBlue, 11);
                  bool tradeResult = false;
                  if (candidateTradeSide == "BUY") {
                     tradeResult = trade.Buy(tradeLotSize, _Symbol, 0, 0, 0, filteredEvents[i].event);
                  } else if (candidateTradeSide == "SELL") {
                     tradeResult = trade.Sell(tradeLotSize, _Symbol, 0, 0, 0, filteredEvents[i].event);
                  }
                  if (tradeResult) {
                     Print("Trade executed for candidate event: ", filteredEvents[i].event, " Side: ", candidateTradeSide);
                     int size = ArraySize(triggeredNewsEvents);
                     ArrayResize(triggeredNewsEvents, size + 1);
                     triggeredNewsEvents[size] = i;
                     tradeExecuted = true;
                     tradedNewsTime = eventTime;
                  } else {
                     Print("Trade execution failed for candidate event: ", filteredEvents[i].event, " Error: ", GetLastError());
                  }
                  break;
               }
            }
         } else {
            //---- Live mode: Unchanged
            MqlCalendarValue values[];
            int totalValues = CalendarValueHistory(values, lowerBound, upperBound, NULL, NULL);
            for (int i = 0; i < totalValues; i++) {
               if (values[i].time == candidateEventTime) {
                  MqlCalendarEvent event;
                  if (!CalendarEventById(values[i].event_id, event)) continue;
                  if (currentTime >= values[i].time) {
                     Print("Skipping candidate ", event.name, " because current time is past event time.");
                     continue;
                  }
                  MqlCalendarValue calValue;
                  if (!CalendarValueById(values[i].id, calValue)) {
                     Print("Error retrieving calendar value for candidate event: ", event.name);
                     continue;
                  }
                  double forecast = calValue.GetForecastValue();
                  double previous = calValue.GetPreviousValue();
                  if (forecast == 0.0 || previous == 0.0 || forecast == previous) {
                     Print("Skipping candidate ", event.name, " due to invalid forecast/previous values.");
                     continue;
                  }
                  string newsInfo = "Trading on news: " + event.name +
                                    " ("+TimeToString(values[i].time, TIME_MINUTES)+")";
                  Print(newsInfo);
                  createLabel1("NewsTradeInfo", 355, 22, newsInfo, clrBlue, 11);
                  bool tradeResult = false;
                  if (candidateTradeSide == "BUY") {
                     tradeResult = trade.Buy(tradeLotSize, _Symbol, 0, 0, 0, event.name);
                  } else if (candidateTradeSide == "SELL") {
                     tradeResult = trade.Sell(tradeLotSize, _Symbol, 0, 0, 0, event.name);
                  }
                  if (tradeResult) {
                     Print("Trade executed for candidate event: ", event.name, " Side: ", candidateTradeSide);
                     int size = ArraySize(triggeredNewsEvents);
                     ArrayResize(triggeredNewsEvents, size + 1);
                     triggeredNewsEvents[size] = (int)values[i].event_id;
                     tradeExecuted = true;
                     tradedNewsTime = values[i].time;
                  } else {
                     Print("Trade execution failed for candidate event: ", event.name, " Error: ", GetLastError());
                  }
                  break;
               }
            }
         }
      } else {
         int remainingSeconds = (int)(candidateEventTime - currentTime);
         int hrs = remainingSeconds / 3600;
         int mins = (remainingSeconds % 3600) / 60;
         int secs = remainingSeconds % 60;
         string countdownText = "News in: " + IntegerToString(hrs) + "h " +
                               IntegerToString(mins) + "m " + IntegerToString(secs) + "s";
         if (ObjectFind(0, "NewsCountdown") < 0) {
            createButton1("NewsCountdown", 50, 17, 300, 30, countdownText, clrWhite, 12, clrBlue, clrBlack);
            Print("Pre-trade countdown created: ", countdownText);
         } else {
            updateLabel1("NewsCountdown", countdownText);
            Print("Pre-trade countdown updated: ", countdownText);
         }
      }
   } else {
      if (ObjectFind(0, "NewsCountdown") >= 0) {
         ObjectDelete(0, "NewsCountdown");
         ObjectDelete(0, "NewsTradeInfo");
         Print("Pre-trade countdown deleted.");
      }
   }
}

//+------------------------------------------------------------------+
//| Get importance level                                              |
//+------------------------------------------------------------------+
ENUM_CALENDAR_EVENT_IMPORTANCE get_importance_level(string &impact_label[], ENUM_CALENDAR_EVENT_IMPORTANCE &importance_levels[], string selected_label) {
   for (int i = 0; i < ArraySize(impact_label); i++) {
      if (impact_label[i] == selected_label) {
         return importance_levels[i];
      }
   }
   return CALENDAR_IMPORTANCE_NONE;
}

//+------------------------------------------------------------------+
//| Destroy dashboard                                                 |
//+------------------------------------------------------------------+
void destroy_Dashboard() {
   ObjectDelete(0,"MAIN_REC");
   ObjectDelete(0,"SUB_REC1");
   ObjectDelete(0,"SUB_REC2");
   ObjectDelete(0,"HEADER_LABEL");
   ObjectDelete(0,"TIME_LABEL");
   ObjectDelete(0,"IMPACT_LABEL");
   ObjectsDeleteAll(0,"ARRAY_CALENDAR");
   ObjectsDeleteAll(0,"ARRAY_NEWS");
   ObjectsDeleteAll(0,"DATA_HOLDERS");
   ObjectsDeleteAll(0,"IMPACT_LABEL");
   ObjectDelete(0,"FILTER_LABEL");
   ObjectDelete(0,"FILTER_CURR_BTN");
   ObjectDelete(0,"FILTER_IMP_BTN");
   ObjectDelete(0,"FILTER_TIME_BTN");
   ObjectDelete(0,"CANCEL_BTN");
   ObjectsDeleteAll(0,"CURRENCY_BTNS");
   ChartRedraw(0);
}

//+------------------------------------------------------------------+
//| Delete trade objects                                              |
//+------------------------------------------------------------------+
void deleteTradeObjects() {
   ObjectDelete(0, "NewsCountdown");
   ObjectDelete(0, "NewsTradeInfo");
   ChartRedraw();
}

//+------------------------------------------------------------------+
//| Update filter info                                                |
//+------------------------------------------------------------------+
void UpdateFilterInfo() {
   string filterInfo = "Filters: ";
   if (enableCurrencyFilter) {
      filterInfo += "Currency: ";
      for (int i = 0; i < ArraySize(curr_filter_selected); i++) {
         filterInfo += curr_filter_selected[i];
         if (i < ArraySize(curr_filter_selected) - 1) filterInfo += ",";
      }
      filterInfo += "; ";
   } else {
      filterInfo += "Currency: Off; ";
   }
   if (enableImportanceFilter) {
      filterInfo += "Impact: ";
      for (int i = 0; i < ArraySize(imp_filter_selected); i++) {
         filterInfo += EnumToString(imp_filter_selected[i]);
         if (i < ArraySize(imp_filter_selected) - 1) filterInfo += ",";
      }
      filterInfo += "; ";
   } else {
      filterInfo += "Impact: Off; ";
   }
   if (enableTimeFilter) {
      filterInfo += "Time: Up to " + EnumToString(end_time);
   } else {
      filterInfo += "Time: Off";
   }
   if (!MQLInfoInteger(MQL_TESTER) || debugLogging) Print("Filter Info: ", filterInfo); // Modified: Conditional logging
}

//+------------------------------------------------------------------+
//| Check for news events                                             |
//+------------------------------------------------------------------+
bool isNewsEvent() {
   int totalNews = 0;
   bool isNews = false;
   MqlCalendarValue values[];
   datetime startTime = TimeTradeServer() - PeriodSeconds(PERIOD_D1);
   datetime endTime = TimeTradeServer() + PeriodSeconds(PERIOD_D1);
   int valuesTotal = CalendarValueHistory(values,startTime,endTime,NULL,NULL);
   Print("TOTAL VALUES = ",valuesTotal," || Array size = ",ArraySize(values));
   datetime timeRange = PeriodSeconds(PERIOD_D1);
   datetime timeBefore = TimeTradeServer() - timeRange;
   datetime timeAfter = TimeTradeServer() + timeRange;
   Print("Current time = ",TimeTradeServer());
   Print("FURTHEST TIME LOOK BACK = ",timeBefore," >>> LOOK FORE = ",timeAfter);
   for (int i = 0; i < valuesTotal; i++) {
      MqlCalendarEvent event;
      CalendarEventById(values[i].event_id, event);
      MqlCalendarCountry country;
      CalendarCountryById(event.country_id, country);
      if (StringFind(_Symbol, country.currency) >= 0) {
         if (event.importance == CALENDAR_IMPORTANCE_MODERATE) {
            if (values[i].time <= TimeTradeServer() && values[i].time >= timeBefore) {
               Print(event.name," > ", country.currency," > ", EnumToString(event.importance)," Time= ",values[i].time," (ALREADY RELEASED)");
               totalNews++;
            }
            if (values[i].time >= TimeTradeServer() && values[i].time <= timeAfter) {
               Print(event.name," > ", country.currency," > ", EnumToString(event.importance)," Time= ",values[i].time," (NOT YET RELEASED)");
               totalNews++;
            }
         }
      }
   }
   if (totalNews > 0) {
      isNews = true;
      Print(">>>>>>> (FOUND NEWS) TOTAL NEWS = ",totalNews,"/",ArraySize(values));
   } else {
      isNews = false;
      Print(">>>>>>> (NOT FOUND NEWS) TOTAL NEWS = ",totalNews,"/",ArraySize(values));
   }
   return isNews;
}

//+------------------------------------------------------------------+
//| Create rectangle label                                            |
//+------------------------------------------------------------------+
bool createRecLabel(string objName, int xD, int yD, int xS, int yS, color clrBg, int widthBorder,
                    color clrBorder = clrNONE, ENUM_BORDER_TYPE borderType = BORDER_FLAT,
                    ENUM_LINE_STYLE borderStyle = STYLE_SOLID) {
   ResetLastError();
   if (!ObjectCreate(0, objName, OBJ_RECTANGLE_LABEL, 0, 0, 0)) {
      Print(__FUNCTION__, ": failed to create rec label! Error code = ", _LastError);
      return false;
   }
   ObjectSetInteger(0, objName, OBJPROP_XDISTANCE, xD);
   ObjectSetInteger(0, objName, OBJPROP_YDISTANCE, yD);
   ObjectSetInteger(0, objName, OBJPROP_XSIZE, xS);
   ObjectSetInteger(0, objName, OBJPROP_YSIZE, yS);
   ObjectSetInteger(0, objName, OBJPROP_CORNER, CORNER_LEFT_UPPER);
   ObjectSetInteger(0, objName, OBJPROP_BGCOLOR, clrBg);
   ObjectSetInteger(0, objName, OBJPROP_BORDER_TYPE, borderType);
   ObjectSetInteger(0, objName, OBJPROP_STYLE, borderStyle);
   ObjectSetInteger(0, objName, OBJPROP_WIDTH, widthBorder);
   ObjectSetInteger(0, objName, OBJPROP_COLOR, clrBorder);
   ObjectSetInteger(0, objName, OBJPROP_BACK, false);
   ObjectSetInteger(0, objName, OBJPROP_STATE, false);
   ObjectSetInteger(0, objName, OBJPROP_SELECTABLE, false);
   ObjectSetInteger(0, objName, OBJPROP_SELECTED, false);
   ChartRedraw(0);
   return true;
}

//+------------------------------------------------------------------+
//| Create button                                                     |
//+------------------------------------------------------------------+
bool createButton(string objName, int xD, int yD, int xS, int yS, string txt = "", color clrTxt = clrBlack,
                  int fontSize = 12, color clrBg = clrNONE, color clrBorder = clrNONE, string font = "Arial Rounded MT Bold") {
   ResetLastError();
   if (!ObjectCreate(0, objName, OBJ_BUTTON, 0, 0, 0)) {
      Print(__FUNCTION__, ": failed to create the button! Error code = ", _LastError);
      return false;
   }
   ObjectSetInteger(0, objName, OBJPROP_XDISTANCE, xD);
   ObjectSetInteger(0, objName, OBJPROP_YDISTANCE, yD);
   ObjectSetInteger(0, objName, OBJPROP_XSIZE, xS);
   ObjectSetInteger(0, objName, OBJPROP_YSIZE, yS);
   ObjectSetInteger(0, objName, OBJPROP_CORNER, CORNER_LEFT_UPPER);
   ObjectSetString(0, objName, OBJPROP_TEXT, txt);
   ObjectSetInteger(0, objName, OBJPROP_COLOR, clrTxt);
   ObjectSetInteger(0, objName, OBJPROP_FONTSIZE, fontSize);
   ObjectSetString(0, objName, OBJPROP_FONT, font);
   ObjectSetInteger(0, objName, OBJPROP_BGCOLOR, clrBg);
   ObjectSetInteger(0, objName, OBJPROP_BORDER_COLOR, clrBorder);
   ObjectSetInteger(0, objName, OBJPROP_BACK, false);
   ObjectSetInteger(0, objName, OBJPROP_STATE, false);
   ObjectSetInteger(0, objName, OBJPROP_SELECTABLE, false);
   ObjectSetInteger(0, objName, OBJPROP_SELECTED, false);
   ChartRedraw(0);
   return true;
}

//+------------------------------------------------------------------+
//| Create label                                                      |
//+------------------------------------------------------------------+
bool createLabel(string objName, int xD, int yD, string txt, color clrTxt = clrBlack, int fontSize = 12,
                 string font = "Arial Rounded MT Bold") {
   ResetLastError();
   if (!ObjectCreate(0, objName, OBJ_LABEL, 0, 0, 0)) {
      Print(__FUNCTION__, ": failed to create the label! Error code = ", _LastError);
      return false;
   }
   ObjectSetInteger(0, objName, OBJPROP_XDISTANCE, xD);
   ObjectSetInteger(0, objName, OBJPROP_YDISTANCE, yD);
   ObjectSetInteger(0, objName, OBJPROP_CORNER, CORNER_LEFT_UPPER);
   ObjectSetString(0, objName, OBJPROP_TEXT, txt);
   ObjectSetInteger(0, objName, OBJPROP_COLOR, clrTxt);
   ObjectSetInteger(0, objName, OBJPROP_FONTSIZE, fontSize);
   ObjectSetString(0, objName, OBJPROP_FONT, font);
   ObjectSetInteger(0, objName, OBJPROP_BACK, false);
   ObjectSetInteger(0, objName, OBJPROP_STATE, false);
   ObjectSetInteger(0, objName, OBJPROP_SELECTABLE, false);
   ObjectSetInteger(0, objName, OBJPROP_SELECTED, false);
   ChartRedraw(0);
   return true;
}

//+------------------------------------------------------------------+
//| Update label                                                      |
//+------------------------------------------------------------------+
bool updateLabel(string objName, string txt) {
   ResetLastError();
   if (!ObjectSetString(0, objName, OBJPROP_TEXT, txt)) {
      Print(__FUNCTION__, ": failed to update the label! Error code = ", _LastError);
      return false;
   }
   ChartRedraw(0);
   return true;
}

//+------------------------------------------------------------------+
//| Create button (trade UI)                                          |
//+------------------------------------------------------------------+
bool createButton1(string objName, int x, int y, int width, int height, string text, color txtColor,
                   int fontSize, color bgColor, color borderColor) {
   if (!ObjectCreate(0, objName, OBJ_BUTTON, 0, 0, 0)) {
      Print("Error creating button ", objName, " : ", GetLastError());
      return false;
   }
   ObjectSetInteger(0, objName, OBJPROP_XDISTANCE, x);
   ObjectSetInteger(0, objName, OBJPROP_YDISTANCE, y);
   ObjectSetInteger(0, objName, OBJPROP_XSIZE, width);
   ObjectSetInteger(0, objName, OBJPROP_YSIZE, height);
   ObjectSetString(0, objName, OBJPROP_TEXT, text);
   ObjectSetInteger(0, objName, OBJPROP_COLOR, txtColor);
   ObjectSetInteger(0, objName, OBJPROP_FONTSIZE, fontSize);
   ObjectSetString(0, objName, OBJPROP_FONT, "Arial Bold");
   ObjectSetInteger(0, objName, OBJPROP_BGCOLOR, bgColor);
   ObjectSetInteger(0, objName, OBJPROP_BORDER_COLOR, borderColor);
   ObjectSetInteger(0, objName, OBJPROP_CORNER, CORNER_LEFT_UPPER);
   ObjectSetInteger(0, objName, OBJPROP_BACK, true);
   ChartRedraw();
   return true;
}

//+------------------------------------------------------------------+
//| Create label (trade UI)                                           |
//+------------------------------------------------------------------+
bool createLabel1(string objName, int x, int y, string text, color txtColor, int fontSize) {
   if (!ObjectCreate(0, objName, OBJ_LABEL, 0, 0, 0)) {
      Print("Error creating label ", objName, " : ", GetLastError());
      return false;
   }
   ObjectSetInteger(0, objName, OBJPROP_XDISTANCE, x);
   ObjectSetInteger(0, objName, OBJPROP_YDISTANCE, y);
   ObjectSetString(0, objName, OBJPROP_TEXT, text);
   ObjectSetInteger(0, objName, OBJPROP_COLOR, txtColor);
   ObjectSetInteger(0, objName, OBJPROP_FONTSIZE, fontSize);
   ObjectSetString(0, objName, OBJPROP_FONT, "Arial Bold");
   ObjectSetInteger(0, objName, OBJPROP_CORNER, CORNER_LEFT_UPPER);
   ChartRedraw();
   return true;
}

//+------------------------------------------------------------------+
//| Update label (trade UI)                                           |
//+------------------------------------------------------------------+
bool updateLabel1(string objName, string text) {
   if (ObjectFind(0, objName) < 0) {
      Print("updateLabel1: Object ", objName, " not found.");
      return false;
   }
   ObjectSetString(0, objName, OBJPROP_TEXT, text);
   ChartRedraw();
   return true;
}

//+------------------------------------------------------------------+
//| Compare string arrays for changes                                 |
//+------------------------------------------------------------------+
bool isChangeInStringArrays(string &arr1[], string &arr2[]) {
   bool isChange = false;
   int size1 = ArraySize(arr1);
   int size2 = ArraySize(arr2);
   if (size1 != size2) {
      Print("Arrays have different sizes. Size of Array 1: ", size1, ", Size of Array 2: ", size2);
      return true;
   }
   for (int i = 0; i < size1; i++) {
      if (StringCompare(arr1[i], arr2[i]) != 0) {
         Print("Change detected at index ", i, ": '", arr1[i], "' vs '", arr2[i], "'");
         return true;
      }
   }
   return false;
}