#include "ExecutionAlgorithm.mqh"

//+------------------------------------------------------------------+
//| Time-Weighted Average Price (TWAP) Algorithm                     |
//+------------------------------------------------------------------+
class CTWAP : public CExecutionAlgorithm
{
private:
   int               m_intervals;           // Number of time intervals
   int               m_currentInterval;     // Current interval
   datetime          m_nextExecutionTime;   // Next execution time
   double            m_intervalVolume;      // Volume per interval
   bool              m_useRandomization;    // Whether to randomize order sizes
   double            m_randomizationFactor; // Factor for randomization (0-1)
   ENUM_ORDER_TYPE   m_orderType;           // Order type (buy or sell)
   
public:
   // Constructor
   CTWAP(string symbol, double volume, datetime startTime, datetime endTime, 
         int intervals, ENUM_ORDER_TYPE orderType, bool useRandomization = false, 
         double randomizationFactor = 0.2, int slippage = 3);
   
   // Destructor
   ~CTWAP();
   
   // Implementation of virtual methods
   virtual bool      Initialize() override;
   virtual bool      Execute() override;
   virtual bool      Update() override;
   virtual bool      Terminate() override;
   
   // TWAP specific methods
   void              CalculateIntervalVolume();
   datetime          CalculateNextExecutionTime();
   double            GetRandomizedVolume(double baseVolume);
};

//+------------------------------------------------------------------+
//| Constructor                                                       |
//+------------------------------------------------------------------+
CTWAP::CTWAP(string symbol, double volume, datetime startTime, datetime endTime, 
            int intervals, ENUM_ORDER_TYPE orderType, bool useRandomization, 
            double randomizationFactor, int slippage)
   : CExecutionAlgorithm(symbol, volume, startTime, endTime, slippage)
{
   m_intervals = intervals;
   m_currentInterval = 0;
   m_useRandomization = useRandomization;
   m_randomizationFactor = randomizationFactor;
   m_orderType = orderType;
   m_nextExecutionTime = 0;
   m_intervalVolume = 0.0;
}

//+------------------------------------------------------------------+
//| Destructor                                                        |
//+------------------------------------------------------------------+
CTWAP::~CTWAP()
{
   // Clean up resources if needed
}

//+------------------------------------------------------------------+
//| Initialize the TWAP algorithm                                     |
//+------------------------------------------------------------------+
bool CTWAP::Initialize()
{
   if(!CExecutionAlgorithm::Initialize())
      return false;
      
   // Calculate the volume for each interval
   CalculateIntervalVolume();
   
   // Calculate the time for the first execution
   m_nextExecutionTime = CalculateNextExecutionTime();
   
   m_isActive = true;
   
   Print("TWAP algorithm initialized for ", m_symbol, 
         ". Total volume: ", DoubleToString(m_totalVolume, 2), 
         ", Intervals: ", m_intervals);
   
   return true;
}

//+------------------------------------------------------------------+
//| Execute the TWAP algorithm                                        |
//+------------------------------------------------------------------+
bool CTWAP::Execute()
{
   if(!m_isActive)
      return false;
      
   // Check if it's time to execute the next order
   datetime currentTime = TimeCurrent();
   
   if(currentTime < m_nextExecutionTime)
      return true; // Not time yet
      
   // Calculate the volume for this execution
   double volumeToExecute = m_useRandomization ? 
                           GetRandomizedVolume(m_intervalVolume) : 
                           m_intervalVolume;
                           
   // Ensure we don't exceed the remaining volume
   if(volumeToExecute > m_remainingVolume)
      volumeToExecute = m_remainingVolume;
      
   // Get current market price
   double price = 0.0;
   if(m_orderType == ORDER_TYPE_BUY)
      price = SymbolInfoDouble(m_symbol, SYMBOL_ASK);
   else
      price = SymbolInfoDouble(m_symbol, SYMBOL_BID);
      
   // Place the order
   if(!PlaceOrder(m_orderType, volumeToExecute, price))
   {
      Print("TWAP: Failed to place order for interval ", m_currentInterval);
      return false;
   }
   
   // Update interval counter
   m_currentInterval++;
   
   // Calculate the time for the next execution
   if(m_currentInterval < m_intervals && m_remainingVolume > 0)
      m_nextExecutionTime = CalculateNextExecutionTime();
   else
      m_isActive = false; // All intervals completed or no volume left
      
   Print("TWAP: Executed ", DoubleToString(volumeToExecute, 2), 
         " at price ", DoubleToString(price, _Digits), 
         ". Remaining: ", DoubleToString(m_remainingVolume, 2));
         
   return true;
}

//+------------------------------------------------------------------+
//| Update the TWAP algorithm state                                   |
//+------------------------------------------------------------------+
bool CTWAP::Update()
{
   if(!m_isActive)
      return false;
      
   // Check if the end time has been reached
   if(TimeCurrent() >= m_endTime)
   {
      Print("TWAP: End time reached. Terminating algorithm.");
      return Terminate();
   }
   
   // Check if all volume has been executed
   if(m_remainingVolume <= 0)
   {
      Print("TWAP: All volume executed. Terminating algorithm.");
      return Terminate();
   }
   
   return true;
}

//+------------------------------------------------------------------+
//| Terminate the TWAP algorithm                                      |
//+------------------------------------------------------------------+
bool CTWAP::Terminate()
{
   if(!m_isActive)
      return false;
      
   // If there's remaining volume and we want to execute it all at once
   if(m_remainingVolume > 0)
   {
      Print("TWAP: Terminating with remaining volume: ", DoubleToString(m_remainingVolume, 2));
      
      // Get current market price
      double price = 0.0;
      if(m_orderType == ORDER_TYPE_BUY)
         price = SymbolInfoDouble(m_symbol, SYMBOL_ASK);
      else
         price = SymbolInfoDouble(m_symbol, SYMBOL_BID);
         
      // Place the final order for remaining volume
      if(!PlaceOrder(m_orderType, m_remainingVolume, price))
      {
         Print("TWAP: Failed to place final order");
      }
   }
   
   m_isActive = false;
   
   Print("TWAP algorithm terminated. Total executed volume: ", 
         DoubleToString(m_executedVolume, 2));
         
   return true;
}

//+------------------------------------------------------------------+
//| Calculate the volume for each interval                            |
//+------------------------------------------------------------------+
void CTWAP::CalculateIntervalVolume()
{
   // Simple equal distribution across intervals
   m_intervalVolume = m_totalVolume / m_intervals;
   
   Print("TWAP: Interval volume calculated: ", DoubleToString(m_intervalVolume, 2));
}

//+------------------------------------------------------------------+
//| Calculate the time for the next execution                         |
//+------------------------------------------------------------------+
datetime CTWAP::CalculateNextExecutionTime()
{
   // Calculate the duration of each interval
   int totalSeconds = (int)(m_endTime - m_startTime);
   int intervalSeconds = totalSeconds / m_intervals;
   
   // Calculate the next execution time
   datetime nextTime;
   
   if(m_currentInterval == 0) {
      // First interval - start at the defined start time
      nextTime = m_startTime;
   } else {
      // For subsequent intervals, ensure proper spacing from current time
      datetime currentTime = TimeCurrent();
      nextTime = currentTime + intervalSeconds;
      
      // Make sure we don't exceed the end time
      if(nextTime > m_endTime)
         nextTime = m_endTime;
   }
   
   Print("TWAP: Next execution time set to ", TimeToString(nextTime));
   return nextTime;
}

//+------------------------------------------------------------------+
//| Get a randomized volume based on the interval volume              |
//+------------------------------------------------------------------+
double CTWAP::GetRandomizedVolume(double baseVolume)
{
   // Generate a random factor between (1-factor) and (1+factor)
   double randomFactor = 1.0 + ((MathRand() / (double)32767) * 2 - 1) * m_randomizationFactor;
   
   // Apply the random factor to the base volume
   double randomizedVolume = baseVolume * randomFactor;
   
   // Ensure the volume is within valid bounds
   double minVolume = SymbolInfoDouble(m_symbol, SYMBOL_VOLUME_MIN);
   double maxVolume = SymbolInfoDouble(m_symbol, SYMBOL_VOLUME_MAX);
   double stepVolume = SymbolInfoDouble(m_symbol, SYMBOL_VOLUME_STEP);
   
   // Normalize to volume step
   randomizedVolume = MathFloor(randomizedVolume / stepVolume) * stepVolume;
   
   // Ensure within min/max bounds
   randomizedVolume = MathMax(minVolume, MathMin(maxVolume, randomizedVolume));
   
   return randomizedVolume;
}
//+------------------------------------------------------------------+
