#include "ExecutionAlgorithm.mqh"

//+------------------------------------------------------------------+
//| Performance analysis for execution algorithms                     |
//+------------------------------------------------------------------+
class CPerformanceAnalyzer
{
private:
   CExecutionAlgorithm *m_algorithm;        // Algorithm to analyze
   string            m_symbol;               // Trading symbol
   double            m_benchmarkPrice;       // Benchmark price (e.g., VWAP from market)
   double            m_implementationShortfall; // Implementation shortfall
   double            m_marketImpact;         // Market impact
   double            m_slippageCost;         // Slippage cost
   double            m_timingCost;           // Timing cost
   double            m_totalCost;            // Total execution cost
   double            m_marketVWAP;           // Market VWAP for comparison
   double            m_marketTWAP;           // Market TWAP for comparison
   datetime          m_startTime;            // Start time of execution
   datetime          m_endTime;              // End time of execution
   
public:
   // Constructor
   CPerformanceAnalyzer(CExecutionAlgorithm *algorithm, string symbol, 
                        datetime startTime, datetime endTime);
   
   // Analysis methods
   double            CalculateImplementationShortfall();
   double            CalculateMarketImpact();
   double            CalculateSlippageCost();
   double            CalculateTimingCost();
   double            CalculateTotalCost();
   
   // Comparison methods
   double            CalculateMarketVWAP();
   double            CalculateMarketTWAP();
   double            CompareToMarketVWAP();
   double            CompareToMarketTWAP();
   
   // Reporting
   void              GenerateReport(string filename);
   string            GetPerformanceSummary();
};

//+------------------------------------------------------------------+
//| Constructor                                                       |
//+------------------------------------------------------------------+
CPerformanceAnalyzer::CPerformanceAnalyzer(CExecutionAlgorithm *algorithm, string symbol, 
                                         datetime startTime, datetime endTime)
{
   m_algorithm = algorithm;
   m_symbol = symbol;
   m_startTime = startTime;
   m_endTime = endTime;
   
   m_benchmarkPrice = 0.0;
   m_implementationShortfall = 0.0;
   m_marketImpact = 0.0;
   m_slippageCost = 0.0;
   m_timingCost = 0.0;
   m_totalCost = 0.0;
   m_marketVWAP = 0.0;
   m_marketTWAP = 0.0;
}

//+------------------------------------------------------------------+
//| Calculate implementation shortfall                                |
//+------------------------------------------------------------------+
double CPerformanceAnalyzer::CalculateImplementationShortfall()
{
   // Implementation shortfall is the difference between the decision price
   // and the actual average execution price
   
   // Get the average execution price from the algorithm
   double avgExecutionPrice = m_algorithm.GetAverageExecutionPrice();
   
   // For this example, we'll use the opening price as the decision price
   MqlRates rates[];
   if(CopyRates(m_symbol, PERIOD_D1, m_startTime, 1, rates) <= 0)
   {
      Print("Failed to copy rates for implementation shortfall calculation");
      return 0.0;
   }
   
   double decisionPrice = rates[0].open;
   
   // Calculate the shortfall (positive means worse execution)
   // For buy orders: execution price - decision price
   // For sell orders: decision price - execution price
   // We'll assume buy orders for simplicity
   m_implementationShortfall = avgExecutionPrice - decisionPrice;
   
   return m_implementationShortfall;
}

//+------------------------------------------------------------------+
//| Calculate market impact                                           |
//+------------------------------------------------------------------+
double CPerformanceAnalyzer::CalculateMarketImpact()
{
   // Market impact is the price movement caused by our own trading
   // It's difficult to measure precisely, but we can estimate it
   
   // Get the average execution price from the algorithm
   double avgExecutionPrice = m_algorithm.GetAverageExecutionPrice();
   
   // Calculate market VWAP for the period
   double marketVWAP = CalculateMarketVWAP();
   
   // Market impact is the difference between our execution price and market VWAP
   // For buy orders: our price - VWAP
   // For sell orders: VWAP - our price
   // We'll assume buy orders for simplicity
   m_marketImpact = avgExecutionPrice - marketVWAP;
   
   return m_marketImpact;
}

//+------------------------------------------------------------------+
//| Calculate slippage cost                                           |
//+------------------------------------------------------------------+
double CPerformanceAnalyzer::CalculateSlippageCost()
{
   // Slippage is the difference between the expected price and the actual price
   // For this example, we'll use the difference between the limit price and execution price
   
   // Get the average execution price from the algorithm
   double avgExecutionPrice = m_algorithm.GetAverageExecutionPrice();
   
   // For this example, we'll use the opening price as the expected price
   MqlRates rates[];
   if(CopyRates(m_symbol, PERIOD_D1, m_startTime, 1, rates) <= 0)
   {
      Print("Failed to copy rates for slippage calculation");
      return 0.0;
   }
   
   double expectedPrice = rates[0].open;
   
   // Calculate slippage (positive means worse execution)
   // For buy orders: execution price - expected price
   // For sell orders: expected price - execution price
   // We'll assume buy orders for simplicity
   m_slippageCost = avgExecutionPrice - expectedPrice;
   
   return m_slippageCost;
}

//+------------------------------------------------------------------+
//| Calculate timing cost                                             |
//+------------------------------------------------------------------+
double CPerformanceAnalyzer::CalculateTimingCost()
{
   // Timing cost is the cost due to price movements during the execution period
   // It's the difference between the average execution price and the arrival price
   
   // Get the average execution price from the algorithm
   double avgExecutionPrice = m_algorithm.GetAverageExecutionPrice();
   
   // For this example, we'll use the opening price as the arrival price
   MqlRates rates[];
   if(CopyRates(m_symbol, PERIOD_D1, m_startTime, 1, rates) <= 0)
   {
      Print("Failed to copy rates for timing cost calculation");
      return 0.0;
   }
   
   double arrivalPrice = rates[0].open;
   
   // Calculate timing cost (positive means worse execution)
   // For buy orders: execution price - arrival price
   // For sell orders: arrival price - execution price
   // We'll assume buy orders for simplicity
   m_timingCost = avgExecutionPrice - arrivalPrice;
   
   return m_timingCost;
}

//+------------------------------------------------------------------+
//| Calculate total execution cost                                    |
//+------------------------------------------------------------------+
double CPerformanceAnalyzer::CalculateTotalCost()
{
   // Calculate all cost components
   CalculateImplementationShortfall();
   CalculateMarketImpact();
   CalculateSlippageCost();
   CalculateTimingCost();
   
   // Total cost is the sum of all components
   m_totalCost = m_implementationShortfall + m_marketImpact + m_slippageCost + m_timingCost;
   
   return m_totalCost;
}

//+------------------------------------------------------------------+
//| Calculate market VWAP                                             |
//+------------------------------------------------------------------+
double CPerformanceAnalyzer::CalculateMarketVWAP()
{
   // Calculate the Volume-Weighted Average Price from market data
   
   // Load market data for the execution period
   MqlRates rates[];
   int copied = CopyRates(m_symbol, PERIOD_M1, m_startTime, m_endTime, rates);
   
   if(copied <= 0)
   {
      Print("Failed to copy rates for VWAP calculation. Error: ", GetLastError());
      return 0.0;
   }
   
   double volumeSum = 0.0;
   double priceVolumeSum = 0.0;
   
   for(int i = 0; i < copied; i++)
   {
      // Calculate typical price
      double typicalPrice = (rates[i].high + rates[i].low + rates[i].close) / 3.0;
      
      // Add to sums
      volumeSum += rates[i].tick_volume;
      priceVolumeSum += typicalPrice * rates[i].tick_volume;
   }
   
   // Calculate VWAP
   if(volumeSum > 0)
      m_marketVWAP = priceVolumeSum / volumeSum;
   else
      m_marketVWAP = 0.0;
      
   return m_marketVWAP;
}

//+------------------------------------------------------------------+
//| Calculate market TWAP                                             |
//+------------------------------------------------------------------+
double CPerformanceAnalyzer::CalculateMarketTWAP()
{
   // Calculate the Time-Weighted Average Price from market data
   
   // Load market data for the execution period
   MqlRates rates[];
   int copied = CopyRates(m_symbol, PERIOD_M1, m_startTime, m_endTime, rates);
   
   if(copied <= 0)
   {
      Print("Failed to copy rates for TWAP calculation. Error: ", GetLastError());
      return 0.0;
   }
   
   double priceSum = 0.0;
   
   for(int i = 0; i < copied; i++)
   {
      // Calculate typical price
      double typicalPrice = (rates[i].high + rates[i].low + rates[i].close) / 3.0;
      
      // Add to sum
      priceSum += typicalPrice;
   }
   
   // Calculate TWAP
   if(copied > 0)
      m_marketTWAP = priceSum / copied;
   else
      m_marketTWAP = 0.0;
      
   return m_marketTWAP;
}

//+------------------------------------------------------------------+
//| Compare execution to market VWAP                                  |
//+------------------------------------------------------------------+
double CPerformanceAnalyzer::CompareToMarketVWAP()
{
   // Get the average execution price from the algorithm
   double avgExecutionPrice = m_algorithm.GetAverageExecutionPrice();
   
   // Calculate market VWAP if not already calculated
   if(m_marketVWAP == 0.0)
      CalculateMarketVWAP();
      
   // Calculate the difference (positive means worse execution for buy orders)
   // For buy orders: execution price - VWAP
   // For sell orders: VWAP - execution price
   // We'll assume buy orders for simplicity
   double difference = avgExecutionPrice - m_marketVWAP;
   
   return difference;
}

//+------------------------------------------------------------------+
//| Compare execution to market TWAP                                  |
//+------------------------------------------------------------------+
double CPerformanceAnalyzer::CompareToMarketTWAP()
{
   // Get the average execution price from the algorithm
   double avgExecutionPrice = m_algorithm.GetAverageExecutionPrice();
   
   // Calculate market TWAP if not already calculated
   if(m_marketTWAP == 0.0)
      CalculateMarketTWAP();
      
   // Calculate the difference (positive means worse execution for buy orders)
   // For buy orders: execution price - TWAP
   // For sell orders: TWAP - execution price
   // We'll assume buy orders for simplicity
   double difference = avgExecutionPrice - m_marketTWAP;
   
   return difference;
}

//+------------------------------------------------------------------+
//| Generate a performance report                                     |
//+------------------------------------------------------------------+
void CPerformanceAnalyzer::GenerateReport(string filename)
{
   // Calculate all metrics
   CalculateTotalCost();
   CalculateMarketVWAP();
   CalculateMarketTWAP();
   CompareToMarketVWAP();
   CompareToMarketTWAP();
   
   // Create the report
   int fileHandle = FileOpen(filename, FILE_WRITE|FILE_TXT);
   
   if(fileHandle == INVALID_HANDLE)
   {
      Print("Failed to open file for writing: ", filename, ". Error: ", GetLastError());
      return;
   }
   
   // Write report header
   FileWrite(fileHandle, "Performance Analysis Report");
   FileWrite(fileHandle, "==========================");
   FileWrite(fileHandle, "");
   
   // Write execution details
   FileWrite(fileHandle, "Execution Details:");
   FileWrite(fileHandle, "Symbol: ", m_symbol);
   FileWrite(fileHandle, "Start Time: ", TimeToString(m_startTime));
   FileWrite(fileHandle, "End Time: ", TimeToString(m_endTime));
   FileWrite(fileHandle, "Average Execution Price: ", DoubleToString(m_algorithm.GetAverageExecutionPrice(), _Digits));
   FileWrite(fileHandle, "");
   
   // Write market benchmarks
   FileWrite(fileHandle, "Market Benchmarks:");
   FileWrite(fileHandle, "Market VWAP: ", DoubleToString(m_marketVWAP, _Digits));
   FileWrite(fileHandle, "Market TWAP: ", DoubleToString(m_marketTWAP, _Digits));
   FileWrite(fileHandle, "");
   
   // Write cost analysis
   FileWrite(fileHandle, "Cost Analysis:");
   FileWrite(fileHandle, "Implementation Shortfall: ", DoubleToString(m_implementationShortfall, _Digits));
   FileWrite(fileHandle, "Market Impact: ", DoubleToString(m_marketImpact, _Digits));
   FileWrite(fileHandle, "Slippage Cost: ", DoubleToString(m_slippageCost, _Digits));
   FileWrite(fileHandle, "Timing Cost: ", DoubleToString(m_timingCost, _Digits));
   FileWrite(fileHandle, "Total Cost: ", DoubleToString(m_totalCost, _Digits));
   FileWrite(fileHandle, "");
   
   // Write comparison to benchmarks
   FileWrite(fileHandle, "Benchmark Comparison:");
   FileWrite(fileHandle, "Difference from VWAP: ", DoubleToString(CompareToMarketVWAP(), _Digits));
   FileWrite(fileHandle, "Difference from TWAP: ", DoubleToString(CompareToMarketTWAP(), _Digits));
   FileWrite(fileHandle, "");
   
   // Write summary
   FileWrite(fileHandle, "Summary:");
   FileWrite(fileHandle, GetPerformanceSummary());
   
   // Close the file
   FileClose(fileHandle);
   
   Print("Performance report generated: ", filename);
}

//+------------------------------------------------------------------+
//| Get a performance summary                                         |
//+------------------------------------------------------------------+
string CPerformanceAnalyzer::GetPerformanceSummary()
{
   string summary = "";
   
   // Calculate all metrics if not already calculated
   if(m_totalCost == 0.0)
      CalculateTotalCost();
      
   if(m_marketVWAP == 0.0)
      CalculateMarketVWAP();
      
   if(m_marketTWAP == 0.0)
      CalculateMarketTWAP();
      
   // Get the average execution price
   double avgExecutionPrice = m_algorithm.GetAverageExecutionPrice();
   
   // Compare to benchmarks
   double vwapDiff = avgExecutionPrice - m_marketVWAP;
   double twapDiff = avgExecutionPrice - m_marketTWAP;
   
   // Create summary text
   summary += "Execution completed with an average price of " + DoubleToString(avgExecutionPrice, _Digits) + ".\n";
   summary += "Market VWAP was " + DoubleToString(m_marketVWAP, _Digits) + " (";
   
   if(vwapDiff > 0)
      summary += "execution was worse by " + DoubleToString(vwapDiff, _Digits) + ").\n";
   else if(vwapDiff < 0)
      summary += "execution was better by " + DoubleToString(MathAbs(vwapDiff), _Digits) + ").\n";
   else
      summary += "execution was equal).\n";
      
   summary += "Market TWAP was " + DoubleToString(m_marketTWAP, _Digits) + " (";
   
   if(twapDiff > 0)
      summary += "execution was worse by " + DoubleToString(twapDiff, _Digits) + ").\n";
   else if(twapDiff < 0)
      summary += "execution was better by " + DoubleToString(MathAbs(twapDiff), _Digits) + ").\n";
   else
      summary += "execution was equal).\n";
      
   summary += "Total execution cost was " + DoubleToString(m_totalCost, _Digits) + " points.";
   
   return summary;
}
//+------------------------------------------------------------------+
