//+------------------------------------------------------------------+
//|                                    MultiTimeframeRegimes.txt.mq5 |
//|                                                      Sahil Bagdi |
//|                         https://www.mql5.com/en/users/sahilbagdi |
//+------------------------------------------------------------------+
#property copyright "Sahil Bagdi"
#property link      "https://www.mql5.com/en/users/sahilbagdi"
#property version   "1.00"
#property indicator_chart_window
#property indicator_buffers 0
#property indicator_plots   0

// Include the Market Regime Detector
#include <MarketRegimeEnum.mqh>
#include <MarketRegimeDetector.mqh>

// Input parameters
input int      LookbackPeriod = 100;       // Lookback period for calculations
input double   TrendThreshold = 0.2;       // Threshold for trend detection (0.1-0.5)
input double   VolatilityThreshold = 1.5;  // Threshold for volatility detection (1.0-3.0)

// Timeframes to analyze
input bool     UseM1 = false;              // Use 1-minute timeframe
input bool     UseM5 = false;              // Use 5-minute timeframe
input bool     UseM15 = true;              // Use 15-minute timeframe
input bool     UseM30 = true;              // Use 30-minute timeframe
input bool     UseH1 = true;               // Use 1-hour timeframe
input bool     UseH4 = true;               // Use 4-hour timeframe
input bool     UseD1 = true;               // Use Daily timeframe
input bool     UseW1 = false;              // Use Weekly timeframe
input bool     UseMN1 = false;             // Use Monthly timeframe

// Global variables
CMarketRegimeDetector *Detectors[];
ENUM_TIMEFRAMES Timeframes[];
int TimeframeCount = 0;

//+------------------------------------------------------------------+
//| Custom indicator initialization function                         |
//+------------------------------------------------------------------+
int OnInit()
{
    // Initialize timeframes array
    InitializeTimeframes();
    
    // Create detectors for each timeframe
    ArrayResize(Detectors, TimeframeCount);
    
    for(int i = 0; i < TimeframeCount; i++)
    {
        Detectors[i] = new CMarketRegimeDetector(LookbackPeriod);
        if(Detectors[i] == NULL)
        {
            Print("Failed to create Market Regime Detector for timeframe ", EnumToString(Timeframes[i]));
            return INIT_FAILED;
        }
        
        // Configure the detector
        Detectors[i].SetTrendThreshold(TrendThreshold);
        Detectors[i].SetVolatilityThreshold(VolatilityThreshold);
        Detectors[i].Initialize();
    }
    
    // Set indicator name
    IndicatorSetString(INDICATOR_SHORTNAME, "Multi-Timeframe Regime Analysis");
    
    return INIT_SUCCEEDED;
}

//+------------------------------------------------------------------+
//| Initialize timeframes array based on user inputs                 |
//+------------------------------------------------------------------+
void InitializeTimeframes()
{
    // Count selected timeframes
    TimeframeCount = 0;
    if(UseM1)  TimeframeCount++;
    if(UseM5)  TimeframeCount++;
    if(UseM15) TimeframeCount++;
    if(UseM30) TimeframeCount++;
    if(UseH1)  TimeframeCount++;
    if(UseH4)  TimeframeCount++;
    if(UseD1)  TimeframeCount++;
    if(UseW1)  TimeframeCount++;
    if(UseMN1) TimeframeCount++;
    
    // Resize and fill timeframes array
    ArrayResize(Timeframes, TimeframeCount);
    
    int index = 0;
    if(UseM1)  Timeframes[index++] = PERIOD_M1;
    if(UseM5)  Timeframes[index++] = PERIOD_M5;
    if(UseM15) Timeframes[index++] = PERIOD_M15;
    if(UseM30) Timeframes[index++] = PERIOD_M30;
    if(UseH1)  Timeframes[index++] = PERIOD_H1;
    if(UseH4)  Timeframes[index++] = PERIOD_H4;
    if(UseD1)  Timeframes[index++] = PERIOD_D1;
    if(UseW1)  Timeframes[index++] = PERIOD_W1;
    if(UseMN1) Timeframes[index++] = PERIOD_MN1;
}

//+------------------------------------------------------------------+
//| Custom indicator iteration function                              |
//+------------------------------------------------------------------+
int OnCalculate(const int rates_total,
                const int prev_calculated,
                const datetime &time[],
                const double &open[],
                const double &high[],
                const double &low[],
                const double &close[],
                const long &tick_volume[],
                const long &volume[],
                const int &spread[])
{
    // Check if there's enough data
    if(rates_total < LookbackPeriod)
        return 0;
    
    // Process data for each timeframe
    string commentText = "Multi-Timeframe Regime Analysis\n\n";
    
    for(int i = 0; i < TimeframeCount; i++)
    {
        // Get price data for this timeframe
        double tfClose[];
        ArraySetAsSeries(tfClose, true);
        int copied = CopyClose(Symbol(), Timeframes[i], 0, LookbackPeriod, tfClose);
        
        if(copied != LookbackPeriod)
        {
            Print("Failed to copy price data for timeframe ", EnumToString(Timeframes[i]));
            continue;
        }
        
        // Process data with the detector
        if(!Detectors[i].ProcessData(tfClose, LookbackPeriod))
        {
            Print("Failed to process data for timeframe ", EnumToString(Timeframes[i]));
            continue;
        }
        
        // Add timeframe information to comment
        commentText += TimeframeToString(Timeframes[i]) + ": ";
        commentText += Detectors[i].GetRegimeDescription();
        commentText += " (Trend: " + DoubleToString(Detectors[i].GetTrendStrength(), 2);
        commentText += ", Vol: " + DoubleToString(Detectors[i].GetVolatility(), 2) + ")";
        commentText += "\n";
    }
    
    // Display the multi-timeframe analysis
    Comment(commentText);
    
    // Return the number of calculated bars
    return rates_total;
}

//+------------------------------------------------------------------+
//| Convert timeframe enum to readable string                        |
//+------------------------------------------------------------------+
string TimeframeToString(ENUM_TIMEFRAMES timeframe)
{
    switch(timeframe)
    {
        case PERIOD_M1:  return "M1";
        case PERIOD_M5:  return "M5";
        case PERIOD_M15: return "M15";
        case PERIOD_M30: return "M30";
        case PERIOD_H1:  return "H1";
        case PERIOD_H4:  return "H4";
        case PERIOD_D1:  return "D1";
        case PERIOD_W1:  return "W1";
        case PERIOD_MN1: return "MN1";
        default:         return "Unknown";
    }
}

//+------------------------------------------------------------------+
//| Custom indicator deinitialization function                       |
//+------------------------------------------------------------------+
void OnDeinit(const int reason)
{
    // Clean up
    for(int i = 0; i < TimeframeCount; i++)
    {
        if(Detectors[i] != NULL)
        {
            delete Detectors[i];
            Detectors[i] = NULL;
        }
    }
    
    // Clear the comment
    Comment("");
}
