//+------------------------------------------------------------------+
//|                                               MarketRegimeEA.mq5 |
//|                                                      Sahil Bagdi |
//|                         https://www.mql5.com/en/users/sahilbagdi |
//+------------------------------------------------------------------+
#property copyright "Sahil Bagdi"
#property link      "https://www.mql5.com/en/users/sahilbagdi"
#property version   "1.00"
#property strict

// Include the Market Regime Detector
#include <MarketRegimeEnum.mqh>
#include <MarketRegimeDetector.mqh>

// EA input parameters
input int      LookbackPeriod = 100;       // Lookback period for calculations
input int      SmoothingPeriod = 10;       // Smoothing period for regime transitions
input double   TrendThreshold = 0.2;       // Threshold for trend detection (0.1-0.5)
input double   VolatilityThreshold = 1.5;  // Threshold for volatility detection (1.0-3.0)

// Trading parameters
input double   TrendingLotSize = 0.1;      // Lot size for trending regimes
input double   RangingLotSize = 0.05;      // Lot size for ranging regimes
input double   VolatileLotSize = 0.02;     // Lot size for volatile regimes
input int      TrendingStopLoss = 100;     // Stop loss in points for trending regimes
input int      RangingStopLoss = 50;       // Stop loss in points for ranging regimes
input int      VolatileStopLoss = 150;     // Stop loss in points for volatile regimes
input int      TrendingTakeProfit = 200;   // Take profit in points for trending regimes
input int      RangingTakeProfit = 80;     // Take profit in points for ranging regimes
input int      VolatileTakeProfit = 300;   // Take profit in points for volatile regimes

// Global variables
CMarketRegimeDetector *Detector = NULL;
int OnBarCount = 0;
datetime LastBarTime = 0;

//+------------------------------------------------------------------+
//| Expert initialization function                                   |
//+------------------------------------------------------------------+
int OnInit() {
    // Create and initialize the Market Regime Detector
    Detector = new CMarketRegimeDetector(LookbackPeriod, SmoothingPeriod);
    if(Detector == NULL) {
        Print("Failed to create Market Regime Detector");
        return INIT_FAILED;
    }

    // Configure the detector
    Detector.SetTrendThreshold(TrendThreshold);
    Detector.SetVolatilityThreshold(VolatilityThreshold);
    Detector.Initialize();

    // Initialize variables
    OnBarCount = 0;
    LastBarTime = 0;

    return INIT_SUCCEEDED;
}

//+------------------------------------------------------------------+
//| Expert deinitialization function                                 |
//+------------------------------------------------------------------+
void OnDeinit(const int reason) {
    // Clean up
    if(Detector != NULL) {
        delete Detector;
        Detector = NULL;
    }

    // Clear the comment
    Comment("");
}

//+------------------------------------------------------------------+
//| Expert tick function                                             |
//+------------------------------------------------------------------+
void OnTick() {
    // Check for new bar
    datetime currentBarTime = iTime(Symbol(), PERIOD_CURRENT, 0);
    if(currentBarTime == LastBarTime)
        return; // No new bar

    LastBarTime = currentBarTime;
    OnBarCount++;

    // Wait for enough bars to accumulate
    if(OnBarCount < LookbackPeriod) {
        Comment("Accumulating data: ", OnBarCount, " of ", LookbackPeriod, " bars");
        return;
    }

    // Get price data
    double close[];
    ArraySetAsSeries(close, true);
    int copied = CopyClose(Symbol(), PERIOD_CURRENT, 0, LookbackPeriod, close);

    if(copied != LookbackPeriod) {
        Print("Failed to copy price data: copied = ", copied, " of ", LookbackPeriod);
        return;
    }

    // Process data with the detector
    if(!Detector.ProcessData(close, LookbackPeriod)) {
        Print("Failed to process data with Market Regime Detector");
        return;
    }

    // Get current market regime
    ENUM_MARKET_REGIME currentRegime = Detector.GetCurrentRegime();

    // Display current regime information
    string regimeText = "Current Market Regime: " + Detector.GetRegimeDescription();
    string trendText = "Trend Strength: " + DoubleToString(Detector.GetTrendStrength(), 4);
    string volatilityText = "Volatility: " + DoubleToString(Detector.GetVolatility(), 4);

    Comment(regimeText + "\n" + trendText + "\n" + volatilityText);

    // Execute trading strategy based on market regime
    ExecuteRegimeBasedStrategy(currentRegime);
}

//+------------------------------------------------------------------+
//| Execute trading strategy based on market regime                  |
//+------------------------------------------------------------------+
void ExecuteRegimeBasedStrategy(ENUM_MARKET_REGIME regime) {
    // Check if we already have open positions
    if(PositionsTotal() > 0)
        return; // Don't open new positions if we already have one

    // Get current market information
    double ask = SymbolInfoDouble(Symbol(), SYMBOL_ASK);
    double bid = SymbolInfoDouble(Symbol(), SYMBOL_BID);
    double point = SymbolInfoDouble(Symbol(), SYMBOL_POINT);

    // Determine trading parameters based on regime
    double lotSize = 0.0;
    int stopLoss = 0;
    int takeProfit = 0;
    ENUM_ORDER_TYPE orderType = ORDER_TYPE_BUY;

    switch(regime) {
    case REGIME_TRENDING_UP: {
        lotSize = TrendingLotSize;
        stopLoss = TrendingStopLoss;
        takeProfit = TrendingTakeProfit;
        orderType = ORDER_TYPE_BUY; // Buy in uptrend
        break;
    }

    case REGIME_TRENDING_DOWN: {
        lotSize = TrendingLotSize;
        stopLoss = TrendingStopLoss;
        takeProfit = TrendingTakeProfit;
        orderType = ORDER_TYPE_SELL; // Sell in downtrend
        break;
    }

    case REGIME_RANGING: {
        // In ranging markets, we can use mean-reversion strategies
        // For simplicity, we'll use RSI to determine overbought/oversold
        double rsi[];
        ArraySetAsSeries(rsi, true);
        int rsiCopied = CopyBuffer(iRSI(Symbol(), PERIOD_CURRENT, 14, PRICE_CLOSE), 0, 0, 2, rsi);

        if(rsiCopied != 2)
            return;

        lotSize = RangingLotSize;
        stopLoss = RangingStopLoss;
        takeProfit = RangingTakeProfit;

        if(rsi[0] < 30) // Oversold
            orderType = ORDER_TYPE_BUY;
        else if(rsi[0] > 70) // Overbought
            orderType = ORDER_TYPE_SELL;
        else
            return; // No signal
        break;
    }

    case REGIME_VOLATILE: {
        // In volatile markets, we can use breakout strategies
        // For simplicity, we'll use Bollinger Bands
        double upper[], lower[];
        ArraySetAsSeries(upper, true);
        ArraySetAsSeries(lower, true);

        int bbCopied1 = CopyBuffer(iBands(Symbol(), PERIOD_CURRENT, 20, 2, 0, PRICE_CLOSE), 1, 0, 2, upper);
        int bbCopied2 = CopyBuffer(iBands(Symbol(), PERIOD_CURRENT, 20, 2, 0, PRICE_CLOSE), 2, 0, 2, lower);

        if(bbCopied1 != 2 || bbCopied2 != 2)
            return;

        lotSize = VolatileLotSize;
        stopLoss = VolatileStopLoss;
        takeProfit = VolatileTakeProfit;

        double close[];
        ArraySetAsSeries(close, true);
        int copied = CopyClose(Symbol(), PERIOD_CURRENT, 0, 2, close);

        if(copied != 2)
            return;

        if(close[1] < upper[1] && close[0] > upper[0]) // Breakout above upper band
            orderType = ORDER_TYPE_BUY;
        else if(close[1] > lower[1] && close[0] < lower[0]) // Breakout below lower band
            orderType = ORDER_TYPE_SELL;
        else
            return; // No signal

        break;
    }
    default:
        return; // No trading in undefined regime
    }

    // Calculate stop loss and take profit levels
    double slLevel = 0.0;
    double tpLevel = 0.0;

    if(orderType == ORDER_TYPE_BUY) {
        slLevel = ask - stopLoss * point;
        tpLevel = ask + takeProfit * point;
    } else if(orderType == ORDER_TYPE_SELL) {
        slLevel = bid + stopLoss * point;
        tpLevel = bid - takeProfit * point;
    }

    // Execute trade
    MqlTradeRequest request;
    MqlTradeResult result;

    ZeroMemory(request);
    ZeroMemory(result);

    request.action = TRADE_ACTION_DEAL;
    request.symbol = Symbol();
    request.volume = lotSize;
    request.type = orderType;
    request.price = (orderType == ORDER_TYPE_BUY) ? ask : bid;
    request.sl = slLevel;
    request.tp = tpLevel;
    request.deviation = 10;
    request.magic = 123456; // Magic number for this EA
    request.comment = "Market Regime: " + Detector.GetRegimeDescription();
    request.type_filling = ORDER_FILLING_FOK;

    bool success = OrderSend(request, result);

    if(success) {
        Print("Trade executed successfully: ", result.retcode, " ", result.comment);
    } else {
        Print("Trade execution failed: ", result.retcode, " ", result.comment);
    }
}
//+------------------------------------------------------------------+
