//+------------------------------------------------------------------+
//|                                         SignalBollingerBands.mqh |
//|                             Copyright 2000-2024, MetaQuotes Ltd. |
//|                                             https://www.mql5.com |
//+------------------------------------------------------------------+
#include <Expert\ExpertSignal.mqh>
// wizard description start
//+------------------------------------------------------------------+
//| Description of the class                                         |
//| Title=Signals of indicator 'Bollinger Bands'                     |
//| Type=SignalAdvanced                                              |
//| Name=Bollinger Bands                                             |
//| ShortName=BollingerBands                                         |
//| Class=CSignalBollingerBands                                      |
//| Page=signal_bollingerbands                                       |
//| Parameter=PeriodMA,int,20,Period of averaging                    |
//| Parameter=Shift,int,0,Time shift                                 |
//| Parameter=Applied,ENUM_APPLIED_PRICE,PRICE_CLOSE,Prices series   |
//| Parameter=Deviation,double,2.0,Deviation                         |
//| Parameter=Pattern_0,int,50,Pattern 0                             |
//| Parameter=Pattern_1,int,50,Pattern 1                             |
//| Parameter=Pattern_2,int,50,Pattern 2                             |
//| Parameter=Pattern_3,int,50,Pattern 3                             |
//| Parameter=Pattern_4,int,50,Pattern 4                             |
//| Parameter=Pattern_5,int,50,Pattern 5                             |
//| Parameter=Pattern_6,int,50,Pattern 6                             |
//| Parameter=Pattern_7,int,50,Pattern 7                             |
//| Parameter=PatternsUsed,uchar,255,Patterns Used BitMap            |
//+------------------------------------------------------------------+
// wizard description end
//+------------------------------------------------------------------+
//| Class CSignalBollingerBands.                                     |
//| Purpose: Class of generator of trade signals based on            |
//|          the 'BollingerBands' indicator.                         |
//| Is derived from the CExpertSignal class.                         |
//+------------------------------------------------------------------+
class CSignalBollingerBands : public CExpertSignal
{
protected:
   CiBands           m_bands;            // object-indicator
   //--- adjusted parameters
   int               m_ma_period;      // the "period of averaging" parameter of the indicator
   int               m_ma_shift;       // the "time shift" parameter of the indicator
   ENUM_APPLIED_PRICE m_ma_applied;    // the "object of averaging" parameter of the indicator
   double            m_deviation;      // the "deviation" parameter of the indicator
   double            m_limit_in;       // threshold sensitivity of the 'rollback zone'
   double            m_limit_out;      // threshold sensitivity of the 'break through zone'
   //--- "weights" of market models (0-100)
   int               m_pattern_0;      // model 0 "Price Crossing the Upper Band or the Lower Band"
   int               m_pattern_1;      // model 1 "Price Bouncing Off Lower Band or Upper Band "
   int               m_pattern_2;      // model 2 "Price Squeeze Followed by a Breakout Above Upper Band or Below Lower Band "
   int               m_pattern_3;      // model 3 "Price Double Bottoms Near Lower Band or Double Top Near Upper Band "
   int               m_pattern_4;      // model 4 "Price Bounces Off the Middle Band from Above & Bounce Off from Below "
   int               m_pattern_5;      // model 5 "Volume Divergence at Lower Band or Upper Band "
   int               m_pattern_6;      // model 6 "Bands Widening After Downtrend or After Uptrend "
   int               m_pattern_7;      // model 7 "Bands Orientation and Angle Changes "
   uchar             m_patterns_used;  // bit-map integer for used pattens

public:
                     CSignalBollingerBands(void);
                    ~CSignalBollingerBands(void);
   //--- methods of setting adjustable parameters
   void              PeriodMA(int value)
   {  m_ma_period = value;
   }
   void              Shift(int value)
   {  m_ma_shift = value;
   }
   void              Applied(ENUM_APPLIED_PRICE value)
   {  m_ma_applied = value;
   }
   void              Deviation(double value)
   {  m_deviation = value;
   }
   void              LimitIn(double value)
   {  m_limit_in = value;
   }
   void              LimitOut(double value)
   {  m_limit_out = value;
   }
   //--- methods of adjusting "weights" of market models
   void              Pattern_0(int value)
   {  m_pattern_0 = value;
   }
   void              Pattern_1(int value)
   {  m_pattern_1 = value;
   }
   void              Pattern_2(int value)
   {  m_pattern_2 = value;
   }
   void              Pattern_3(int value)
   {  m_pattern_3 = value;
   }
   void              Pattern_4(int value)
   {  m_pattern_4 = value;
   }
   void              Pattern_5(int value)
   {  m_pattern_5 = value;
   }
   void              Pattern_6(int value)
   {  m_pattern_6 = value;
   }
   void              Pattern_7(int value)
   {  m_pattern_7 = value;
   }
   void              PatternsUsed(int value)
   {                 PatternsUsage(value);
   }
   //--- method of verification of settings
   virtual bool      ValidationSettings(void);
   //--- method of creating the indicator and timeseries
   virtual bool      InitIndicators(CIndicators *indicators);
   //--- methods of checking if the market models are formed
   virtual int       LongCondition(void);
   virtual int       ShortCondition(void);

protected:
   //--- method of initialization of the indicator
   bool              InitMA(CIndicators *indicators);
   //--- methods of getting data
   double            Upper(int ind)
   {                 return(m_bands.Upper(ind));
   }
   double            Lower(int ind)
   {                 return(m_bands.Lower(ind));
   }
   double            Base(int ind)
   {                 return(m_bands.Base(ind));
   }
   double            Close(int ind)
   {                 return(m_close.GetData(ind));
   }
   double            Range(int ind)
   {                 return(m_high.GetData(ind) - m_low.GetData(ind));
   }
   double            Gap(int ind)
   {                 return(Upper(ind) - Lower(ind));
   }
   //--- methods to check for patterns
   bool              IsPattern_0(ENUM_POSITION_TYPE T);
   bool              IsPattern_1(ENUM_POSITION_TYPE T);
   bool              IsPattern_2(ENUM_POSITION_TYPE T);
   bool              IsPattern_3(ENUM_POSITION_TYPE T);
   bool              IsPattern_4(ENUM_POSITION_TYPE T);
   bool              IsPattern_5(ENUM_POSITION_TYPE T);
   bool              IsPattern_6(ENUM_POSITION_TYPE T);
   bool              IsPattern_7(ENUM_POSITION_TYPE T);
};
//+------------------------------------------------------------------+
//| Constructor                                                      |
//+------------------------------------------------------------------+
CSignalBollingerBands::CSignalBollingerBands(void) : m_ma_period(20),
   m_ma_shift(0),
   m_ma_applied(PRICE_CLOSE),
   m_deviation(2.0),
   m_limit_in(0.2),
   m_limit_out(0.2),
   m_pattern_0(50),
   m_pattern_1(50),
   m_pattern_2(50),
   m_pattern_3(50),
   m_pattern_4(50),
   m_pattern_5(50),
   m_pattern_6(50),
   m_pattern_7(50),
   m_patterns_used(255)
{
//--- initialization of protected data
   m_used_series = USE_SERIES_OPEN + USE_SERIES_HIGH + USE_SERIES_LOW + USE_SERIES_CLOSE;
   PatternsUsage(m_patterns_used);
}
//+------------------------------------------------------------------+
//| Destructor                                                       |
//+------------------------------------------------------------------+
CSignalBollingerBands::~CSignalBollingerBands(void)
{
}
//+------------------------------------------------------------------+
//| Validation settings protected data.                              |
//+------------------------------------------------------------------+
bool CSignalBollingerBands::ValidationSettings(void)
{
//--- validation settings of additional filters
   if(!CExpertSignal::ValidationSettings())
      return(false);
//--- initial data checks
   if(m_ma_period <= 0)
   {  printf(__FUNCTION__ + ": period MA must be greater than 0");
      return(false);
   }
//--- ok
   return(true);
}
//+------------------------------------------------------------------+
//| Create indicators.                                               |
//+------------------------------------------------------------------+
bool CSignalBollingerBands::InitIndicators(CIndicators *indicators)
{
//--- check pointer
   if(indicators == NULL)
      return(false);
//--- initialization of indicators and timeseries of additional filters
   if(!CExpertSignal::InitIndicators(indicators))
      return(false);
//--- create and initialize MA indicator
   if(!InitMA(indicators))
      return(false);
//--- ok
   return(true);
}
//+------------------------------------------------------------------+
//| Initialize MA indicators.                                        |
//+------------------------------------------------------------------+
bool CSignalBollingerBands::InitMA(CIndicators *indicators)
{
//--- check pointer
   if(indicators == NULL)
      return(false);
//--- add object to collection
   if(!indicators.Add(GetPointer(m_bands)))
   {  printf(__FUNCTION__ + ": error adding object");
      return(false);
   }
//--- initialize object
   if(!m_bands.Create(m_symbol.Name(), m_period, m_ma_period, m_ma_shift, m_deviation, m_ma_applied))
   {  printf(__FUNCTION__ + ": error initializing object");
      return(false);
   }
//--- ok
   return(true);
}
//+------------------------------------------------------------------+
//| "Voting" that price will grow.                                   |
//+------------------------------------------------------------------+
int CSignalBollingerBands::LongCondition(void)
{  int result  = 0, results = 0;
//--- if the model 0 is used and "Price Crossing the Upper Band or the Lower Band"
   if(((m_patterns_used & 0x01) != 0) && IsPattern_0(POSITION_TYPE_BUY))
   {  result += m_pattern_0;
      results++;
   }
//--- if the model 1 is used and "Price Bouncing Off Lower Band or Upper Band "
   if(((m_patterns_used & 0x02) != 0) && IsPattern_1(POSITION_TYPE_BUY))
   {  result += m_pattern_1;
      results++;
   }
//--- if the model 2 is used and "Price Squeeze Followed by a Breakout Above Upper Band or Below Lower Band "
   if(((m_patterns_used & 0x04) != 0) && IsPattern_2(POSITION_TYPE_BUY))
   {  result += m_pattern_2;
      results++;
   }
//--- if the model 3 is used and "Price Double Bottoms Near Lower Band or Double Top Near Upper Band "
   if(((m_patterns_used & 0x08) != 0) && IsPattern_3(POSITION_TYPE_BUY))
   {  result += m_pattern_3;
      results++;
   }
//--- if the model 4 is used and "Price Bounces Off the Middle Band from Above & Bounce Off from Below "
   if(((m_patterns_used & 0x10) != 0) && IsPattern_4(POSITION_TYPE_BUY))
   {  result += m_pattern_4;
      results++;
   }
//--- if the model 5 is used and "Volume Divergence at Lower Band or Upper Band  "
   if(((m_patterns_used & 0x20) != 0) && IsPattern_5(POSITION_TYPE_BUY))
   {  result += m_pattern_5;
      results++;
   }
//--- if the model 6 is used and "Bands Widening After Downtrend or After Uptrend "
   if(((m_patterns_used & 0x40) != 0) && IsPattern_6(POSITION_TYPE_BUY))
   {  result += m_pattern_6;
      results++;
   }
//--- if the model 7 is used and "Bands Orientation and Angle Changes "
   if(((m_patterns_used & 0x80) != 0) && IsPattern_7(POSITION_TYPE_BUY))
   {  result += m_pattern_7;
      results++;
   }
//--- return the result
   if(results > 0)
   {  return(int(round(result / results)));
   }
   return(0);
}
//+------------------------------------------------------------------+
//| "Voting" that price will fall.                                   |
//+------------------------------------------------------------------+
int CSignalBollingerBands::ShortCondition(void)
{  int result  = 0, results = 0;
//--- if the model 0 is used and "Price Crossing the Upper Band or the Lower Band"
   if(((m_patterns_used & 0x01) != 0) && IsPattern_0(POSITION_TYPE_SELL))
   {  result += m_pattern_0;
      results++;
   }
//--- if the model 1 is used and "Price Bouncing Off Lower Band or Upper Band "
   if(((m_patterns_used & 0x02) != 0) && IsPattern_1(POSITION_TYPE_SELL))
   {  result += m_pattern_1;
      results++;
   }
//--- if the model 2 is used and "Price Squeeze Followed by a Breakout Above Upper Band or Below Lower Band "
   if(((m_patterns_used & 0x04) != 0) && IsPattern_2(POSITION_TYPE_SELL))
   {  result += m_pattern_2;
      results++;
   }
//--- if the model 3 is used and "Price Double Bottoms Near Lower Band or Double Top Near Upper Band "
   if(((m_patterns_used & 0x08) != 0) && IsPattern_3(POSITION_TYPE_SELL))
   {  result += m_pattern_3;
      results++;
   }
//--- if the model 4 is used and "Price Bounces Off the Middle Band from Above & Bounce Off from Below "
   if(((m_patterns_used & 0x10) != 0) && IsPattern_4(POSITION_TYPE_SELL))
   {  result += m_pattern_4;
      results++;
   }
//--- if the model 5 is used and "Volume Divergence at Lower Band or Upper Band  "
   if(((m_patterns_used & 0x20) != 0) && IsPattern_5(POSITION_TYPE_SELL))
   {  result += m_pattern_5;
      results++;
   }
//--- if the model 6 is used and "Bands Widening After Downtrend or After Uptrend "
   if(((m_patterns_used & 0x40) != 0) && IsPattern_6(POSITION_TYPE_SELL))
   {  result += m_pattern_6;
      results++;
   }
//--- if the model 7 is used and "Bands Orientation and Angle Changes "
   if(((m_patterns_used & 0x80) != 0) && IsPattern_7(POSITION_TYPE_SELL))
   {  result += m_pattern_7;
      results++;
   }
//--- return the result
   if(results > 0)
   {  return(int(round(result / results)));
   }
   return(0);
}
//+------------------------------------------------------------------+
//| Check for Pattern 0.                                             |
//+------------------------------------------------------------------+
bool CSignalBollingerBands::IsPattern_0(ENUM_POSITION_TYPE T)
{  m_bands.Refresh(-1);
   m_close.Refresh(-1);
   if(T == POSITION_TYPE_BUY && Close(StartIndex() + 1) < Lower(StartIndex() + 1) && Close(StartIndex()) > Lower(StartIndex()))
   {  return(true);
   }
   else if(T == POSITION_TYPE_SELL && Close(StartIndex() + 1) > Upper(StartIndex() + 1) && Close(StartIndex()) < Upper(StartIndex()))
   {  return(true);
   }
   return(false);
}
//+------------------------------------------------------------------+
//| Check for Pattern 1.                                             |
//+------------------------------------------------------------------+
bool CSignalBollingerBands::IsPattern_1(ENUM_POSITION_TYPE T)
{  m_bands.Refresh(-1);
   m_close.Refresh(-1);
   if(T == POSITION_TYPE_BUY && Close(StartIndex() + 2) > Lower(StartIndex() + 2) && Close(StartIndex() + 1) <= Lower(StartIndex() + 1) && Close(StartIndex()) > Lower(StartIndex()))
   {  return(true);
   }
   else if(T == POSITION_TYPE_SELL && Close(StartIndex() + 2) < Upper(StartIndex() + 2) && Close(StartIndex() + 1) >= Upper(StartIndex() + 1) && Close(StartIndex()) < Upper(StartIndex()))
   {  return(true);
   }
   return(false);
}
//+------------------------------------------------------------------+
//| Check for Pattern 2.                                             |
//+------------------------------------------------------------------+
bool CSignalBollingerBands::IsPattern_2(ENUM_POSITION_TYPE T)
{  m_bands.Refresh(-1);
   m_close.Refresh(-1);
   if(Upper(StartIndex()) > Upper(StartIndex() + 1) && Upper(StartIndex() + 1) < Upper(StartIndex() + 2)  && Lower(StartIndex()) < Lower(StartIndex() + 1) && Lower(StartIndex() + 1) > Lower(StartIndex() + 2))
   {  if(T == POSITION_TYPE_BUY && Close(StartIndex()) >= Upper(StartIndex()))
      {  return(true);
      }
      else if(T == POSITION_TYPE_SELL && Close(StartIndex()) <= Lower(StartIndex()))
      {  return(true);
      }
   }
   return(false);
}
//+------------------------------------------------------------------+
//| Check for Pattern 3.                                             |
//+------------------------------------------------------------------+
bool CSignalBollingerBands::IsPattern_3(ENUM_POSITION_TYPE T)
{  m_bands.Refresh(-1);
   m_close.Refresh(-1);
   m_high.Refresh(-1);
   m_low.Refresh(-1);
   if
   (
      T == POSITION_TYPE_BUY &&
      Close(StartIndex() + 4) < Close(StartIndex() + 3) &&
      Close(StartIndex() + 3) > Close(StartIndex() + 2) &&
      Close(StartIndex() + 2) < Close(StartIndex() + 1) &&
      Close(StartIndex() + 1) > Close(StartIndex()) &&
      Close(m_close.MinIndex(StartIndex(), 5)) - Upper(StartIndex()) >= -1.0 * Range(StartIndex())
   )
   {  return(true);
   }
   else if
   (
      T == POSITION_TYPE_SELL &&
      Close(StartIndex() + 4) > Close(StartIndex() + 3) &&
      Close(StartIndex() + 3) < Close(StartIndex() + 2) &&
      Close(StartIndex() + 2) > Close(StartIndex() + 1) &&
      Close(StartIndex() + 1) < Close(StartIndex()) &&
      Lower(StartIndex()) - Close(m_close.MaxIndex(StartIndex(), 5)) >= -1.0 * Range(StartIndex())
   )
   {  return(true);
   }
   return(false);
}
//+------------------------------------------------------------------+
//| Check for Pattern 4.                                             |
//+------------------------------------------------------------------+
bool CSignalBollingerBands::IsPattern_4(ENUM_POSITION_TYPE T)
{  m_bands.Refresh(-1);
   m_close.Refresh(-1);
   if(T == POSITION_TYPE_BUY && Close(StartIndex() + 2) > Base(StartIndex() + 2) && Close(StartIndex() + 1) <= Base(StartIndex() + 1) && Close(StartIndex()) > Base(StartIndex()))
   {  return(true);
   }
   else if(T == POSITION_TYPE_SELL && Close(StartIndex() + 2) < Base(StartIndex() + 2) && Close(StartIndex() + 1) >= Base(StartIndex() + 1) && Close(StartIndex()) < Base(StartIndex()))
   {  return(true);
   }
   return(false);
}
//+------------------------------------------------------------------+
//| Check for Pattern 5.                                             |
//+------------------------------------------------------------------+
bool CSignalBollingerBands::IsPattern_5(ENUM_POSITION_TYPE T)
{  m_bands.Refresh(-1);
   m_close.Refresh(-1);
   m_high.Refresh(-1);
   m_low.Refresh(-1);
   if(Range(StartIndex()) < Range(StartIndex() + 1) && Range(StartIndex() + 1) < Range(StartIndex() + 2))
   {  if(T == POSITION_TYPE_BUY && fabs(m_close.GetData(StartIndex()) - Lower(StartIndex())) < fabs(m_close.GetData(StartIndex()) - Base(StartIndex())))
      {  return(true);
      }
      else if(T == POSITION_TYPE_SELL && fabs(m_close.GetData(StartIndex()) - Upper(StartIndex())) < fabs(m_close.GetData(StartIndex()) - Base(StartIndex())))
      {  return(true);
      }
   }
   return(false);
}
//+------------------------------------------------------------------+
//| Check for Pattern 6.                                             |
//+------------------------------------------------------------------+
bool CSignalBollingerBands::IsPattern_6(ENUM_POSITION_TYPE T)
{  m_bands.Refresh(-1);
   m_close.Refresh(-1);
   m_high.Refresh(-1);
   m_low.Refresh(-1);
   if(Gap(StartIndex()) > Gap(StartIndex() + 1) && Gap(StartIndex() + 1) > Range(StartIndex() + 2))
   {  if(T == POSITION_TYPE_BUY && m_close.GetData(StartIndex() + 2) < m_close.GetData(StartIndex() + 3) && m_close.GetData(StartIndex() + 3) < m_close.GetData(StartIndex() + 4))
      {  return(true);
      }
      else if(T == POSITION_TYPE_SELL && m_close.GetData(StartIndex() + 2) > m_close.GetData(StartIndex() + 3) && m_close.GetData(StartIndex() + 3) > m_close.GetData(StartIndex() + 4))
      {  return(true);
      }
   }
   return(false);
}
//+------------------------------------------------------------------+
//| Check for Pattern 7.                                             |
//+------------------------------------------------------------------+
bool CSignalBollingerBands::IsPattern_7(ENUM_POSITION_TYPE T)
{  m_bands.Refresh(-1);
   m_close.Refresh(-1);
   m_high.Refresh(-1);
   m_low.Refresh(-1);
   if
   (
      T == POSITION_TYPE_BUY &&
      Lower(StartIndex()) > Lower(StartIndex() + 1) &&
      Lower(StartIndex() + 1) > Lower(StartIndex() + 2) &&
      Upper(StartIndex()) >= Upper(StartIndex() + 1) &&
      Upper(StartIndex() + 1) >= Upper(StartIndex() + 2) &&
      Lower(StartIndex()) - Lower(StartIndex() + 2) >= m_deviation * (Upper(StartIndex()) - Upper(StartIndex() + 2))
   )
   {  return(true);
   }
   else if
   (
      T == POSITION_TYPE_SELL &&
      Upper(StartIndex()) < Upper(StartIndex() + 1) &&
      Upper(StartIndex() + 1) < Upper(StartIndex() + 2) &&
      Lower(StartIndex()) <= Lower(StartIndex() + 1) &&
      Lower(StartIndex() + 1) <= Lower(StartIndex() + 2) &&
      Upper(StartIndex() + 2) - Upper(StartIndex()) >= m_deviation * (Lower(StartIndex() + 2) - Lower(StartIndex()))
   )
   {  return(true);
   }
   return(false);
}
//+------------------------------------------------------------------+
