//+------------------------------------------------------------------+
//|                                                           np.mqh |
//|                                  Copyright 2024, MetaQuotes Ltd. |
//|                                             https://www.mql5.com |
//+------------------------------------------------------------------+
#property copyright "Copyright 2024, MetaQuotes Ltd."
#property link      "https://www.mql5.com"
#include<Math\Alglib\linalg.mqh>
#define BEGIN 0
#define STEP 1
#define END LONG_MAX
#define BEGIN_REVERSE -1
#define STEP_REVERSE -1
#define END_REVERSE LONG_MIN

namespace np
{
namespace internal
{
//+------------------------------------------------------------------+
//|calculate size of container                                       |
//+------------------------------------------------------------------+
long calculatesize(long initial,long stop,long step)
  {
//---calculate len
   long len = fabs(initial-stop)/fabs(step);
//---
   if(len*fabs(step) < fabs(initial-stop))
      len += fabs(initial-stop) - len*fabs(step);
//---
   return len;
  }
//+------------------------------------------------------------------+
//| process start parameter                                          |
//+------------------------------------------------------------------+
long normalizestart(long start,ulong size)
  {
   long s = (start<0 && ulong(fabs(start))<size)?long(size)+start:(start<0 && ulong(fabs(start))>=size)?0:start;
   return s;
  }
//+------------------------------------------------------------------+
//|process stop parameter                                            |
//+------------------------------------------------------------------+
long normalizestop(long stop, ulong size)
  {
   long s = (stop>0 && ulong(stop)>=size)?long(size):(stop<0 && ulong(fabs(stop))>size)?-1:(stop<0 && ulong(fabs(stop))<=size)?long(size)+stop:stop;
   return s;
  }
//+------------------------------------------------------------------+
//| validate all calculated params                                   |
//+------------------------------------------------------------------+
bool invalidparams(long start,long stop, long step, ulong size)
  {
   if(step==0 || start==stop || ulong(start)>=size || start<0 || ulong(fabs(stop))>size || (step<0 && start<stop) || (step>0 && start>stop))
      return true;
   return false;
  }
}
//+------------------------------------------------------------------+
//| The function sorts (in) vector and indices[] simultaneously      |
//| using QuickSort algorithm. After sorting the initial ordering    |
//| of the elements is located in the indices[] array.               |
//|                                                                  |
//| Arguments:                                                       |
//| in          : vector with values to sort                         |
//| ascending   : direction of sort (true:ascending)                 |
//| (false:descending)                                               |
//| first       : First element index                                |
//| last        : Last element index                                 |
//|                                                                  |
//| Return value: None                                               |
//+------------------------------------------------------------------+
template <typename T>
bool quickSort(vector<T> &in,bool ascending,long first,long last)
  {
   long    i,j;
   T p_double,t_double;
//--- check
   if(first<0 || last<0)
     {
      return false;
     }
//--- sort
   i=first;
   j=last;
   while(i<last)
     {
      //--- ">>1" is quick division by 2
      p_double=in[(first+last)>>1];
      while(i<j)
        {
         while((ascending && in[i]<p_double) || (!ascending && in[i]>p_double))
           {
            //--- control the output of the in bounds
            if(i==in.Size()-1)
               break;
            i++;
           }
         while((ascending && in[j]>p_double) || (!ascending && in[j]<p_double))
           {
            //--- control the output of the in bounds
            if(j==0)
               break;
            j--;
           }
         if(i<=j)
           {
            //-- swap elements i and j
            t_double=in[i];
            in[i]=in[j];
            in[j]=t_double;
            //--
            i++;
            //--- control the output of the in bounds
            if(j==0)
               break;
            j--;
           }
        }
      if(first<j)
         quickSort(in,ascending,first,j);
      first=i;
      j=last;
     }

   return true;
  }
//+------------------------------------------------------------------+
//| The function sorts (in) vector and indices[] simultaneously      |
//| using QuickSort algorithm. After sorting the initial ordering    |
//| of the elements is located in the indices[] array.               |
//|                                                                  |
//| Arguments:                                                       |
//| in          : vector with values to sort                         |
//| ascending   : direction of sort (true:ascending)                 |
//| (false:descending)                                               |
//| indices     : array of indexes                                   |
//| first       : First element index                                |
//| last        : Last element index                                 |
//|                                                                  |
//| Return value: None                                               |
//+------------------------------------------------------------------+
template <typename T>
bool quickSortIndices(vector<T> &in,bool ascending,long &indices[],long first,long last)
  {
   long    i,j,t_int;
   T p_double,t_double;
//--- check
   if(first<0 || last<0)
     {
      return false;
     }
//--- sort
   i=first;
   j=last;
   while(i<last)
     {
      //--- ">>1" is quick division by 2
      p_double=in[(first+last)>>1];
      while(i<j)
        {
         while((ascending && in[i]<p_double) || (!ascending && in[i]>p_double))
           {
            //--- control the output of the in bounds
            if(i==in.Size()-1)
               break;
            i++;
           }
         while((ascending && in[j]>p_double) || (!ascending && in[j]<p_double))
           {
            //--- control the output of the in bounds
            if(j==0)
               break;
            j--;
           }
         if(i<=j)
           {
            //-- swap elements i and j
            t_double=in[i];
            in[i]=in[j];
            in[j]=t_double;
            //-- swap indices i and j
            t_int=indices[i];
            indices[i]=indices[j];
            indices[j]=t_int;
            i++;
            //--- control the output of the in bounds
            if(j==0)
               break;
            j--;
           }
        }
      if(first<j)
         quickSortIndices(in,ascending,indices,first,j);
      first=i;
      j=last;
     }
   return true;
  }

//+------------------------------------------------------------------+
//| sort a vector                                                    |
//+------------------------------------------------------------------+
template<typename T>
bool sort(vector<T>&in, bool ascending=true)
  {
   if(in.Size()<1)
      return false;

   return quickSort(in,ascending,0,in.Size()-1);
  }
//+------------------------------------------------------------------+
//| sort matrix by columns or rows                                   |
//+------------------------------------------------------------------+
template<typename T>
bool sort(matrix<T>&in, bool by_rows, bool ascending=true)
  {
   if(by_rows)
     {
      for(ulong i = 0; i<in.Rows(); i++)
        {
         vector row = in.Row(i);
         if(!sort(row,ascending))
            return false;
         in.Row(row,i);
        }
     }
   else
     {
      for(ulong i = 0; i<in.Cols(); i++)
        {
         vector col = in.Col(i);
         if(!sort(col,ascending))
            return false;
         in.Col(col,i);
        }
     }
   return true;
  }
//+------------------------------------------------------------------+
//|  extract a portion of vector within a range at specific interval |
//|  with support for negative indexing python style                 |
//| returns new vector from start to stop-1 at step intervals        |
//| start - index of first element (inclusive)                       |
//| stop  - index past last element (exclusive)                      |
//| step  - step size                                                |
//+------------------------------------------------------------------+
template<typename T>
vector<T> sliceVector(vector<T> &in, const long index_start=BEGIN, const long index_stop = END, const long index_step = STEP)
  {
//---local variables
   long st,sp,stp,size;
//---check function parameters
   st = internal::normalizestart(index_start,in.Size());
   sp = internal::normalizestop(index_stop,in.Size());
   stp = index_step;
//---error handling
   if(internal::invalidparams(st,sp,stp,in.Size()))
     {
      Print(__FUNCTION__," invalid function parameters ");
      return vector<T>::Zeros(0);
     }
//---calculate size of new vector
   size = internal::calculatesize(st,sp,stp);
//---initialize output vector
   vector<T> out(ulong(size));
//---assign elements to output
   for(long pos = st, index=0; index<size; index++,pos+=stp)
      out[index] = in[pos];
//---done
   return out;
  }

//+------------------------------------------------------------------+
//| reverse a vector                                                 |
//+------------------------------------------------------------------+
template<typename T>
bool reverseVector(vector<T> &in)
  {
   if(in.Size()<2)
      return in.Size()!=0;

   ulong size = in.Size()-1;
   ulong half = (in.Size())>>1;
   for(ulong i = 0; i<half; i++)
     {
      T temp = in[i];
      in[i] = in[size - i];
      in[size - i] = temp;
     }
   return true;
  }
//+------------------------------------------------------------------+
//|  extract a portion a matrix within a range at specific intervals |
//|  with support for negative indexing python style                 |
//| returns new matrix                                               |
//| row_start - row index of first row (inclusive)                   |
//| row_stop  - row index past last row (exclusive)                  |
//| row_step  - row step size                                        |
//| column_start - columnindex of first column (inclusive)           |
//| column_stop  - column index past last columun (exclusive)        |
//| column_step  - column step size                                  |
//+------------------------------------------------------------------+
template<typename T>
matrix<T> sliceMatrix(matrix<T> &in, const long row_start=BEGIN, const long row_stop = END, const long row_step = STEP,const long column_start=BEGIN, const long column_stop = END, const long column_step = STEP)
  {
//---local variables
   long rst,rsp,rstp,rsize,cst,csp,cstp,csize;
//---check function parameters for row manipulation
   rst =internal::normalizestart(row_start,in.Rows());
   rsp =internal::normalizestop(row_stop,in.Rows());
   rstp = row_step;
//---error handling
   if(internal::invalidparams(rst,rsp,rstp,in.Rows()))
     {
      Print(__FUNCTION__," invalid function parameters for row specification ");
      return matrix<T>::Zeros(0,0);
     }
//---calculate size of new matrix rows
   rsize = internal::calculatesize(rst,rsp,rstp);
//---check function parameters for row manipulation
   cst = internal::normalizestart(column_start,in.Cols());
   csp = internal::normalizestop(column_stop,in.Cols());
   cstp = column_step;
//---error handling
   if(internal::invalidparams(cst,csp,cstp,in.Cols()))
     {
      Print(__FUNCTION__," invalid function parameters for column specification ");
      return matrix<T>::Zeros(0,0);
     }
//---calculate size of new matrix columns
   csize = internal::calculatesize(cst,csp,cstp);
//---initialize output matrix
   matrix<T> out(ulong(rsize),ulong(csize));
//---assign elements to output
   for(long rpos = rst,row_index = 0; row_index<rsize; rpos+=rstp,row_index++)
      for(long cpos = cst,column_index = 0; column_index<csize; column_index++,cpos+=cstp)
         out[row_index][column_index] = in[rpos][cpos];
//---done
   return out;
  }
//+------------------------------------------------------------------+
//|  extract a portion of matrix by selecting range of columns and   |
//|  optional interval                                               |
//|  with support for negative indexing python style                 |
//| returns new matrix                                               |
//| column_start - columnindex of first column (inclusive)           |
//| column_stop  - column index past last columun (exclusive)        |
//| column_step  - column step size                                  |
//+------------------------------------------------------------------+
template<typename T>
matrix<T> sliceMatrixCols(matrix<T> &in, const long column_start=BEGIN, const long column_stop = END, const long column_step = STEP)
  {
   return sliceMatrix(in,BEGIN,END,STEP,column_start,column_stop,column_step);
  }
//+------------------------------------------------------------------+
//|reverse columns of matrix                                         |
//+------------------------------------------------------------------+
template<typename T>
bool reverseMatrixCols(matrix<T> &in)
  {

   ulong cols = in.Cols()-1;

   if(in.Cols()<2)
      return in.Cols()!=0;

   ulong cols_end = (in.Cols())>>1;

   for(ulong i = 0; i<cols_end; i++)
      if(!in.SwapCols(i,cols-i))
         return false;

   return true;
  }
//+------------------------------------------------------------------+
//|  extract a portion of matrix by selecting range of rows and      |
//|  optional interval                                               |
//|  with support for negative indexing python style                 |
//| returns new matrix                                               |
//| row_start - row index of first row (inclusive)                   |
//| row_stop  - row index past last row (exclusive)                  |
//| row_step  - row step size                                        |
//+------------------------------------------------------------------+
template<typename T>
matrix<T> sliceMatrixRows(matrix<T> &in, const long row_start=BEGIN, const long row_stop = END, const long row_step = STEP)
  {
   return sliceMatrix(in,row_start,row_stop,row_step);
  }
//+------------------------------------------------------------------+
//|reverse rows of matrix                                            |
//+------------------------------------------------------------------+
template<typename T>
bool reverseMatrixRows(matrix<T> &in)
  {
   ulong rows = in.Rows()-1;

   if(in.Rows()<2)
      return in.Rows()!=0;

   ulong rows_end = (in.Rows())>>1;

   for(ulong i = 0; i<rows_end; i++)
      if(!in.SwapRows(i,rows-i))
         return false;

   return true;
  }
//+------------------------------------------------------------------+
//|  assign values to a portion of matrix selected within range      |
//|  of columns and                                                  |
//|  optional interval                                               |
//|  with support for negative indexing python style                 |
//| dest_start - column index of first column (inclusive)            |
//| dest_stop  - column index past last columun (exclusive)          |
//| dest_step  - column step size                                    |
//| indexes refer to positions in copyto                             |
//+------------------------------------------------------------------+
template <typename T>
bool matrixCopyCols(matrix<T> &copyto,matrix &copyfrom, const long dest_start=BEGIN, const long dest_stop = END, const long dest_step = STEP)
  {
   return matrixCopy(copyto,copyfrom,BEGIN,END,STEP,dest_start,dest_stop,dest_step);
  }
//+------------------------------------------------------------------+
//|  assign values to a portion of matrix selected within range      |
//|  of rows and                                                     |
//|  optional interval                                               |
//|  with support for negative indexing python style                 |
//| dest_start - row index of first row (inclusive)                  |
//| dest_stop  - row index past last row (exclusive)                 |
//| dest_step  - row step size                                       |
//| indexes refer to positions in copyto                             |
//+------------------------------------------------------------------+
template <typename T>
bool matrixCopyRows(matrix<T> &copyto, matrix &copyfrom, const long dest_start=BEGIN, const long dest_stop = END, const long dest_step = STEP)
  {
   return matrixCopy(copyto,copyfrom,dest_start,dest_stop,dest_step);
  }
//+------------------------------------------------------------------+
//|return vector of arbitrary elements specified as  collection      |
//| of indices                                                       |
//+------------------------------------------------------------------+
template <typename T>
vector<T> select(vector<T> &in,const ulong &indices[])
  {
   ulong size = (ulong)MathMin(in.Size(),indices.Size());

   if(size<1 || indices[ArrayMaximum(indices,0,int(size))]>=in.Size())
     {
      Print(__FUNCTION__, " invalid function parameter ");
      return vector<T> vector::Zeros(0);
     }

   vector<T> out(size);

   for(ulong i=0; i<size; i++)
      out[i] = in[indices[i]];

   return out;
  }

//+------------------------------------------------------------------+
//|return  arbitrary columns of a matrix specified as                |
//|collection of indices                                             |
//+------------------------------------------------------------------+
template <typename T>
matrix<T> selectMatrixCols(matrix<T> &in,const ulong &indices[])
  {
   ulong size = (ulong)MathMin(in.Cols(),indices.Size());

   if(size<1 || indices[ArrayMaximum(indices,0,int(size))]>=in.Cols())
     {
      Print(__FUNCTION__," invalid function parameters ");
      return matrix<T>::Zeros(0,0);
     }

   matrix<T> out(in.Rows(),size);

   for(ulong i=0; i<size; i++)
      out.Col(in.Col(indices[i]),i);

   return out;
  }
//+------------------------------------------------------------------+
//|return  arbitrary rows of a matrix specified as                   |
//|collection of indices                                             |
//+------------------------------------------------------------------+
template <typename T>
matrix<T> selectMatrixRows(matrix<T> &in,const ulong &indices[])
  {
   ulong size = (ulong)MathMin(in.Rows(),indices.Size());

   if(size<1  || indices[ArrayMaximum(indices,0,int(size))]>=in.Rows())
     {
      Print(__FUNCTION__," invalid function parameters ");
      return matrix<T>::Zeros(0,0);
     }

   matrix<T> out(size,in.Cols());

   for(ulong i=0; i<size; i++)
      out.Row(in.Row(indices[i]),i);

   return out;
  }
//+------------------------------------------------------------------+
//|copy vector copyfrom into specified positions of vector copyto    |
//|with support for negative indexing python style                   |
//+------------------------------------------------------------------+
template <typename T>
bool vectorCopy(vector<T> &copyto,vector<T> &copyfrom,const long dest_start=BEGIN, const long dest_stop = END, const long dest_step = STEP)
  {
//---local variables
   long st,sp,stp,size;
//---check function parameters
   st = internal::normalizestart(dest_start,copyto.Size());
   sp = internal::normalizestop(dest_stop,copyto.Size());
   stp = dest_step;
//---error handling
   if(internal::invalidparams(st,sp,stp,copyto.Size()))
     {
      Print(__FUNCTION__," invalid function parameters ");
      return false;
     }
//---calculate size of new vector
   size = internal::calculatesize(st,sp,stp);
//---error handling
   if(copyfrom.Size()<ulong(size))
     {
      Print(__FUNCTION__," source vector is of insufficient size ");
      return false;
     }
//---assign elements
   for(long pos = st,index=0; index<size; index++,pos+=stp)
      copyto[pos] = copyfrom[index];
//---done
   return true;
  }

//+------------------------------------------------------------------+
//| assign selected elements of vector the value /fill_value/        |
//+------------------------------------------------------------------+
template <typename T>
bool vectorFill(vector<T> &copyto, T fill_value,const long dest_start=BEGIN, const long dest_stop = END, const long dest_step = STEP)
  {
//---local variables
   long st,sp,stp,size;
//---check function parameters
   st = internal::normalizestart(dest_start,copyto.Size());
   sp = internal::normalizestop(dest_stop,copyto.Size());
   stp = dest_step;
//---error handling
   if(internal::invalidparams(st,sp,stp,copyto.Size()))
     {
      Print(__FUNCTION__," invalid function parameters ");
      return vector<T>::Zeros(0);
     }
//---calculate size of new vector
   size = internal::calculatesize(st,sp,stp);
//---assign elements
   for(long pos = st,index=0; index<size; index++,pos+=stp)
      copyto[pos] = fill_value;
//---done
   return true;
  }
//+------------------------------------------------------------------+
//|copy matrix copyfrom into specified positions of matrix copyto    |
//|with support for negative indexing python style                   |
//+------------------------------------------------------------------+
template<typename T>
bool matrixCopy(matrix<T> &copyto,matrix &copyfrom, const long row_dest_start=BEGIN, const long row_dest_stop = END, const long row_dest_step = STEP,const long column_dest_start=BEGIN, const long column_dest_stop = END, const long column_dest_step = STEP)
  {
//---local variables
   long rst,rsp,rstp,rsize,cst,csp,cstp,csize;
//---check function parameters for row manipulation
   rst =internal::normalizestart(row_dest_start,copyto.Rows());
   rsp =internal::normalizestop(row_dest_stop,copyto.Rows());
   rstp = row_dest_step;
//---error handling
   if(internal::invalidparams(rst,rsp,rstp,copyto.Rows()))
     {
      Print(__FUNCTION__," invalid function parameters for row specification ");
      return matrix<T>::Zeros(0,0);
     }
//---calculate size of new matrix rows
   rsize = internal::calculatesize(rst,rsp,rstp);
//---check function parameters for row manipulation
   cst = internal::normalizestart(column_dest_start,copyto.Cols());
   csp = internal::normalizestop(column_dest_stop,copyto.Cols());
   cstp = column_dest_step;
//---error handling
   if(internal::invalidparams(cst,csp,cstp,copyto.Cols()))
     {
      Print(__FUNCTION__," invalid function parameters for column specification ");
      return matrix<T>::Zeros(0,0);
     }
//---calculate size of new matrix columns
   csize = internal::calculatesize(cst,csp,cstp);
//---error handling
   if(copyfrom.Cols() < ulong(csize) || copyfrom.Rows() < ulong(rsize))
     {
      Print(__FUNCTION__, " source matrix has insufficient number of columns and/or rows ");
      return false;
     }
//---assign elements
   for(long rpos = rst,row_index = 0; row_index<rsize; rpos+=rstp,row_index++)
      for(long cpos = cst,column_index = 0; column_index<csize; column_index++,cpos+=cstp)
         copyto[rpos][cpos]=copyfrom[row_index][column_index];
//---done
   return true;
  }
//+------------------------------------------------------------------+
//| assign selected elements of vector the value /fill_value/        |
//+------------------------------------------------------------------+
template<typename T>
bool matrixFill(matrix<T> &copyto,T fill_value, const long row_dest_start=BEGIN, const long row_dest_stop = END, const long row_dest_step = STEP,const long column_dest_start=BEGIN, const long column_dest_stop = END, const long column_dest_step = STEP)
  {
//---local variables
   long rst,rsp,rstp,rsize,cst,csp,cstp,csize;
//---check function parameters for row manipulation
   rst =internal::normalizestart(row_dest_start,copyto.Rows());
   rsp =internal::normalizestop(row_dest_stop,copyto.Rows());
   rstp = row_dest_step;
//---error handling
   if(internal::invalidparams(rst,rsp,rstp,copyto.Rows()))
     {
      Print(__FUNCTION__," invalid function parameters for row specification ");
      return matrix<T>::Zeros(0,0);
     }
//---calculate size of new matrix rows
   rsize = internal::calculatesize(rst,rsp,rstp);
//---check function parameters for row manipulation
   cst = internal::normalizestart(column_dest_start,copyto.Cols());
   csp = internal::normalizestop(column_dest_stop,copyto.Cols());
   cstp = column_dest_step;
//---error handling
   if(internal::invalidparams(cst,csp,cstp,copyto.Cols()))
     {
      Print(__FUNCTION__," invalid function parameters for column specification ");
      return matrix<T>::Zeros(0,0);
     }
//---calculate size of new matrix columns
   csize = internal::calculatesize(cst,csp,cstp);
//---assign elements
   for(long rpos = rst,row_index = 0; row_index<rsize; rpos+=rstp,row_index++)
      for(long cpos = cst,column_index = 0; column_index<csize; column_index++,cpos+=cstp)
         copyto[rpos][cpos]=fill_value;
//---done
   return true;
  }
//+------------------------------------------------------------------+
//| returns vector                                                   |
//+------------------------------------------------------------------+
template<typename T>
vector arange(const ulong size,  T value = 0.0, T step_value = 1.0)
  {
   vector<T> out(size);

   for(ulong i = 0; i<size; i++, value+=step_value)
      out[i] = value;

   return out;
  }

//+------------------------------------------------------------------+
//| outputs array                                                    |
//+------------------------------------------------------------------+
template<typename T>
bool arange(T &in_out[],const int size,  T value = 0, T step_value = 1)
  {
   if(!size)
      return false;

   if(ArrayResize(in_out,fabs(size))!=fabs(size))
     {
      Print(__FUNCTION__, " error ", GetLastError());
      return false;
     }

   for(int i = 0; i<size; i++, value+=step_value)
      in_out[i] = value;

   return true;
  }
//+------------------------------------------------------------------+
//| fill diagonal                                                    |
//+------------------------------------------------------------------+
template<typename T>
bool fillDiagonal(matrix<T> &in_out, T fill_value)
  {
   if(in_out.Rows()!=in_out.Cols())
     {
      Print(__FUNCTION__ "Invalid Input. Matrix is not square ");
      return false;
     }

   for(ulong i = 0; i<in_out.Rows(); i++)
      in_out[i][i] = fill_value;

   return true;
  }
//+-----------------------------------------------------------------------+
//| Generate a Vandermonde matrix                                         |
//| in : vector                                                           |
//| num_columns : ulong,  default: size of in input vector.               |
//| ascending : bool, (default: false) Order of the powers of the columns.|
//| If true, the powers increase                                          |
//| from left to right, if false (the default) they are reversed          |
//+-----------------------------------------------------------------------+
template<typename T>
matrix<T> vander(vector<T> &in, ulong num_columns = 0,bool ascending = false)
  {
   ulong n = num_columns?num_columns:in.Size();

   matrix<T> out(in.Size(),n);

   for(ulong i = 0; i<n; i++)
      out.Col(pow(in,ascending?i:n-1-i),i);

   return out;
  }
//+------------------------------------------------------------------+
//|generic vector to array                                           |
//+------------------------------------------------------------------+
template<typename T>
bool vecAsArray(const vector<T> &in, T &out[])
  {
//---
   if(in.Size()<1)
     {
      Print(__FUNCTION__," Empty vector");
      return false;
     }
//---
   if(ulong(out.Size())!=in.Size() && ArrayResize(out,int(in.Size()))!=int(in.Size()))
     {
      Print(__FUNCTION__," resize error ", GetLastError());
      return false;
     }
//---
   for(uint i = 0; i<out.Size(); i++)
      out[i]=in[i];
//---
   return true;
//---
  }
//+------------------------------------------------------------------+
//|generic matrix to 1-D array                                       |
//+------------------------------------------------------------------+
template<typename T>
bool matAsArray(const matrix<T> &in, T &out[])
  {
//---
   if(in.Cols()+in.Rows()<1)
     {
      Print(__FUNCTION__," Empty vector");
      return false;
     }
//---
   if(ulong(out.Size())!=in.Cols()*in.Rows() && ArrayResize(out,int(in.Cols()*in.Rows()))!=int(in.Cols()*in.Rows()))
     {
      Print(__FUNCTION__," resize error ", GetLastError());
      return false;
     }
//---
   for(uint i = 0; i<out.Size(); i++)
      out[i]=in.Flat(i);
//---
   return true;
//---
  }
//+------------------------------------------------------------------+
//| difference vector according to parameter degreee                 |
//+------------------------------------------------------------------+
template<typename T>
vector<T> diff(vector<T> &in, ulong degree = 1)
  {
//---
   if(in.Size()<1 || degree<1 || in.Size()<degree+1)
     {
      Print(__FUNCTION__," Invalid parameter ");
      return vector<T>::Zeros(0);
     }
//---
   vector<T> yy,zz;
//---
   yy =  sliceVector(in,long(degree));
   zz =  sliceVector(in,BEGIN,long(in.Size()-degree));
//---
   return yy-zz;
  }
//+------------------------------------------------------------------+
//| difference matrix about the rows or columns                      |
//+------------------------------------------------------------------+
template<typename T>
matrix<T> diff(matrix<T> &in, ulong degree = 1, bool by_row = true)
  {
//---
   if(in.Cols()+in.Rows()<1 || degree<1 || (by_row && in.Cols()<degree+1) || (!by_row && in.Rows()<degree+1))
     {
      Print(__FUNCTION__," Invalid function parameter ");
      return matrix::Zeros(0,0);
     }
//---
   matrix<T> yy,zz;
//---
   if(by_row)
     {
      yy =  sliceMatrix(in,BEGIN,END,STEP,long(degree));
      zz =  sliceMatrix(in,BEGIN,END,STEP,BEGIN,long(in.Cols()-degree));
     }
   else
     {
      yy =  sliceMatrix(in,long(degree));
      zz =  sliceMatrix(in,BEGIN,long(in.Rows()-degree));
     }
//---
   return yy-zz;
  }

//+------------------------------------------------------------------+
//| generate vector of num_repetitions of input vector elements      |
//+------------------------------------------------------------------+
template<typename T>
vector<T> repeat_vector_elements(vector<T> &vect,ulong num_repetitions)
  {
   vector<T> out(num_repetitions*vect.Size());

   for(ulong i = 0; i<out.Size(); i+=vect.Size())
      vectorCopy(out,vect,i,i+vect.Size());

   return out;
  }
//+------------------------------------------------------------------+
//| generate vector of num_repetitions of input matrix elements      |
//+------------------------------------------------------------------+
template<typename T>
vector<T> repeat_matrix_elements(matrix<T> &in, ulong num_repetitions)
  {
   vector<T> out(num_repetitions*in.Cols()*in.Rows());

   for(ulong i = 0; i<out.Size(); i+=(in.Cols()*num_repetitions))
     {
      vector copy = repeat_vector_elements(in.Row(i/(in.Cols()*num_repetitions)),num_repetitions);
      vectorCopy(out,copy,i,i+copy.Size());
     }

   return out;
  }
//+------------------------------------------------------------------+
//| generate matrix by repeating vector as rows or columns           |
//+------------------------------------------------------------------+
template<typename T>
matrix<T> repeat_vector_as_rows_cols(vector<T>&in, ulong num_repetitions, bool as_rows = true)
  {
   matrix<T> out(as_rows?num_repetitions:in.Size(),as_rows?in.Size():num_repetitions);

   for(ulong i = 0; i<num_repetitions; i++)
      as_rows?out.Row(in,i):out.Col(in,i);

   return out;
  }
//+------------------------------------------------------------------+
//| generate matrix of num_repetitions of input columns or rows      |
//+------------------------------------------------------------------+
template<typename T>
matrix<T> repeat_matrix_rows_cols(matrix<T> &in, ulong num_repetitions, bool by_rows)
  {
   if(num_repetitions<1)
     {
      Print(__FUNCTION__, " invalid function parameter ");
      return matrix<T>::Zeros(0,0);
     }

   matrix<T> out((by_rows)?num_repetitions*in.Rows():in.Rows(),(!by_rows)?num_repetitions*in.Cols():in.Cols());

   if(by_rows)
      for(ulong i = 0; i<out.Rows(); i++)
         out.Row(in.Row(i/num_repetitions),i);
   else
      for(ulong i = 0; i<out.Cols(); i++)
         out.Col(in.Col(i/num_repetitions),i);

   return out;
  }

//+------------------------------------------------------------------+
//| generate matrix of an arbitrary number of repetitions of each row|
//| or column as specified by elements of input array                |
//+------------------------------------------------------------------+
template<typename T>
matrix<T> repeat_matrix_rows_cols(matrix<T> &in, ulong &reps[],bool by_rows)
  {
   ulong max = 0 ;

   for(uint i = 0; i<reps.Size(); i++)
      max+=reps[i];


   if((by_rows && in.Rows()!=ulong(reps.Size())) ||
      (!by_rows && in.Cols()!=ulong(reps.Size()))||
      reps[ArrayMinimum(reps)]<1)
     {
      Print(__FUNCTION__, " invalid function parameters ");
      return matrix<T>::Zeros(0,0);
     }

   matrix<T> out((by_rows)?max:in.Rows(),(!by_rows)?max:in.Cols());

   if(by_rows)
      for(ulong i = 0, j = 0; i<out.Rows(); i+=reps[j],j++)
        {
         matrix sliced =sliceRows(in,j,j+1);
         matrix rpl = repeat_matrix_rows_cols(sliced,reps[j],true);
         copyRows(out,rpl,i,i+reps[j]);
        }
   else
      for(ulong i = 0, j = 0; i<out.Cols(); i+=reps[j],j++)
        {
         matrix sliced =sliceCols(in,j,j+1);
         matrix rpl = repeat_matrix_rows_cols(sliced,reps[j],false);
         copyCols(out,rpl,i,i+reps[j]);
        }

   return out;
  }
//+------------------------------------------------------------------+
//| The function extracts the unique values from the vector.         |
//|                                                                  |
//| Arguments:                                                       |
//| in     : vector with values                                      |
//|                                                                  |
//|                                                                  |
//| Return value: out vector of unique values  .                     |
//+------------------------------------------------------------------+
template<typename T>
vector<T> unique(vector<T> &in)
  {
//--- check array size
   ulong size=in.Size();
   if(size==0)
     {
      Print(__FUNCTION__," Invalid function parameter ");
      return(vector<T>::Zeros(0));
     }

//--- prepare additional in
   bool checked[];
   if(ArrayResize(checked,int(size))!=int(size))
     {
      Print(__FUNCTION__," error ",GetLastError());
      return(vector<T>::Zeros(0));
     }
   ArrayFill(checked,0,int(size),false);

//--- prepare out in
   vector<T>out(size);
//--- find unique elements
   ulong unique_count=0;
   T value=0;
   while(true)
     {
      bool flag=false;
      for(ulong i=unique_count; i<size; i++)
        {
         if(!flag && !checked[i])
           {
            value=in[i];
            out[unique_count]=in[i];
            unique_count++;
            checked[i]=true;
            flag=true;
           }
         else
            if(flag && value==in[i])
               checked[i]=true;
        }
      if(!flag)
         break;
     }
//--- resize target in
   out.Resize(unique_count);
//---
   return(out);
  }
//+------------------------------------------------------------------+
//| The function extracts the unique values from the vector.         |
//|                                                                  |
//| Arguments:                                                       |
//| in     : vector with values                                      |
//| out    : vector of unique values                                 |
//| indices: index positions in original vector                      |
//| counts : number of occurances of each unique value               |
//| Return bool: true on success and false on error                  |
//+------------------------------------------------------------------+
template<typename T>
bool  unique(vector<T> &in, vector<T> &out, long &indices[],long &counts[])
  {
//--- check vector size
   ulong size=in.Size();
   if(size==0)
     {
      Print(__FUNCTION__," Invalid function parameter ");
      return false;
     }

//--- prepare internal and output buffers
   bool checked[];
   if(ArrayResize(checked,int(size))!=int(size) ||
      ArrayResize(indices,int(size))!=int(size) ||
      ArrayResize(counts,int(size))!=int(size) ||
      !out.Resize(size))
     {
      Print(__FUNCTION__, " error ", GetLastError());
      return(false);
     }
   ArrayFill(checked,0,int(size),false);
   ArrayFill(indices,0,int(size),-1);
   ArrayFill(counts,0,int(size),1);

//--- prepare out vector
   out.Resize(size);
//--- find unique elements
   ulong unique_count=0;
   T value=0;
   while(true)
     {
      bool flag=false;
      for(ulong i=unique_count; i<size; i++)
        {
         if(!flag && !checked[i])
           {
            value=in[i];
            out[unique_count]=in[i];
            indices[unique_count]=long(i);
            unique_count++;
            checked[i]=true;
            flag=true;
           }
         else
            if(flag && value==in[i])
              {
               checked[i]=true;
               counts[unique_count-1]++;
              }
        }
      if(!flag)
         break;
     }
//---
   return (out.Resize(unique_count) && ArrayResize(indices,int(unique_count))==int(unique_count) && ArrayResize(counts,int(unique_count))==int(unique_count));
//---
  }
//+------------------------------------------------------------------+
//| numpy pinverse implementation with optional cutoff               |
//+------------------------------------------------------------------+
bool pinv(matrix &in, matrix &out, double cutoff=1.e-15)
  {
   matrix a = in;
   matrix u,v;
   vector s;
   
   if(!a.SVD(u,v,s))
     {
      Print(__FUNCTION__, " error ", GetLastError());
      return false;
     }
     
   double max = s.Max();
   cutoff*=max;

   for(ulong i=0; i<s.Size(); i++)
     {
      if(s[i]>cutoff)
         s[i]=1.0/s[i];
      else
         s[i]=0.0;
     }
     
   out = v.MatMul((np::repeat_vector_as_rows_cols(s,u.Transpose().Cols(),false))*(u.Transpose()));
   return true;
  }
}
//+------------------------------------------------------------------+

//+------------------------------------------------------------------+
