//+------------------------------------------------------------------+
//|                                             PrincipalFactors.mq5 |
//|                                  Copyright 2024, MetaQuotes Ltd. |
//|                                             https://www.mql5.com |
//+------------------------------------------------------------------+
#property copyright "Copyright 2024, MetaQuotes Ltd."
#property link      "https://www.mql5.com"
#property version   "1.00"
#include<pfa.mqh>
#include<ErrorDescription.mqh>
#include<EasyAndFastGUI\WndCreate.mqh>
//+------------------------------------------------------------------+
//|    which matrix                                                  |
//+------------------------------------------------------------------+
enum SELECT_VIEW
  {
   ENUM_DATA=0,
   ENUM_FL,
   ENUM_RFL,
   ENUM_CORR
  };
//+------------------------------------------------------------------+
//|indicator type                                                    |
//+------------------------------------------------------------------+
enum SELECT_INDICATOR
  {
   ATR=0,//ATR
   MA//MA
  };
//--- input parameters
input uint     period_inc=2;//lookback increment
input uint     max_lookback=50;
input ENUM_MA_METHOD         AppliedMA = MODE_SMA;
input ENUM_APPLIED_PRICE     AppliedPrice = PRICE_CLOSE;
input datetime SampleStartDate=D'2019.12.31';
input datetime SampleStopDate=D'2022.12.31';
input string   SetSymbol="BTCUSD";
input ENUM_TIMEFRAMES SetTF = PERIOD_D1;

//+------------------------------------------------------------------+
//| Gui application class                                            |
//+------------------------------------------------------------------+
class CApp:public CWndCreate
  {
protected:

   CWindow           m_window;                    // main window
   CTable            m_table;                     // matrix viewer
   CTextLabel        m_kmostat;                   //kmo text label
   CTextLabel        m_bstpvalue;                 //bst text label
   CComboBox         m_appliedrotation;           //factor rotation option
   CComboBox         m_view;                      //which matrix to view
   Cpfa              m_fa;                        //local Cpfa object
   double            m_KMO,m_pvalue;              //test statistics
   string            m_csv_header;                //csv file header
   string            m_col_labels[];              //column headings
   string            m_row_labels[];              //row labels
   string            m_factor_labels[];           //factor labels
   int               size_sample,                 //training set size
                     size_observations,           //size of of both training and testing sets combined
                     maxperiod,                   //maximum lookback
                     indicator_handle;            //long moving average indicator handle
   vector            indicator[];                 // indicator values;
   matrix            feature_matrix,              //full matrix of features;
                     factorloadings,              //factor loading matrix
                     rfactorloadings;             // rotated factor loadings
   int               c_selected,r_selected;       //utility flags

   bool              CollectData(void);
   bool              FillTable(matrix &vdata);
   bool              SetRowLabels(string &rowlabels[],int shift=0);
   bool              SetColLabels(string &collabels[]);
   void              Reset(int view,int rotate);

public:
                     CApp(void);                 //constructor
                    ~CApp(void);                 //destructor

   void              OnInitEvent(void);
   void              OnDeinitEvent(const int reason);

   virtual void      OnEvent(const int id, const long &lparam, const double &dparam,const string &sparam);

   bool              CreateGUI(void);


  };
//+------------------------------------------------------------------+
//| Constructor                                                      |
//+------------------------------------------------------------------+
CApp::CApp(void)
  {
   m_csv_header = "    ";
   indicator_handle=INVALID_HANDLE;
   c_selected=0;
   r_selected=-1;
  }
//+------------------------------------------------------------------+
//| Destructor                                                       |
//+------------------------------------------------------------------+
CApp::~CApp(void)
  {

  }
//+------------------------------------------------------------------+
//| On initialization                                                |
//+------------------------------------------------------------------+
void CApp::OnInitEvent(void)
  {
  }
//+------------------------------------------------------------------+
//| On DeInitilization                                               |
//+------------------------------------------------------------------+
void CApp::OnDeinitEvent(const int reason)
  {
   CWndEvents::Destroy();
  }
//+------------------------------------------------------------------+
//| Event handler                                                    |
//+------------------------------------------------------------------+
void CApp::OnEvent(const int id,const long &lparam,const double &dparam,const string &sparam)
  {
   if(id==CHARTEVENT_CUSTOM+ON_CLICK_COMBOBOX_ITEM)
     {
      int n_select=m_view.GetListViewPointer().SelectedItemIndex();
      int rn_select=m_appliedrotation.GetListViewPointer().SelectedItemIndex();
      if((lparam==m_appliedrotation.Id() && n_select && n_select<3 && rn_select!=r_selected) || (lparam == m_view.Id() && n_select!=c_selected))
         Reset(n_select,rn_select);
     }
   return;
  }
//+------------------------------------------------------------------+
//|  Create the graphical user interface                             |
//+------------------------------------------------------------------+
bool CApp::CreateGUI(void)
  {
//---first check and initialize main buffer
   if(!CollectData())
      return(false);
//---initialize window creation
   if(!CWndCreate::CreateWindow(m_window,"Principal Factor Analysis",1,1,700,600,true,true,true,false))
      return(false);
//---create combobox1
   string rots[] = {"Varimax","Promax"};
   if(!CWndCreate::CreateCombobox(m_appliedrotation,"Factor Rotation",m_window,0,false,7,25,280,180,rots,50,0))
      return(false);
//---create combobox2
   string which[] = {"Data Set","Factor Loadings","Rotated Factor Loadings","Correlation Matrix"};
   if(!CWndCreate::CreateCombobox(m_view,"Select Matrix ",m_window,0,false,300,25,300,180,which,75,0))
      return(false);
//---Kmo test result text label
   if(!CWndCreate::CreateTextLabel(m_kmostat,"KMO Statistic:  "+DoubleToString(m_KMO),m_window,0,7,50,200))
      return(false);
//---BST pvalue
   if(!CWndCreate::CreateTextLabel(m_bstpvalue,"Bartlett's Sphericity P-value:  "+DoubleToString(m_pvalue),m_window,0,300,50,200))
      return(false);
//---create table
   if(!CWndCreate::CreateTable(m_table,m_window,0,int(feature_matrix.Cols()+1),int(feature_matrix.Rows()),m_col_labels,0,75,700,450,false,false))
      return(false);
//---fill table
   if(!FillTable(feature_matrix))
      return(false);
//---set row labels
   if(!SetRowLabels(m_row_labels))
      return(false);
//---init events
   CWndEvents::CompletedGUI();
//---
   return(true);
  }

//+------------------------------------------------------------------+
//| Initialize data buffers                                          |
//+------------------------------------------------------------------+
bool CApp::CollectData(void)
  {
//---get relative shift of sample set
   int samplestart,samplestop,num_features;
   samplestart=iBarShift(SetSymbol!=""?SetSymbol:NULL,SetTF,SampleStartDate);
   samplestop=iBarShift(SetSymbol!=""?SetSymbol:NULL,SetTF,SampleStopDate);
   num_features = int((max_lookback/period_inc)*2);
//---check for errors from ibarshift calls
   if(samplestart<0 || samplestop<0)
     {
      Print(ErrorDescription(GetLastError()));
      return false;
     }
//---set the size of the sample sets
   size_observations=(samplestart - samplestop) + 1 ;
   maxperiod=int(max_lookback);
//---check for input errors
   if(size_observations<=0 || maxperiod<=0)
     {
      Print("Invalid inputs ");
      return false;
     }
//---allocate memory
   if(ArrayResize(indicator,num_features)<num_features)
     {
      Print(ErrorDescription(GetLastError()));
      return false;
     }
//----get the full collection of indicator values
   int period_len;
   int k=0;
//---
   for(SELECT_INDICATOR select_indicator = 0; select_indicator<2; select_indicator++)
     {
      for(int iperiod=0; iperiod<int(indicator.Size()/2); iperiod++)
        {
         period_len=int((iperiod+1) * period_inc);
         m_csv_header+=","+EnumToString(select_indicator)+"_"+string(period_len);
         int try
               =10;
         while(try)
           {
            switch(select_indicator)
              {
               case ATR:
                  indicator_handle=iATR(SetSymbol!=""?SetSymbol:NULL,SetTF,period_len);
                  break;
               case MA:
                  indicator_handle=iMA(SetSymbol!=""?SetSymbol:NULL,SetTF,period_len,0,AppliedMA,AppliedPrice);
                  break;
              }

            if(indicator_handle==INVALID_HANDLE)
               try
                  --;
            else
               break;
           }

         if(indicator_handle==INVALID_HANDLE)
           {
            Print("Invalid indicator handle ",EnumToString(select_indicator)," ", GetLastError());
            return false;
           }

         try
               = 0;
         while(!indicator[k].CopyIndicatorBuffer(indicator_handle,0,samplestop,size_observations) && try
                  <10)
              {
               try
                  ++;
               Sleep(5000);
              }
         if(try
               <10)
               ++k;
         else
           {
            Print("error copying to indicator buffers ",GetLastError());
            return false;
           }

         if(indicator_handle!=INVALID_HANDLE && IndicatorRelease(indicator_handle))
            indicator_handle=INVALID_HANDLE;
        }
     }

//---resize matrix
   if(!feature_matrix.Resize(size_observations,indicator.Size()))
     {
      Print(ErrorDescription(GetLastError()));
      return false;
     }
//---copy collected data to matrix
   for(ulong i = 0; i<feature_matrix.Cols(); i++)
      if(!feature_matrix.Col(indicator[i],i))
        {
         Print(ErrorDescription(GetLastError()));
         return false;
        }
//---
   int headings = StringSplit(m_csv_header,StringGetCharacter(",",0),m_col_labels);
//---
   if(ArrayResize(m_row_labels,int(feature_matrix.Rows()))!=int(feature_matrix.Rows()) ||
      ArrayResize(m_factor_labels,int(m_col_labels.Size()))!=int(m_col_labels.Size()))
     {
      Print(ErrorDescription(GetLastError()));
      return false;
     }
//---
   m_factor_labels[0] = "     ";
   for(uint i = 1; i<m_factor_labels.Size(); i++)
      m_factor_labels[i] = "Factor "+IntegerToString(i);
//---
   for(uint i = 0; i<m_row_labels.Size(); i++)
      m_row_labels[i] = IntegerToString(i);
//---
   if(headings!=int(feature_matrix.Cols()+1))
     {
      Print(ErrorDescription(GetLastError()));
      return false;
     }
//---kmo test
   vector kmo_vect;
   kmo(feature_matrix,kmo_vect,m_KMO);
//---Bartlett sphericity test
   double bs_stat;
   bartlet_sphericity(feature_matrix,bs_stat,m_pvalue);
//---Extract the principal factors
   if(!m_fa.fit(feature_matrix))
      return false;;
//---
   factorloadings = m_fa.get_factor_loadings();
   vector cumvar = m_fa.get_cum_var_contributions();
//---
   for(uint i = 1; i<m_factor_labels.Size(); i++)
      m_factor_labels[i] +=" ("+DoubleToString(cumvar[i-1],2)+"%)";
//---
   return true;
//---
  }
//+------------------------------------------------------------------+
//|  fill the table                                                  |
//+------------------------------------------------------------------+
bool CApp::FillTable(matrix &vdata)
  {
   for(uint i = 0; i<uint(vdata.Rows())&&i<m_table.RowsTotal();i++)
     {
      for(uint j = 1; j<m_table.ColumnsTotal(); j++)
         m_table.SetValue(j,i,DoubleToString(vdata[i][j-1]),10,false);
     }
   return true;
  }
//+------------------------------------------------------------------+
//|  reset rowlabels                                                 |
//+------------------------------------------------------------------+
bool CApp::SetRowLabels(string &rowlabels[],int shift=0)
  {
   for(uint i = shift; i<rowlabels.Size(); i++)
     {
      m_table.SetValue(0,i-shift,rowlabels[i],0,false);
     }
   return true;
  }
//+------------------------------------------------------------------+
//|  reset rowlabels                                                 |
//+------------------------------------------------------------------+
bool CApp::SetColLabels(string &collabels[])
  {
   for(uint i = 0; i<m_table.ColumnsTotal(); i++)
      m_table.SetHeaderText(i,collabels[i]);

   return true;
  }
//+------------------------------------------------------------------+
//| Reset the matrix                                                 |
//+------------------------------------------------------------------+
void CApp::Reset(int view, int rotate)
  {
   c_selected = view;
   r_selected = rotate;

   switch((SELECT_VIEW)view)
     {
      case ENUM_RFL:
         rfactorloadings = m_fa.rotate_factorloadings((ENUM_FACTOR_ROTATION)rotate);
         m_table.Rebuilding(int(rfactorloadings.Cols()+1),int(rfactorloadings.Rows()));
         FillTable(rfactorloadings);
         SetColLabels(m_factor_labels);
         SetRowLabels(m_col_labels,1);
         break;
      case ENUM_FL:
         m_table.Rebuilding(int(factorloadings.Cols()+1),int(factorloadings.Rows()));
         FillTable(factorloadings);
         SetColLabels(m_factor_labels);
         SetRowLabels(m_col_labels,1);
         break;;
      case ENUM_DATA:
         m_table.Rebuilding(int(feature_matrix.Cols()+1),int(feature_matrix.Rows()));
         FillTable(feature_matrix);
         SetColLabels(m_col_labels);
         SetRowLabels(m_row_labels);
         break;
      case ENUM_CORR:
        {
         matrix corr = m_fa.get_correlation_matrix();
         m_table.Rebuilding(int(corr.Cols()+1),int(corr.Rows()));
         FillTable(corr);
         SetColLabels(m_col_labels);
         SetRowLabels(m_col_labels,1);
        }
      break;
     }

   m_table.Update(true);

  }

CApp app;
//+------------------------------------------------------------------+
//| Expert initialization function                                   |
//+------------------------------------------------------------------+
int OnInit(void)
  {
   ulong tick_counter=::GetTickCount();
//---
   app.OnInitEvent();
//---
   if(!app.CreateGUI())
     {
      ::Print(__FUNCTION__," > error");
      return(INIT_FAILED);
     }

   return(INIT_SUCCEEDED);
  }
//+------------------------------------------------------------------+
//| Expert deinitialization function                                 |
//+------------------------------------------------------------------+
void OnDeinit(const int reason)
  {
   app.OnDeinitEvent(reason);
  }
//+------------------------------------------------------------------+
//| Expert tick function                                             |
//+------------------------------------------------------------------+
void OnTick(void)
  {
  }
//+------------------------------------------------------------------+
//| Timer function                                                   |
//+------------------------------------------------------------------+
void OnTimer(void)
  {
   app.OnTimerEvent();
  }
//+------------------------------------------------------------------+
//| Trade function                                                   |
//+------------------------------------------------------------------+
void OnTrade(void)
  {
  }
//+------------------------------------------------------------------+
//| ChartEvent function                                              |
//+------------------------------------------------------------------+
void OnChartEvent(const int    id,
                  const long   &lparam,
                  const double &dparam,
                  const string &sparam)
  {
   app.ChartEvent(id,lparam,dparam,sparam);
  }
//+------------------------------------------------------------------+

//+------------------------------------------------------------------+
