//+------------------------------------------------------------------+
//|                                                      Network.mqh |
//|                                  Copyright 2024, MetaQuotes Ltd. |
//|                                             https://www.mql5.com |
//+------------------------------------------------------------------+
#property copyright "Copyright 2024, MetaQuotes Ltd."
#property link      "https://www.mql5.com"
#property version   "1.00"
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
enum Epadding
{  PADDING_NONE = -1,
   PADDING_ZERO = 0,
   PADDING_EDGE = 1,
   PADDING_REFLECT = 2,
};
enum Epooling
{  POOLING_NONE = -1,
   POOLING_AVERAGE = 0,
   POOLING_MAX = 1
};
//+------------------------------------------------------------------+
//| Single Layer Convoluted Neural Network Class.                    |
//| (Can be array-stacked for multiple layers)                       |
//+------------------------------------------------------------------+
class Ccnn
{
protected:

   struct            Skernel
   {  matrix         weights;
      double         bias;
                     Skernel()
      {              weights.Init(0, 0);
         bias = 0.5;
      };
   };
   int               kernels;

   Epadding          padding;
   Epooling          pooling;

   vector            target_sizes;

   int               padding_stride;
   int               pooling_stride;
   double            alpha;


   double            LeakyReLU(double X);
   double            LeakyReLUDerivative(double X);

public:
   bool              validated;
   
   //
   Skernel           kernel[];
   matrix            inputs;
   matrix            target[];
   //
   matrix            output[];

   int               Kernels()
   {                 return(kernels);
   }
   
   void              Pad();
   void              Convolve();
   void              Activate();
   void              Pool();
   void              Evolve(double LearningRate = 0.05);

   void              Get(matrix &Target[])
   {  for(int i = 0; i < int(Target.Size()); i++)
      {  if(Target[i].Rows() == Target[i].Cols() && Target[i].Cols() == target_sizes[i])
         {  target[i].Copy(Target[i]);
         }
      }
   }

   void              Set(matrix &Inputs, bool Init = true)
   {  inputs.Init(0, 0);
      inputs.Copy(Inputs);
      if(Init)
      {  for (int f = 0; f < kernels; f++)
         {  if(padding != PADDING_NONE)
            {  output[f].Init(inputs.Rows(), inputs.Rows());
            }
            else
            {  output[f].Init(inputs.Rows() - kernel[f].weights.Rows() + 1, inputs.Rows() - kernel[f].weights.Rows() + 1);
            }
         }
      }
   }


   void              Ccnn(int InputsSize, int &KernelSizes[], Epadding Pad, Epooling Pool, double InitialWeight, double InitialBias, double Alpha, int PaddingStride = 1, int PoolingStride = 1)
   {  validated = false;
      if(Pad == PADDING_NONE && Pool != POOLING_NONE && InputsSize <= 4)
      {  return;
      }
      if(Pad == PADDING_NONE && Pool == POOLING_NONE && InputsSize <= 2)
      {  return;
      }
      if(Pad != PADDING_NONE && Pool != POOLING_NONE && InputsSize <= 2)
      {  return;
      }
      kernels =    ArraySize(KernelSizes);
      if(PaddingStride >= 1 && kernels >= 1 && kernels <= USHORT_MAX && ((KernelSizes[ArrayMinimum(KernelSizes)] > 0 && Pad != PADDING_NONE) || (KernelSizes[ArrayMinimum(KernelSizes)] > 2 && Pad == PADDING_NONE))  && KernelSizes[ArrayMaximum(KernelSizes)] < InputsSize)
      {  inputs.Init(InputsSize, InputsSize);
         ArrayResize(kernel, kernels);
         ArrayResize(output, kernels);
         ArrayResize(target, kernels);
         target_sizes.Init(kernels);
         padding = Pad;
         pooling = Pool;
         padding_stride = PaddingStride;
         pooling_stride = PoolingStride;
         alpha = Alpha;
         for(int i = 0; i < kernels; i++)
         {  kernel[i].weights.Init(KernelSizes[i], KernelSizes[i]);
            kernel[i].weights.Fill(InitialWeight);
            kernel[i].bias = InitialBias;
            if(Pad != PADDING_NONE)
            {  output[i].Init(InputsSize, InputsSize);
               if(Pool == POOLING_NONE)
               {  target_sizes[i] = InputsSize;
               }
               else if(Pool != POOLING_NONE)
               {  target_sizes[i] = InputsSize - 2;
               }
            }
            else
            {  output[i].Init(InputsSize - KernelSizes[i] + 1, InputsSize - KernelSizes[i] + 1);
               if(Pool == POOLING_NONE)
               {  target_sizes[i] = InputsSize - 2;
               }
               else if(Pool != POOLING_NONE)
               {  target_sizes[i] = InputsSize - 4;
               }
            }
         }
         validated = true;
      }
      else
      {  printf(__FUNCSIG__ + " invalid network settings. ");
      }
   };
   void              ~Ccnn(void) { };
};
//+------------------------------------------------------------------+
//| Leaky ReLU activation function                                   |
//+------------------------------------------------------------------+
double Ccnn::LeakyReLU(double X)
{  return ((X > 0.0) ? X : alpha*X);
}
//+------------------------------------------------------------------+
//| Derivative of the Leaky ReLU function                            |
//+------------------------------------------------------------------+
double Ccnn::LeakyReLUDerivative(double X)
{  return ((X > 0.0) ? 1.0 : alpha);
}
//+------------------------------------------------------------------+
//| Pad                                                              |
//+------------------------------------------------------------------+
void Ccnn::Pad()
{  if(!validated)
   {  printf(__FUNCSIG__ + " network invalid! ");
      return;
   }
   if(padding != PADDING_NONE)
   {  matrix _padded;
      _padded.Init(inputs.Rows() + 2, inputs.Cols() + 2);
      _padded.Fill(0.0);
      for(int i = 0; i < int(_padded.Cols()); i++)
      {  for(int j = 0; j < int(_padded.Rows()); j++)
         {  if(i == 0 || i == int(_padded.Cols()) - 1 || j == 0 || j == int(_padded.Rows()) - 1)
            {  if(padding == PADDING_ZERO)
               {  _padded[j][i] = 0.0;
               }
               else if(padding == PADDING_EDGE)
               {  if(i == 0 && j == 0)
                  {  _padded[j][i] = inputs[0][0];
                  }
                  else if(i == 0 && j == int(_padded.Rows()) - 1)
                  {  _padded[j][i] = inputs[inputs.Rows() - 1][0];
                  }
                  else if(i == int(_padded.Cols()) - 1 && j == 0)
                  {  _padded[j][i] = inputs[0][inputs.Cols() - 1];
                  }
                  else if(i == int(_padded.Cols()) - 1 && j == int(_padded.Rows()) - 1)
                  {  _padded[j][i] = inputs[inputs.Rows() - 1][inputs.Cols() - 1];
                  }
                  else if(i == 0)
                  {  _padded[j][i] = inputs[j - 1][i];
                  }
                  else if(j == 0)
                  {  _padded[j][i] = inputs[j][i - 1];
                  }
                  else if(i == int(_padded.Cols()) - 1)
                  {  _padded[j][i] = inputs[j - 1][inputs.Cols() - 1];
                  }
                  else if(j == int(_padded.Rows()) - 1)
                  {  _padded[j][i] = inputs[inputs.Rows() - 1][i - 1];
                  }
               }
               else if(padding == PADDING_REFLECT)
               {  if(i == 0 && j == 0)
                  {  _padded[j][i] = inputs[1][1];
                  }
                  else if(i == 0 && j == int(_padded.Rows()) - 1)
                  {  _padded[j][i] = inputs[inputs.Rows() - 2][1];
                  }
                  else if(i == int(_padded.Cols()) - 1 && j == 0)
                  {  _padded[j][i] = inputs[1][inputs.Cols() - 2];
                  }
                  else if(i == int(_padded.Cols()) - 1 && j == int(_padded.Rows()) - 1)
                  {  _padded[j][i] = inputs[inputs.Rows() - 2][inputs.Cols() - 2];
                  }
                  else if(i == 0)
                  {  _padded[j][i] = inputs[j - 1][1];
                  }
                  else if(j == 0)
                  {  _padded[j][i] = inputs[1][i - 1];
                  }
                  else if(i == int(_padded.Cols()) - 1)
                  {  _padded[j][i] = inputs[j - 1][inputs.Cols() - 2];
                  }
                  else if(j == int(_padded.Rows()) - 1)
                  {  _padded[j][i] = inputs[inputs.Rows() - 2][i - 1];
                  }
               }
            }
            else
            {  _padded[j][i] = inputs[j - 1][i - 1];
            }
         }
      }
      //
      Set(_padded, false);
   }
}
//+------------------------------------------------------------------+
//| Convolve through all kernels                                     |
//+------------------------------------------------------------------+
void Ccnn::Convolve()
{  if(!validated)
   {  printf(__FUNCSIG__ + " network invalid! ");
      return;
   }
// Loop through kernel at set padding_stride
   for (int f = 0; f < kernels; f++)
   {  bool _stop = false;
      int _stride_row = 0, _stride_col = 0;
      output[f].Fill(0.0);
      for (int g = 0; g < int(output[f].Cols()); g++)
      {  for (int h = 0; h < int(output[f].Rows()); h++)
         {  for (int i = 0; i < int(kernel[f].weights.Cols()); i++)
            {  for (int j = 0; j < int(kernel[f].weights.Rows()); j++)
               {  output[f][h][g] += (kernel[f].weights[j][i] * inputs[_stride_row + j][_stride_col + i]);
               }
            }
            output[f][h][g] += kernel[f].bias;
            _stride_col += padding_stride;
            if(_stride_col + int(kernel[f].weights.Cols()) > int(inputs.Cols()))
            {  _stride_col = 0;
               _stride_row += padding_stride;
               if(_stride_row + int(kernel[f].weights.Rows()) > int(inputs.Rows()))
               {  _stride_col = 0;
                  _stride_row = 0;
               }
            }
         }
      }
   }
}
//+------------------------------------------------------------------+
//| Activate                                                       |
//+------------------------------------------------------------------+
void Ccnn::Activate()
{  if(!validated)
   {  printf(__FUNCSIG__ + " network invalid! ");
      return;
   }
   for(int h = 0; h < int(output.Size()); h++)
   {  for (int i = 0; i < int(output[h].Cols()); i++)
      {  for (int j = 0; j < int(output[h].Rows()); j++)
         {  output[h][i][j] = LeakyReLU(output[h][i][j]);
         }
      }
   }
}
//+------------------------------------------------------------------+
//| Pool                                                             |
//+------------------------------------------------------------------+
void Ccnn::Pool()
{  if(!validated)
   {  printf(__FUNCSIG__ + " network invalid! ");
      return;
   }
   if(pooling != POOLING_NONE)
   {  for(int f = 0; f < int(output.Size()); f++)
      {  matrix _pooled;
         if(output[f].Cols() > 2 && output[f].Rows() > 2)
         {  _pooled.Init(output[f].Rows() - 2, output[f].Cols() - 2);
            _pooled.Fill(0.0);
            for (int g = 0; g < int(_pooled.Cols()); g++)
            {  for (int h = 0; h < int(_pooled.Rows()); h++)
               {  if(pooling == POOLING_MAX)
                  {  _pooled[h][g] = DBL_MIN;
                  }
                  for (int i = 0; i < int(output[f].Cols()); i++)
                  {  for (int j = 0; j < int(output[f].Rows()); j++)
                     {  if(pooling == POOLING_MAX)
                        {  _pooled[h][g] = fmax(output[f][j][i], _pooled[h][g]);
                        }
                        else if(pooling == POOLING_AVERAGE)
                        {  _pooled[h][g] += output[f][j][i];
                        }
                     }
                  }
                  if(pooling == POOLING_AVERAGE)
                  {  _pooled[h][g] /= double(output[f].Cols()) * double(output[f].Rows());
                  }
               }
            }
            output[f].Copy(_pooled);
         }
      }
   }
}
//+------------------------------------------------------------------+
//| Evolve pass through the neural network to update kernel          |
//| and biases using gradient descent                                |
//+------------------------------------------------------------------+
void Ccnn::Evolve(double LearningRate = 0.05)
{  if(!validated)
   {  printf(__FUNCSIG__ + " network invalid! ");
      return;
   }
   
   for(int f = 0; f < kernels; f++)
   {  matrix _output_error = target[f] - output[f];
      // Calculate output layer gradients
      matrix _output_gradients;
      _output_gradients.Init(output[f].Rows(),output[f].Cols());
      for (int g = 0; g < int(output[f].Rows()); g++)
      {  for (int h = 0; h < int(output[f].Cols()); h++)
         {  _output_gradients[g][h] =  LeakyReLUDerivative(output[f][g][h]) * _output_error[g][h];
         }
      }
      
      // Update output layer kernel weights and biases
      int _stride_row = 0, _stride_col = 0;
      for (int g = 0; g < int(output[f].Cols()); g++)
      {  for (int h = 0; h < int(output[f].Rows()); h++)
         {  double _bias_sum = 0.0;
            for (int i = 0; i < int(kernel[f].weights.Cols()); i++)
            {  for (int j = 0; j < int(kernel[f].weights.Rows()); j++)
               {  kernel[f].weights[j][i] += (LearningRate * _output_gradients[_stride_row + j][_stride_col + i]); // output[f][_stride_row + j][_stride_col + i]);
                  _bias_sum += _output_gradients[_stride_row + j][_stride_col + i];
               }
            }
            kernel[f].bias += LearningRate * _bias_sum;
            _stride_col += padding_stride;
            if(_stride_col + int(kernel[f].weights.Cols()) > int(_output_gradients.Cols()))
            {  _stride_col = 0;
               _stride_row += padding_stride;
               if(_stride_row + int(kernel[f].weights.Rows()) > int(_output_gradients.Rows()))
               {  _stride_col = 0;
                  _stride_row = 0;
               }
            }
         }
      }
   }
}
//+------------------------------------------------------------------+
