
// cSocketApp.cpp

#include "cSocketApp.h"

// Constructors
cSocketApp::cSocketApp(): mu16_Port(PORT), me_DataMode(E_NORMAL),
	                      mb_ParentThreadClosed(FALSE),mh_Thread(NULL) 
{
	InitializeCriticalSection(&mk_Critical);
	ms8_TempBuffer = new char[TEMP_BUFFER_SIZE];
};

cSocketApp::cSocketApp(DWORD u32_Port, eDataMode DataMode):
	                         mu16_Port((USHORT)u32_Port), me_DataMode(DataMode),
	                         mb_ParentThreadClosed(FALSE),mh_Thread(NULL) 
{
	InitializeCriticalSection(&mk_Critical);
	ms8_TempBuffer = new char[TEMP_BUFFER_SIZE]; 
}

// Destructor							 
cSocketApp::~cSocketApp()
{
	DeleteCriticalSection(&mk_Critical);
	CloseSockets();
	if (ms8_TempBuffer) delete ms8_TempBuffer; 
}
// Creates a Server socket.
// Can return: 1) Zero(0)  --> OK, we now are listening on port mu16_Port
//             2) Non zero --> WSA error, see WSA error codes
// u32_BindIP = 0          --> listen on all network adapters
// u32_BindIP = 10.1.0.143 --> listen only on the network adapter with local IP 10.1.0.143
// u32_BindIP = 10.1.2.208 --> listen only on the network adapter with local IP 10.1.2.208
// u32_EventTimeout = the timeout after which ProcessEvents() will abort waiting for an event
// If u32_MaxIdleTime > 0  --> automatically disconnect clients which are idle for a longer time (in seconds)
// Switch this application into Server mode and listen on u32_Port for Client connections
DWORD cSocketApp::Listen() 
{
	TCP::cSocket::cHash<DWORD, DWORD> i_cHashLocalIPs;    // Key = 0, Value = local IP
	// Get all local IP addresses for each network adapter
	DWORD u32_Err = mi_Socket.GetLocalIPs(&i_cHashLocalIPs);
	
	if (u32_Err)
		return u32_Err;
	
	DWORD u32_BindIP = i_cHashLocalIPs.GetValueByIndex(0);
	
	DWORD u32_EventTimeout =  INFINITE;

	u32_Err = mi_Socket.Listen(u32_BindIP, mu16_Port, u32_EventTimeout, MAX_SERVER_IDLE_TIME);
	
	if (u32_Err) 
	{
		CloseSockets();
		return u32_Err ;
	}

	// runs until an error occurred or all sockets have closed
	DWORD u32_ID;
	mh_Thread = CreateThread(0, 0, ProcessEventThread, this, 0, &u32_ID);
	
	return 0;
}
// Creates a Client socket.
// Can return: 1) Zero(0)  --> OK, we now are connected  to server with IP = u32_ServerIP on port mu16_Port, 
//                                 handle fo connection(h_Socket) is passed to caller via reference parameter SOCKET &h_Socket
//             2) Non zero --> WSA error, see WSA error codes
// u32_ServerIP = 0x6401a8c0 -> 192.168.1.100
// If MAX_CLIENT_IDLE_TIME > 0  -> automatically disconnect from server if idle for a longer time (in seconds)
// u32_EventTimeout = the timeout after which ProcessEvents() will abort waiting for an event, in our case = INFINITE (endless)
// *************************** ATTENTION ************************************
// When this funcion returns without error the socket is connected!
// Here we wait for the FD_CONNECT event arise without/with error!
// *************************** ATTENTION ************************************
// Switch this application into Client mode and connect to a server
DWORD cSocketApp::ConnectTo(DWORD u32_ServerIP, SOCKET &h_Socket)
{
	DWORD u32_EventTimeout = INFINITE;
	
	DWORD u32_Err = mi_Socket.ConnectTo(u32_ServerIP, mu16_Port, u32_EventTimeout, MAX_CLIENT_IDLE_TIME);
	
	if (u32_Err) 
	{
		CloseSockets();
		return u32_Err;
	}
	
	// runs until an error occurred or socket was closed and removed
	DWORD u32_ID;
	mh_Thread = CreateThread(0, 0, ProcessEventThread, this, 0, &u32_ID);
	
	TCP::cSocket::cHash<SOCKET,DWORD> i_cHashSocketList;  // Key = socket handle, Value = peer IP

	// wait for FD_CONNECT event without error ...
	while (mi_Socket.GetSocketCount())
	{
		if (mi_Socket.GetState() & TCP::cSocket::E_Connected)
		{
			DWORD u32_Error = mi_Socket.GetAllConnectedSockets(&i_cHashSocketList);  //Key = socket handle, Value = peer IP
			if (u32_Error)
				return u32_Error;

			h_Socket	 = i_cHashSocketList.GetKeyByIndex(0);  // return h_Socket value no caller

			return 0;
		}
		continue;
	}
	return mu32_ProcessEventsError;
}

// Get all client's connections, for both server and client users. 
kCon cSocketApp::GetAllConnections()
{
	// The variable mi_SocketList is manipulated from two threads - from main application thread and from events handler thread.
	// The critical section assures thread safety
	EnterCriticalSection(&mk_Critical);

	mi_SocketList.UpdateSocketList(&mi_Socket);

	kCon k_Con;
	
	DWORD u32_Count = mi_SocketList.GetSocketCount();
	for (DWORD i=0; i<u32_Count; i++)
	{
		SOCKET h_Socket = mi_SocketList.GetSocketByIndex(i);
		DWORD  u32_IP   = mi_SocketList.GetIpByIndex(i);

		k_Con.k_Client[i].h_Client     = h_Socket;
		k_Con.k_Client[i].u32_ClientIP = u32_IP;
	}
	k_Con.u32_ClientCount = u32_Count;
	//
	LeaveCriticalSection(&mk_Critical);
	//
	return k_Con;
};

// This function is for use on a server only
// It can be called to force a disconnect of a specific client from the server
DWORD cSocketApp::DisconnectClient(SOCKET h_Socket)
{
	return mi_Socket.DisconnectClient(h_Socket);
}

// Sends data to the given socket
// Can return: 1) Zero(0) --> OK ! u32_SendBufLen bytes of data from s8_SendBuf was send to remote socket that is peer for parm socket h_Socket
//			   2) ERROR_INVALID_PARAMETER --> h_Socket socket does'not exist ! or s8_SendBuf = NULL or u32_SendBufLen = 0 or u32_SendBufLen > (TEMP_BUFFER_SIZE-4)
//             3) WSAEWOULDBLOCK  -> The data will be send after the next FD_WRITE event, no any action needed
//             4) WSA_IO_PENDING  -> Error: A previous Send operation is still pending. This data will not be sent, try later
//             5) Any other value -> Severe error -> event loop aborted !
DWORD cSocketApp::SendTo(SOCKET h_Socket, char* s8_SendBuf, DWORD u32_SendBufLen)
{
	// The variable mi_SocketList is manipulated from two threads - from main application thread and from network events handler thread.
	// The critical section assures thread safety
	EnterCriticalSection(&mk_Critical);

	mi_SocketList.UpdateSocketList(&mi_Socket);

	if (!mi_SocketList.FindSocket(h_Socket) || !s8_SendBuf || !u32_SendBufLen || u32_SendBufLen > (TEMP_BUFFER_SIZE-4))
	{
		LeaveCriticalSection(&mk_Critical);
		return ERROR_INVALID_PARAMETER;  
	}

	char* s8_Data;
	DWORD u32_Len = u32_SendBufLen;
	
	if (me_DataMode == E_PREFIXED)
	{
		s8_Data = ms8_TempBuffer;
		u32_Len += 4; // set to total length of datablock
		((DWORD*)s8_Data)[0] = u32_Len;
		memcpy(s8_Data+4, s8_SendBuf, u32_SendBufLen);
	}
	else
		s8_Data = s8_SendBuf;

	DWORD u32_Err = mi_Socket.SendTo(h_Socket, s8_Data, u32_Len);

	switch (u32_Err)
	{
	case 0:
		LeaveCriticalSection(&mk_Critical);
		return 0;

	case WSAEWOULDBLOCK:
	LeaveCriticalSection(&mk_Critical);
		// WSAEWOULDBLOCK -> The data will be send after the next FD_WRITE event.
		return WSAEWOULDBLOCK;

	case WSA_IO_PENDING:
		LeaveCriticalSection(&mk_Critical);
		// WSA_IO_PENDING -> Error: A previous Send operation is still pending. This data will not be sent. Try latter
		return WSA_IO_PENDING;

	default:
		LeaveCriticalSection(&mk_Critical);
		// -> Error:  GetErrMsg(u32_Err));
		// Severe error -> abort event loop
		CloseSockets();
		return u32_Err; 
	};
}

// Read data recieved for the given socket.
// ReadFrom(...) does'not read any data from winsock, it simply read data from socket buffer prepared by network events handler
// working as separate thread.
// Data recieved for the given socket is stored in mi_SocketList.mk_Socket[GetIndexBySocket(SOCKET h_Socket)].pi_RecvMem-->i_TCP::cSocket::cMemory
// 
// Can return 1) Zero(0) : u32_Count = 0 --> no any data recieved/remained from peer for this socket
//                         u32_Count > 0 -->  contains byte's count of data placed in s8_ReadBuf 
//                         u32_Count = Negative_Value   --> Length of s8_ReadBuf - u32_ReadBufLen is too small, no data placed in s8_ReadBuf,
//                                                          abs(Negative_Value) - required length of s8_ReadBuf,
//                                                          prepare new larger s8_ReadBuf and try once more 
//
//            2) ERROR_INVALID_PARAMETER --> h_Socket socket does'not exist ! or s8_ReadBuf = NULL or u32_ReadBufLen = 0 
DWORD cSocketApp::ReadFrom(SOCKET h_Socket, char* s8_ReadBuf, int u32_ReadBufLen, int &u32_Count)
{
	// The variable mi_SocketList is manipulated from two threads - from main application thread and from network events handler thread.
	// The critical section assures thread safety
	EnterCriticalSection(&mk_Critical);

	mi_SocketList.UpdateSocketList(&mi_Socket);

	cSocketList::kSocket* pk_Socket = mi_SocketList.FindSocket(h_Socket);

	if (!pk_Socket || !s8_ReadBuf || !u32_ReadBufLen)
	{
		LeaveCriticalSection(&mk_Critical);
		return ERROR_INVALID_PARAMETER;  
	}

	TCP::cSocket::cMemory* pi_RecvMem = pk_Socket->pi_RecvMem;

	if (!pi_RecvMem)
	{
		u32_Count = 0;     // no any data recieved from peer for this socket !
		LeaveCriticalSection(&mk_Critical);
		return 0;
	}

	char*  s8_Buf = pi_RecvMem->GetBuffer();
	INT    u32_Len = pi_RecvMem->GetLength(); 
	INT u32_Blocksize;
	INT s32_Pos; 

	switch (me_DataMode)
	{
		// E_NORMAL - caller will get all data recieved for this socket
		case E_NORMAL: if (!u32_Len)
					   {
						   u32_Count = 0;  // no any data remained for this socket !
						   LeaveCriticalSection(&mk_Critical);
						   return 0;
					   }
					   if (u32_Len > u32_ReadBufLen)
					   {
						   u32_Count = -u32_Len; // u32_Count = Negative_Value, s8_ReadBuf is too small abs(Negative_Value) - required length of s8_ReadBuf
						   LeaveCriticalSection(&mk_Critical);
						   return 0;
					   }
					   memcpy(s8_ReadBuf, s8_Buf, u32_Len);
					   u32_Count = u32_Len;      // u32_Count =  actual byte's count of data placed in s8_ReadBuf
					   // Delete all read data from the receive memory of this socket
					   pi_RecvMem->DeleteLeft(u32_Len);
					   LeaveCriticalSection(&mk_Critical);
					   return 0;
		// E_PREFIXED - caller will get only next block of data recieved for this socket
		case E_PREFIXED: if (!u32_Len)
						 {
							u32_Count = 0;  // no any data remained for this socket !
							LeaveCriticalSection(&mk_Critical);
							return 0;
					     }
						 u32_Blocksize = ((DWORD*)s8_Buf)[0];
						 if (u32_Blocksize-4 > u32_ReadBufLen)
					     {
							u32_Count = -(u32_Blocksize-4); // u32_Count = Negative_Value, s8_ReadBuf is too small, abs(Negative_Value) - required length of s8_ReadBuf
							LeaveCriticalSection(&mk_Critical);
							return 0;
					     }
					     memcpy(s8_ReadBuf, s8_Buf+4, u32_Blocksize-4);
					     u32_Count = u32_Blocksize-4;      // u32_Count =  actual byte's count of data block placed in s8_ReadBuf
					     // Delete this read data block from the receive memory of this socket
					     pi_RecvMem->DeleteLeft(u32_Blocksize);
					     LeaveCriticalSection(&mk_Critical);
						 return 0;
		// E_TELNET - caller will get only next data block starting from s8_Buf+0 and ending with '\n' inclusively recieved for this socket				 
		case E_TELNET:   if (!u32_Len)
					     {
							u32_Count = 0;  // no any data remained for this socket !
							LeaveCriticalSection(&mk_Critical);
							return 0;
					     }
						 // find first '\n' character in s8_Buf
						 s32_Pos = 0; 
						 for (int i = 0; i < u32_Len; i++)
						 {
							if (s8_Buf[i] == '\n')
							{
								s32_Pos = i; break;
							}
		                 }
						 if (s32_Pos+1 > u32_ReadBufLen)
					     {
							u32_Count = -(s32_Pos+1); // u32_Count = Negative_Value, s8_ReadBuf is too small, abs(Negative_Value) - required length of s8_ReadBuf
							LeaveCriticalSection(&mk_Critical);
							return 0;
					     }
					     memcpy(s8_ReadBuf, s8_Buf, s32_Pos+1);
					     u32_Count = s32_Pos+1;      // u32_Count =  actual byte's count of data block placed in s8_ReadBuf
					     // Delete this read data block from the receive memory of this socket
					     pi_RecvMem->DeleteLeft(s32_Pos+1);
					     LeaveCriticalSection(&mk_Critical);
						 return 0;
	}
}

// Close all open sockets (if any), for both server and client users, remove all sockets. 
void cSocketApp::CloseSockets() 
{
	// Signal to terminate endless loop of network events handler thread
	mb_ParentThreadClosed = true;

	if (mi_Socket.GetSocketCount())
	{
		mi_Socket.Close();
	}

	mi_SocketList.RemoveAll();

	// Wait for ProcessEventThread(...) thread really returns to Windows !
	// After that,  cSocketApp instance might be normally destroyed by ~cSocketApp()
	WaitForSingleObject(mh_Thread, INFINITE); // wait for ProcessEventThread(...) really returns to Windows

	CloseHandle(mh_Thread);   // ??? !!!
}

// Returns total socket count including Listen(...) socket himself and, of course, ConnectTo(...) socket
DWORD cSocketApp::GetSocketCount() // returns total socket count including Listen(...) socket himself and, of course, ConnectTo(...) socket
{
	return mi_Socket.GetSocketCount();
}

// static
ULONG WINAPI cSocketApp::ProcessEventThread(void* p_Param)
{
	cSocketApp* p_This = (cSocketApp*)p_Param;
	p_This->ProcessEvents();                                                   printf("\nExit from ProcessEventThread!!!\n");
	//CloseHandle(p_This->mh_Thread);   // ??? !!!
	return 0;
}

// Process all events which occur on one of the open sockets
void cSocketApp::ProcessEvents()
{
	BOOL b_Server = (mi_Socket.GetState() & TCP::cSocket::E_Server);
	DWORD u32_Err;
	while (TRUE) // Loop runs until the Parent thread was closed or a severe error occurred
	{
		TCP::cSocket::cMemory* pi_RecvMem;
		SOCKET  h_Socket;
		DWORD u32_Event, u32_IP, u32_Read, u32_Sent;
		u32_Err = mi_Socket.ProcessEvents(&u32_Event, &u32_IP, &h_Socket, &pi_RecvMem, &u32_Read,  &u32_Sent);
		
		// Parent thread was closed -> !Immediately! stop all output into Parent thread .
		// Otherwise the application will not shut down correctly and the EXE keeps running. (only visible in Task Manager)
		// There may appear a lot of other strange things when the Events thread still runs while the Parent thread already finished!
		if (mb_ParentThreadClosed) 
			return;  // return NOT break!

		if (u32_Err == ERROR_TIMEOUT) // 50 ms interval has elapsed when not INFINITE
			continue;
		
		// The variable mi_SocketList is manipulated from two threads - from main application thread and from this network events thread.
		// The critical section assures thread safety
		EnterCriticalSection(&mk_Critical);

		// Update the list of all connected sockets in mi_SocketList
		mi_SocketList.UpdateSocketList(&mi_Socket); 
		if (u32_Event) // ATTENTION: u32_Event may be == 0 -> do nothing.
		{
			if (u32_Event & FD_READ && pi_RecvMem) // pi_RecvMem may be NULL if an error occurred!!
			{
				// get pointer to actual socket element
				cSocketList::kSocket* pk_Socket = mi_SocketList.FindSocket(h_Socket);	
				switch (me_DataMode)
				{
					case E_NORMAL:   ProcessReceivedDataNormal(pk_Socket, pi_RecvMem); break;
					case E_PREFIXED: ProcessReceivedDataPrefix(pk_Socket, pi_RecvMem); break;
					case E_TELNET:   ProcessReceivedDataTelnet(pk_Socket, pi_RecvMem); break;
				}
				
			}
			
		}

		LeaveCriticalSection(&mk_Critical);

		if (u32_Err) 
			mu32_ProcessEventsError = u32_Err;
		//

		if (u32_Err)
		{
			// mi_Socket.Close() has been called -> break...
			if (u32_Err == WSAENOTCONN)
				break;

			// Print all the other error messages
			//Print(_T("ProcessEvent Error %s"), GetErrMsg(u32_Err));
			
			// An error normally means that the socket has a problem -> abort the loop.
			// A few errors should not abort the processing:
			if (u32_Err != WSAECONNABORTED && // e.g. after the other side was killed in TaskManager 
				u32_Err != WSAECONNRESET   && // Connection reset by peer.
				u32_Err != WSAECONNREFUSED && // FD_ACCEPT with already 62 clients connected
				u32_Err != WSAESHUTDOWN)      // Sending data to a socket just in the short timespan 
				break;                        //   between shutdown() and closesocket()
		}
	}; // end loop
	
	mi_Socket.Close();

	if (b_Server) printf("Stop Listening. Error %d\n\n", u32_Err);
	else          printf("Connection abandoned. Error %d\n\n", u32_Err);
}

// ##################################################################################################
//                                  PROCESS RECEIVED DATA
// ##################################################################################################

// Mode NORMAL:
// This simple "data processor" prints the data blocks immediately and unchanged as they arrive from the network
void cSocketApp::ProcessReceivedDataNormal(cSocketList::kSocket* pk_Socket, TCP::cSocket::cMemory* pi_RecvMem)
{
	if (!pk_Socket->pi_RecvMem)
			 pk_Socket->pi_RecvMem = new TCP::cSocket::cMemory(MEMORY_INITIAL_SIZE);
		
	char*  s8_Buf = pi_RecvMem->GetBuffer();
	DWORD u32_Len = pi_RecvMem->GetLength();

	pk_Socket->pi_RecvMem->Append(s8_Buf, u32_Len);
	// Delete all received data from the receive memory
	pi_RecvMem->DeleteLeft(u32_Len);
}

// Mode PREFIX:
// Each datablock comes prefixed with a DWORD which contains the total length of the datablock.
// So it is easy to determine if a block has been received completely.
// The data is accumulated in pi_RecvMem which works like a FIFO memory.
// This is the recommended principle for transmitting binary data.
// To test this mode set SEND_LARGE_DATA to 100 and set READ_BUFFER_SIZE to 30
void cSocketApp::ProcessReceivedDataPrefix(cSocketList::kSocket* pk_Socket, TCP::cSocket::cMemory* pi_RecvMem)
{
	while (TRUE) // There may arrive multiple datablocks at once -> loop until FIFO is empty
	{
		if (!pk_Socket->pi_RecvMem)
			 pk_Socket->pi_RecvMem = new TCP::cSocket::cMemory(MEMORY_INITIAL_SIZE);

		char*  s8_Buf = pi_RecvMem->GetBuffer();
		DWORD u32_Len = pi_RecvMem->GetLength();
		if (u32_Len < 4)
			return; // There must always be at least 1 Dword

		DWORD u32_Blocksize = ((DWORD*)s8_Buf)[0];
		if (u32_Blocksize > u32_Len)
		{
			return; // The block is not yet complete -> accumulate more data in pi_RecvMem
		}

		//Received entire datablock 
		pk_Socket->pi_RecvMem->Append(s8_Buf, u32_Blocksize);
		// Only delete the data that has been processed and leave the rest.
		// ATTENTION: DeleteLeft(u32_Len) would result in data loss!!
		pi_RecvMem->DeleteLeft(u32_Blocksize);
	}
}

// Mode TELNET:
// This function demonstrates how single characters received from a Telnet client
// are accumulated in pi_RecvMem, which works like a FIFO memory, until a line feed is found.
// When a line is complete it is copied to the pk_Socket->pi_RecvMem and deleted from pi_RecvMem.
void cSocketApp::ProcessReceivedDataTelnet(cSocketList::kSocket* pk_Socket, TCP::cSocket::cMemory* pi_RecvMem)
{
	#if _UNICODE
		wprintf(TEXT("Telnet does not use Unicode. Please compile the Telnet  as MBCS!"));
		return;
	#endif;

	if (!pk_Socket->pi_RecvMem)
			 pk_Socket->pi_RecvMem = new TCP::cSocket::cMemory(MEMORY_INITIAL_SIZE);

	// If you send "Hello\nWorld\n" from SocketDemo instead of using a real Telnet client this requires a loop
	while (TRUE) 
	{
		char*  s8_Buf = pi_RecvMem->GetBuffer();
		DWORD u32_Len = pi_RecvMem->GetLength();

		DWORD s32_Pos = 0; 
		for (DWORD i = 0; i < u32_Len; i++)
		{
			if (s8_Buf[i] == '\n')
			{
				s32_Pos = i; break;
			}
		}

		if (!s32_Pos)
			return; // The line is not yet complete -> accumulate more characters, waiting for linefeed...

		pk_Socket->pi_RecvMem->Append(s8_Buf, s32_Pos+1);
		
		// Received entire line
		// Delete all characters including the "\n" itself from the receive memory
		// but leave all characters in RecvMem which follow the "\n"
		pi_RecvMem->DeleteLeft(s32_Pos+1);
	};
}

// ##################################################################################################
//                                            END
// ##################################################################################################

/***************************************************************************************
****************************************************************************************

	embedded class cSocketList

	Description:
	Stores for all open sockets: their Handle, IP address, recieve buffer whith data to be grabbed by main application parent thread
	                             by means of method - DWORD cSocketApp::ReadFrom(SOCKET h_Socket, char* s8_ReadBuf, DWORD u32_ReadBufLen, INT &u32_Count).

	CLIENT:
	- uses only Index=0 which holds the socket that is connected to the server.

	SERVER:
	- Uses up to WSA_MAXIMUM_WAIT_EVENTS-2  sockets at Index=0 to Index=(WSA_MAXIMUM_WAIT_EVENTS-2)-1.
	- Each of the following sockets Index=0,2,3...61 may be connected to one client.

	Author: 
	Elm (www.netcult.ch/elmue)

****************************************************************************************
****************************************************************************************/

cSocketApp::cSocketList::cSocketList(void): mu32_Count(0), mi_cHashSocketList(62) 
{
}

cSocketApp::cSocketList::~cSocketList(void)
{
	RemoveAll();
}

// FindSocket(SOCKET h_Socket)
// returns the pointer of the given socket element in the socket list
// returns NULL if not found
cSocketApp::cSocketList::kSocket* cSocketApp::cSocketList::FindSocket(SOCKET h_Socket)
{
	for (DWORD i=0; i<mu32_Count; i++)
	{
		if (mk_Socket[i].h_Socket == h_Socket)
			return &mk_Socket[i];
	}
	return NULL;
}

// Add(SOCKET h_Socket, DWORD u32_IP)
cSocketApp::cSocketList::kSocket* cSocketApp::cSocketList::Add(SOCKET h_Socket, DWORD u32_IP)
{
	kSocket* pk_Socket = FindSocket(h_Socket);
	if (pk_Socket)
		return pk_Socket;

	memset(&mk_Socket[mu32_Count], 0, sizeof(kSocket));

	mk_Socket [mu32_Count].h_Socket = h_Socket;
	mk_Socket [mu32_Count].u32_IP   = u32_IP;

	return &mk_Socket[mu32_Count++];
}

// UpdateSocketList(TCP::cSocket* pi_Socket)
// Update the list of all connected sockets in mi_SocketList, 
// some sockets elements may be removed, new sockets elements may be added
DWORD cSocketApp::cSocketList::UpdateSocketList(TCP::cSocket* pi_Socket)
{
	DWORD u32_Err = pi_Socket->GetAllConnectedSockets(&mi_cHashSocketList); // mi_cHashSocketList : Key = socket handle, Value = peer IP 

	if (u32_Err)
		return u32_Err;

	//if (u32_Err) Print(_T("Error getting connected Sockets: %s"), GetErrMsg(u32_Err));

	// remove not actual sockets from mi_SocketList
	for (DWORD i = 0; i<mu32_Count; i++)
	{
		SOCKET h_Socket = GetSocketByIndex(i);

		if (!mi_cHashSocketList.GetValueByKey(h_Socket))
			Remove(i);
	}

	// add new sockets in mi_SocketList
	DWORD u32_Count = mi_cHashSocketList.GetCount();
	for (DWORD i=0; i<u32_Count; i++)
	{
		SOCKET h_Socket = mi_cHashSocketList.GetKeyByIndex(i);
		DWORD  u32_IP   = mi_cHashSocketList.GetValueByIndex(i);

		Add(h_Socket, u32_IP);
	}

	return 0;
}

// GetSocketByIndex(DWORD u32_Index)
// returns the socket by the index in the socket list
// returns NULL if not found
SOCKET cSocketApp::cSocketList::GetSocketByIndex(DWORD u32_Index)
{
	if (u32_Index >= mu32_Count)
		return NULL;
	return mk_Socket[u32_Index].h_Socket;
};

// GetIpByIndex(DWORD u32_Index)
// returns the IP by the index in the socket list
// returns NULL if not found
DWORD cSocketApp::cSocketList::GetIpByIndex(DWORD u32_Index)
{
	if (u32_Index >= mu32_Count)
		return NULL;
	return mk_Socket[u32_Index].u32_IP;
};

// Remove(DWORD u32_Index)
BOOL cSocketApp::cSocketList::Remove(DWORD u32_Index)
{
	if (u32_Index >= mu32_Count)
		return FALSE;

	if (mk_Socket[u32_Index].pi_RecvMem) delete mk_Socket[u32_Index].pi_RecvMem;

	// Close the gap by copying the last socket to the deleted location
	mu32_Count--;
	mk_Socket[u32_Index] = mk_Socket[mu32_Count];
	
	return TRUE;
}

// RemoveAll()
void cSocketApp::cSocketList::RemoveAll()
{
	while (mu32_Count)
	{
		Remove(mu32_Count-1);
	}
}

// GetSocketCount()
DWORD cSocketApp::cSocketList::GetSocketCount()
{
	return mu32_Count;
}



