//+------------------------------------------------------------------+
//|                                                 PermuteRates.mqh |
//|                                  Copyright 2023, MetaQuotes Ltd. |
//|                                             https://www.mql5.com |
//+------------------------------------------------------------------+
#property copyright "Copyright 2023, MetaQuotes Ltd."
#property link      "https://www.mql5.com"
#include <UniformRandom.mqh>
//+------------------------------------------------------------------+
//| struct to handle relative values of rate data to be worked on    |
//+------------------------------------------------------------------+
struct CRelRates
  {
   double rel_open;
   double rel_high;
   double rel_low;
   double rel_close;
  };
  
//+------------------------------------------------------------------+
//| Class to enable permuation of a collection of ticks in an array  |
//+------------------------------------------------------------------+
class CPermuteRates
  {
private :
   MqlRates          m_rates[];        //original rates to be shuffled
   CRelRates         m_differenced[];  //log difference of rates 
   bool              m_initialized;    //flag to signal state of random number object
   CUniFrand         *m_random;      //random number generator
   
public :
   //constructor
                     CPermuteRates(void);
   //desctructor
                    ~CPermuteRates(void);
   bool              Initialize(MqlRates &in_rates[]);                 
   bool              Permute(MqlRates &out_rates[]);
  };
//+------------------------------------------------------------------+
//|  constructor                                                     |
//+------------------------------------------------------------------+
CPermuteRates::CPermuteRates(void)
{
//--- 
 m_random= new CUniFrand();
//---
 m_random.SetSeed(MathRand());
//---
 m_initialized=false;
//---    
}
//+------------------------------------------------------------------+
//| destructor                                                       |
//+------------------------------------------------------------------+
CPermuteRates::~CPermuteRates(void)
{
 delete m_random;
 
 ArrayFree(m_rates);
 
 ArrayFree(m_differenced);
}
//+------------------------------------------------------------------+
//| Initialize internal buffers and prepare                                                                  |
//+------------------------------------------------------------------+
bool CPermuteRates::Initialize(MqlRates &in_rates[])
{
 //---check the random number object
   if(m_random==NULL)
    {
     Print("Critical internal error, failed to initialize random number generator");
     return false;
    }
 //---set or reset initialization flag  
   m_initialized=false;   
 //---check arraysize
   if(in_rates.Size()<5)
     {
      Print("Insufficient amount of data supplied ");
      return false;
     }
//---copy ticks to local array
   if(ArrayCopy(m_rates,in_rates)!=int(in_rates.Size()))
     {
      Print("Error copying ticks ", GetLastError());
      return false;
     }  
//---ensure the size of m_differenced array
   if(m_differenced.Size()!=m_rates.Size()-1)
      ArrayResize(m_differenced,m_rates.Size()-1);
//---fill m_differenced with differenced values, excluding the first tick
   for(uint i=1; i<m_rates.Size() && !IsStopped(); i++)
     {
      m_differenced[i-1].rel_open=MathLog(m_rates[i].open)-MathLog(m_rates[i-1].close);
      m_differenced[i-1].rel_high=MathLog(m_rates[i].high)-MathLog(m_rates[i].open);
      m_differenced[i-1].rel_low=MathLog(m_rates[i].low)-MathLog(m_rates[i].open);
      m_differenced[i-1].rel_close=MathLog(m_rates[i].close)-MathLog(m_rates[i].open);
     }     
//---
   if(IsStopped())
      return false;     
//---
   m_initialized=true;
//---
   return true;
} 
//+------------------------------------------------------------------+
//| Permute the bars                                                 |
//+------------------------------------------------------------------+
bool CPermuteRates::Permute(MqlRates &out_rates[])
{
//--- 
 if(!m_initialized)
  {
   Print("Initialization error");
   ZeroMemory(out_rates);
   return false;
  }
//---
 int i,j;
 double temp=0.0;
//---
   i=ArraySize(m_rates)-2;
//---
   while (i > 1 && !IsStopped()) 
    {     
      j = (int) (m_random.RandomDouble() * i) ;
      if(j >= i)   
         j = i - 1 ;
      --i ;
      temp = m_differenced[i+1].rel_open ;
      m_differenced[i+1].rel_open = m_differenced[j+1].rel_open ;
      m_differenced[j+1].rel_open = temp ;
    }  
//--- 
   i =ArraySize(m_rates)-2;
//---    
   while (i > 1  && !IsStopped()) 
    {  
      j = (int) (m_random.RandomDouble() * i) ;
      if (j >= i)         
         j = i - 1 ;
      --i ;
      temp = m_differenced[i].rel_high;
      m_differenced[i].rel_high = m_differenced[j].rel_high ;
      m_differenced[j].rel_high = temp ;
      temp = m_differenced[i].rel_low ;
      m_differenced[i].rel_low = m_differenced[j].rel_low ;
      m_differenced[j].rel_low = temp ;
      temp = m_differenced[i].rel_close ;
      m_differenced[i].rel_close = m_differenced[j].rel_close ;
      m_differenced[j].rel_close = temp ;
    }   
//---
 if(ArrayCopy(out_rates,m_rates)!=int(m_rates.Size()))
   {
    ZeroMemory(out_rates);
    Print("Copy error ", GetLastError());
    return false;
   }
//---      
 for (i=1 ; i<ArraySize(out_rates) && !IsStopped() ; i++) 
      {
        out_rates[i].open  = MathExp(((MathLog(out_rates[i-1].close)) + m_differenced[i-1].rel_open)) ;
        out_rates[i].high  = MathExp(((MathLog(out_rates[i].open)) + m_differenced[i-1].rel_high)) ;
        out_rates[i].low   = MathExp(((MathLog(out_rates[i].open)) + m_differenced[i-1].rel_low)) ;
        out_rates[i].close = MathExp(((MathLog(out_rates[i].open)) + m_differenced[i-1].rel_close)) ;
      }     
//---
 if(IsStopped())
   return false;
//---   
 return true;    
//---
}   