//+------------------------------------------------------------------+
//|                                                    NewSymbol.mqh |
//|                                           Copyright 2019, denkir |
//|                             https://www.mql5.com/ru/users/denkir |
//+------------------------------------------------------------------+
#property copyright "Copyright 2019, denkir"
#property link      "https://www.mql5.com/ru/users/denkir"
//--- include
#include <Object.mqh>
#include<Files\File.mqh>
//---
#define ATTEMTS 3
#define PAUSE 75

//+------------------------------------------------------------------+
//| Class CNewSymbol.                                                |
//| Purpose: Base class for a custom symbol.                         |
//+------------------------------------------------------------------+
class CNewSymbol : public CObject
  {
   //--- === Data members === ---
private:
   string            m_name;
   string            m_path;
   MqlTick           m_tick;
   ulong             m_from_msc;
   ulong             m_to_msc;
   uint              m_batch_size;
   bool              m_is_selected;
   //--- === Methods === ---
public:
   //--- constructor/destructor
   void              CNewSymbol(void);
   void             ~CNewSymbol(void) {};
   //--- create/delete
   int               Create(const string _name,const string _path="",const string _origin_name=NULL,
                            const uint _batch_size=1e6,const bool _is_selected=false);
   bool              Delete(void);
   //--- methods of access to protected data
   string            Name(void) const { return(m_name); }
   bool              RefreshRates(void);
   //--- fast access methods to the integer symbol properties
   bool              Select(void) const;
   bool              Select(const bool select);
   //--- service methods
   bool              Clone(const string _origin_symbol,const ulong _from_msc=0,const ulong _to_msc=0);
   bool              LoadTicks(const string _src_file_name);
   //--- API
   bool              SetProperty(ENUM_SYMBOL_INFO_DOUBLE _property,double _val) const;
   bool              SetProperty(ENUM_SYMBOL_INFO_INTEGER _property,long _val) const;
   bool              SetProperty(ENUM_SYMBOL_INFO_STRING _property,string _val) const;
   double            GetProperty(ENUM_SYMBOL_INFO_DOUBLE _property) const;
   long              GetProperty(ENUM_SYMBOL_INFO_INTEGER _property) const;
   string            GetProperty(ENUM_SYMBOL_INFO_STRING _property) const;
   bool              SetSessionQuote(const ENUM_DAY_OF_WEEK _day_of_week,const uint _session_index,
                                     const datetime _from,const datetime _to);
   bool              SetSessionTrade(const ENUM_DAY_OF_WEEK _day_of_week,const uint _session_index,
                                     const datetime _from,const datetime _to);
   int               RatesDelete(const datetime _from,const datetime _to);
   int               RatesReplace(const datetime _from,const datetime _to,const MqlRates &_rates[]);
   int               RatesUpdate(const MqlRates &_rates[]) const;
   int               TicksAdd(const MqlTick &_ticks[]) const;
   int               TicksDelete(const long _from_msc,long _to_msc) const;
   int               TicksReplace(const MqlTick &_ticks[]) const;
   //---
private:
   template<typename PT>
   bool              CloneProperty(const string _origin_symbol,const PT _prop_type) const;
   int               CloneTicks(const MqlTick &_ticks[]) const;
   int               CloneTicks(const string _origin_symbol) const;
  };
//+------------------------------------------------------------------+
//| Constructor                                                      |
//+------------------------------------------------------------------+
void CNewSymbol::CNewSymbol(void)
  {
   m_name=m_path=NULL;
   m_from_msc=m_to_msc=0;
   m_batch_size=0;
   m_is_selected=false;
   ::ZeroMemory(m_tick);
  };
//+------------------------------------------------------------------+
//| Create a custom symbol                                           |
//| Codes:                                                           |
//|       -1 - failed to create;                                     |
//|        0 - a symbol exists, no need to create;                   |
//|        1 - successfully created.                                 |
//+------------------------------------------------------------------+
int CNewSymbol::Create(const string _name,const string _path="",const string _origin_name=NULL,
                           const uint _batch_size=1e6,const bool _is_selected=false)
  {
   int res_code=-1;
   m_name=m_path=NULL;
   if(_batch_size<1e2)
     {
      ::Print(__FUNCTION__+": a batch size must be greater than 100!");
     }
   else
     {
      ::ResetLastError();
      //--- attempt to create a custom symbol
      if(!::CustomSymbolCreate(_name,_path,_origin_name))
        {
         if(::SymbolInfoInteger(_name,SYMBOL_CUSTOM))
           {
            //::PrintFormat(__FUNCTION__+": a custom symbol \"%s\" already exists!",_name);
            res_code=0;
           }
         else
           {
            ::PrintFormat(__FUNCTION__+": failed to create a custom symbol. Error code: %d",::GetLastError());
           }
        }
      else
         res_code=1;
      if(res_code>=0)
        {
         m_name=_name;
         m_path=_path;
         m_batch_size=_batch_size;
         //--- if the custom symbol must be selected in the "Market Watch"
         if(_is_selected)
           {
            if(!this.Select())
               if(!this.Select(true))
                 {
                  ::PrintFormat(__FUNCTION__+": failed to set the \"Market Watch\" symbol flag. Error code: %d",::GetLastError());
                  return false;
                 }
           }
         else
           {
            if(this.Select())
               if(!this.Select(false))
                 {
                  ::PrintFormat(__FUNCTION__+": failed to unset the \"Market Watch\" symbol flag. Error code: %d",::GetLastError());
                  return false;
                 }
           }
         m_is_selected=_is_selected;
        }
     }
//---
   return res_code;
  }
//+------------------------------------------------------------------+
//| Delete                                                           |
//+------------------------------------------------------------------+
bool CNewSymbol::Delete(void)
  {
   ::ResetLastError();
   if(this.Select())
      if(!this.Select(false))
        {
         ::PrintFormat(__FUNCTION__+": failed to set the \"Market Watch\" symbol flag. Error code: %d",::GetLastError());
         return false;
        }
   if(!::CustomSymbolDelete(m_name))
     {
      ::PrintFormat(__FUNCTION__+": failed to delete the custom symbol \"%s\". Error code: %d",m_name,::GetLastError());
      return false;
     }
//---
   return true;
  }
//+------------------------------------------------------------------+
//| Refresh cached data                                              |
//+------------------------------------------------------------------+
bool CNewSymbol::RefreshRates(void)
  {
   for(int att=0; att<ATTEMTS; att++)
     {
      MqlTick ticks_arr[];
      if(::CopyTicks(m_name,ticks_arr,COPY_TICKS_INFO,0,1)==1)
        {
         m_tick=ticks_arr[0];
         return true;
        }
      ::Sleep(PAUSE);
     }
//---
   return false;
  }
//+------------------------------------------------------------------+
//| Get the property value "SYMBOL_SELECT"                           |
//+------------------------------------------------------------------+
bool CNewSymbol::Select(void) const
  {
   return (bool)::SymbolInfoInteger(m_name,SYMBOL_SELECT);
  }
//+------------------------------------------------------------------+
//| Set the property value "SYMBOL_SELECT"                           |
//+------------------------------------------------------------------+
bool CNewSymbol::Select(const bool select)
  {
   return ::SymbolSelect(m_name,select);
  }
//+------------------------------------------------------------------+
//| Double property                                                  |
//+------------------------------------------------------------------+
bool CNewSymbol::SetProperty(ENUM_SYMBOL_INFO_DOUBLE _property,double _val) const
  {
   return ::CustomSymbolSetDouble(m_name,_property,_val);
  }
//+------------------------------------------------------------------+
//| Integer property                                                 |
//+------------------------------------------------------------------+
bool CNewSymbol::SetProperty(ENUM_SYMBOL_INFO_INTEGER _property,long _val) const
  {
   return ::CustomSymbolSetInteger(m_name,_property,_val);
  }
//+------------------------------------------------------------------+
//| String property                                                  |
//+------------------------------------------------------------------+
bool CNewSymbol::SetProperty(ENUM_SYMBOL_INFO_STRING _property,string _val) const
  {
   return ::CustomSymbolSetString(m_name,_property,_val);
  }
//+------------------------------------------------------------------+
//| Double property                                                  |
//+------------------------------------------------------------------+
double CNewSymbol::GetProperty(ENUM_SYMBOL_INFO_DOUBLE _property) const
  {
   return ::SymbolInfoDouble(m_name,_property);
  }
//+------------------------------------------------------------------+
//| Integer property                                                 |
//+------------------------------------------------------------------+
long CNewSymbol::GetProperty(ENUM_SYMBOL_INFO_INTEGER _property) const
  {
   return ::SymbolInfoInteger(m_name,_property);
  }
//+------------------------------------------------------------------+
//| String property                                                  |
//+------------------------------------------------------------------+
string CNewSymbol::GetProperty(ENUM_SYMBOL_INFO_STRING _property) const
  {
   return ::SymbolInfoString(m_name,_property);
  }
//+------------------------------------------------------------------+
//| Clone a symbol                                                   |
//+------------------------------------------------------------------+
bool CNewSymbol::Clone(const string _origin_symbol,const ulong _from_msc=0,const ulong _to_msc=0)
  {
   if(!::StringCompare(m_name,_origin_symbol))
     {
      ::Print(__FUNCTION__+": the origin symbol name must be different!");
      return false;
     }
   ::ResetLastError();
//--- if to load history
   if(_to_msc>0)
     {
      if(_to_msc<_from_msc)
        {
         ::Print(__FUNCTION__+": wrong settings for a time interval!");
         return false;
        }
      m_from_msc=_from_msc;
      m_to_msc=_to_msc;
     }
   else
      m_from_msc=m_to_msc=0;
//--- double
   ENUM_SYMBOL_INFO_DOUBLE dbl_props[]=
     {
      SYMBOL_MARGIN_HEDGED,
      SYMBOL_MARGIN_INITIAL,
      SYMBOL_MARGIN_MAINTENANCE,
      SYMBOL_OPTION_STRIKE,
      SYMBOL_POINT,
      SYMBOL_SESSION_PRICE_LIMIT_MAX,
      SYMBOL_SESSION_PRICE_LIMIT_MIN,
      SYMBOL_SESSION_PRICE_SETTLEMENT,
      SYMBOL_SWAP_LONG,
      SYMBOL_SWAP_SHORT,
      SYMBOL_TRADE_ACCRUED_INTEREST,
      SYMBOL_TRADE_CONTRACT_SIZE,
      SYMBOL_TRADE_FACE_VALUE,
      SYMBOL_TRADE_LIQUIDITY_RATE,
      SYMBOL_TRADE_TICK_SIZE,
      SYMBOL_TRADE_TICK_VALUE,
      SYMBOL_VOLUME_LIMIT,
      SYMBOL_VOLUME_MAX,
      SYMBOL_VOLUME_MIN,
      SYMBOL_VOLUME_STEP
     };
   for(int prop_idx=0; prop_idx<::ArraySize(dbl_props); prop_idx++)
     {
      ENUM_SYMBOL_INFO_DOUBLE curr_property=dbl_props[prop_idx];
      if(!this.CloneProperty(_origin_symbol,curr_property))
         {
          ::PrintFormat(__FUNCTION__+": failed to set %s property, error is %d",EnumToString(curr_property), GetLastError());
          return false;
         }
     }
//--- integer
   ENUM_SYMBOL_INFO_INTEGER int_props[]=
     {
      SYMBOL_BACKGROUND_COLOR,
      SYMBOL_CHART_MODE,
      SYMBOL_DIGITS,
      SYMBOL_EXPIRATION_MODE,
      SYMBOL_EXPIRATION_TIME,
      SYMBOL_FILLING_MODE,
      SYMBOL_MARGIN_HEDGED_USE_LEG,
      SYMBOL_OPTION_MODE,
      SYMBOL_OPTION_RIGHT,
      SYMBOL_ORDER_GTC_MODE,
      SYMBOL_ORDER_MODE,
      SYMBOL_SPREAD,
      SYMBOL_SPREAD_FLOAT,
      SYMBOL_START_TIME,
      SYMBOL_SWAP_MODE,
      SYMBOL_SWAP_ROLLOVER3DAYS,
      SYMBOL_TICKS_BOOKDEPTH,// https://www.mql5.com/ru/forum/170952/page85#comment_7227865
      SYMBOL_TRADE_CALC_MODE,
      SYMBOL_TRADE_EXEMODE,
      SYMBOL_TRADE_FREEZE_LEVEL,
      SYMBOL_TRADE_MODE,
      SYMBOL_TRADE_STOPS_LEVEL
     };
   for(int prop_idx=0; prop_idx<::ArraySize(int_props); prop_idx++)
     {
      ENUM_SYMBOL_INFO_INTEGER curr_property=int_props[prop_idx];
      if(!this.CloneProperty(_origin_symbol,curr_property))
         {
          ::PrintFormat(__FUNCTION__+": failed to set %s property, error is %d",EnumToString(curr_property), GetLastError());
          return false;
         }
     }
//--- string
   ENUM_SYMBOL_INFO_STRING str_props[]=
     {
      SYMBOL_BASIS,
      SYMBOL_CURRENCY_BASE,
      SYMBOL_CURRENCY_MARGIN,
      SYMBOL_CURRENCY_PROFIT,
      SYMBOL_DESCRIPTION,
      SYMBOL_FORMULA,
      SYMBOL_ISIN,
      SYMBOL_PAGE,
      SYMBOL_PATH
     };
   for(int prop_idx=0; prop_idx<::ArraySize(str_props); prop_idx++)
     {
      ENUM_SYMBOL_INFO_STRING curr_property=str_props[prop_idx];
      if(!this.CloneProperty(_origin_symbol,curr_property))
         {
          ::PrintFormat(__FUNCTION__+": failed to set %s property, error is %d",EnumToString(curr_property), GetLastError());
          return false;
         }
     }
//--- history
   if(_to_msc>0)
     {
      if(this.CloneTicks(_origin_symbol)==-1)
         return false;
     }
//---
   return true;
  }
//+------------------------------------------------------------------+
//| Load ticks                                                       |
//+------------------------------------------------------------------+
bool CNewSymbol::LoadTicks(const string _src_file_name)
  {
   int symbol_digs=(int)this.GetProperty(SYMBOL_DIGITS);;
//--- delete ticks
   if(this.TicksDelete(0,LONG_MAX)<0)
      return false;
//--- open a file
   CFile curr_file;
   ::ResetLastError();
   int file_ha=curr_file.Open(_src_file_name,FILE_READ|FILE_CSV,',');
   if(file_ha==INVALID_HANDLE)
     {
      ::PrintFormat(__FUNCTION__+": failed to open a %s file!",_src_file_name);
      return false;
     }
   curr_file.Seek(0,SEEK_SET);
//--- read data from a file
   MqlTick batch_arr[];
   if(::ArrayResize(batch_arr,m_batch_size)!=m_batch_size)
     {
      ::Print(__FUNCTION__+": failed to allocate memory for a batch array!");
      return false;
     }
   ::ZeroMemory(batch_arr);
   uint tick_idx=0;
   bool is_file_ending=false;
   uint tick_cnt=0;
   do
     {
      is_file_ending=curr_file.IsEnding();
      string dates_str[2];
      if(!is_file_ending)
        {
         //--- time
         string time_str=::FileReadString(file_ha);
         if(::StringLen(time_str)<1)
           {
            ::Print(__FUNCTION__+": no datetime string - the current tick skipped!");
            ::PrintFormat("The unprocessed string: %s",time_str);
            continue;
           }
         string sep=".";
         ushort u_sep;
         string result[];
         u_sep=::StringGetCharacter(sep,0);
         int str_num=::StringSplit(time_str,u_sep,result);
         if(str_num!=4)
           {
            ::Print(__FUNCTION__+": no substrings - the current tick skipped!");
            ::PrintFormat("The unprocessed string: %s",time_str);
            continue;
           }
         //--- datetime
         datetime date_time=::StringToTime(result[0]+"."+result[1]+"."+result[2]);
         long time_msc=(long)(1e3*date_time+::StringToInteger(result[3]));
         //--- bid
         double bid_val=::FileReadNumber(file_ha);
         if(bid_val<.0)
           {
            ::Print(__FUNCTION__+": no bid price - the current tick skipped!");
            continue;
           }
         //--- ask
         double ask_val=::FileReadNumber(file_ha);
         if(ask_val<.0)
           {
            ::Print(__FUNCTION__+": no ask price - the current tick skipped!");
            continue;
           }
         //--- volumes
         for(int jtx=0; jtx<2; jtx++)
            ::FileReadNumber(file_ha);
         //--- fill in the current tick
         MqlTick curr_tick= {0};
         curr_tick.time=date_time;
         curr_tick.time_msc=(long)(1e3*date_time+::StringToInteger(result[3]));
         curr_tick.bid=::NormalizeDouble(bid_val,symbol_digs);
         curr_tick.ask=::NormalizeDouble(ask_val,symbol_digs);
         //--- flags
         if(m_tick.bid!=curr_tick.bid)
            curr_tick.flags|=TICK_FLAG_BID;
         if(m_tick.ask!=curr_tick.ask)
            curr_tick.flags|=TICK_FLAG_ASK;
         if(curr_tick.flags==0)
            curr_tick.flags=TICK_FLAG_BID|TICK_FLAG_ASK;;
         if(tick_idx==m_batch_size)
           {
            //--- add ticks to the custom symbol
            if(m_is_selected)
              {
               if(this.TicksAdd(batch_arr)!=m_batch_size)
                  return false;
              }
            else
              {
               if(this.TicksReplace(batch_arr)!=m_batch_size)
                  return false;
              }
            tick_cnt+=m_batch_size;
            //--- log
            for(uint idx=0,batch_idx=0; idx<uint(::ArraySize(dates_str)); idx++,batch_idx+=(m_batch_size-1))
               dates_str[idx]=::TimeToString(batch_arr[batch_idx].time,TIME_DATE|TIME_SECONDS);
            ::PrintFormat("\nTicks loaded from %s to %s.",dates_str[0],dates_str[1]);
            //--- reset
            ::ZeroMemory(batch_arr);
            tick_idx=0;
           }
         batch_arr[tick_idx]=curr_tick;
         m_tick=curr_tick;
         tick_idx++;
        }
      //--- end of file
      else
        {
         uint new_size=tick_idx;
         if(new_size>0)
           {
            MqlTick last_batch_arr[];
            if(::ArrayCopy(last_batch_arr,batch_arr,0,0,new_size)!=new_size)
              {
               ::Print(__FUNCTION__+": failed to copy a batch array!");
               return false;
              }
            //--- add ticks to the custom symbol
            if(m_is_selected)
              {
               if(this.TicksAdd(last_batch_arr)!=new_size)
                  return false;
              }
            else
              {
               if(this.TicksReplace(last_batch_arr)!=new_size)
                  return false;
              }
            tick_cnt+=new_size;
            //--- log
            for(uint idx=0,batch_idx=0; idx<uint(::ArraySize(dates_str)); idx++,batch_idx+=(tick_idx-1))
               dates_str[idx]=::TimeToString(batch_arr[batch_idx].time,TIME_DATE|TIME_SECONDS);
            ::PrintFormat("\nTicks loaded from %s to %s.",dates_str[0],dates_str[1]);
           }
        }
     }
   while(!is_file_ending && !::IsStopped());
   ::PrintFormat("\nLoaded ticks number: %I32u",tick_cnt);
   curr_file.Close();
//---
   MqlTick ticks_arr[];
   if(::CopyTicks(m_name,ticks_arr,COPY_TICKS_INFO,1,1)!=1)
     {
      ::Print(__FUNCTION__+": failed to copy the first tick!");
      return false;
     }
   m_from_msc=ticks_arr[0].time_msc;
   if(::CopyTicks(m_name,ticks_arr,COPY_TICKS_INFO,0,1)!=1)
     {
      ::Print(__FUNCTION__+": failed to copy the last tick!");
      return false;
     }
   m_to_msc=ticks_arr[0].time_msc;
//---
   return true;
  }

//+------------------------------------------------------------------+
//| Set session quote                                                |
//+------------------------------------------------------------------+
bool CNewSymbol::SetSessionQuote(const ENUM_DAY_OF_WEEK _day_of_week,const uint _session_index,
                                     const datetime _from,const datetime _to)
  {
   if(!::SymbolInfoInteger(m_name,SYMBOL_CUSTOM))
     {
      ::PrintFormat(__FUNCTION__+": a symbol %s does not exist!",m_name);
      return false;
     }
//---
   ::ResetLastError();
   if(!::CustomSymbolSetSessionQuote(m_name,_day_of_week,_session_index,_from,_to))
     {
      ::PrintFormat(__FUNCTION__+": failed to set a quote session! Error code: %d",::GetLastError());
      return false;
     }
//---
   return true;
  }
//+------------------------------------------------------------------+
//| Set session quote                                                |
//+------------------------------------------------------------------+
bool CNewSymbol::SetSessionTrade(const ENUM_DAY_OF_WEEK _day_of_week,const uint _session_index,
                                     const datetime _from,const datetime _to)
  {
   if(!::SymbolInfoInteger(m_name,SYMBOL_CUSTOM))
     {
      ::PrintFormat(__FUNCTION__+": a symbol %s does not exist!",m_name);
      return false;
     }
//---
   ::ResetLastError();
   if(!::CustomSymbolSetSessionTrade(m_name,_day_of_week,_session_index,_from,_to))
     {
      ::PrintFormat(__FUNCTION__+" failed to set a trade session! Error code: %d",::GetLastError());
      return false;
     }
//---
   return true;
  }
//+------------------------------------------------------------------+
//| Delete rates                                                     |
//+------------------------------------------------------------------+
int CNewSymbol::RatesDelete(const datetime _from,const datetime _to)
  {
   ::ResetLastError();
   int deleted_rates_num=::CustomRatesDelete(m_name,_from,_to);
   if(deleted_rates_num<0)
      ::PrintFormat(__FUNCTION__+": failed to delete rates! Error code: %d",::GetLastError());
//---
   return deleted_rates_num;
  }
//+------------------------------------------------------------------+
//| Replace rates                                                    |
//+------------------------------------------------------------------+
int CNewSymbol::RatesReplace(const datetime _from,const datetime _to,const MqlRates  &_rates[])
  {
   ::ResetLastError();
   int replaced_rates_num=::CustomRatesReplace(m_name,_from,_to,_rates);
   if(replaced_rates_num<0)
      ::PrintFormat(__FUNCTION__+": failed to replace rates! Error code: %d",::GetLastError());
//---
   return replaced_rates_num;
  }
//+------------------------------------------------------------------+
//| Update rates                                                     |
//+------------------------------------------------------------------+
int CNewSymbol::RatesUpdate(const MqlRates &_rates[]) const
  {
   ::ResetLastError();
   int updated_rates_num=::CustomRatesUpdate(m_name,_rates);
   if(updated_rates_num<0)
      ::PrintFormat(__FUNCTION__+": failed to update rates! Error code: %d",::GetLastError());
//---
   return updated_rates_num;
  }
//+------------------------------------------------------------------+
//| Add ticks                                                        |
//+------------------------------------------------------------------+
int CNewSymbol::TicksAdd(const MqlTick &_ticks[]) const
  {
   int added_ticks_num=0;
   int ticks_arr_size=::ArraySize(_ticks);
   if(ticks_arr_size>0)
     {
      ::ResetLastError();
      added_ticks_num=::CustomTicksAdd(m_name,_ticks);
      if(added_ticks_num<0)
         ::PrintFormat(__FUNCTION__+": failed to add ticks! Error code: %d",::GetLastError());
     }
//---
   return added_ticks_num;
  }
//+------------------------------------------------------------------+
//| Delete ticks                                                     |
//+------------------------------------------------------------------+
int CNewSymbol::TicksDelete(const long _from_msc,long _to_msc) const
  {
   ::ResetLastError();
   int deleted_ticks_num=::CustomTicksDelete(m_name,_from_msc,_to_msc);
   if(deleted_ticks_num<0)
      ::PrintFormat(__FUNCTION__+": failed to delete ticks! Error code: %d",::GetLastError());
//---
   return deleted_ticks_num;
  }
//+------------------------------------------------------------------+
//| Replace ticks                                                    |
//+------------------------------------------------------------------+
int CNewSymbol::TicksReplace(const MqlTick &_ticks[]) const
  {
   int replaced_ticks_num=WRONG_VALUE;
   ulong from_msc,to_msc;
   int ticks_arr_size=::ArraySize(_ticks);
   if(ticks_arr_size>0)
     {
      from_msc=_ticks[0].time_msc;
      to_msc=_ticks[ticks_arr_size-1].time_msc;
      ::ResetLastError();
      replaced_ticks_num=::CustomTicksReplace(m_name,from_msc,to_msc,_ticks);
      if(replaced_ticks_num<0)
         ::PrintFormat(__FUNCTION__+": failed to replace ticks! Error code: %d",::GetLastError());
     }
//---
   return replaced_ticks_num;
  }
//+------------------------------------------------------------------+
//| Clone property                                                   |
//+------------------------------------------------------------------+
template<typename PT>
bool CNewSymbol::CloneProperty(const string _origin_symbol,const PT _prop_type) const
  {
   bool is_prop_cloned=false;
   string prop_type=typename(_prop_type);
   if(!::StringCompare(prop_type,"enum ENUM_SYMBOL_INFO_DOUBLE"))
     {
      ENUM_SYMBOL_INFO_DOUBLE curr_property=(ENUM_SYMBOL_INFO_DOUBLE)_prop_type;
      double curr_prop_val;
      if(!::SymbolInfoDouble(_origin_symbol,curr_property,curr_prop_val))
        {
         ::PrintFormat(__FUNCTION__+": failed to get a double-property %s!",::EnumToString(_prop_type));
         return false;
        }
      ::ResetLastError();
      if(!this.SetProperty(curr_property,curr_prop_val))
        {
         ::PrintFormat(__FUNCTION__+": failed to set a double-property %s! Error code: %d",
                       ::EnumToString(_prop_type),::GetLastError());
         return false;
        }
      is_prop_cloned=true;
     }
   else
      if(!::StringCompare(prop_type,"enum ENUM_SYMBOL_INFO_INTEGER"))
        {
         ENUM_SYMBOL_INFO_INTEGER curr_property=(ENUM_SYMBOL_INFO_INTEGER)_prop_type;
         long curr_prop_val;
         if(!::SymbolInfoInteger(_origin_symbol,curr_property,curr_prop_val))
           {
            ::PrintFormat(__FUNCTION__+": failed to get an int-property %s!",::EnumToString(curr_property));
            return false;
           }
         ::ResetLastError();
         if(!this.SetProperty(curr_property,curr_prop_val))
           {
            ::PrintFormat(__FUNCTION__+": failed to set an int-property %s! Error code: %d",
                          ::EnumToString(curr_property),::GetLastError());
            return false;
           }
         is_prop_cloned=true;
        }
      else
         if(!::StringCompare(prop_type,"enum ENUM_SYMBOL_INFO_STRING"))
           {
            ENUM_SYMBOL_INFO_STRING curr_property=(ENUM_SYMBOL_INFO_STRING)_prop_type;
            string curr_prop_val;
            if(!::SymbolInfoString(_origin_symbol,curr_property,curr_prop_val))
              {
               ::PrintFormat(__FUNCTION__+": failed to get a string-property %s!",::EnumToString(curr_property));
               return false;
              }
            ::ResetLastError();
            if(!this.SetProperty(curr_property,curr_prop_val))
              {
               ::PrintFormat(__FUNCTION__+": failed to set a string-property %s! Error code: %d",
                             ::EnumToString(curr_property),::GetLastError());
               return false;
              }
            is_prop_cloned=true;
           }
//---
   return is_prop_cloned;
  }
//+------------------------------------------------------------------+
//| Clone ticks                                                      |
//+------------------------------------------------------------------+
int CNewSymbol::CloneTicks(const MqlTick &_ticks[]) const
  {
   int cloned_ticks_num=WRONG_VALUE;
//---
   ::ResetLastError();
   int ticks_arr_size=ArraySize(_ticks);
//--- try to replace
   for(int att=0; att<ATTEMTS; att++)
     {
      int ticks_num=0;
      if(m_is_selected)
         ticks_num=this.TicksAdd(_ticks);
      else
         ticks_num=this.TicksReplace(_ticks);
      if(ticks_num==ticks_arr_size)
        {
         cloned_ticks_num=ticks_num;
         break;
        }
      ::Sleep(PAUSE);
     }
//---
   return cloned_ticks_num;
  }
//+------------------------------------------------------------------+
//| Clone ticks                                                      |
//+------------------------------------------------------------------+
int CNewSymbol::CloneTicks(const string _origin_symbol) const
  {
   int cloned_ticks_num=WRONG_VALUE;
   MqlTick ticks[];
   ::ResetLastError();
   if(::CopyTicksRange(_origin_symbol,ticks,COPY_TICKS_INFO,m_from_msc,m_to_msc)<0)
      ::PrintFormat(__FUNCTION__+": failed to copy ticks for a %s symbol! Error code: %d",
                    _origin_symbol,::GetLastError());
   else
      cloned_ticks_num=this.CloneTicks(ticks);
//---
   return cloned_ticks_num;
  }
//+------------------------------------------------------------------+

//--- [EOF]
