//+------------------------------------------------------------------+
//|                                           PermutedSymbolData.mqh |
//|                                  Copyright 2023, MetaQuotes Ltd. |
//|                                             https://www.mql5.com |
//+------------------------------------------------------------------+
#property copyright "Copyright 2023, MetaQuotes Ltd."
#property link      "https://www.mql5.com"
#include<PermuteTicks.mqh>
#include<PermuteRates.mqh>
#include<NewSymbol.mqh>

//+------------------------------------------------------------------+
//|Permute rates or ticks of symbol                                  |
//+------------------------------------------------------------------+
enum ENUM_RATES_TICKS
{
 ENUM_USE_RATES=0,//Use rates
 ENUM_USE_TICKS//Use ticks
};
//+------------------------------------------------------------------+
//| defines:max number of data download attempts and array resize    |
//+------------------------------------------------------------------+
#define MAX_DOWNLOAD_ATTEMPTS 10 
#define RESIZE_RESERVE 100
//+------------------------------------------------------------------+
//|CPermuteSymbolData class                                          |
//| creates custom symbols from an existing base symbol's  data      |
//|  symbols represent permutations of base symbol's data            |
//+------------------------------------------------------------------+
class CPermuteSymbolData
{
 private:
   ENUM_RATES_TICKS m_use_rates_or_ticks;//permute either ticks or rates
   string           m_basesymbol;        //base symbol
   string           m_symbols_id;        //common identifier added to names of new symbols 
   datetime         m_datarangestart;    //beginning date for range of base symbol's data
   datetime         m_datarangestop;     //ending date for range of base symbol's data
   uint             m_permutations;      //number of permutations and ultimately the number of new symbols to create
   MqlTick          m_baseticks[];       //base symbol's tick
   MqlTick          m_permutedticks[];   //permuted ticks;
   MqlRates         m_baserates[];       //base symbol's rates
   MqlRates         m_permutedrates[];   //permuted rates;
   CPermuteRates   *m_rates_shuffler;    //object used to shuffle rates
   CPermuteTicks   *m_ticks_shuffler;    //object used to shuffle ticks
   CNewSymbol      *m_csymbols[];        //array of created symbols
   
 public: 
   CPermuteSymbolData(const ENUM_RATES_TICKS mode);
   ~CPermuteSymbolData(void);                      
   bool Initiate(const string base_symbol,const string symbols_id,const datetime start_date,const datetime stop_date);
   uint Generate(const uint permutations );
};
//+------------------------------------------------------------------+
//|Constructor                                                       |
//+------------------------------------------------------------------+
CPermuteSymbolData::CPermuteSymbolData(const ENUM_RATES_TICKS mode):m_basesymbol(NULL),
                       m_symbols_id(NULL),
                       m_permutations(0),
                       m_datarangestart(0),
                       m_datarangestop(0)
{
  m_use_rates_or_ticks = mode;
  m_rates_shuffler=NULL;
  m_ticks_shuffler=NULL; 
} 
//+------------------------------------------------------------------+
//|Destructor                                                        |
//+------------------------------------------------------------------+
CPermuteSymbolData::~CPermuteSymbolData(void)
{
 if(CheckPointer(m_ticks_shuffler)==POINTER_DYNAMIC)
        delete m_ticks_shuffler;
        
 if(CheckPointer(m_rates_shuffler)==POINTER_DYNAMIC)
        delete m_rates_shuffler;
               
 for(uint i=0;i<m_csymbols.Size();i++)
    if(CheckPointer(m_csymbols[i])==POINTER_DYNAMIC)
        delete m_csymbols[i];
}
//+-----------------------------------------------------------------------------------------+
//|set and check parameters for symbol creation, download data and initialize data shuffler |
//+-----------------------------------------------------------------------------------------+
bool CPermuteSymbolData::Initiate(const string base_symbol,const string symbols_id,const datetime start_date,const datetime stop_date)
{
//---reset number of permutations previously done
 m_permutations=0;
//---set base symbol
 m_basesymbol=base_symbol;
//---make sure base symbol is selected, ie, visible in WatchList 
 if(!SymbolSelect(m_basesymbol,true))
  {
   Print("Failed to select ", m_basesymbol," error ", GetLastError());
   return false;
  }
//---set symbols id 
 m_symbols_id=symbols_id;
//---check, set data date range
 if(start_date>=stop_date)
   {
    Print("Invalid date range ");
    return false;
   }
 else
   {
    m_datarangestart= start_date;
    m_datarangestop = stop_date;
   }  
//---download data
   Comment("Downloading data");
   uint attempts=0;
   int downloaded=-1;
    while(attempts<MAX_DOWNLOAD_ATTEMPTS && !IsStopped())
     {
      downloaded=(m_use_rates_or_ticks==ENUM_USE_TICKS)?CopyTicksRange(m_basesymbol,m_baseticks,COPY_TICKS_ALL,long(m_datarangestart)*1000,long(m_datarangestop)*1000):CopyRates(m_basesymbol,PERIOD_M1,m_datarangestart,m_datarangestop,m_baserates);
      if(downloaded<=0)
        {
         Sleep(500);
         ++attempts;
        }
      else 
        break;   
     }
//---check download result
   if(downloaded<=0)
    {
     Print("Failed to download data for ",m_basesymbol," error ", GetLastError());
     Comment("");
     return false;
    }      
           
  //Print(downloaded," Ticks downloaded ", " data start ",m_basedata[0].time, " data end ", m_basedata[m_basedata.Size()-1].time);  
//---return shuffler initialization result 
  switch(m_use_rates_or_ticks)  
      {
       case ENUM_USE_TICKS:
         {
          if(m_ticks_shuffler==NULL)
             m_ticks_shuffler=new CPermuteTicks();
          return m_ticks_shuffler.Initialize(m_baseticks);
         } 
       case ENUM_USE_RATES: 
         {
          if(m_rates_shuffler==NULL)
             m_rates_shuffler=new CPermuteRates(); 
          return m_rates_shuffler.Initialize(m_baserates);
         }
       default:
          return false; 
      }               
}                      
//+------------------------------------------------------------------+
//| generate symbols return newly created or refreshed symbols       |
//+------------------------------------------------------------------+
uint CPermuteSymbolData::Generate(const uint permutations)
{
//---check permutations
 if(!permutations)
   {
    Print("Invalid parameter value for Permutations ");
    Comment("");
    return 0;
   } 
//---resize m_csymbols
  if(m_csymbols.Size()!=m_permutations+permutations)
    ArrayResize(m_csymbols,m_permutations+permutations,RESIZE_RESERVE);
//---
  string symspath=m_basesymbol+m_symbols_id+"_PermutedData"; 
  //int exists;
//---do more permutations
  for(uint i=m_permutations; i<m_csymbols.Size() && !IsStopped(); i++)
      {
       if(CheckPointer(m_csymbols[i])==POINTER_INVALID)
             m_csymbols[i]=new CNewSymbol();
              
       if(m_csymbols[i].Create(m_basesymbol+m_symbols_id+"_"+string(i+1),symspath,m_basesymbol)<0)
            continue;
       
       Comment("Processing Symbol "+m_basesymbol+m_symbols_id+"_"+string(i+1) );
       if(!m_csymbols[i].Clone(m_basesymbol) || 
         (m_use_rates_or_ticks==ENUM_USE_TICKS && !m_ticks_shuffler.Permute(m_permutedticks)) ||
         (m_use_rates_or_ticks==ENUM_USE_RATES && !m_rates_shuffler.Permute(m_permutedrates))  )
            break;
       else
           {
            m_csymbols[i].Select(true);
            Comment("Adding permuted data");
            if(m_use_rates_or_ticks==ENUM_USE_TICKS)
                m_permutations+=(m_csymbols[i].TicksReplace(m_permutedticks)>0)?1:0;  
            else
                m_permutations+=(m_csymbols[i].RatesUpdate(m_permutedrates)>0)?1:0; 
           }              
       }   
//---return successfull number of permutated symbols
 Comment("");
//--- 
 if(IsStopped())
   return 0;
//---   
 return m_permutations;
}