//+------------------------------------------------------------------+
//|                                                        Janus.mqh |
//|                        Copyright 2023, MetaQuotes Software Corp. |
//|                                             https://www.mql5.com |
//+------------------------------------------------------------------+
#property copyright "Copyright 2023, MetaQuotes Software Corp."
#property link      "https://www.mql5.com"
#include<Math\Stat\Math.mqh>
#include <Arrays\ArrayObj.mqh>
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
enum ENUM_INDEX_TYPE
  {
   INDEX_FOREX_MAJORS=0,//forex majors only list
   INDEX_CUSTOM,//custom symbol list
  };
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
enum ENUM_PRICE
  {
   CLOSE=0,//close price
   MEDIAN,//median price
   TYPICAL,//typical price
   WEIGHTED//weighted price
  };
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
enum ENUM_DIFF_TYPE
  {
   DIFF_PERCENT=0,//percent difference
   DIFF_LOG//log difference
  };


//+------------------------------------------------------------------+
//|Class which manage the single Symbol                              |
//+------------------------------------------------------------------+
class CSymbolData
  {

private:
   string            m_name;    // name of the symbol
   ENUM_TIMEFRAMES   m_timeframe;// timeframe
   int               m_length;  // length for copy rates
   MqlRates          m_rates[]; // store rates
   datetime          m_first;   // first date on server or local history

   datetime          SetFirstDate(void)
     {
      datetime first_date=-1;
      if((datetime)SymbolInfoInteger(m_name,SYMBOL_TIME)>0)
         first_date=(datetime)SeriesInfoInteger(m_name,m_timeframe,SERIES_FIRSTDATE);
      //---
      if(first_date==WRONG_VALUE || first_date==0)
        {
         if(TerminalInfoInteger(TERMINAL_CONNECTED))
           {
            while(!SeriesInfoInteger(m_name,m_timeframe,SERIES_SERVER_FIRSTDATE,first_date) && !IsStopped())
               Sleep(10);
           }
        }
      //---
#ifdef DEBUG
      Print(m_name," FirstDate ",first_date);
#endif
      return first_date;
     }

public:
                     CSymbolData(string name,ENUM_TIMEFRAMES tf=PERIOD_CURRENT)
     {
      m_name = name;
      m_length = 0;
      m_timeframe = tf;
      SymbolSelect(m_name,true);
     }

                    ~CSymbolData(void)
     {
      ArrayFree(m_rates);
     }
   datetime          GetFirstDate(void)
     {
      m_first = SetFirstDate();
      return m_first;
     }
   string            GetName(void)
     {
      return m_name;
     }

   int               GetLength(void)
     {
      return m_length;
     }

   void              SetLength(const int set_length)
     {
      if(set_length>0)
        {
         m_length=set_length;
         ArrayResize(m_rates,m_length,m_length);
         ArraySetAsSeries(m_rates,true);
        }
     }

   bool              Update(void)
     {
      int copied = CopyRates(m_name,m_timeframe,0,m_length,m_rates);
#ifdef DEBUG
      Print("copied ", copied, " requested ", m_length);
#endif
      //--
      return copied == m_length;
     };

   MqlRates          GetRateAtPos(const int i)
     {
      if(i<0 || i>=m_length)
        {
#ifdef DEBUG
         Print("Array out of range ",i,".Size of array is ",m_length);
#endif
         return (i<0)?m_rates[0]:m_rates[m_length-1];
        }
      return m_rates[i];
     }
  };

//+------------------------------------------------------------------+
//| Class that mange the collection of symbols                       |
//+------------------------------------------------------------------+
class CSymbolCollection
  {

private:

   int               m_calculation_length; // global length
   ENUM_TIMEFRAMES   m_collection_timeframe;  // timeframe of data
   string            m_raw_symbols;    // delimited symbol list
   datetime          m_synced_first;  // synced first bar opentime for all symbols
   bool              m_synced;        // flag of whether all symbols are synchronized
   CSymbolData       *m_collection[]; // Collection of Symbol Pointer
   int               m_collection_length; // Collection of Symbol Length
   //+------------------------------------------------------------------+
   //|                                                                  |
   //+------------------------------------------------------------------+
   bool               CheckSymbolBars(const string __sym)
     {
      int bars=-1;
      bars=iBarShift(__sym,m_collection_timeframe,m_synced_first)+1;//SeriesInfoInteger(__sym,PERIOD_CURRENT,SERIES_BARS_COUNT);
#ifdef DEBUG
      Print("Bars found in history for ",__sym," ",bars);
#endif
      if(bars>=m_calculation_length)
         return(true);
      //---
      return(SyncSymbol(__sym));
     }
   //+------------------------------------------------------------------+
   //|                                                                  |
   //+------------------------------------------------------------------+
   bool               SyncSymbol(const string __sym)
     {
      //--- load data step by step
      bool downloaded=false;
      datetime times[1];
      int bars=-1;
      
     /* if(MQLInfoInteger(MQL_PROGRAM_TYPE)==PROGRAM_INDICATOR)
        {
#ifdef DEBUG
         Print(" cannot download ",__sym," history from an indicator");
#endif
         return(downloaded);
        }*/
          
#ifdef DEBUG
      Print(" downloading ",__sym," history");
#endif
      while(!IsStopped() && !downloaded && TerminalInfoInteger(TERMINAL_CONNECTED))
        {
         if(MQLInfoInteger(MQL_PROGRAM_TYPE)==PROGRAM_INDICATOR)
           {
#ifdef DEBUG
            Print("Indicators will not download history data, use script to download history first");
#endif            
            break;
           } 
         //---
         while(!SeriesInfoInteger(__sym,m_collection_timeframe,SERIES_SYNCHRONIZED) && !IsStopped())
            Sleep(5);
         //---
         bars=Bars(__sym,PERIOD_CURRENT);
         if(bars>=m_calculation_length)
           {
            downloaded=true;
            break;
           }
         //--- copying of next part forces data loading
         if(CopyTime(__sym,m_collection_timeframe,m_calculation_length-1,1,times)==1)
           {
            downloaded=true;
            break;
           }
         //---
         Sleep(5);
        }
#ifdef DEBUG
      if(downloaded)
         Print(bars," ",__sym," bars downloaded ");
      else
         Print("Downloading ",__sym," bars failed");
#endif
      return(downloaded);
     }

public:

                     CSymbolCollection(const ENUM_TIMEFRAMES tf=PERIOD_CURRENT)
     {
      m_raw_symbols="";
      m_collection_length = 0;
      m_calculation_length = -1;
      m_synced_first=0;
      m_synced=false;
      m_collection_timeframe=tf;
     }

                    ~CSymbolCollection(void)
     {

      for(int i=0; i<m_collection_length; i++)
        {
         if(CheckPointer(m_collection[i])==POINTER_DYNAMIC)
            delete m_collection[i];
        }
     }
   //+------------------------------------------------------------------+
   //|return the set timeframe for bars stored in the collection        |
   //+------------------------------------------------------------------+
   ENUM_TIMEFRAMES   GetTimeFrame(void)
     {
      return(m_collection_timeframe);
     }
   
   //+------------------------------------------------------------------+
   //|Checks the history avaiable and sycns it across all symbols       |
   //+------------------------------------------------------------------+
   bool              CheckHistory(const int size)
     {
      if(size<=0)
         return(false);

      int available=iBarShift(NULL,m_collection_timeframe,m_synced_first)+1;

      if(available<size)
         m_calculation_length=available;
      else
         m_calculation_length=size;

#ifdef DEBUG
      Print("synced first date is ", m_synced_first);
      Print("Proposed size of history ",m_calculation_length);
#endif

      if(m_calculation_length<=0)
         return(false);

      ResetLastError();

      for(int i=0; i<m_collection_length; i++)
        {
         m_synced=CheckSymbolBars(m_collection[i].GetName());
         if(!m_synced)
           {
            Print("Not Enough history data for ", m_collection[i].GetName(), " > ", GetLastError());
            return(m_synced);
           }
         m_collection[i].SetLength(m_calculation_length);
        }

      return m_synced;

     }


   //+------------------------------------------------------------------+
   //| Add a symbol by name to the collection                           |
   //+------------------------------------------------------------------+
   int               Add(string name)
     {
      CSymbolData *ref = new CSymbolData(name,m_collection_timeframe);
      datetime f=ref.GetFirstDate();
      int found=GetIndex(name);
      if(f==WRONG_VALUE || found>-1)
        {
#ifdef DEBUG
         if(f==WRONG_VALUE)
            Print("Failed to retrieve information for symbol ",name,". Symbol removed from collection");
         if(found>-1)
            Print("Symbol ",name,"already part of collection");
#endif
         delete ref;
         return(m_collection_length);
        }
      ArrayResize(m_collection, m_collection_length + 1,1);
      m_collection[m_collection_length] = ref;
      //---
      if(f>m_synced_first)
         m_synced_first=f;
      //---

      return(++m_collection_length);

     }
   //+------------------------------------------------------------------+
   //|Return symbol name                                                |
   //+------------------------------------------------------------------+
   string            GetSymbolNameAtPos(int pos)
     {
      return m_collection[pos].GetName();
     }
   //+------------------------------------------------------------------+
   //|return index of symbol                                            |
   //+------------------------------------------------------------------+
   int               GetIndex(const string symbol_name)
     {
      for(int i=0; i<m_collection_length; i++)
        {
         if(symbol_name==m_collection[i].GetName())
            return(i);
        }
      //---fail
      return(-1);
     }
   //+------------------------------------------------------------------+
   //| Return Collection length                                         |
   //+------------------------------------------------------------------+
   int               GetCollectionLength(void)
     {
      return m_collection_length;
     }

   //+------------------------------------------------------------------+
   //| Update every currency rates                                      |
   //+------------------------------------------------------------------+
   bool              Update(void)
     {

      int i;
      for(i = 0; i < m_collection_length; i++)
        {
         bool res = m_collection[i].Update();
         if(res==false)
           {
            Print("missing data on " + m_collection[i].GetName());
            return false;
           }
        }
      return true;
     }
   //+------------------------------------------------------------------+
   //|                                                                  |
   //+------------------------------------------------------------------+
   int               GetHistoryBarsLength(void)
     {
      return m_calculation_length;
     }
   //+------------------------------------------------------------------+
   //| Return MqlRates of currency at position                          |
   //+------------------------------------------------------------------+
   MqlRates          GetRateAtPos(int pos, int i)
     {
      return m_collection[pos].GetRateAtPos(i);
     }

   //+------------------------------------------------------------------+
   //| Return Open price of currency at position                        |
   //+------------------------------------------------------------------+
   double            GetOpenAtPos(int pos, int i)
     {
      return m_collection[pos].GetRateAtPos(i).open;
     }

   //+------------------------------------------------------------------+
   //| Return Close price of currency at position                       |
   //+------------------------------------------------------------------+
   double            GetCloseAtPos(int pos, int i)
     {
      return m_collection[pos].GetRateAtPos(i).close;
     }

   //+------------------------------------------------------------------+
   //| Return High price of currency at position                        |
   //+------------------------------------------------------------------+
   double            GetHighAtPos(int pos, int i)
     {
      return m_collection[pos].GetRateAtPos(i).high;
     }

   //+------------------------------------------------------------------+
   //| Return Low price of currency at position                         |
   //+------------------------------------------------------------------+
   double            GetLowAtPos(int pos, int i)
     {
      return m_collection[pos].GetRateAtPos(i).low;
     }

   //+------------------------------------------------------------------+
   //| Return Median price of currency at position                      |
   //+------------------------------------------------------------------+
   double            GetMedianAtPos(int pos, int i)
     {
      return (GetHighAtPos(pos,i) + GetLowAtPos(pos, i))/2;
     }

   //+------------------------------------------------------------------+
   //| Return Typical price of currency at position                     |
   //+------------------------------------------------------------------+
   double            GetTypicalAtPos(int pos, int i)
     {
      return (GetHighAtPos(pos,i) + GetLowAtPos(pos, i) + GetCloseAtPos(pos,i))/3;
     }

   //+------------------------------------------------------------------+
   //| Return Weighted price of currency at position                    |
   //+------------------------------------------------------------------+
   double            GetWeightedAtPos(int pos, int i)
     {
      return (GetHighAtPos(pos,i) + GetLowAtPos(pos, i) + GetCloseAtPos(pos,i) * 2)/4;
     }
  };
//+------------------------------------------------------------------+


const int  CURRENT_SYMBOL_INDEX = 0;



//+------------------------------------------------------------------+
//|Janus class for calculating janus family of indicators.           |
//+------------------------------------------------------------------+
class CJanus
  {
private:
   CSymbolCollection* m_symbol_list;    //object container of symbols
   ENUM_PRICE         m_price_type;     //applied price for calculations
   ENUM_DIFF_TYPE     m_diff_type;      //method of differencing applied
   ENUM_INDEX_TYPE    m_index_type;     //type of index
   int                m_hist_size;      // synchronized size of history across all selected symbols in collection
   ENUM_TIMEFRAMES    m_list_timeframe; //timeframe for bars to be used in calculations
   //---private methods
   double            market_return(const uint barshift, const uint symbolshift);
   void              market_offense_defense(const uint barshift,const uint symbolshift,const uint rs_period,double& out_offense,double& out_defense);
   double            rs(const uint barshift,const uint symbolshift,const uint rs_period,uint lag=0);
   double            rs_off_def(const uint barshift, const uint symbolshift,const uint lag,const double median,const double index_offense, const double index_defense, double &array[]);
   //---
public:
   //constructor
                     CJanus(void):m_symbol_list(NULL),
                     m_price_type(WRONG_VALUE),
                     m_diff_type(WRONG_VALUE),
                     m_index_type(WRONG_VALUE),
                     m_list_timeframe(WRONG_VALUE),
                     m_hist_size(0)
     {

     }
   // destructor
                    ~CJanus(void)
     {
      if(CheckPointer(m_symbol_list)==POINTER_DYNAMIC)
         delete m_symbol_list;
     }
   //public methods
   bool              Initialize(const ENUM_PRICE set_price_type, const ENUM_DIFF_TYPE set_diff_type, const ENUM_INDEX_TYPE set_index_type, ENUM_TIMEFRAMES set_timeframe,const int history_size, const string symbol_list);
   bool              Update(void);
   int               HistorySize(void);
   string            GetSymbolAt(const int sym_ind);
   int               GetSymbolsTotal(void);
   double            CalculateReturn(const uint barshift, const string symbol__);
   double            CalculateBenchMarkReturn(const uint barshift);
   double            CalculateSummedIndexReturn(const uint barshift);
   void              CalculateBenchMarkOffenseDefense(const uint barshift,const uint rs_period,double& out_offense,double& out_defense);
   void              CalculateSymbolOffenseDefense(const uint barshift,const string symbol__,const uint rs_period,double& out_offense,double& out_defense);
   double            CalculateRelativeStrength(const uint barshift,const string symbol__,const uint rs_period,uint lag=0);
   void              CalculateRelativeStrengthLeaderLaggard(const uint barshift,const uint rs_period,const double rs_percent_top,double& leader,double& laggard);
   double            CalculateRelativeStrengthSpread(const uint barshift,const uint rs_period,const double rs_percent_top);
   double            CalculateRelativeStrengthSpreadChange(const uint barshift,const uint rs_period,const double rs_percent_top);

  };

//+------------------------------------------------------------------+
//|helper method for calculating market offense and defense figures  |
//+------------------------------------------------------------------+
double CJanus::rs_off_def(const uint barshift,const uint symbolshift,const uint lag,const double median,const double index_offense,const double index_defense,double &array[])
  {

   double offense,defense;
   offense=defense=0;
   int size= ArraySize(array);

   for(int i=(int)lag; i<size; i++)
     {
      if(array[i]>=median)
         offense+=market_return(barshift+i,symbolshift);
      else
         defense+=market_return(barshift+i,symbolshift);
     }

   return((100/MathSqrt(2.0))*(offense/index_offense - defense/index_defense));

  }
//+------------------------------------------------------------------------+
//|method that initializes properties, checks and synchronizes history data|
//+------------------------------------------------------------------------+
bool CJanus::Initialize(const ENUM_PRICE set_price_type,const ENUM_DIFF_TYPE set_diff_type,const ENUM_INDEX_TYPE set_index_type,ENUM_TIMEFRAMES set_timeframe,const int history_size,const string symbol_list)
  {

   if(history_size<=0)
      return false;

   if(symbol_list=="" && set_index_type>0)
      return false;

   m_list_timeframe=set_timeframe;

   m_index_type=set_index_type;

   m_price_type=set_price_type;

   m_diff_type=set_diff_type;

   m_hist_size=history_size;



   string forex_symbols=(m_index_type==INDEX_FOREX_MAJORS)?"usd,gbp,eur,jpy,cad,aud,chf,nzd":symbol_list;

   int inside=0;

   if(CheckPointer(m_symbol_list)==POINTER_INVALID)
      m_symbol_list=new CSymbolCollection(m_list_timeframe);

   if(m_symbol_list==NULL)
      return(false);

   inside=m_symbol_list.Add(_Symbol);

   if(m_index_type<1)
     {
      if(!StringToLower(forex_symbols))
        {
         Print("error with string tolower ",__LINE__," ",__FUNCTION__);
         delete m_symbol_list;
         return(false);
        }

      string market_symbol="";
      string symbol_name="";
      string sym_basis="";
      string sym_alt="";

      int total=SymbolsTotal(false);

      for(int i=0; i<total; i++)
        {
         symbol_name=SymbolName(i,false);
         market_symbol=symbol_name;

         if(market_symbol==_Symbol)
            continue;

         sym_basis=SymbolInfoString(symbol_name,SYMBOL_CURRENCY_BASE);
         sym_alt=StringSubstr(symbol_name,StringFind(market_symbol,sym_basis)+3,3);

         if(!StringToLower(sym_basis) || !StringToLower(sym_alt) || !StringToLower(symbol_name))
           {
            Print("error with string tolower ",__LINE__," ",__FUNCTION__);
            delete m_symbol_list;
            return(false);
           }

         if(StringFind(symbol_name,sym_basis)>=0 && StringFind(forex_symbols,sym_alt)>=0 && StringFind(forex_symbols,sym_basis)>=0)
            inside=m_symbol_list.Add(market_symbol);
        }
     }
   else
     {
      string collect_syms[];

      int done=StringSplit(forex_symbols,StringGetCharacter(",",0),collect_syms);

      if(done<=0)
        {
         delete m_symbol_list;
         return(false);
        }

      for(int i=0; i<done; i++)
        {
         if(!SymbolInfoInteger(collect_syms[i],SYMBOL_EXIST))
           {
            Print("Symbol ",collect_syms[i]," does not exist");
            continue;
           }

         if(collect_syms[i] == _Symbol)
            continue;

         inside=m_symbol_list.Add(collect_syms[i]);
        }
     }

   if(inside<2)
     {
      Print("Building index requires more than one symbol");
      delete m_symbol_list;
      return(false);
     }

   return(m_symbol_list.CheckHistory(m_hist_size));
  }

//+------------------------------------------------------------------+
//|Method that refreshes price quotes once every bar                 |
//+------------------------------------------------------------------+

bool CJanus::Update(void)
  {
   static datetime last_time;

   if(iTime(NULL,m_list_timeframe,0)>last_time)
     {
      last_time=iTime(NULL,m_list_timeframe,0);
      return(m_symbol_list.Update());
     }
   else
      return(true);
  }
//+------------------------------------------------------------------+
//|returns the number of bars available in history                   |
//+------------------------------------------------------------------+
int CJanus::HistorySize(void)
  {
   return(m_symbol_list.GetHistoryBarsLength());
  }
//+------------------------------------------------------------------+
//|returns the number of symbols added to the symbol collection      |
//+------------------------------------------------------------------+
int CJanus::GetSymbolsTotal(void)
  {
   return(m_symbol_list.GetCollectionLength());
  }
//+------------------------------------------------------------------+
//|returns the name of the symbol by index                           |
//+------------------------------------------------------------------+
string CJanus::GetSymbolAt(const int sym_ind)
  {
   return(m_symbol_list.GetSymbolNameAtPos(sym_ind));
  }  
//+------------------------------------------------------------------+
//|private method that calculates the returns                        |
//+------------------------------------------------------------------+
double CJanus::market_return(const uint barshift, const uint symbolshift)
  {
   double curr,prev;
   curr=0;
   prev=1.e-60;

   switch(m_price_type)
     {
      case CLOSE:
         curr=m_symbol_list.GetCloseAtPos(symbolshift,barshift);
         prev=m_symbol_list.GetCloseAtPos(symbolshift,barshift+1);
         break;
      case MEDIAN:
         curr=m_symbol_list.GetMedianAtPos(symbolshift,barshift);
         prev=m_symbol_list.GetMedianAtPos(symbolshift,barshift+1);
         break;
      case TYPICAL:
         curr=m_symbol_list.GetTypicalAtPos(symbolshift,barshift);
         prev=m_symbol_list.GetTypicalAtPos(symbolshift,barshift+1);
         break;
      case WEIGHTED:
         curr=m_symbol_list.GetWeightedAtPos(symbolshift,barshift);
         prev=m_symbol_list.GetWeightedAtPos(symbolshift,barshift+1);
         break;
      default:
         return WRONG_VALUE;
     }

   if(prev==0)
      return(WRONG_VALUE);

   switch(m_diff_type)
     {
      case DIFF_PERCENT:
         return(((curr-prev)/prev)*100);
      case DIFF_LOG:
         return(MathLog(curr/prev));
      default:
         return(WRONG_VALUE);
     }
  }
//+------------------------------------------------------------------+
//|public method to calculate returns for single bar                 |
//+------------------------------------------------------------------+
double CJanus::CalculateReturn(const uint barshift, const string symbol_)
  {
   int sshift=m_symbol_list.GetIndex(symbol_);
   if(sshift>-1)
      return(market_return(barshift,sshift));
   else
      return(WRONG_VALUE);
  }
//+------------------------------------------------------------------+
//|public method to calculate index returns                          |
//+------------------------------------------------------------------+
double CJanus::CalculateBenchMarkReturn(const uint barshift)
  {
   double sorted[];
   int size=m_symbol_list.GetCollectionLength();

   if(size<=0)
      return(WRONG_VALUE);

   ArrayResize(sorted,size);

   for(int i=0; i<size; i++)
     {
      sorted[i]=market_return(barshift,i);
     }

   if(!ArraySort(sorted))
     {
      Print("sorting error ",__LINE__," ",__FUNCTION__);
      return(0);
     }

   return(MathMedian(sorted));
  }
//+------------------------------------------------------------------+
//|calculates the sum of returns across all symbols in collection    |
//+------------------------------------------------------------------+
double CJanus::CalculateSummedIndexReturn(const uint barshift)
  {
   int size=m_symbol_list.GetCollectionLength();

   if(size<=0)
      return(WRONG_VALUE);

   double value=0;

   for(int i=0; i<size; i++)
     {
      value+=market_return(barshift,i);
     }

   return(value);
  }
//+------------------------------------------------------------------+
//|private method that calculates the relative strength              |
//+------------------------------------------------------------------+
double CJanus::rs(const uint barshift,const uint symbolshift,const uint rs_period,uint lag=0)
  {
   if(lag>=rs_period)
      return(WRONG_VALUE);

   double index[],sorted[],median,i_offense,i_defense;
   median=i_offense=i_defense=0;

   ArrayResize(index,rs_period);
   ArrayResize(sorted,rs_period);

   int begin=(int)lag;

   for(int i=0; i<(int)rs_period; i++)
     {
      index[i]=CalculateBenchMarkReturn(barshift+i);
      if(i>=begin)
         sorted[i-begin]=index[i];
     }

   if(!ArraySort(sorted))
     {
      Print("sorting error ",__LINE__," ",__FUNCTION__);
      return(EMPTY_VALUE);
     }

   median=MathMedian(sorted);

   i_offense=1.e-30;
   i_defense=-1.e-30;

   for(int i=begin; i<(int)rs_period; i++)
     {
      if(index[i]>=median)
         i_offense+=index[i]-median;
      else
         i_defense+=index[i]-median;
     }

   if(i_offense<0 || i_defense>0)
     {
#ifdef DEBUG
      Print("error invalid figures ","Offensive ",i_offense," Defensive ",i_defense);
#endif
      return(WRONG_VALUE);
     }

   return(rs_off_def(barshift,symbolshift,lag,median,i_offense,i_defense,index));
  }
//+------------------------------------------------------------------+
//|private method that calculates market defense and offense values  |
//+------------------------------------------------------------------+
void CJanus::market_offense_defense(const uint barshift,const uint symbolshift,const uint rs_period,double& out_offense,double& out_defense)
  {
   out_defense=out_offense=0.0;

   double index[],sorted[],median,i_offense,i_defense;
   median=i_offense=i_defense=0;

   ArrayResize(index,rs_period);
   ArrayResize(sorted,rs_period);

   int begin=0;

   for(int i=0; i<(int)rs_period; i++)
     {
      index[i]=CalculateBenchMarkReturn(barshift+i);
      if(i>=begin)
         sorted[i-begin]=index[i];
     }

   if(!ArraySort(sorted))
     {
      Print("sorting error ",__LINE__," ",__FUNCTION__);
      return;
     }

   median=MathMedian(sorted);

   i_offense=1.e-30;
   i_defense=-1.e-30;

   for(int i=begin; i<(int)rs_period; i++)
     {
      if(index[i]>=median)
         i_offense+=index[i]-median;
      else
         i_defense+=index[i]-median;
     }

   if(i_offense<0 || i_defense>0)
     {
#ifdef DEBUG
      Print("error invalid figures ","Offensive ",i_offense," Defensive ",i_defense);
#endif
      return;
     }

   double m_offense,m_defense;
   m_offense=m_defense=0;

   for(int i=0; i<(int)rs_period; i++)
     {
      if(index[i]>=median)
         m_offense+=market_return(barshift+i,symbolshift);
      else
         m_defense+=market_return(barshift+i,symbolshift);
     }

   out_defense= (m_defense/i_defense) * 100;
   out_offense= (m_offense/i_offense) * 100;

  }
//+------------------------------------------------------------------+
//|public method to calculate Index offense and defense scores       |
//+------------------------------------------------------------------+
void CJanus::CalculateBenchMarkOffenseDefense(const uint barshift,const uint rs_period,double& out_offense,double& out_defense)
  {
   out_defense=out_offense=WRONG_VALUE;

   double index[],sorted[],median,i_offense,i_defense;
   median=i_offense=i_defense=0;

   ArrayResize(index,rs_period);
   ArrayResize(sorted,rs_period);

   int begin=0;

   for(int i=0; i<(int)rs_period; i++)
     {
      index[i]=CalculateBenchMarkReturn(barshift+i);
      if(i>=begin)
         sorted[i-begin]=index[i];
     }

   if(!ArraySort(sorted))
     {
      Print("sorting error ",__LINE__," ",__FUNCTION__);
      return;
     }

   median=MathMedian(sorted);

   i_offense=1.e-30;
   i_defense=-1.e-30;

   for(int i=begin; i<(int)rs_period; i++)
     {
      if(index[i]>=median)
         i_offense+=index[i]-median;
      else
         i_defense+=index[i]-median;
     }

   if(i_offense<0 || i_defense>0)
     {
#ifdef DEBUG
      Print("error invalid figures ","Offensive ",i_offense," Defensive ",i_defense);
#endif
      return;
     }

   out_offense=i_offense;
   out_defense=i_defense;

   return;
  }
//+------------------------------------------------------------------+
//|public method to calculate relative strength                      |
//+------------------------------------------------------------------+
double CJanus::CalculateRelativeStrength(const uint barshift,const string symbol__,const uint rs_period,uint lag=0)
  {
   int sshift=m_symbol_list.GetIndex(symbol__);
   if(sshift>-1)
      return(rs(barshift,sshift,rs_period,lag));
   else
      return(WRONG_VALUE);
  }


//+------------------------------------------------------------------+
//|public method to calculate relative strength leaders and laggards |
//+------------------------------------------------------------------+
void CJanus::CalculateRelativeStrengthLeaderLaggard(const uint barshift,const uint rs_period,const double rs_percent_top,double& leader,double& laggard)
  {
   leader=laggard=0;
   
   uint lag=rs_period;

   double sorted[];
   int iwork[],k,n,isub;
   k=isub=-1;

   int size=m_symbol_list.GetCollectionLength();

   ArrayResize(sorted,size);
   ArrayResize(iwork,size);

   for(int i=0; i<size; i++)
     {
      sorted[i]=rs(barshift,uint(i),rs_period,lag);
      iwork[i]=i;
     }

   MathQuickSortAscending(sorted,iwork,0,size-1);

   k=(int)(rs_percent_top*(size+1))-1;
   if(k<0)
      k=0;
   n=k+1;

   while(k>=0)
     {
      isub=iwork[k];
      for(uint i=0; i<lag; i++)
        {
         laggard+=market_return(barshift+i,isub);
        }
      isub=iwork[size-1-k];
      for(uint i=0; i<lag; i++)
        {
         leader+=market_return(barshift+i,isub);
        }
      --k;
     }
   leader/=n*lag;
   laggard/=n*lag;

   return;
  }
//+------------------------------------------------------------------+
//|public method to calculate market offense and defense                                                                  |
//+------------------------------------------------------------------+
void CJanus::CalculateSymbolOffenseDefense(const uint barshift,const string symbol__,const uint rs_period,double &out_offense,double &out_defense)
  {
   out_defense=out_offense=0.0;
   int sshift=m_symbol_list.GetIndex(symbol__);
   if(sshift>-1)
      market_offense_defense(barshift,sshift,rs_period,out_offense,out_defense);
   else
      return;
  }

//+------------------------------------------------------------------+
//|public method to calculate the relative strength spread.          |
//+------------------------------------------------------------------+
double CJanus::CalculateRelativeStrengthSpread(const uint barshift,const uint rs_period,const double rs_percent_top)
  {
   double sorted[],width,div;
   int k=0;
   int size=m_symbol_list.GetCollectionLength();

   width=div=0;
   ArrayResize(sorted,size);

   for(int i=0; i<size; i++)
     {
      sorted[i]=rs(barshift,uint(i),rs_period);
     }

   if(!ArraySort(sorted))
     {
      Print("sorting error ",__LINE__," ",__FUNCTION__);
      return(WRONG_VALUE);
     }

   k=(int)(rs_percent_top*(size+1))-1;

   if(k<0)
      k=0;
   double n=double(k+1);

   while(k>=0)
     {
      width+=sorted[size-1-k]-sorted[k];
      --k;
     }

   return(width/=n);
  }
//+------------------------------------------------------------------+
//|public method that returns difference of relative strength        |
//+------------------------------------------------------------------+
double CJanus::CalculateRelativeStrengthSpreadChange(const uint barshift,const uint rs_period,const double rs_percent_top)
  {
   return(CalculateRelativeStrengthSpread(barshift,rs_period,rs_percent_top) - CalculateRelativeStrengthSpread(barshift+1,rs_period,rs_percent_top));
  }

//+------------------------------------------------------------------+
