//+------------------------------------------------------------------+
//|                                                    CSignalDA.mqh |
//|                   Copyright 2009-2013, MetaQuotes Software Corp. |
//|                                              http://www.mql5.com |
//+------------------------------------------------------------------+
#include <Math\Alglib\dataanalysis.mqh>
#include <Expert\ExpertSignal.mqh>
// wizard description start
//+------------------------------------------------------------------+
//| Description of the class                                         |
//| Title=Signals of indicator 'Discriminant Analysis'               |
//| Type=SignalAdvanced                                              |
//| Name=DiscriminantAnalysis                                        |
//| ShortName=DiscriminantAnalysis                                   |
//| Class=CSignalDA                                                  |
//| Page=signal_da                                                   |
//| Parameter=SignalPoints,int,5,Signal Data Points                  |
//| Parameter=SignalType,int,3,Signal Data Type                      |
//| Parameter=SignalRegulizer,double,0.05,Signal Regulizer           |
//+------------------------------------------------------------------+
// wizard description end
//+------------------------------------------------------------------+
//| Class CSignalDA.                                                 |
//| Purpose: Class of generator of trade signals based on            |
//|          the 'Discriminant Analysis' indicator.                  |
//| Is derived from the CExpertSignal class.                         |
//+------------------------------------------------------------------+
#define     __S_VARS 5
#define     __S_CLASSES 3
//
#define		__S_PHI (sqrt(5.0)*0.5)-0.5

#define     __S_BULLISH 2
#define     __S_WHIPSAW 1
#define     __S_BEARISH 0
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
class CSignalDA : public CExpertSignal
  {
protected:
   CLDA              LDA;                    // object-indicator
   
   //--- adjusted parameters
   int               m_signal_points;        // the data points
   int               m_signal_type;          // the data type
   double            m_signal_regulizer;     // the neutrality definer 

public:
   
                     CSignalDA(void);
                    ~CSignalDA(void);
   //--- methods of setting adjustable parameters
   void              SignalPoints(int value)             { m_signal_points=value;         }
   void              SignalType(int value)               { m_signal_type=value;           }
   void              SignalRegulizer(double value)       { m_signal_regulizer=value;      }
   //--- method of verification of settings
   virtual bool      ValidationSettings(void);
   //--- method of creating the indicator and timeseries
   virtual bool      InitIndicators(CIndicators *indicators);
   //--- methods of checking if the market models are formed
   virtual int       LongCondition(void);
   virtual int       ShortCondition(void);

protected:
   
   double            ProcessDA(int Index);
   double            Data(int Index,bool Variables=true);
   double            Range(int Index){ int _h,_l; return(m_high.MaxValue(StartIndex()+Index,2,_h)-m_low.MinValue(StartIndex()+Index,2,_l)); }
  };
//+------------------------------------------------------------------+
//| Constructor                                                      |
//+------------------------------------------------------------------+
CSignalDA::CSignalDA(void) : m_signal_points(5),m_signal_type(3),m_signal_regulizer(0.05)
  {
//--- initialization of protected data
   m_used_series=USE_SERIES_OPEN+USE_SERIES_HIGH+USE_SERIES_LOW+USE_SERIES_CLOSE;
  }
//+------------------------------------------------------------------+
//| Destructor                                                       |
//+------------------------------------------------------------------+
CSignalDA::~CSignalDA(void)
  {
  }
//+------------------------------------------------------------------+
//| Validation settings protected data.                              |
//+------------------------------------------------------------------+
bool CSignalDA::ValidationSettings(void)
  {
//--- validation settings of additional filters
   if(!CExpertSignal::ValidationSettings())
      return(false);
//--- initial data checks
   if(m_signal_points<3)
     {
      printf(__FUNCTION__+": signal points must be greater than 2");
      return(false);
     }
   if(m_signal_type>=4)
     {
      printf(__FUNCTION__+": signal type must be less than 4");
      return(false);
     }
   if(m_signal_regulizer<=0.0)
     {
      printf(__FUNCTION__+": signal regulizer must be greater than 0.0");
      return(false);
     }
//--- ok
   return(true);
  }
//+------------------------------------------------------------------+
//| Create indicators.                                               |
//+------------------------------------------------------------------+
bool CSignalDA::InitIndicators(CIndicators *indicators)
  {
//--- check pointer
   if(indicators==NULL)
      return(false);
//--- initialization of indicators and timeseries of additional filters
   if(!CExpertSignal::InitIndicators(indicators))
      return(false);
//--- ok
   return(true);
  }
//+------------------------------------------------------------------+
//| Discriminant Ananlysis method                                    |
//| INPUT PARAMETERS                                                 |
//|     Index   -   int, read index within price buffer.             |
//| OUTPUT                                                           |
//|     double  -   Normalized value (-1.0 to +1.0) for signal       |
//+------------------------------------------------------------------+
double CSignalDA::ProcessDA(int Index)
   {
      double _da=0.0;
      
      int _info=0;
      CMatrixDouble _w,_xy,_z;
      _xy.Resize(m_signal_points,__S_VARS+1);
      
      for(int p=0;p<m_signal_points;p++)
      {
         for(int v=0;v<__S_VARS;v++)
         {
            _xy[p].Set(v,Data(Index+p+v+1));
         }
         
         _xy[p].Set(__S_VARS,Data(Index+p,false));
      }
      
      LDA.FisherLDAN(_xy,m_signal_points,__S_VARS,__S_CLASSES,_info,_w);
      
      if(_info>0)
      {
         double _centroids[__S_CLASSES],_unknown_centroid=0.0; ArrayInitialize(_centroids,0.0);
         
         _z.Resize(1,6);
         for(int v=0;v<__S_VARS;v++)
         {
            _z[0].Set(v,Data(Index));
         }
         
         for(int p=0;p<m_signal_points;p++)
         {
            for(int v=0;v<__S_VARS;v++)
            {
               _centroids[int(_xy[p][5])]+= (_w[0][v]*_xy[p][v]);
            }
         }
         
         // best eigen vector is the first 
         for(int v=0;v<__S_VARS;v++){ _unknown_centroid+= (_w[0][v]*_z[0][v]); }
         
         //
         /*for(int c=0;c<__S_CLASSES;c++)
         {
            printf(__FUNCSIG__+" in: "+IntegerToString(c)+" known centroid: "+DoubleToString(_centroids[c])); 
         }
         printf(__FUNCSIG__+" unknown centroid: "+DoubleToString(_unknown_centroid)+
         ", range: "+DoubleToString(Range(Index))+
         ", change: "+DoubleToString(Close(Index)-Close(Index+1))+
         ", measure: "+DoubleToString(m_signal_regulizer*Range(Index))
         );*///return(0);
         //
         
         if(fabs(_centroids[__S_BULLISH]-_unknown_centroid)<fabs(_centroids[__S_BEARISH]-_unknown_centroid) && fabs(_centroids[__S_BULLISH]-_unknown_centroid)<fabs(_centroids[__S_WHIPSAW]-_unknown_centroid))
         {
            _da=(1.0-(fabs(_centroids[__S_BULLISH]-_unknown_centroid)/(fabs(_centroids[__S_BULLISH]-_unknown_centroid)+fabs(_centroids[__S_WHIPSAW]-_unknown_centroid)+fabs(_centroids[__S_BEARISH]-_unknown_centroid))));
         }
         else if(fabs(_centroids[__S_BEARISH]-_unknown_centroid)<fabs(_centroids[__S_BULLISH]-_unknown_centroid) && fabs(_centroids[__S_BEARISH]-_unknown_centroid)<fabs(_centroids[__S_WHIPSAW]-_unknown_centroid))
         {
            _da=-1.0*(1.0-(fabs(_centroids[__S_BEARISH]-_unknown_centroid)/(fabs(_centroids[__S_BULLISH]-_unknown_centroid)+fabs(_centroids[__S_WHIPSAW]-_unknown_centroid)+fabs(_centroids[__S_BEARISH]-_unknown_centroid))));
         }
      }
      else
      {
         if(_info==-4){ printf(__FUNCSIG__+" internal EVD subroutine hasn't converged "); }
         else if(_info==-2){ printf(__FUNCSIG__+" there is a point with class number outside of [0..NClasses-1] "); }
         else if(_info==-1){ printf(__FUNCSIG__+" incorrect parameters was passed  either: NPoints<0, or NVars<1, or NClasses<2) "); }
      }
      
      return(_da);
   }
//+------------------------------------------------------------------+
//| Data Set method                                                  |
//| INPUT PARAMETERS                                                 |
//|     Index   -   int, read index within price buffer.             |
//|                                                                  |
//|     Type    -   position type for determining use of             |
//|                 lows or highs.                                   |
//|                                                                  |
//|     Variables                                                    |
//|             -   whether data component is variables or .         |
//|                  classifier.                                     |
//| OUTPUT                                                           |
//|     double  -   Data depending on data set type                  |
//|                                                                  |
//| DATA SET TYPES                                                   |
//| 1. Discretized variables.                                        |
//|                                                                  |
//| 2. Normalized variables.                                         |
//|                                                                  |
//| 3. Continuized variables.                                        |
//|                                                                  |
//| 4. Raw data variables.                                           |
//|                                                                  |
//+------------------------------------------------------------------+
double CSignalDA::Data(int Index,bool Variables=true)
   {
      m_close.Refresh(-1);
         
      m_low.Refresh(-1);
      m_high.Refresh(-1);
            
      if(Variables)
      {
         if(m_signal_type==0)
         {
            return(fabs(Close(StartIndex()+Index)-Close(StartIndex()+Index+1))<m_signal_regulizer*Range(Index)?1.0:((Close(StartIndex()+Index)>Close(StartIndex()+Index+1))?2.0:0.0));
         }
         else if(m_signal_type==1)
         {
            if(fabs(Close(StartIndex()+Index)-Close(StartIndex()+Index+1))<m_signal_regulizer*Range(Index))
            {
               return(0.0);
            }
            return((Close(StartIndex()+Index)-Close(StartIndex()+Index+1))/fmax(m_symbol.Point(),fmax(High(StartIndex()+Index),High(StartIndex()+Index+1))-fmin(Low(StartIndex()+Index),Low(StartIndex()+Index+1))));
         }
         else if(m_signal_type==2)
         {
            if(fabs(Close(StartIndex()+Index)-Close(StartIndex()+Index+1))<m_signal_regulizer*Range(Index))
            {
               return(0.0);
            }
            return(Close(StartIndex()+Index)-Close(StartIndex()+Index+1));
         }
         else if(m_signal_type==3)
         {
            if(fabs(Close(StartIndex()+Index)-Close(StartIndex()+Index+1))<m_signal_regulizer*Range(Index))
            {
               return(Close(StartIndex()+Index+1));
            }
            return(Close(StartIndex()+Index));
         }
      }
      
      return(fabs(Close(StartIndex()+Index)-Close(StartIndex()+Index+1))<m_signal_regulizer*Range(Index)?1.0:((Close(StartIndex()+Index)>Close(StartIndex()+Index+1))?2.0:0.0));
   }
//+------------------------------------------------------------------+
//| "Voting" that price will grow.                                   |
//+------------------------------------------------------------------+
int CSignalDA::LongCondition(void)
  {
   
   int result=0;
   int idx   =StartIndex();
   double _da=ProcessDA(idx);
   
   if(_da>0.0)
     {
      result=int(round(100.0*_da));
     }
     
   //printf(__FUNCSIG__+" DA: "+DoubleToString(_da)+" result: "+IntegerToString(result));//return(0);
//--- return the result
   return(fmin(100,result));
  }
//+------------------------------------------------------------------+
//| "Voting" that price will fall.                                   |
//+------------------------------------------------------------------+
int CSignalDA::ShortCondition(void)
  {
   int result  =0;
   int idx     =StartIndex();
   double _da=ProcessDA(idx);
   
   if(_da<0.0)
     {
      result=int(round(-100.0*_da));
     }
   //printf(__FUNCSIG__+" DA: "+DoubleToString(_da)+" result: "+IntegerToString(result));//return(0);
//--- return the result
   return(fmin(100,result));
  }
//+------------------------------------------------------------------+
