//+------------------------------------------------------------------+
//|                                                Google_Charts.mqh |
//|                        Copyright 2010, MetaQuotes Software Corp. |
//|                                              http://www.mql5.com |
//+------------------------------------------------------------------+
#property copyright "Copyright 2010, MetaQuotes Software Corp."
#property link      "http://www.mql5.com"

#include <Strings\String.mqh>
#include <Arrays\ArrayDouble.mqh>
#include <Arrays\ArrayInt.mqh>
#include <Arrays\ArrayString.mqh>
#include <Files\FileBin.mqh>

#import "ColorToText.dll"
//+------------------------------------------------------------------+
//Function of conversion of a color to a string   
// R - red tint component
// G - green tint component
// B - blue tint component
//Returned value: a color in 16 bit string representation
//+------------------------------------------------------------------+
string ColorToTxt(int R,int G,int B);
#import "PNG_to_BMP.dll"//import of DLL with a function of conversion of PNG images to BMP
//+------------------------------------------------------------------+
//Funciton of conversion of a PNG image to BMP
// src - Path to a source file
// dst - Where a converted file should be put at
//Returned value: true-successful, false-unsuccessful
//+------------------------------------------------------------------+
bool Convert_PNG(string src,string dst);
#import "wininet.dll"//import of DLL with functions for working with the internet
//+------------------------------------------------------------------+
//Check if there is an internet connection and try to create it (Windows API)     
// x - Reserved, must be set to 0
//Returned value: 0-if successful, otherwise another value
//+------------------------------------------------------------------+
int InternetAttemptConnect(int x);
//+------------------------------------------------------------------+
//Initializes a structure for functions of the WinInet library to work (Windows API)   
// sAgent - Name of an application that creates connection
// lAccessType - Access specifier
// sProxyName - Name of proxy server
// sProxyBypass - List of parameters of host
// lFlags - Additional flags
//Returned value: Descriptor for the following functions to work, NULL in case of failure
//+------------------------------------------------------------------+
int InternetOpenW(string sAgent,int lAccessType,
                  string sProxyName="",string sProxyBypass="",
                  int lFlags=0);
//+------------------------------------------------------------------+
//Sends request to server (Windows API)   
// hInternetSession - descriptor 
// sUrl - Request line
// sHeaders - Header
// lHeadersLength - Header length
// lFlags - Additional flags
//Returned value: Session descriptor, NULL in case of failure
//+------------------------------------------------------------------+                  
int InternetOpenUrlW(int hInternetSession,string sUrl,
                     string sHeaders="",int lHeadersLength=0,
                     int lFlags=0,int lContext=0);
//+------------------------------------------------------------------+
//Reads information received from server as a result of the request (Windows API)   
// hFile - descriptor 
// sBuffer - Where to write received information 
// lNumBytesToRead - Amount of information to be read
// lNumberOfBytesRead - Amount of actually read information
//Returned value: Session descriptor, NULL in case of failure
//+------------------------------------------------------------------+                       
int InternetReadFile(int hFile,char &sBuffer[],int lNumBytesToRead,
                     int &lNumberOfBytesRead[]);
//+------------------------------------------------------------------+
//Releases transmitted descriptor (Windows API)   
// hInet -descriptor 
//Returned value: 0-if successful, otherwise another value
//+------------------------------------------------------------------+                          
int InternetCloseHandle(int hInet);
#import "kernel32.dll"
//+------------------------------------------------------------------+
//Function of deletion of a file (Windows API)    
// path - path to a file
//Returned value: true-successful, false-unsuccessful
//+------------------------------------------------------------------+
bool DeleteFileW(string path);
#import
//+------------------------------------------------------------------+
//Enumeration of diagram types     
//+------------------------------------------------------------------+
enum DIAGRAM_TYPE
  {
   ///Base
   DIAGRAM_TYPE_BASE=0,
   ///Linear chart
   DIAGRAM_TYPE_LINEXY=1,
   ///Formula
   DIAGRAM_TYPE_FORMULA=2,
   ///Histogram
   DIAGRAM_TYPE_BAR=3,
   ///Circle
   DIAGRAM_TYPE_PIE=4,
   ///Graph
   DIAGRAM_TYPE_GRAPH=5,
   ///Radar
   DIAGRAM_TYPE_RADAR=6,
   ///QR Code
   DIAGRAM_TYPE_QRCODE=7,
   ///Venn
   DIAGRAM_TYPE_VENN=8,
   ///Candlesticks
   DIAGRAM_TYPE_CANDLESTRICK=9,
   ///Scatter
   DIAGRAM_TYPE_SCATTER=10,
   ///Map
   DIAGRAM_TYPE_MAP=11
  };
//+------------------------------------------------------------------+
//Enumeration of errors generated by the library
//Notice: To get  an error code, use the GetLastError() function 
//+------------------------------------------------------------------+
enum DIAGRAM_ERR
  {
   ///no errors
   DIAGRAM_NO_ERR=1,
   ///unequal amount of date transmitted in sets
   DIAGRAM_ERR_INCORRECT_DATA_AMOUNT=2,
   ///incorrect type of object for binding
   //Notice: Only graphical objects of the OBJ_BITMAP and OBJ_BITMAP_LABEL types are supported
   DIAGRAM_ERR_INCORRECT_OBJ_TYPE=3,
   ///graphical object doesn't exist
   DIAGRAM_ERR_OBJ_NOT_EXIST=4,
   ///incorrect line identifier
   //Notice: possibly the line has been deleted or it hasn't been created yet
   DIAGRAM_ERR_INCORRECT_LINE_ID=5,
   ///incorrect filling type 
   DIAGRAM_ERR_INVALID_FILLING_TYPE=6,
   ///incorrect marker type
   ///see DIAGRAM_LINE_MARKERS
   DIAGRAM_ERR_INVALID_MARKER_TYPE=7,
   ///incorrect identifier of the second line for filling
   DIAGRAM_ERR_INCORRECT_FILL_INDEX=8,
   ///too many lines
   //Notice: maximum number of lines is 100
   DIAGRAM_ERR_TOO_MANY_LINES=9,
   ///error of image conversion
   DIAGRAM_ERR_IMAGE_CONVERSION_FAILED=10,
   ///no internet connection
   DIAGRAM_ERR_INTERNET_CONNECT_FAILED=11,
   ///too many labels
   //Notice: maximum number of labels is 100
   DIAGRAM_ERR_TOO_MANY_LABELS=12,
   ///incorrect label type
   ///see DIAGRAM_LABELS_TYPE
   DIAGRAM_ERR_INVALID_LABEL_TYPE=13,
   ///incorrect label identifier
   //Notice: possibly the label has been deleted or it hasn't been created yet
   DIAGRAM_ERR_INCORRECT_LABEL_ID=14,
   ///too many edges in a graph
   //Notice: maximum number of edges is 100
   DIAGRAM_ERR_TOO_MANY_EDGES=15,
   ///too many nodes
   //Notice: maximum number of nodes is 100   
   DIAGRAM_ERR_TOO_MANY_NODES=16,
   ///incorrect node identifier
   //Notice: possibly the node has been deleted or it hasn't been created yet
   DIAGRAM_ERR_INCORRECT_NODE_ID=17,
   ///incorrect edge identifier
   //Notice: possibly the edge has been deleted or it hasn't been created yet   
   DIAGRAM_ERR_INCORRECT_EDGE_ID=18,
   ///too long request (max. length is 2048)
   //Notice: try to decrease the amount of data, number of digits for truncation and elements of formatting of filling type, etc.
   DIAGRAM_ERR_TOO_LARGE_REQUEST=19
  };
//+------------------------------------------------------------------+
//Enumeration of encodings for CQRCodeChrat
//+------------------------------------------------------------------+
enum DIAGRAM_QRCODE_ENCODING
  {
   DIAGRAM_QRCODE_ENCODING_UTF_8=1,
   DIAGRAM_QRCODE_ENCODING_Shift_JIS=2,
   DIAGRAM_QRCODE_ENCODING_ISO_8859_1=3
  };
//+------------------------------------------------------------------+
//Enumeration of error correction levels for CQRCodeChrat
//+------------------------------------------------------------------+
enum DIAGRAM_QRCODE_ERROR_CORRECTION
  {
   DIAGRAM_QRCODE_ERROR_CORRECTION_LOW=1,
   DIAGRAM_QRCODE_ERROR_CORRECTION_MEDIUM=2,
   DIAGRAM_QRCODE_ERROR_CORRECTION_HIGH=3,
  };
//+------------------------------------------------------------------+
//Enumeration of types of graphic engines for CGraphChrat
//+------------------------------------------------------------------+
enum DIAGRAM_GRAPH_ENGINES
  {
   DIAGRAM_GRAPH_ENGINE_DOT=1,
   DIAGRAM_GRAPH_ENGINE_NEATO=2,
   DIAGRAM_GRAPH_ENGINE_TWOPI=3,
   DIAGRAM_GRAPH_ENGINE_CIRCO=4,
   DIAGRAM_GRAPH_ENGINE_FDP=5
  };
//+------------------------------------------------------------------+
//Enumeration of acceptable chart axes
//+------------------------------------------------------------------+
enum DIAGRAM_AXIS
  {
   ///X-axis
   DIAGRAM_AXIS_BOTTOM=1,
   ///upper axis
   DIAGRAM_AXIS_TOP=2,
   ///Y-axis
   DIAGRAM_AXIS_LEFT=4,
   ///right axis
   DIAGRAM_AXIS_RIGHT=8
  };
//+------------------------------------------------------------------+
//Enumeration of acceptable position of legend
//+------------------------------------------------------------------+
enum DIAGRAM_LEGEND_POSITION
  {
   DIAGRAM_LEGEND_POSITION_RIGHT_VERTICAL=1,
   DIAGRAM_LEGEND_POSITION_BOTTOM_HORIZONTAL=2,
   DIAGRAM_LEGEND_POSITION_BOTTOM_VERTICAL=3,
   DIAGRAM_LEGEND_POSITION_TOP_HORIZONTAL=4,
   DIAGRAM_LEGEND_POSITION_TOP_VERTICAL=5,
   DIAGRAM_LEGEND_POSITION_LEFT_VERTICAL=6
  };
//+------------------------------------------------------------------+
//Enumeration of acceptable markers of lines
//+------------------------------------------------------------------+
enum DIAGRAM_LINE_MARKERS
  {
   DIAGRAM_LINE_MARKERS_ARROW=1,
   DIAGRAM_LINE_MARKERS_CROSS=2,
   DIAGRAM_LINE_MARKERS_DIAMOND=3,
   DIAGRAM_LINE_MARKERS_CIRCLE=4,
   DIAGRAM_LINE_MARKERS_SQARE=5
  };
//+------------------------------------------------------------------+
//Enumeration of label types
//+------------------------------------------------------------------+
enum DIAGRAM_LABELS_TYPE
  {
   DIAGRAM_LABEL_TYPE_FLAG=1,
   DIAGRAM_LABELS_TYPE_TEXT=2,
   DIAGRAM_LABELS_TYPE_ANNOTATION=3
  };
//+------------------------------------------------------------------+
//Enumeration of acceptable horizontal location of labels
//+------------------------------------------------------------------+
enum DIAGRAM_LABELS_HORIZONTAL_PLACEMENT
  {
   DIAGRAM_LABELS_HORIZONTAL_PLACEMENT_LEFT=1,
   DIAGRAM_LABELS_HORIZONTAL_PLACEMENT_CENTER=2,
   DIAGRAM_LABELS_HORIZONTAL_PLACEMENT_RIGHT=3
  };
//+------------------------------------------------------------------+
//Enumeration of acceptable vertical location of labels
//+------------------------------------------------------------------+
enum DIAGRAM_LABELS_VERTICAL_PLACEMENT
  {
   DIAGRAM_LABELS_VERTICAL_PLACEMENT_BOTTOM=1,
   DIAGRAM_LABELS_VERTICAL_PLACEMENT_MIDDLE=2,
   DIAGRAM_LABELS_VERTICAL_PLACEMENT_TOP=3
  };
//+------------------------------------------------------------------+
//Enumeration of scales for CMapChart
//+------------------------------------------------------------------+
enum DIAGRAM_MAP_AREA
  {
   DIAGRAM_MAP_AREA_WORLD=0,
   DIAGRAM_MAP_AREA_AFRICA=1,
   DIAGRAM_MAP_AREA_ASIA=2,
   DIAGRAM_MAP_AREA_EUROPE=3,
   DIAGRAM_MAP_AREA_MIDDLE_EAST=4,
   DIAGRAM_MAP_AREA_SOUTH_AMERICA=5,
   DIAGRAM_MAP_AREA_USA=6
  };
//+------------------------------------------------------------------+
//Gets color as a string
// Color - color
//Returned value: a string with color in 16 bit string representation
//+------------------------------------------------------------------+
string GetColorStr(color Color)
  {
   CString res;
   string R,G,B;
   res.Assign(ColorToString(Color));
   int first=res.Find(0,",");
   int last=res.FindRev(",");
   R=(res.Left(first));
   G=(res.Mid(first+1, last-first-1));
   B=(res.Right(res.Len()-last-1));
   return ColorToTxt(R,G,B);
  }
//+------------------------------------------------------------------+
//Replaces spaces in a string with a specified symbol
// str - String
// new_str - Symbol
//Returned value: A string with replaced spaces
//+------------------------------------------------------------------+
string Remove_spaces(string str,string new_str="+")
  {
   CString res;
   res.Append(str);
   res.Replace(" ", new_str);
   return res.Str();
  }
//+------------------------------------------------------------------+
//\class CDiagram
//\brief Base Class    
//+------------------------------------------------------------------+
class CDiagram
  {
protected:
   long              chart_ID;
   string            name;
   long              obj_type;
   int               X;
   int               Y;
   string            title;
   color             title_color;
   int               title_font_size;
   int               legend_position;
   int               margin_left;
   int               margin_right;
   int               margin_top;
   int               margin_bottom;
   int               legend_width_;
   int               legend_height_;
   color             fill_chart;
   color             fill_bg;
public:
   bool              Attach(long _chart_ID,string _name);
   bool              Deatch();
   bool              ObjExist();
   bool              ObjExist(long _chart_ID,string _name);
   bool              GetChart();
   void              SetSize(int X_size,int Y_size);
   void              SetTitle(string text,color color_,int font_size);
   //+------------------------------------------------------------------+
   //Displays legend
   // position - legend position
   //+------------------------------------------------------------------+
   void ShowLegend(DIAGRAM_LEGEND_POSITION position=DIAGRAM_LEGEND_POSITION_RIGHT_VERTICAL)
     {legend_position=position;}
   //+------------------------------------------------------------------+
   //Hides legend
   //+------------------------------------------------------------------+      
   void HideLegend() {legend_position=0;}
   void              SetMargins(int left,int right,int top,int bottom,int legend_width,int legend_height);
   //+------------------------------------------------------------------+
   //Sets color of filling of chart
   // chart - color of chart filling
   // background - background color 
   //+------------------------------------------------------------------+
   void SetFill(color chart,color background=White) {fill_chart=chart; fill_bg=background;}
   void              CDiagram();
   //+------------------------------------------------------------------+
   //Virtual function of creation of a request to the Google server
   //Returned value: Request line   
   //+------------------------------------------------------------------+   
   virtual string CreateRequest() {return "";}
   //+------------------------------------------------------------------+
   //Virtual function of identification of a diagram type
   //Returned value: Diagram type      
   //+------------------------------------------------------------------+      
   virtual DIAGRAM_TYPE Type() {return DIAGRAM_TYPE_BASE;}
  };
//+------------------------------------------------------------------+
//Constructor of the base class   
//+------------------------------------------------------------------+
void CDiagram::CDiagram()
  {
   X=300;
   Y=300;
   name="";
   title="";
   title_color=Black;
   title_font_size=0;
   legend_position=0;
   fill_chart=-1;
   fill_bg=-1;
  }
//+------------------------------------------------------------------+
//Set diagram size
// X_size - width
// Y_size - height
//Notice: Maximum height and width of a chart (except CMapChart) is 1000x1000 pixels, all the values greater than allowed are cut
//Notice: Maximum height and width of  the CMapChart charts is 440x220 pixels; all the vales greater than allowed are automatically cut
//Notice: In addition, maximum area of a chart cannot exceed 300000 pixels. If the limit is exceeded the 1000x300 pixels size will be automatically set 
//+------------------------------------------------------------------+
void CDiagram::SetSize(int X_size,int Y_size)
  {
   X=X_size; Y=Y_size;
   if(X>1000) X=1000;//check size of a chart
   if(Y>1000) Y=1000;//not to be too big
   if(X<1) X=1;//or small
   if(Y<1) Y=1;
   if(X*Y>300000) {X=1000; Y=300;}//and to fit by the area
  }
//+------------------------------------------------------------------+
//Set sizes of the chart edge and the legend
// left - size of the left edge 
// right - size of the right edge 
// top - size of the upper edge 
// bottom - size of the lower edge 
// legend_width - width of the legend
// legend_height - height of the legend
//Notice: All values are in pixels
//+------------------------------------------------------------------+
void CDiagram::SetMargins(int left,int right,int top,int bottom,int legend_width=0,int legend_height=0)
  {
   margin_left=left;
   margin_right=right;
   margin_top=top;
   margin_bottom=bottom;
   legend_width_=legend_width;
   legend_height_=legend_height;
  }
//+------------------------------------------------------------------+
//Setting header of a diagram
// text   -Text of header
// color_   -Color of header 
// font_size   -Font size of header
//+------------------------------------------------------------------+
void CDiagram::SetTitle(string text,color color_=Black,int font_size=10)
  {
   title=Remove_spaces(text,"+");
   title_color=color_;
   title_font_size=font_size;
  }
//+------------------------------------------------------------------+
//Check if the object the diagram is bound to exists
//Returned value: true-object exists, false-doesn't exist
//+------------------------------------------------------------------+
bool CDiagram::ObjExist()
  {
   return(ObjectFind(chart_ID,name)>=0);
  }
//+------------------------------------------------------------------+
//Check if any graphic object exists
// _chart_ID   -Chart identifier
// _name   -Object name
//Returned value: true-object exists, false-doesn't exist
//+------------------------------------------------------------------+
bool CDiagram::ObjExist(long _chart_ID,string _name)
  {
   return(ObjectFind(_chart_ID,_name)>=0);
  }
//+------------------------------------------------------------------+
//Binding chart to a graphical object
// _chart_ID   -Chart identifier
// _name   -Object name
//Returned value: true-successful, false-unsuccessful 
//Notice: To get additional information about an error, call the GetLastError() function
//+------------------------------------------------------------------+
bool CDiagram::Attach(long _chart_ID,string _name)
  {
   if(!ObjExist(_chart_ID,_name))
     {SetUserError(DIAGRAM_ERR_OBJ_NOT_EXIST); return false;}
   obj_type=ObjectGetInteger(_chart_ID,_name,OBJPROP_TYPE);
   if(obj_type!=OBJ_BITMAP && obj_type!=OBJ_BITMAP_LABEL)
     {SetUserError(DIAGRAM_ERR_INCORRECT_OBJ_TYPE); return false;}
   chart_ID=_chart_ID;
   name=_name;
   return true;
  }
//+------------------------------------------------------------------+
//Unbinds chart from a graphic object
//Returned value: true-successful, false-unsuccessful 
//Notice: To get additional information about an error, call the GetLastError() function
//+------------------------------------------------------------------+
bool CDiagram::Deatch()
  {
   CString src;
   src.Assign(TerminalInfoString(TERMINAL_DATA_PATH));
   src.Append("\MQL5\Files\\"+name+".png");
   src.Replace("\\","\\\\");
   CString dst;
   dst.Assign(TerminalInfoString(TERMINAL_DATA_PATH));
   dst.Append("\MQL5\Images\\"+name+".bmp");
   dst.Replace("\\","\\\\");
   DeleteFileW(dst.Str());
   DeleteFileW(src.Str());
   if(!ObjExist()) {SetUserError(DIAGRAM_ERR_OBJ_NOT_EXIST); return false;}
   switch(obj_type)
     {
      case OBJ_BITMAP:
        {
         ObjectSetString(chart_ID,name,OBJPROP_BMPFILE,"");
         return true;
        }
      case OBJ_BITMAP_LABEL:
        {
         ObjectSetString(chart_ID,name,OBJPROP_BMPFILE,0,"");
         ObjectSetString(chart_ID,name,OBJPROP_BMPFILE,1,"");
         return true;
        }
      default: return false;
     }
   name="";
   chart_ID=-1;
   ChartRedraw();
   return true;
  }
//+------------------------------------------------------------------+
//Gets chart from the Google server and places it to a bound graphical object
//Returned value: true-successful, false-unsuccessful 
//Notice: To get additional information about an error, call the GetLastError() function
//+------------------------------------------------------------------+
bool CDiagram::GetChart()
  {
   if(!ObjExist()) {SetUserError(DIAGRAM_ERR_OBJ_NOT_EXIST); return false;}
   string request=CreateRequest();
//Print(request);
   if(StringLen(request)>2048) {SetUserError(DIAGRAM_ERR_TOO_LARGE_REQUEST); return false;}
//try to create a connection
   int rv=InternetAttemptConnect(0);
   if(rv!=0) {SetUserError(DIAGRAM_ERR_INTERNET_CONNECT_FAILED); return false;}
//initialize structures
   int hInternetSession=InternetOpenW("Microsoft Internet Explorer", 0, "", "", 0);
   if(hInternetSession<=0) {SetUserError(DIAGRAM_ERR_INTERNET_CONNECT_FAILED); return false;}
//send a request
   int hURL=InternetOpenUrlW(hInternetSession, request, "", 0, 0, 0);
   if(hURL<=0) SetUserError(DIAGRAM_ERR_INTERNET_CONNECT_FAILED);
//prepare paths for the converter 
   CString src;
   src.Assign(TerminalInfoString(TERMINAL_PATH));
   src.Append("\MQL5\Files\\"+name+".png");
   src.Replace("\\","\\\\");
   CString dst;
   dst.Assign(TerminalInfoString(TERMINAL_PATH));
   dst.Append("\MQL5\Images\\"+name+".bmp");
   dst.Replace("\\","\\\\");
   DeleteFileW(dst.Str());
   DeleteFileW(src.Str());
   CFileBin chart_file;//file the result is calculated to
                       //create it
   chart_file.Open(name+".png",FILE_BIN|FILE_WRITE);
//****************************   
   int dwBytesRead[1];// amount of read data
   char readed[1000];//information itself
                     //read information received from the server after requesting
   while(InternetReadFile(hURL,readed,1000,dwBytesRead))
     {
      if(dwBytesRead[0]<=0) break;//no information - exit
      chart_file.WriteCharArray(readed,0,dwBytesRead[0]);//write information to a file
     }
   InternetCloseHandle(hInternetSession);//close connection
   chart_file.Close();//and the file
                      //convert the file
   if(!Convert_PNG(src.Str(),dst.Str())) SetUserError(DIAGRAM_ERR_IMAGE_CONVERSION_FAILED);
//bind the file to a graphical object
   switch(obj_type)
     {
      case OBJ_BITMAP:
        {
         ObjectSetString(chart_ID,name,OBJPROP_BMPFILE,name+".bmp");
         return true;
        }
      case OBJ_BITMAP_LABEL:
        {
         ObjectSetString(chart_ID,name,OBJPROP_BMPFILE,0,name+".bmp");
         ObjectSetString(chart_ID,name,OBJPROP_BMPFILE,1,name+".bmp");
         return true;
        }
      default: return false;
     }
//redraw the chart   
   ChartRedraw();
  }
//+------------------------------------------------------------------+
//\class CLineXYChart
//\brief Class for drawing a line chart
//Notice: inherited from CDiagram  
//+------------------------------------------------------------------+
class CLineXYChart : public CDiagram
  {
protected:
   int               total_lines;
   int               total_labels;
   int               axes;
   double            xmax_val,xmin_val;
   double            ymax_val,ymin_val;
   double            bottom_axe_step;
   double            top_axe_step;
   double            left_axe_step;
   double            right_axe_step;
   int               bottom_axe_line;
   int               top_axe_line;
   int               left_axe_line;
   int               right_axe_line;
   bool              grid;
   double            grid_x,grid_y;
   string            bottom_axe_label;
   string            top_axe_label;
   string            left_axe_label;
   string            right_axe_label;
   int               bottom_axe_index;
   int               top_axe_index;
   int               left_axe_index;
   int               right_axe_index;
   color             bottom_axe_color;
   color             top_axe_color;
   color             left_axe_color;
   color             right_axe_color;
   int               bottom_axe_size;
   int               top_axe_size;
   int               left_axe_size;
   int               right_axe_size;
   int               bottom_axe_digits;
   int               top_axe_digits;
   int               left_axe_digits;
   int               right_axe_digits;
   color             axe_label_color;
   int               axe_label_size;
   //Structure for storing information about lines
   struct Line
     {
      //identifier of line
      int               ID;
      //information about axes of a line
      CArrayDouble      X_data;
      CArrayDouble      Y_data;
      CArrayDouble      Z_data;
      CArrayDouble      H_data;
      //total amount of data
      int               data_amount;
      //color
      color             color_;
      //number of digits to be truncated
      int               digits;
      //legend
      string            legend;
      //style
      int               thickness;
      int               dash_length;
      int               space_length;
      //markers
      int               marker_type;
      color             marker_color;
      int               marker_size;
      double            marker_z_order;
      //filling
      int               fill_type;
      color             fill_color;
      int               fill_start;
      int               fill_end;
      //min and max values of axes
      double            x_min,y_min,x_max,y_max;
      //scaling of line
      bool              scaling;
      double            stretch;
      //dimension for CPieChart
      int               dimensional;
      //constructor
      void Line(){ID=-1;}
     };
   ///Structure for storing information about labels
   struct Label
     {
      //label identifier
      int               ID;
      //identifier of a line the label belongs to
      int               line_ID;
      //type
      int               type;
      //anchor point
      int               data_point;
      //text
      string            text;
      //color
      color             color_;
      //size
      int               size;
      //level of overlaying
      double            z_order;
      //horizontal position
      int               h_placement;
      //vertical position
      int               v_placement;
      //constructor
      void Label() {ID=-1; line_ID=-1;}
     };
   Line              Lines[100];
   Label             Labels[100];
   void              GetLinesMaxMin();
   void              GetChartMaxMin();
   void              ReqGrid(CString &res);
   void              ReqAxisLabels(CString &res);
   void              ReqSize(CString &res);
   void              ReqTitle(CString &res);
   void              ReqFill(CString &res);
   void              ReqMargins(CString &res);
   void              ReqLineData(CString &res);
   void              ReqLineColor(CString &res);
   void              ReqLineStyle(CString &res);
   void              ReqLineLegend(CString &res);
   void              ReqLineScaling(CString &res);
   void              ReqLineMarkers(CString &res);
   void              ReqLineFilling(CString &res);
   void              ReqLineLabels(CString &res);
public:
   int               AddLine(double &Y_data[],double &X_data[],color color_,int digits,string legend);
   int               AddLine(double &Data[],color color_,int digits,string legend);
   int               AddLine(CArrayDouble *Y_data,CArrayDouble *X_data,color color_,int digits,string legend);
   int               AddLine(CArrayDouble *Data,color color_,int digits,string legend);
   bool              SetLineData(int line_ID,double &Y_data[],double &X_data[]);
   bool              SetLineData(int line_ID,double &Data[]);
   bool              SetLineData(int line_ID,CArrayDouble *Y_data,CArrayDouble *X_data);
   bool              SetLineData(int line_ID,CArrayDouble *Data);
   void              DeleteLine(int line_ID);
   bool              SetLineStyle(int line_ID,int thickness,int dash_length,int space_length);
   bool              SetLineMarker(int line_ID,DIAGRAM_LINE_MARKERS marker_type,color marker_color,int marker_size,double marker_z_order);
   bool              SetLineFilling(int line_ID,color fill_color,int fill_line_ID);
   bool              SetLineFilling(int line_ID,color fill_color,int start_index,int end_index);
   bool              SetLineScaling(int line_ID,bool scaling,double stretch);
   int               AddLabel(int line_ID,DIAGRAM_LABELS_TYPE type,int data_point,string text,color color_,int size);
   bool              SetLabelPlacement(int label_ID,double z_order,
                                       DIAGRAM_LABELS_HORIZONTAL_PLACEMENT h_placement,DIAGRAM_LABELS_VERTICAL_PLACEMENT v_placement);
   void              DeleteLabel(int label_ID);
   int               GetLineID(int pos);
   int               GetLinePos(int line_ID);
   int               GetLabelID(int pos);
   int               GetLabelPos(int line_ID);
   //+------------------------------------------------------------------+
   //Gets number of lines at a chart
   //Returned value: Number of lines at a chart
   //+------------------------------------------------------------------+
   int LinesTotal() {return total_lines;}
   //+------------------------------------------------------------------+
   //Gets number of labels at a chart
   //Returned value: Number of labels at a chart
   //+------------------------------------------------------------------+   
   int LabelsTotal() {return total_labels;}
   void              SetAxis(int Axis,int line_ID,double step,color color_,int size,int digits);
   //+------------------------------------------------------------------+
   //Sets a grid
   // x_step -step of the grid for the  axis
   // y_step -step of the grid for the Y axis
   //Notice: x_step and y_step are relative values withing the range 0..100 taken in percentage. I.e. number of lines of the grid for the X axis is 100/x_step, for Y axis it is 100/y_step
   //+------------------------------------------------------------------+   
   void SetGrid(double x_step=20,double y_step=20) {grid=true; grid_x=x_step; grid_y=y_step;}
   void HideGrid() {grid=false;}
   void              SetAxisLabels(string bottom,string top,string left,string right,color color_,int size);
   string            CreateRequest();
   //+------------------------------------------------------------------+
   //Virtual function of identification of a diagram type
   //Returned value: Diagram type      
   //+------------------------------------------------------------------+         
   DIAGRAM_TYPE Type() {return DIAGRAM_TYPE_LINEXY;}
   void              CLineXYChart();
  };
//+------------------------------------------------------------------+
///Constructor
void CLineXYChart::CLineXYChart()
  {
   total_lines=0;
   total_labels=0;
   axes=0;
   xmax_val=DBL_MIN; xmin_val=DBL_MAX;
   ymax_val=DBL_MIN; ymin_val=DBL_MAX;
   bottom_axe_line=top_axe_line=left_axe_line=right_axe_line=-2;
   grid=false;
   bottom_axe_label=top_axe_label=left_axe_label=right_axe_label="";
   bottom_axe_index=top_axe_index=left_axe_index=right_axe_index=-1;
  }
//+------------------------------------------------------------------+
//Set labels for chart axes
// bottom -Label of X axis
// top -Label of the upper axis
// left -Label of the Y axis
// right -Label of the right axis
// color_ -Color of labels
// size -Font size
//+------------------------------------------------------------------+
void CLineXYChart::SetAxisLabels(string bottom="",string top="",string left="",string right="",color color_=Black,int size=14)
  {
   bottom_axe_label=Remove_spaces(bottom);
   top_axe_label=Remove_spaces(top);
   left_axe_label=Remove_spaces(left);
   right_axe_label=Remove_spaces(right);
   axe_label_color=color_;
   axe_label_size=size;
  }
//+------------------------------------------------------------------+
//Setting the scaling of line
// line_ID -LIne identifier
// scaling -false-binding the line to the scale of the main axis, true-to the secondary axis
// stretch -Scale of the line in percentage to the initial one (0-leave the scale without changes)
//Notice: To get additional information about an error, call the GetLastError() function
//Returned value: true-if successful, otherwise-false
//+------------------------------------------------------------------+ 
bool CLineXYChart::SetLineScaling(int line_ID,bool scaling,double stretch=0)
  {
   if(line_ID<0 || line_ID>99) {SetUserError(DIAGRAM_ERR_INCORRECT_LINE_ID); return false;}
   for(int i=0;i<100;i++)
      if(Lines[i].ID==line_ID)
        {
         Lines[i].scaling=scaling;
         Lines[i].stretch=stretch;
         return true;
        }
   SetUserError(DIAGRAM_ERR_INCORRECT_LINE_ID);
   return false;
  }
//+------------------------------------------------------------------+
//Setting chart axis
// Axis -combination of flags of the DIAGRAM_AXIS type
// line_ID -LIne identifier
// step -Step of axis
// color_ -Color of axis
// size -Size of axis
// digits -Number of digits to be displayed
//Notice: To get additional information about an error, call the GetLastError() function
//Returned value: true-if successful, otherwise-false
//+------------------------------------------------------------------+
void CLineXYChart::SetAxis(int Axis,int line_ID=-1,double step=0,color color_=Black,int size=10,int digits=2)
  {
   int tmp_axe=Axis;
   if(tmp_axe  &DIAGRAM_AXIS_BOTTOM)
     {
      bottom_axe_line=line_ID;
      bottom_axe_step=step;
      bottom_axe_digits=digits;
      bottom_axe_color=color_;
      bottom_axe_size=size;
     }
   tmp_axe=Axis;
   if(tmp_axe  &DIAGRAM_AXIS_TOP)
     {
      top_axe_line=line_ID;
      top_axe_step=step;
      top_axe_digits=digits;
      top_axe_color=color_;
      top_axe_size=size;
     }
   tmp_axe=Axis;
   if(tmp_axe  &DIAGRAM_AXIS_LEFT)
     {
      left_axe_line=line_ID;
      left_axe_step=step;
      left_axe_digits=digits;
      left_axe_color=color_;
      left_axe_size=size;
     }

   tmp_axe=Axis;
   if(tmp_axe  &DIAGRAM_AXIS_RIGHT)
     {
      right_axe_line=line_ID;
      right_axe_step=step;
      right_axe_digits=digits;
      right_axe_color=color_;
      right_axe_size=size;
     }
  }
//+------------------------------------------------------------------+
void CLineXYChart::GetChartMaxMin()
  {
   GetLinesMaxMin();
   xmax_val=DBL_MIN; xmin_val=DBL_MAX;
   ymax_val=DBL_MIN; ymin_val=DBL_MAX;
   for(int i=0;i<100;i++)
      if(Lines[i].ID>-1 && !Lines[i].scaling)
        {
         if(Lines[i].x_max>xmax_val) xmax_val=Lines[i].x_max;
         if(Lines[i].x_min<xmin_val) xmin_val=Lines[i].x_min;
         if(Lines[i].y_max>ymax_val) ymax_val=Lines[i].y_max;
         if(Lines[i].y_min<ymin_val) ymin_val=Lines[i].y_min;
        }
  }
//+------------------------------------------------------------------+
void CLineXYChart::GetLinesMaxMin()
  {
   for(int i=0;i<100;i++)
      if(Lines[i].ID>-1)
        {
         Lines[i].x_max=DBL_MIN; Lines[i].x_min=DBL_MAX;
         Lines[i].y_max=DBL_MIN; Lines[i].y_min=DBL_MAX;
         for(int j=0;j<Lines[i].X_data.Total();j++)
           {
            if(Lines[i].X_data.At(j)>Lines[i].x_max) Lines[i].x_max=Lines[i].X_data.At(j);
            if(Lines[i].X_data.At(j)<Lines[i].x_min) Lines[i].x_min=Lines[i].X_data.At(j);
           }
         for(int j=0;j<Lines[i].Y_data.Total();j++)
           {
            if(Lines[i].Y_data.At(j)>Lines[i].y_max) Lines[i].y_max=Lines[i].Y_data.At(j);
            if(Lines[i].Y_data.At(j)<Lines[i].y_min) Lines[i].y_min=Lines[i].Y_data.At(j);
           }
         Lines[i].y_max=Lines[i].y_max+(Lines[i].y_max-Lines[i].y_min)*Lines[i].stretch/100.0;
         Lines[i].y_min=Lines[i].y_min-(Lines[i].y_max-Lines[i].y_min)*Lines[i].stretch/100.0;
        }
  }
//+------------------------------------------------------------------+
//Gets identifier of a line by its position in the list
// pos - Position in the list
//Notice: To get additional information about an error, call the GetLastError() function
//Returned value: line identifier in case of success, otherwise -1
//+------------------------------------------------------------------+
int CLineXYChart::GetLineID(int pos)
  {
   int cnt=0;
   for(int i=0;i<100;i++)
      if(Lines[i].ID>-1)
        {
         if(cnt==pos) return Lines[i].ID;
         cnt++;
        }
   return -1;
  }
//+------------------------------------------------------------------+
//Gets position of a line in the list by its identifier
// line_ID - Line identifier
//Notice: To get additional information about an error, call the GetLastError() function
//Returned value: line identifier in case of success, otherwise -1
//+------------------------------------------------------------------+
int CLineXYChart::GetLinePos(int line_ID)
  {
   int cnt=0;
   for(int i=0;i<100;i++)
      if(Lines[i].ID>-1)
        {
         if(Lines[i].ID==line_ID) return cnt;
         cnt++;
        }
   return -1;
  }
//+------------------------------------------------------------------+
//Gets identifier of a label by its position in the list
// pos - Position in the list
//Notice: To get additional information about an error, call the GetLastError() function
//Returned value: Identifier of the label in case of success, otherwise -1
//+------------------------------------------------------------------+
int CLineXYChart::GetLabelID(int pos)
  {
   int cnt=0;
   for(int i=0;i<100;i++)
      if(Labels[i].ID>-1)
        {
         if(cnt==pos) return Labels[i].ID;
         cnt++;
        }
   return -1;
  }
//+------------------------------------------------------------------+
//Gets position of a label in the list by its identifier
// label_ID - Label identifier
//Notice: To get additional information about an error, call the GetLastError() function
//Returned value: Identifier of the label in case of success, otherwise -1
//+------------------------------------------------------------------+
int CLineXYChart::GetLabelPos(int label_ID)
  {
   int cnt=0;
   for(int i=0;i<100;i++)
      if(Labels[i].ID>-1)
        {
         if(Labels[i].ID==label_ID) return cnt;
         cnt++;
        }
   return -1;
  }
//+------------------------------------------------------------------+
//Add label
// line_ID -Identifier of a line the label will be placed at
// type -Label type
// data_point -Anchor point of the label
// text -Text of the label
// color_ -Color
// size -Size
//Notice: To get additional information about an error, call the GetLastError() function
//Returned value: Identifier of the label in case of success, otherwise -1
//+------------------------------------------------------------------+
int CLineXYChart::AddLabel(int line_ID,DIAGRAM_LABELS_TYPE type,int data_point,string text,color color_,int size)

  {
   for(int i=0;i<100;i++)
      if(Labels[i].ID==-1)
        {
         Labels[i].line_ID=line_ID;
         Labels[i].type=type;
         Labels[i].data_point=data_point;
         Labels[i].text=text;
         Labels[i].color_=color_;
         Labels[i].size=size;
         Labels[i].z_order=0;
         Labels[i].h_placement=1;
         Labels[i].v_placement=1;
         Labels[i].ID=total_labels++;
         return Labels[i].ID;
        }
   SetUserError(DIAGRAM_ERR_TOO_MANY_LABELS);
   return -1;
  }
//+------------------------------------------------------------------+
//Set position of a label relatively to an anchor point
// label_ID -Label identifier
// z_order - Order of overlaying
// h_placement - Horizontal position
// v_placement - Vertical position
//Notice: To get additional information about an error, call the GetLastError() function
//Returned value: true-if successful, otherwise-false
//+------------------------------------------------------------------+
bool CLineXYChart::SetLabelPlacement(int label_ID,double z_order,
                                     DIAGRAM_LABELS_HORIZONTAL_PLACEMENT h_placement,DIAGRAM_LABELS_VERTICAL_PLACEMENT v_placement)
  {
   for(int i=0;i<100;i++)
      if(Labels[i].ID==label_ID)
        {
         Labels[i].z_order=0;
         Labels[i].h_placement=1;
         Labels[i].v_placement=1;
         Labels[i].ID=total_labels++;
         return true;
        }
   SetUserError(DIAGRAM_ERR_INCORRECT_LABEL_ID);
   return false;
  }
//+------------------------------------------------------------------+
//Sets style of a line
// line_ID -LIne identifier
// thickness -Thickness
// dash_length -Dash length
// space_length -Space length
//Notice: To get additional information about an error, call the GetLastError() function
//Returned value: true-if successful, otherwise-false
//+------------------------------------------------------------------+
bool CLineXYChart::SetLineStyle(int line_ID,int thickness,int dash_length,int space_length)
  {
   if(line_ID<0 || line_ID>99) {SetUserError(DIAGRAM_ERR_INCORRECT_LINE_ID); return false;}
   for(int i=0;i<100;i++)
      if(Lines[i].ID==line_ID)
        {
         Lines[i].thickness=thickness;
         Lines[i].dash_length=dash_length;
         Lines[i].space_length=space_length;
         return true;
        }
   SetUserError(DIAGRAM_ERR_INCORRECT_LINE_ID);
   return false;
  }
//+------------------------------------------------------------------+
//Sets a data set for a line
// line_ID -LIne identifier
// Y_data -Data for the Y axis
// X_data -Data for the X axis
//Notice: To get additional information about an error, call the GetLastError() function
//Returned value: true-if successful, otherwise-false
//+------------------------------------------------------------------+
bool CLineXYChart::SetLineData(int line_ID,double &Y_data[],double &X_data[])
  {
   if(line_ID<0 || line_ID>99) {SetUserError(DIAGRAM_ERR_INCORRECT_LINE_ID); return false;}
   for(int i=0;i<100;i++)
      if(Lines[i].ID==line_ID)
        {
         Lines[i].Y_data.AssignArray(Y_data);
         Lines[i].X_data.AssignArray(X_data);
         return true;
        }
   SetUserError(DIAGRAM_ERR_INCORRECT_LINE_ID);
   return false;

  }
//+------------------------------------------------------------------+
//Sets a data set for a line
// line_ID -LIne identifier
// Data -Data for the Y axis
//Notice: To get additional information about an error, call the GetLastError() function
//Notice: In case this function is called, data for the Y axis is automatically set as 0...N sequence
//Returned value: true-if successful, otherwise-false
//+------------------------------------------------------------------+
bool CLineXYChart::SetLineData(int line_ID,double &Data[])
  {
   if(line_ID<0 || line_ID>99) {SetUserError(DIAGRAM_ERR_INCORRECT_LINE_ID); return false;}
   for(int i=0;i<100;i++)
      if(Lines[i].ID==line_ID)
        {
         Lines[i].Y_data.AssignArray(Data);
         double X_Data[];
         int y_size=ArrayRange(Data,0);
         ArrayResize(X_Data,y_size);
         for(int j=0;j<y_size;j++) X_Data[j]=j;
         Lines[i].X_data.AssignArray(X_Data);
         return true;
        }
   SetUserError(DIAGRAM_ERR_INCORRECT_LINE_ID);
   return false;

  }
//+------------------------------------------------------------------+
//Sets a data set for a line
// line_ID -LIne identifier
// Y_data -Data for the Y axis
// X_data -Data for the X axis
//Notice: To get additional information about an error, call the GetLastError() function
//Returned value: true-if successful, otherwise-false
//+------------------------------------------------------------------+
bool CLineXYChart::SetLineData(int line_ID,CArrayDouble *Y_data,CArrayDouble *X_data)
  {
   if(line_ID<0 || line_ID>99) {SetUserError(DIAGRAM_ERR_INCORRECT_LINE_ID); return false;}
   for(int i=0;i<100;i++)
      if(Lines[i].ID==line_ID)
        {
         Lines[i].Y_data.AssignArray(Y_data);
         Lines[i].X_data.AssignArray(X_data);
         return true;
        }
   SetUserError(DIAGRAM_ERR_INCORRECT_LINE_ID);
   return false;

  }
//+------------------------------------------------------------------+
//Sets a data set for a line
// line_ID -LIne identifier
// Data -Data for the Y axis
//Notice: To get additional information about an error, call the GetLastError() function
//Notice: In case this function is called, data for the Y axis is automatically set as 0...N sequence
//Returned value: true-if successful, otherwise-false
//+------------------------------------------------------------------+
bool CLineXYChart::SetLineData(int line_ID,CArrayDouble *Data)
  {
   if(line_ID<0 || line_ID>99) {SetUserError(DIAGRAM_ERR_INCORRECT_LINE_ID); return false;}
   for(int i=0;i<100;i++)
      if(Lines[i].ID==line_ID)
        {
         Lines[i].Y_data.AssignArray(Data);
         double X_Data[];
         int y_size=Data.Total();
         ArrayResize(X_Data,y_size);
         for(int j=0;j<y_size;j++) X_Data[j]=j;
         Lines[i].X_data.AssignArray(X_Data);
         return true;
        }
   SetUserError(DIAGRAM_ERR_INCORRECT_LINE_ID);
   return false;

  }
//+------------------------------------------------------------------+
//Set markers of a line
// line_ID -LIne identifier
// marker_type -Marker type
// marker_color -Color
// marker_size -Size in pixels
// marker_z_order - Order of overlaying
//Notice: To get additional information about an error, call the GetLastError() function
//Returned value: true-if successful, otherwise-false
//+------------------------------------------------------------------+
bool CLineXYChart::SetLineMarker(int line_ID,DIAGRAM_LINE_MARKERS marker_type,color marker_color,int marker_size,double marker_z_order=0)
  {
   if(line_ID<0 || line_ID>99) {SetUserError(DIAGRAM_ERR_INCORRECT_LINE_ID); return false;}
   for(int i=0;i<100;i++)
      if(Lines[i].ID==line_ID)
        {
         Lines[i].marker_type=marker_type;
         Lines[i].marker_color=marker_color;
         Lines[i].marker_size=marker_size;
         Lines[i].marker_z_order=marker_z_order;
         return true;
        }
   SetUserError(DIAGRAM_ERR_INCORRECT_LINE_ID);
   return false;
  }
//+------------------------------------------------------------------+
//Setting up of filling of an area between two lines
// line_ID -LIne identifier
// fill_color -Color of filling
// fill_line_ID -Identifier of the second line
//Notice: To get additional information about an error, call the GetLastError() function
//Returned value: true-if successful, otherwise-false
//+------------------------------------------------------------------+
bool CLineXYChart::SetLineFilling(int line_ID,color fill_color,int fill_line_ID)
  {
   if(line_ID<0 || line_ID>99) {SetUserError(DIAGRAM_ERR_INCORRECT_LINE_ID); return false;}
   for(int i=0;i<100;i++)
      if(Lines[i].ID==line_ID)
        {
         if(GetLinePos(fill_line_ID)<0)
           {SetUserError(DIAGRAM_ERR_INCORRECT_FILL_INDEX); return false;}
         Lines[i].fill_type=1;
         Lines[i].fill_color=fill_color;
         Lines[i].fill_start=fill_line_ID;
         Lines[i].fill_end=0;
         return true;
        }
   SetUserError(DIAGRAM_ERR_INCORRECT_LINE_ID);
   return false;
  }
//+------------------------------------------------------------------+
//Setting up of filling of area under a line
// line_ID -LIne identifier
// fill_color -Color of filling
// start_index -Point on the line, the filling should be started from
// end_index -Point in the line the filling should be stopped at
//Notice: To get additional information about an error, call the GetLastError() function
//Returned value: true-if successful, otherwise-false
//+------------------------------------------------------------------+
bool CLineXYChart::SetLineFilling(int line_ID,color fill_color,int start_index,int end_index)
  {
   if(line_ID<0 || line_ID>99) {SetUserError(DIAGRAM_ERR_INCORRECT_LINE_ID); return false;}
   for(int i=0;i<100;i++)
      if(Lines[i].ID==line_ID)
        {
         Lines[i].fill_type=2;
         Lines[i].fill_color=fill_color;
         Lines[i].fill_start=start_index;
         Lines[i].fill_end=end_index;
         return true;
        }
   SetUserError(DIAGRAM_ERR_INCORRECT_LINE_ID);
   return false;
  }
//+------------------------------------------------------------------+
//Adds line
// Data -Data for the Y axis
// color_ -Color of line
// digits -Number of digits for truncation of information
// legend -Legend
//Notice: To get additional information about an error, call the GetLastError() function
//Notice: In case this function is called, data for the Y axis is automatically set as 0...N sequence
//Returned value: identifier of line in case of success, otherwise - 1
//+------------------------------------------------------------------+
int CLineXYChart::AddLine(double &Data[],color color_=Red,int digits=8,string legend="")
  {
   for(int i=0;i<100;i++)
      if(Lines[i].ID==-1)
        {
         int y_size=ArrayRange(Data,0);
         double X_Data[];
         ArrayResize(X_Data,y_size);
         for(int j=0;j<y_size;j++) X_Data[j]=j;
         Lines[i].X_data.AssignArray(X_Data);
         Lines[i].Y_data.AssignArray(Data);
         Lines[i].data_amount=y_size;
         Lines[i].color_=color_;
         Lines[i].digits=digits;
         Lines[i].legend=Remove_spaces(legend, "+");
         //style
         Lines[i].thickness=1;
         Lines[i].dash_length=1;
         Lines[i].space_length=0;
         //marker
         Lines[i].marker_type=0;
         Lines[i].marker_size=0;
         Lines[i].marker_color=0;
         Lines[i].marker_z_order=0;
         //filling
         Lines[i].fill_type=0;
         Lines[i].fill_color=0;
         Lines[i].fill_start=0;
         Lines[i].fill_end=0;
         //scaling
         Lines[i].scaling=false;
         Lines[i].stretch=0;
         //id    
         Lines[i].dimensional=0;
         Lines[i].ID=total_lines++;
         return Lines[i].ID;
        }
   SetUserError(DIAGRAM_ERR_TOO_MANY_LINES);
   return -1;
  }
//+------------------------------------------------------------------+
//Adds line
// Y_Data -Data for the Y axis
// X_Data -Data for the X axis
// color_ -Color of line
// digits -Number of digits for truncation of information
// legend -Legend
//Notice: To get additional information about an error, call the GetLastError() function
//Returned value: identifier of line in case of success, otherwise - 1
//+------------------------------------------------------------------+
int CLineXYChart::AddLine(double &Y_Data[],double &X_Data[],color color_=Red,int digits=8,string legend="")
  {
   for(int i=0;i<100;i++)
      if(Lines[i].ID==-1)
        {
         int y_size=ArrayRange(Y_Data, 0);
         int x_size=ArrayRange(X_Data, 0);
         if(y_size!=x_size)
            return(AddLine(Y_Data,color_,digits,legend));
         Lines[i].X_data.AssignArray(X_Data);
         Lines[i].Y_data.AssignArray(Y_Data);
         Lines[i].data_amount=y_size;
         Lines[i].color_=color_;
         Lines[i].digits=digits;
         Lines[i].legend=Remove_spaces(legend, "+");
         //style
         Lines[i].thickness=1;
         Lines[i].dash_length=1;
         Lines[i].space_length=0;
         //marker
         Lines[i].marker_type=0;
         Lines[i].marker_size=0;
         Lines[i].marker_color=0;
         Lines[i].marker_z_order=0;
         //filling
         Lines[i].fill_type=0;
         Lines[i].fill_color=0;
         Lines[i].fill_start=0;
         Lines[i].fill_end=0;
         //id    
         Lines[i].dimensional=0;
         Lines[i].ID=total_lines++;
         return Lines[i].ID;
        }
   SetUserError(DIAGRAM_ERR_TOO_MANY_LINES);
   return -1;
  }
//+------------------------------------------------------------------+
//Adds line
// Data -Data for the Y axis
// color_ -Color of line
// digits -Number of digits for truncation of information
// legend -Legend
//Notice: To get additional information about an error, call the GetLastError() function
//Notice: In case this function is called, data for the Y axis is automatically set as 0...N sequence
//Returned value: identifier of line in case of success, otherwise - 1
//+------------------------------------------------------------------+
int CLineXYChart::AddLine(CArrayDouble *Data,color color_=Red,int digits=8,string legend="")
  {
   for(int i=0;i<100;i++)
      if(Lines[i].ID==-1)
        {
         int y_size=Data.Total();
         double X_Data[];
         ArrayResize(X_Data,y_size);
         for(int j=0;j<y_size;j++) X_Data[j]=j;
         Lines[i].X_data.AssignArray(X_Data);
         Lines[i].Y_data.AssignArray(Data);
         Lines[i].data_amount=y_size;
         Lines[i].color_=color_;
         Lines[i].digits=digits;
         Lines[i].legend=Remove_spaces(legend, "+");
         //style
         Lines[i].thickness=1;
         Lines[i].dash_length=1;
         Lines[i].space_length=0;
         //marker
         Lines[i].marker_type=0;
         Lines[i].marker_size=0;
         Lines[i].marker_color=-1;
         Lines[i].marker_z_order=0;
         //filling
         Lines[i].fill_type=0;
         Lines[i].fill_color=0;
         Lines[i].fill_start=0;
         Lines[i].fill_end=0;
         //id    
         Lines[i].dimensional=0;
         Lines[i].ID=total_lines++;
         return Lines[i].ID;
        }
   SetUserError(DIAGRAM_ERR_TOO_MANY_LINES);
   return -1;
  }
//+------------------------------------------------------------------+
//Adds line
// Y_Data -Data for the Y axis
// X_Data -Data for the X axis
// color_ -Color of line
// digits -Number of digits for truncation of information
// legend -Legend
//Notice: To get additional information about an error, call the GetLastError() function
//Returned value: identifier of line in case of success, otherwise - 1
//+------------------------------------------------------------------+
int CLineXYChart::AddLine(CArrayDouble *Y_Data,CArrayDouble *X_Data,color color_=Red,int digits=8,string legend="")
  {
   for(int i=0;i<100;i++)
      if(Lines[i].ID==-1)
        {
         int y_size=Y_Data.Total();
         int x_size=X_Data.Total();
         if(y_size!=x_size)
            return(AddLine(Y_Data,color_,digits,legend));
         Lines[i].X_data.AssignArray(X_Data);
         Lines[i].Y_data.AssignArray(Y_Data);
         Lines[i].data_amount=y_size;
         Lines[i].color_=color_;
         Lines[i].digits=digits;
         Lines[i].legend=Remove_spaces(legend, "+");
         //style
         Lines[i].thickness=1;
         Lines[i].dash_length=1;
         Lines[i].space_length=0;
         //marker
         Lines[i].marker_type=0;
         Lines[i].marker_size=0;
         Lines[i].marker_color=0;
         Lines[i].marker_z_order=0;
         //filling
         Lines[i].fill_type=0;
         Lines[i].fill_color=-1;
         Lines[i].fill_start=0;
         Lines[i].fill_end=0;
         //id    
         Lines[i].dimensional=0;
         Lines[i].ID=total_lines++;
         return Lines[i].ID;
        }
   SetUserError(DIAGRAM_ERR_TOO_MANY_LINES);
   return -1;
  }
//+------------------------------------------------------------------+
//Deletes line from a chart
// line_ID -LIne identifier
//+------------------------------------------------------------------+
void CLineXYChart::DeleteLine(int line_ID)
  {
   if(line_ID<0 || line_ID>99) {SetUserError(DIAGRAM_ERR_INCORRECT_LINE_ID); return;}
   if(Lines[line_ID].ID>-1)
     {
      Lines[line_ID].X_data.Shutdown();
      Lines[line_ID].Y_data.Shutdown();
      Lines[line_ID].Z_data.Shutdown();
      Lines[line_ID].ID=-1;
      total_lines--;
     }
   else SetUserError(DIAGRAM_ERR_INCORRECT_LINE_ID);
  }
//+------------------------------------------------------------------+
//Deletes label from a chart
// label_ID -Label identifier
//+------------------------------------------------------------------+
void CLineXYChart::DeleteLabel(int label_ID)
  {
   if(label_ID<0 || label_ID>99) {SetUserError(DIAGRAM_ERR_INCORRECT_LABEL_ID); return;}
   if(Labels[label_ID].ID>-1)
     {
      Labels[label_ID].ID=-1;
      total_labels--;
     }
   else SetUserError(DIAGRAM_ERR_INCORRECT_LABEL_ID);
  }
//+------------------------------------------------------------------+
//Creates request for the Google server
//Returned value: Request line
//+------------------------------------------------------------------+
string CLineXYChart::CreateRequest()
  {
   GetChartMaxMin();
   CString res;
   res.Assign("http://chart.apis.google.com/chart?cht=lxy");
   ReqSize(res);
   ReqGrid(res);
   ReqAxisLabels(res);
   ReqFill(res);
   ReqTitle(res);
   ReqMargins(res);
   ReqLineData(res);
   ReqLineColor(res);
   ReqLineStyle(res);
   ReqLineMarkers(res);
   ReqLineFilling(res);
   ReqLineLabels(res);
   ReqLineLegend(res);
   ReqLineScaling(res);
   return res.Str();
  }
//+------------------------------------------------------------------+
void CLineXYChart::ReqLineLabels(CString &res)
  {
   if(total_lines>0)
     {
      CString labels_str;
      for(int i=0;i<100;i++)
         if(Labels[i].ID>-1)
           {
            bool fail=false;
            switch(Labels[i].type)
              {
               case 1: {labels_str.Append("f"+Labels[i].text+","); break;}
               case 2: {labels_str.Append("t"+Labels[i].text+","); break;}
               case 3: {labels_str.Append("A"+Labels[i].text+","); break;}
               default: {SetUserError(DIAGRAM_ERR_INVALID_LABEL_TYPE); fail=true;}
              }
            if(!fail)
              {
               labels_str.Append(GetColorStr(Labels[i].color_)+",");
               labels_str.Append(IntegerToString(GetLinePos(Labels[i].line_ID))+",");
               labels_str.Append(IntegerToString(Labels[i].data_point)+",");
               labels_str.Append(IntegerToString(Labels[i].size)+",");
               labels_str.Append(DoubleToString(Labels[i].z_order, 2)+",");
               switch(Labels[i].h_placement)
                 {
                  case 1: {labels_str.Append("l"); break;}
                  case 2: {labels_str.Append("h"); break;}
                  case 3: {labels_str.Append("r"); break;}
                  default: labels_str.Append("l");
                 }
               switch(Labels[i].v_placement)
                 {
                  case 1: {labels_str.Append("b|"); break;}
                  case 2: {labels_str.Append("v|"); break;}
                  case 3: {labels_str.Append("t|"); break;}
                  default: labels_str.Append("b|");
                 }
              }
           }
      //---     
      if(labels_str.Str()!="")
        {
         labels_str.TrimRight("|");
         int pos=res.Find(0,"&chm=");
         if(pos>-1)
           {
            res.Insert(pos+5,labels_str.Str()+"|");
           }
         else
           {
            res.Append("&chm=");
            res.Append(GetPointer(labels_str));
           }
        }
     }
  }
//+------------------------------------------------------------------+
void CLineXYChart::ReqLineStyle(CString &res)
  {
   if(total_lines>0)
     {
      CString style_str;
      for(int i=0;i<100;i++)
         if(Lines[i].ID>-1)
           {
            style_str.Append(IntegerToString(Lines[i].thickness)+",");
            style_str.Append(IntegerToString(Lines[i].dash_length)+",");
            style_str.Append(IntegerToString(Lines[i].space_length)+"|");
           }
      res.Append("&chls=");
      style_str.TrimRight("|");
      res.Append(GetPointer(style_str));
     }
  }
//+------------------------------------------------------------------+
void CLineXYChart::ReqLineScaling(CString &res)
  {
   if(total_lines>0)
     {
      CString scale_str;
      for(int i=0;i<100;i++)
         if(Lines[i].ID>-1)
           {
            if(!Lines[i].scaling)
              {
               scale_str.Append(DoubleToString(xmin_val)+","+DoubleToString(xmax_val)+",");
               scale_str.Append(DoubleToString(ymin_val)+","+DoubleToString(ymax_val)+",");
              }
            else
              {
               scale_str.Append(DoubleToString(Lines[i].x_min)+","+DoubleToString(Lines[i].x_max)+",");
               scale_str.Append(DoubleToString(Lines[i].y_min)+","+DoubleToString(Lines[i].y_max)+",");
              }
           }
      res.Append("&chds=");
      scale_str.TrimRight(",");
      res.Append(GetPointer(scale_str));
     }
  }
//+------------------------------------------------------------------+
void CLineXYChart::ReqLineLegend(CString &res)
  {
   if(total_lines>0)
     {
      CString legend_str;
      for(int i=0;i<100;i++)
         if(Lines[i].ID>-1)
            legend_str.Append(Lines[i].legend+"|");
      if(legend_position>0)
        {
         res.Append("&chdl=");
         legend_str.TrimRight("|");
         res.Append(GetPointer(legend_str));
         res.Append("&chdlp=");
         switch(legend_position)
           {
            case 1: {res.Append("r"); break;}
            case 2: {res.Append("b"); break;}
            case 3: {res.Append("bv"); break;}
            case 4: {res.Append("t"); break;}
            case 5: {res.Append("tv"); break;}
            case 6: {res.Append("l"); break;}
            default: res.Append("r");
           }
        }
     }
  }
//+------------------------------------------------------------------+
void CLineXYChart::ReqLineFilling(CString &res)
  {
   if(total_lines>0)
     {
      CString marker_str;
      for(int i=0;i<100;i++)
         if(Lines[i].ID>-1)
           {
            //---
            if(Lines[i].fill_type>0)
              {
               bool fail=false;
               switch(Lines[i].fill_type)
                 {
                  case 1: {marker_str.Append("b,"); break;}
                  case 2: {marker_str.Append("B,"); break;}
                  default : {SetUserError(DIAGRAM_ERR_INVALID_FILLING_TYPE); fail=true;}
                 }
               if(!fail)
                 {
                  marker_str.Append(GetColorStr(Lines[i].fill_color)+",");
                  switch(Lines[i].fill_type)
                    {
                     case 1:
                       {
                        marker_str.Append(IntegerToString(GetLinePos(Lines[i].fill_start))+",");
                        marker_str.Append(IntegerToString(GetLinePos(Lines[i].ID))+",0|");
                        break;
                       }
                     case 2:
                       {
                        marker_str.Append(IntegerToString(GetLinePos(Lines[i].ID))+",");
                        marker_str.Append(IntegerToString(Lines[i].fill_start)+":");
                        marker_str.Append(IntegerToString(Lines[i].fill_end)+",0|");
                        break;
                       }
                    }
                 }
              }
            //---         
           }
      //---
      if(marker_str.Str()!="")
        {
         marker_str.TrimRight("|");
         int pos=res.Find(0,"&chm=");
         if(pos>-1)
           {
            res.Insert(pos+5,marker_str.Str()+"|");
           }
         else
           {
            res.Append("&chm=");
            res.Append(GetPointer(marker_str));
           }
        }
     }
  }
//+------------------------------------------------------------------+
void CLineXYChart::ReqLineMarkers(CString &res)
  {
   if(total_lines>0)
     {
      CString marker_str;
      for(int i=0;i<100;i++)
         if(Lines[i].ID>-1)
           {
            //---
            if(Lines[i].marker_type>0)
              {
               bool fail=false;
               switch(Lines[i].marker_type)
                 {
                  case 1: {marker_str.Append("a,"); break;}
                  case 2: {marker_str.Append("c,"); break;}
                  case 3: {marker_str.Append("d,"); break;}
                  case 4: {marker_str.Append("o,"); break;}
                  case 5: {marker_str.Append("s,"); break;}
                  default : {SetUserError(DIAGRAM_ERR_INVALID_MARKER_TYPE); fail=true;}
                 }
               if(!fail)
                 {
                  marker_str.Append(GetColorStr(Lines[i].marker_color)+",");
                  marker_str.Append(IntegerToString(GetLinePos(Lines[i].ID))+",-1,");
                  marker_str.Append(IntegerToString(Lines[i].marker_size)+",");
                  marker_str.Append(DoubleToString(Lines[i].marker_z_order, 2)+"|");
                 }
              }
            //---       
           }
      //---
      if(marker_str.Str()!="")
        {
         marker_str.TrimRight("|");
         int pos=res.Find(0,"&chm=");
         if(pos>-1)
           {
            res.Insert(pos+5,marker_str.Str()+"|");
           }
         else
           {
            res.Append("&chm=");
            res.Append(GetPointer(marker_str));
           }
        }
     }
  }
//+------------------------------------------------------------------+
void CLineXYChart::ReqLineColor(CString &res)
  {
   if(total_lines>0)
     {
      CString color_str;
      for(int i=0;i<100;i++)
         if(Lines[i].ID>-1)
            color_str.Append(GetColorStr(Lines[i].color_)+",");
      res.Append("&chco=");
      color_str.TrimRight(",");
      res.Append(GetPointer(color_str));
     }
  }
//+------------------------------------------------------------------+
void CLineXYChart::ReqLineData(CString &res)
  {
   if(total_lines>0)
     {
      CString data_str;
      for(int i=0;i<100;i++)
         if(Lines[i].ID>-1)
           {
            CString xdata_str,ydata_str;
            for(int j=0;j<Lines[i].data_amount;j++)
              {
               xdata_str.Append(DoubleToString(Lines[i].X_data.At(j), Lines[i].digits)+",");
               ydata_str.Append(DoubleToString(Lines[i].Y_data.At(j), Lines[i].digits)+",");
              }
            xdata_str.TrimRight(",");
            ydata_str.TrimRight(",");
            data_str.Append(GetPointer(xdata_str));
            data_str.Append("|");
            data_str.Append(GetPointer(ydata_str));
            data_str.Append("|");
           }
      res.Append("&chd=t:");
      data_str.TrimRight("|");
      res.Append(GetPointer(data_str));
     }
  }
//+------------------------------------------------------------------+
void CLineXYChart::ReqMargins(CString &res)
  {
   if(margin_bottom+margin_left+margin_right+margin_top+legend_height_+legend_width_>0)
     {
      res.Append("&chma=");
      res.Append(IntegerToString(margin_left)+",");
      res.Append(IntegerToString(margin_right)+",");
      res.Append(IntegerToString(margin_top)+",");
      res.Append(IntegerToString(margin_bottom)+"|");
      if(legend_height_+legend_width_>0)
        {
         res.Append(IntegerToString(legend_height_)+",");
         res.Append(IntegerToString(legend_width_));
        }
      else res.TrimRight("|");
     }
  }
//+------------------------------------------------------------------+
void CLineXYChart::ReqFill(CString &res)
  {
   if(fill_chart>-1)
      res.Append("&chf=c,s,"+GetColorStr(fill_chart)+"|bg,s,"+GetColorStr(fill_bg));
  }
//+------------------------------------------------------------------+
void CLineXYChart::ReqTitle(CString &res)
  {
   if(title!="")
      res.Append("&chtt="+title+"&chts="+GetColorStr(title_color)+","+IntegerToString(title_font_size));
  }
//+------------------------------------------------------------------+
void CLineXYChart::ReqSize(CString &res)
  {
   res.Append("&chs=");
   res.Append(IntegerToString(X));
   res.Append("x");
   res.Append(IntegerToString(Y));
  }
//+------------------------------------------------------------------+
void CLineXYChart::ReqGrid(CString &res)
  {
   if(grid)
     {
      res.Append("&chg=");
      res.Append(DoubleToString(grid_x)+","+DoubleToString(grid_y));
     }
  }
//+------------------------------------------------------------------+
void CLineXYChart::ReqAxisLabels(CString &res)
  {
   CString axis_range,axis_types,styles;
   int axe_cnt=0;
   if(bottom_axe_line>-2)
     {
      bottom_axe_index=axe_cnt;
      styles.Append(IntegerToString(bottom_axe_index));
      styles.Append("N*f"+IntegerToString(bottom_axe_digits)+"zsy*,");
      styles.Append(GetColorStr(bottom_axe_color)+",");
      styles.Append(IntegerToString(bottom_axe_size)+",");
      styles.Append("0,lt|");
      //**************************
      axis_types.Append("x,");
      axis_range.Append(IntegerToString(axe_cnt++)+",");
      if(bottom_axe_label!="") {axis_types.Append("x,"); axe_cnt++;}
      if(bottom_axe_line==-1)
        {
         axis_range.Append(DoubleToString(xmin_val)+",");
         axis_range.Append(DoubleToString(xmax_val));
        }
      else
        {
         axis_range.Append(DoubleToString(Lines[GetLinePos(bottom_axe_line)].x_min)+",");
         axis_range.Append(DoubleToString(Lines[GetLinePos(bottom_axe_line)].x_max));
        }
      if(bottom_axe_step!=0)
         axis_range.Append(","+DoubleToString(bottom_axe_step)+"|");
      else axis_range.Append("|");
     }
//---
   if(top_axe_line>-2)
     {
      top_axe_index=axe_cnt;
      styles.Append(IntegerToString(top_axe_index));
      styles.Append("N*f"+IntegerToString(top_axe_digits)+"zsy*,");
      styles.Append(GetColorStr(top_axe_color)+",");
      styles.Append(IntegerToString(top_axe_size)+",");
      styles.Append("0,lt|");
      //**************************      
      axis_types.Append("t,");
      axis_range.Append(IntegerToString(axe_cnt++)+",");
      if(top_axe_label!="") {axis_types.Append("t,"); axe_cnt++;}
      if(top_axe_line==-1)
        {
         axis_range.Append(DoubleToString(xmin_val)+",");
         axis_range.Append(DoubleToString(xmax_val));
        }
      else
        {
         axis_range.Append(DoubleToString(Lines[GetLinePos(top_axe_line)].x_min)+",");
         axis_range.Append(DoubleToString(Lines[GetLinePos(top_axe_line)].x_max));
        }
      if(top_axe_step!=0)
         axis_range.Append(","+DoubleToString(top_axe_step)+"|");
      else axis_range.Append("|");
     }
//---
   if(left_axe_line>-2)
     {
      left_axe_index=axe_cnt;
      styles.Append(IntegerToString(left_axe_index));
      styles.Append("N*f"+IntegerToString(left_axe_digits)+"zsy*,");
      styles.Append(GetColorStr(left_axe_color)+",");
      styles.Append(IntegerToString(left_axe_size)+",");
      styles.Append("0,lt|");
      //**************************         
      axis_types.Append("y,");
      axis_range.Append(IntegerToString(axe_cnt++)+",");
      if(left_axe_label!="") {axis_types.Append("y,"); axe_cnt++;}
      if(left_axe_line==-1)
        {
         axis_range.Append(DoubleToString(ymin_val)+",");
         axis_range.Append(DoubleToString(ymax_val));
        }
      else
        {
         axis_range.Append(DoubleToString(Lines[GetLinePos(left_axe_line)].y_min)+",");
         axis_range.Append(DoubleToString(Lines[GetLinePos(left_axe_line)].y_max));
        }
      if(left_axe_step!=0)
         axis_range.Append(","+DoubleToString(left_axe_step)+"|");
      else axis_range.Append("|");
     }
//---
   if(right_axe_line>-2)
     {
      right_axe_index=axe_cnt;
      styles.Append(IntegerToString(right_axe_index));
      styles.Append("N*f"+IntegerToString(right_axe_digits)+"zsy*,");
      styles.Append(GetColorStr(right_axe_color)+",");
      styles.Append(IntegerToString(right_axe_size)+",");
      styles.Append("0,lt|");
      //**************************         
      axis_types.Append("r,");
      axis_range.Append(IntegerToString(axe_cnt++)+",");
      if(right_axe_label!="") {axis_types.Append("r,"); axe_cnt++;}
      if(right_axe_line==-1)
        {
         axis_range.Append(DoubleToString(ymin_val)+",");
         axis_range.Append(DoubleToString(ymax_val));
        }
      else
        {
         axis_range.Append(DoubleToString(Lines[GetLinePos(right_axe_line)].y_min)+",");
         axis_range.Append(DoubleToString(Lines[GetLinePos(right_axe_line)].y_max));
        }
      if(right_axe_step!=0)
         axis_range.Append(","+DoubleToString(right_axe_step)+"|");
      else axis_range.Append("|");
     }
//---
   res.Append("&chxt=");
   axis_types.TrimRight(",");
   res.Append(GetPointer(axis_types));
   res.Append("&chxr=");
   axis_range.TrimRight("|");
   res.Append(GetPointer(axis_range));
//---
   CString labels,positions;
   if(bottom_axe_index>-1 && bottom_axe_label!="")
     {
      styles.Append(IntegerToString(bottom_axe_index+1)+",");
      styles.Append(GetColorStr(axe_label_color)+",");
      styles.Append(IntegerToString(axe_label_size)+",");
      styles.Append("0,lt|");
      labels.Append(IntegerToString(bottom_axe_index+1)+":|"+bottom_axe_label+"|");
      positions.Append(IntegerToString(bottom_axe_index+1)+","+IntegerToString(50)+"|");
     }
   if(top_axe_index>-1 && top_axe_label!="")
     {
      styles.Append(IntegerToString(top_axe_index+1)+",");
      styles.Append(GetColorStr(axe_label_color)+",");
      styles.Append(IntegerToString(axe_label_size)+",");
      styles.Append("0,lt|");
      labels.Append(IntegerToString(top_axe_index+1)+":|"+top_axe_label+"|");
      positions.Append(IntegerToString(top_axe_index+1)+","+IntegerToString(50)+"|");
     }
   if(left_axe_index>-1 && left_axe_label!="")
     {
      styles.Append(IntegerToString(left_axe_index+1)+",");
      styles.Append(GetColorStr(axe_label_color)+",");
      styles.Append(IntegerToString(axe_label_size)+",");
      styles.Append("0,lt|");
      labels.Append(IntegerToString(left_axe_index+1)+":|"+left_axe_label+"|");
      positions.Append(IntegerToString(left_axe_index+1)+","+IntegerToString(50)+"|");
     }
   if(right_axe_index>-1 && right_axe_label!="")
     {
      styles.Append(IntegerToString(right_axe_index+1)+",");
      styles.Append(GetColorStr(axe_label_color)+",");
      styles.Append(IntegerToString(axe_label_size)+",");
      styles.Append("0,lt|");
      labels.Append(IntegerToString(right_axe_index+1)+":|"+right_axe_label+"|");
      positions.Append(IntegerToString(right_axe_index+1)+","+IntegerToString(50)+"|");
     }
//---
   labels.TrimRight("|");
   positions.TrimRight("|");
   styles.TrimRight("|");
   res.Append("&chxl=");
   res.Append(GetPointer(labels));
   res.Append("&chxp=");
   res.Append(GetPointer(positions));
   res.Append("&chxs=");
   res.Append(GetPointer(styles));
  }
//+------------------------------------------------------------------+
//\class CFormulaChart
//\brief Class for creating formulas
//Notice: Inherited from CLineXYChart
//+------------------------------------------------------------------+
class CFormulaChart : public CLineXYChart
  {
protected:
   string            data;
   color             formula_color;
   void              ReqFormulaColor(CString &res);
public:
   //+------------------------------------------------------------------+
   //Sets a line with formula
   // formula - Line with formula
   //Notice: Formulas must be written using  <a href="http://en.wikipedia.org/wiki/TeX">TeX</a> language
   //+------------------------------------------------------------------+
   void SetFormulaString(string formula) {data=formula;}
   //+------------------------------------------------------------------+
   //Set color of the formula
   // Color - color
   //+------------------------------------------------------------------+   
   void SetFormulaColor(color Color) {formula_color=Color;}
   string            CreateRequest();
   //+------------------------------------------------------------------+
   //Virtual function of identification of a diagram type
   //Returned value: Diagram type      
   //+------------------------------------------------------------------+         
   DIAGRAM_TYPE Type() {return DIAGRAM_TYPE_FORMULA;}
   void              CFormulaChart();
  };
//+------------------------------------------------------------------+
//Constructor
//+------------------------------------------------------------------+
void CFormulaChart::CFormulaChart()
  {
   formula_color=Black;
   data="";
  }
//+------------------------------------------------------------------+
void CFormulaChart::ReqFormulaColor(CString &res)
  {
   res.Append("&chco="+GetColorStr(formula_color));
  }
//+------------------------------------------------------------------+
//Creates request for the Google server
//Returned value: Request line
//+------------------------------------------------------------------+
string CFormulaChart::CreateRequest()
  {
   CString res;
   res.Assign("http://chart.apis.google.com/chart?cht=tx&chl="+data);
   ReqSize(res);
   ReqFill(res);
   ReqTitle(res);
   ReqFormulaColor(res);
   return res.Str();
  }
//+------------------------------------------------------------------+
//\class CGraphChart
//\brief Class for creating graphs
//Notice: Inherited from CLineXYChart
//+------------------------------------------------------------------+
class CGraphChart : public CLineXYChart
  {
protected:
   string            nodes[200];
   int               edges[400][2];
   int               nodes_total;
   int               edges_total;
   bool              graph_type;
   int               engine;
   void              ReqData(CString &res);
public:
   int               AddNode(string text);
   int               AddEdge(int Node_A,int Node_B);
   bool              DeleteNode(int node_ID);
   bool              DeleteEdge(int edge_ID);
   //+------------------------------------------------------------------+
   //Sets type of the graph
   // type - true-with arrows, otherwise without arrows
   //+------------------------------------------------------------------+
   void SetGraphType(bool type) {graph_type=type;}
   //+------------------------------------------------------------------+
   //Sets type of the graphical engine for the graph
   // Engine - type of engine
   //+------------------------------------------------------------------+   
   void SetEngine(DIAGRAM_GRAPH_ENGINES Engine) {engine=Engine;}
   string            CreateRequest();
   //+------------------------------------------------------------------+
   //Virtual function of identification of a diagram type
   //Returned value: Diagram type      
   //+------------------------------------------------------------------+         
   DIAGRAM_TYPE Type() {return DIAGRAM_TYPE_GRAPH;}
   void              CGraphChart();
   //+------------------------------------------------------------------+    
  };
//+------------------------------------------------------------------+
//Constructor
//+------------------------------------------------------------------+      
void CGraphChart::CGraphChart()
  {
   ArrayInitialize(edges,-1);
   nodes_total=edges_total=0;
   graph_type=true;
   for(int i=0;i<200;i++) nodes[i]="";
   engine=1;
  }
//+------------------------------------------------------------------+
//Adds a node
// text - text of the node
//Notice: To get additional information about an error, call the GetLastError() function
//Returned value: Node identifier in case of success, otherwise -1 
//+------------------------------------------------------------------+
int CGraphChart::AddNode(string text)
  {
   for(int i=0;i<200;i++)
             if(nodes[i]=="") {nodes[i]=text; nodes_total++; return i;}
   SetUserError(DIAGRAM_ERR_TOO_MANY_NODES);
   return -1;
  }
//+------------------------------------------------------------------+
//Adds an edge
// Node_A - node 1
// Node_B - node 2
//Notice: To get additional information about an error, call the GetLastError() function
//Returned value: Edge identifier in case of success, otherwise -1 
//+------------------------------------------------------------------+
int CGraphChart::AddEdge(int Node_A,int Node_B)
  {
   if(Node_A>=200 || Node_B>=200 || Node_A<0 || Node_B<0)
     {SetUserError(DIAGRAM_ERR_INCORRECT_NODE_ID); return -1;}
   if(nodes[Node_A]=="" || nodes[Node_B]=="")
     {SetUserError(DIAGRAM_ERR_INCORRECT_NODE_ID); return -1;}
   for(int i=0;i<400;i++)
      if(edges[i][0]==-1)
        {edges[i][0]=Node_A; edges[i][1]=Node_B; edges_total++; return i;}
   SetUserError(DIAGRAM_ERR_TOO_MANY_EDGES);
   return -1;
  }
//+------------------------------------------------------------------+
//Deletes a node and all the edges connected to it
// node_ID - Node identifier
//Notice: To get additional information about an error, call the GetLastError() function
//Returned value: true-if successful, otherwise-false
//+------------------------------------------------------------------+
bool CGraphChart::DeleteNode(int node_ID)
  {
   if(node_ID>=200 || node_ID<0)
     {SetUserError(DIAGRAM_ERR_INCORRECT_NODE_ID); return false;}
   if(nodes[node_ID]=="")
     {SetUserError(DIAGRAM_ERR_INCORRECT_NODE_ID); return false;}
   for(int i=0;i<400;i++)
      if(edges[i][0]==node_ID || edges[i][1]==node_ID) DeleteEdge(i);
   nodes[node_ID]="";
   nodes_total--;
   return true;
  }
//+------------------------------------------------------------------+
//Deletes an edge 
// edge_ID - Edge identifier
//Notice: To get additional information about an error, call the GetLastError() function
//Returned value: true-if successful, otherwise-false
//+------------------------------------------------------------------+
bool CGraphChart::DeleteEdge(int edge_ID)
  {
   if(edge_ID>=400 || edge_ID<0)
     {SetUserError(DIAGRAM_ERR_INCORRECT_EDGE_ID); return false;}
   if(edges[edge_ID][0]==-1)
     {SetUserError(DIAGRAM_ERR_INCORRECT_EDGE_ID); return false;}
   edges[edge_ID][0]=edges[edge_ID][1]=-1;
   edges_total--;
   return true;
  }
//+------------------------------------------------------------------+
//Creates request for the Google server
//Returned value: Request line
//+------------------------------------------------------------------+
string CGraphChart::CreateRequest()
  {
   CString res;
   res.Assign("http://chart.apis.google.com/chart?cht=gv:");
   switch(engine)
     {
      case 1: {res.Append("dot"); break;}
      case 2: {res.Append("neato"); break;}
      case 3: {res.Append("twopi"); break;}
      case 4: {res.Append("circo"); break;}
      case 5: {res.Append("fdp"); break;}
      default: res.Append("dot");
     }
   ReqSize(res);
   ReqData(res);
   return res.Str();
  }
//+------------------------------------------------------------------+
void CGraphChart::ReqData(CString &res)
  {
   if(nodes_total==0) return;
   res.Append("&chl=");
   if(graph_type) res.Append("graph{");
   else res.Append("digraph{");
   for(int i=0;i<400;i++)
      if(edges[i][0]>-1 && edges[i][1]>-1)
        {
         res.Append(nodes[edges[i][0]]);
         if(graph_type) res.Append("--");
         else res.Append("->");
         res.Append(nodes[edges[i][1]]+";");
        }
   res.Append("}");
  }
//+------------------------------------------------------------------+
//\class CQRCode
//\brief Class for creating QR codes
//Notice: Inherited from CLineXYChart
//+------------------------------------------------------------------+
class CQRCode:public CLineXYChart
  {
protected:
   string            data;
   int               encoding;
   int               err;
public:
   //+------------------------------------------------------------------+
   //Sets data a code will be created on the basis of
   // Data - Data
   //+------------------------------------------------------------------+
   void SetData(string Data) {data=Remove_spaces(Data);}
   //+------------------------------------------------------------------+
   //Set level of correction of the code error
   // err_lvl - level of correction
   //+------------------------------------------------------------------+   
   void SetErrCorrection(DIAGRAM_QRCODE_ERROR_CORRECTION err_lvl) {err=err_lvl;}
   //+------------------------------------------------------------------+
   //Sets type of encoding of the code
   // Encoding - Encoding type
   //+------------------------------------------------------------------+   
   void SetEncoding(DIAGRAM_QRCODE_ENCODING Encoding) {encoding=Encoding;}
   string            CreateRequest();
   //+------------------------------------------------------------------+
   //Virtual function of identification of a diagram type
   //Returned value: Diagram type      
   //+------------------------------------------------------------------+         
   DIAGRAM_TYPE Type() {return DIAGRAM_TYPE_QRCODE;}
   ///Constructor
   void CQRCode() {data=""; err=1; encoding=1;}
  };
//+------------------------------------------------------------------+
//Creates request for the Google server
//Returned value: Request line
//+------------------------------------------------------------------+
string CQRCode::CreateRequest()
  {
   CString res;
   res.Assign("http://chart.apis.google.com/chart?cht=qr&chl="+data);
   ReqSize(res);
   res.Append("&choe=");
   switch(encoding)
     {
      case 1: {res.Append("UTF-8"); break;}
      case 2: {res.Append("Shift_JIS"); break;}
      case 3: {res.Append("ISO-8859-1"); break;}
      default: res.Append("UTF-8");
     }
   res.Append("&chld=");
   switch(err)
     {
      case 1: {res.Append("L|0"); break;}
      case 2: {res.Append("M|0"); break;}
      case 3: {res.Append("H|0"); break;}
      default: res.Append("L|0");
     }
   return res.Str();
  }
//+------------------------------------------------------------------+
//\class CVennChart
//\brief Class for creating Venn diagrams
//Notice: Inherited from CLineXYChart
//+------------------------------------------------------------------+
class CVennChart : public CLineXYChart
  {
protected:
   double            A_size,B_size,C_size;
   double            intAB,intAC,intBC,intABC;
   color             A_color,B_color,C_color;
   string            A_label,B_label,C_label;
   void              ReqData(CString &res);
   void              ReqVennLegend(CString &res);
public:
   //+------------------------------------------------------------------+
   //Sets sizes of circles of the diagram
   // A - size of the first circle
   // B - size of the second circle
   // C - size of the third circle  
   //Notice: To get a diagram with two circles set the parameter C=0
   //+------------------------------------------------------------------+
   void SetCircleSizes(double A,double B,double C=0)
     {A_size=A; B_size=B; C_size=C;}
   //+------------------------------------------------------------------+
   //Sets colors of the diagram circles
   // A - color of the first circle
   // B - color of the second circle
   // C - color of the third circle  
   //+------------------------------------------------------------------+      
   void SetCircleColors(color A,color B,color C)
     {A_color=A; B_color=B; C_color=C;}
   //+------------------------------------------------------------------+
   //Sets sizes of intersections 
   // AB - intersection AB
   // AC - Intersection AC
   // BC - Intersection BC 
   // ABC - Intersection ABC    
   //Notice: Values must correspond to the sizes For example, if the size of the A circle is 50, then AB=25 mens that the intersection is a half of the circle  
   //+------------------------------------------------------------------+                   
   void SetIntersections(double AB,double AC,double BC,double ABC)
     {intAB=AB; intAC=AC; intBC=BC; intABC=ABC;}
   //+------------------------------------------------------------------+
   //Sets legends of the diagram circles
   // A - legend of the first circle
   // B - legend of the second circle
   // C - legend of the third circle  
   //+------------------------------------------------------------------+                  
   void SetVennLegend(string A,string B,string C)
     {A_label=A; B_label=B; C_label=C;}
   string            CreateRequest();
   //+------------------------------------------------------------------+
   //Virtual function of identification of a diagram type
   //Returned value: Diagram type      
   //+------------------------------------------------------------------+         
   DIAGRAM_TYPE Type() {return DIAGRAM_TYPE_VENN;}
   void              CVennChart();
  };
//+------------------------------------------------------------------+
//Constructor
//+------------------------------------------------------------------+
void CVennChart::CVennChart()
  {
   A_size=B_size=100; C_size=0;
   A_color=Red; B_color=Green; C_color=Blue;
   A_label="A"; B_label="B"; C_label="C";
   intAB=50; intAC=intBC=intABC=0;
  }
//+------------------------------------------------------------------+
//Creates request for the Google server
//Returned value: Request line
//+------------------------------------------------------------------+
string CVennChart::CreateRequest()
  {
   CString res;
   res.Assign("http://chart.apis.google.com/chart?cht=v");
   ReqSize(res);
   ReqTitle(res);
   ReqMargins(res);
   ReqFill(res);
   ReqVennLegend(res);
   res.Append("&chd=t:");
   if(C_size>0)
     {
      res.Append(DoubleToString(A_size)+","+DoubleToString(B_size)+","+
                 DoubleToString(C_size)+",");
      res.Append(DoubleToString(intAB)+","+DoubleToString(intAC)+","+
                 DoubleToString(intBC)+","+DoubleToString(intABC));
     }
   else
     {
      res.Append(DoubleToString(A_size)+","+DoubleToString(B_size)+",0,");
      res.Append(DoubleToString(intAB));
     }
   res.Append("&chco=");
   res.Append(GetColorStr(A_color)+","+GetColorStr(B_color)+","+GetColorStr(C_color));
   return res.Str();
  }
//+------------------------------------------------------------------+
void CVennChart::ReqVennLegend(CString &res)
  {
   CString legend_str;
   legend_str.Append(A_label+"|"+B_label+"|"+C_label);
   if(legend_position>0)
     {
      res.Append("&chdl=");
      legend_str.TrimRight("|");
      res.Append(GetPointer(legend_str));
      res.Append("&chdlp=");
      switch(legend_position)
        {
         case 1: {res.Append("r"); break;}
         case 2: {res.Append("b"); break;}
         case 3: {res.Append("bv"); break;}
         case 4: {res.Append("t"); break;}
         case 5: {res.Append("tv"); break;}
         case 6: {res.Append("l"); break;}
         default: res.Append("r");
        }
     }
  }
//+------------------------------------------------------------------+
//\class CScatterChart
//\brief Class for creating a scatter chart
//Notice: Inherited from CLineXYChart
//+------------------------------------------------------------------+
class CScatterChart : public CLineXYChart
  {
protected:
   void              ReqData(CString &res);
   void              ReqLineColor(CString &res);
public:
   int               AddLineScatter(double &Y_data[],double &X_data[],double &Z_data[],color color_,int digits,string legend);
   int               AddLineScatter(CArrayDouble *Y_data,CArrayDouble *X_data,CArrayDouble *Z_data,color color_,int digits,string legend);
   string            CreateRequest();
   //+------------------------------------------------------------------+
   //Virtual function of identification of a diagram type
   //Returned value: Diagram type      
   //+------------------------------------------------------------------+         
   DIAGRAM_TYPE Type() {return DIAGRAM_TYPE_SCATTER;}
  };
//+------------------------------------------------------------------+
//Creates request for the Google server
//Returned value: Request line
//+------------------------------------------------------------------+
string CScatterChart::CreateRequest()
  {
   CString res;
   GetChartMaxMin();
   res.Assign("http://chart.apis.google.com/chart?cht=s");
   ReqSize(res);
   ReqGrid(res);
   ReqAxisLabels(res);
   ReqFill(res);
   ReqTitle(res);
   ReqMargins(res);
   ReqData(res);
   ReqLineColor(res);
   ReqLineMarkers(res);
   ReqLineLegend(res);
   ReqLineScaling(res);
   return res.Str();
  }
//+------------------------------------------------------------------+
//Adds scatters to a chart
// Y_data -Data for the Y axis 
// X_data -Data for the X axis
// Z_data -Size of scatters
// color_ -Color
// digits -Number of digits for truncation 
// legend -Legend
//Notice: To get additional information about an error, call the GetLastError() function
//Notice: Maximum size of a scatter is 100
//Returned value: Array identifier in case of success, otherwise -1
//+------------------------------------------------------------------+
int CScatterChart::AddLineScatter(double &Y_data[],double &X_data[],double &Z_data[],color color_=Red,int digits=8,string legend="")
  {
   int id=AddLine(Y_data,X_data,color_,digits,legend);
   if(id>-1)
     {
      int pos=GetLinePos(id);
      if(ArrayRange(Z_data,0)==Lines[pos].data_amount)
         Lines[pos].Z_data.AssignArray(Z_data);
      else
        {
         SetUserError(DIAGRAM_ERR_INCORRECT_DATA_AMOUNT);
         DeleteLine(id);
         return-1;
        }
     }
   return id;
  }
//+------------------------------------------------------------------+
//Adds scatters to a chart
// Y_data -Data for the Y axis 
// X_data -Data for the X axis
// Z_data -Size of scatters
// color_ -Color
// digits -Number of digits for truncation 
// legend -Legend
//Notice: To get additional information about an error, call the GetLastError() function
//Notice: Maximum size of a scatter is 100
//Returned value: Array identifier in case of success, otherwise -1
//+------------------------------------------------------------------+
int CScatterChart::AddLineScatter(CArrayDouble *Y_data,CArrayDouble *X_data,CArrayDouble *Z_data,color color_=Red,int digits=8,string legend="")
  {
   int id=AddLine(Y_data,X_data,color_,digits,legend);
   if(id>-1)
     {
      int pos=GetLinePos(id);
      if(Z_data.Total()==Lines[pos].data_amount)
         Lines[pos].Z_data.AssignArray(Z_data);
      else
        {
         SetUserError(DIAGRAM_ERR_INCORRECT_DATA_AMOUNT);
         DeleteLine(id);
         return-1;
        }
     }
   return id;
  }
//+------------------------------------------------------------------+
void CScatterChart::ReqLineColor(CString &res)
  {
   if(total_lines>0)
     {
      CString color_str;
      for(int i=0;i<100;i++)
         if(Lines[i].ID>-1)
            color_str.Append(GetColorStr(Lines[i].color_)+"|");
      res.Append("&chco=");
      color_str.TrimRight("|");
      res.Append(GetPointer(color_str));
     }
  }
//+------------------------------------------------------------------+
void CScatterChart::ReqData(CString &res)
  {
   if(total_lines>0)
     {
      CString Xdata,Ydata,Zdata;
      CArrayDouble *data_pointers[100][3];
      int max_data=0;
      int cnt=0;
      res.Append("&chd=t:");
      for(int i=0;i<100;i++)
         if(Lines[i].ID>-1)
           {
            data_pointers[cnt][0]=GetPointer(Lines[i].X_data);
            data_pointers[cnt][1]=GetPointer(Lines[i].Y_data);
            data_pointers[cnt][2]=GetPointer(Lines[i].Z_data);
            if(Lines[i].data_amount>max_data) max_data=Lines[i].data_amount;
            cnt++;
           }
      for(int j=0;j<max_data;j++)
         for(int i=0;i<cnt;i++)
           {
            int pos=GetLinePos(i);
            if(data_pointers[i][0].Total()>j)
               Xdata.Append(DoubleToString(data_pointers[i][0].At(j),Lines[pos].digits)+",");
            else Xdata.Append("0,");
            if(data_pointers[i][1].Total()>j)
               Ydata.Append(DoubleToString(data_pointers[i][1].At(j),Lines[pos].digits)+",");
            else Ydata.Append("0,");
            if(data_pointers[i][2].Total()>j)
               Zdata.Append(DoubleToString(data_pointers[i][2].At(j),Lines[pos].digits)+",");
            else Zdata.Append("0,");
           }
      Xdata.TrimRight(",");
      Ydata.TrimRight(",");
      Zdata.TrimRight(",");
      res.Append(GetPointer(Xdata));
      res.Append("|");
      res.Append(GetPointer(Ydata));
      res.Append("|");
      res.Append(GetPointer(Zdata));
     }
  }
//+------------------------------------------------------------------+
//\class CBarChart
//\brief Class for creating histograms
//Notice: Inherited from CLineXYChart
//+------------------------------------------------------------------+
class CBarChart : public CLineXYChart
  {
protected:
   bool              grouped;
   double            space_bars,space_groups;
   void              ReqBarData(CString &res);
   void              ReqLineScaling(CString &res);
   void              ReqAxisLabels(CString &res);
public:
   //+------------------------------------------------------------------+
   //Sets grouping of the histogram
   // Grouped - true-group, false-ungroup
   // Space_between_bars -Spaces between bars
   // Space_between_groups -Spaces between groups
   //+------------------------------------------------------------------+
   void SetGrouped(bool Grouped,double Space_between_bars=4.0/23.0,double Space_between_groups=8.0/23.0)
     {grouped=true; space_bars=Space_between_bars; space_groups=Space_between_groups;}
   string            CreateRequest();
   //+------------------------------------------------------------------+
   //Virtual function of identification of a diagram type
   //Returned value: Diagram type      
   //+------------------------------------------------------------------+         
   DIAGRAM_TYPE Type() {return DIAGRAM_TYPE_BAR;}
  };
//+------------------------------------------------------------------+
//Creates request for the Google server
//Returned value: Request line
//+------------------------------------------------------------------+
string CBarChart::CreateRequest()
  {
   CString res;
   GetChartMaxMin();
   res.Assign("http://chart.apis.google.com/chart?cht=");
   if(grouped)
     {
      res.Append("bvg&chbh=r,");
      res.Append(DoubleToString(space_bars)+",");
      res.Append(DoubleToString(space_groups));
     }
   else res.Append("bvo&chbh=a");
   ReqSize(res);
   ReqGrid(res);
   ReqAxisLabels(res);
   ReqFill(res);
   ReqTitle(res);
   ReqMargins(res);
   ReqBarData(res);
   ReqLineColor(res);
   ReqLineFilling(res);
   ReqLineLabels(res);
   ReqLineLegend(res);
   ReqLineScaling(res);
   return res.Str();
  }
//+------------------------------------------------------------------+
void CBarChart::ReqBarData(CString &res)
  {
   if(total_lines>0)
     {
      CString data;
      res.Append("&chd=t:");
      for(int i=0;i<100;i++)
         if(Lines[i].ID>-1)
           {
            for(int j=0;j<Lines[i].Y_data.Total();j++)
               data.Append(DoubleToString(Lines[i].Y_data.At(j),Lines[i].digits)+",");
            data.TrimRight(",");
            data.Append("|");
           }
      data.TrimRight("|");
      res.Append(GetPointer(data));
     }
  }
//+------------------------------------------------------------------+
void CBarChart::ReqLineScaling(CString &res)
  {
   if(total_lines>0)
     {
      res.Append("&chds=");
      if(ymin_val<0) res.Append(DoubleToString(ymin_val)+",");
      else res.Append("0,");
      res.Append(DoubleToString(ymax_val));
     }
  }
//+------------------------------------------------------------------+
void CBarChart::ReqAxisLabels(CString &res)
  {
   CString axis_range,axis_types,styles;
   int axe_cnt=0;
   if(bottom_axe_line>-2)
     {
      bottom_axe_index=axe_cnt;
      styles.Append(IntegerToString(bottom_axe_index));
      styles.Append("N*f"+IntegerToString(bottom_axe_digits)+"zsy*,");
      styles.Append(GetColorStr(bottom_axe_color)+",");
      styles.Append(IntegerToString(bottom_axe_size)+",");
      styles.Append("0,lt|");
      //**************************        
      axis_types.Append("x,");
      axis_range.Append(IntegerToString(axe_cnt++)+",");
      if(bottom_axe_label!="") {axis_types.Append("x,"); axe_cnt++;}
      axis_range.Append(DoubleToString(xmin_val)+",");
      axis_range.Append(DoubleToString(xmax_val));
      if(bottom_axe_step!=0)
         axis_range.Append(","+DoubleToString(bottom_axe_step)+"|");
      else axis_range.Append("|");
     }
//---
   if(top_axe_line>-2)
     {
      top_axe_index=axe_cnt;
      styles.Append(IntegerToString(top_axe_index));
      styles.Append("N*f"+IntegerToString(top_axe_digits)+"zsy*,");
      styles.Append(GetColorStr(top_axe_color)+",");
      styles.Append(IntegerToString(top_axe_size)+",");
      styles.Append("0,lt|");
      //**************************        
      axis_types.Append("t,");
      axis_range.Append(IntegerToString(axe_cnt++)+",");
      if(top_axe_label!="") {axis_types.Append("t,"); axe_cnt++;}
      axis_range.Append(DoubleToString(xmin_val)+",");
      axis_range.Append(DoubleToString(xmax_val));
      if(top_axe_step!=0)
         axis_range.Append(","+DoubleToString(top_axe_step)+"|");
      else axis_range.Append("|");
     }
//---
   if(left_axe_line>-2)
     {
      left_axe_index=axe_cnt;
      styles.Append(IntegerToString(left_axe_index));
      styles.Append("N*f"+IntegerToString(left_axe_digits)+"zsy*,");
      styles.Append(GetColorStr(left_axe_color)+",");
      styles.Append(IntegerToString(left_axe_size)+",");
      styles.Append("0,lt|");
      //**************************        
      axis_types.Append("y,");
      axis_range.Append(IntegerToString(axe_cnt++)+",");
      if(left_axe_label!="") {axis_types.Append("y,"); axe_cnt++;}
      if(ymin_val<0) axis_range.Append(DoubleToString(ymin_val)+",");
      else axis_range.Append("0,");
      axis_range.Append(DoubleToString(ymax_val));
      if(left_axe_step!=0)
         axis_range.Append(","+DoubleToString(left_axe_step)+"|");
      else axis_range.Append("|");
     }
//---
   if(right_axe_line>-2)
     {
      right_axe_index=axe_cnt;
      styles.Append(IntegerToString(right_axe_index));
      styles.Append("N*f"+IntegerToString(right_axe_digits)+"zsy*,");
      styles.Append(GetColorStr(right_axe_color)+",");
      styles.Append(IntegerToString(right_axe_size)+",");
      styles.Append("0,lt|");
      //**************************      
      axis_types.Append("r,");
      axis_range.Append(IntegerToString(axe_cnt++)+",");
      if(right_axe_label!="") {axis_types.Append("r,"); axe_cnt++;}
      if(ymin_val<0) axis_range.Append(DoubleToString(ymin_val)+",");
      else axis_range.Append("0,");
      axis_range.Append(DoubleToString(ymax_val));
      if(right_axe_step!=0)
         axis_range.Append(","+DoubleToString(right_axe_step)+"|");
      else axis_range.Append("|");
     }
//---
   res.Append("&chxt=");
   axis_types.TrimRight(",");
   res.Append(GetPointer(axis_types));
   res.Append("&chxr=");
   axis_range.TrimRight("|");
   res.Append(GetPointer(axis_range));
//---
   CString labels,positions;
   if(bottom_axe_index>-1 && bottom_axe_label!="")
     {
      styles.Append(IntegerToString(bottom_axe_index+1)+",");
      styles.Append(GetColorStr(axe_label_color)+",");
      styles.Append(IntegerToString(axe_label_size)+",");
      styles.Append("0,lt|");
      labels.Append(IntegerToString(bottom_axe_index+1)+":|"+bottom_axe_label+"|");
      positions.Append(IntegerToString(bottom_axe_index+1)+","+IntegerToString(50)+"|");
     }
   if(top_axe_index>-1 && top_axe_label!="")
     {
      styles.Append(IntegerToString(top_axe_index+1)+",");
      styles.Append(GetColorStr(axe_label_color)+",");
      styles.Append(IntegerToString(axe_label_size)+",");
      styles.Append("0,lt|");
      labels.Append(IntegerToString(top_axe_index+1)+":|"+top_axe_label+"|");
      positions.Append(IntegerToString(top_axe_index+1)+","+IntegerToString(50)+"|");
     }
   if(left_axe_index>-1 && left_axe_label!="")
     {
      styles.Append(IntegerToString(left_axe_index+1)+",");
      styles.Append(GetColorStr(axe_label_color)+",");
      styles.Append(IntegerToString(axe_label_size)+",");
      styles.Append("0,lt|");
      labels.Append(IntegerToString(left_axe_index+1)+":|"+left_axe_label+"|");
      positions.Append(IntegerToString(left_axe_index+1)+","+IntegerToString(50)+"|");
     }
   if(right_axe_index>-1 && right_axe_label!="")
     {
      styles.Append(IntegerToString(right_axe_index+1)+",");
      styles.Append(GetColorStr(axe_label_color)+",");
      styles.Append(IntegerToString(axe_label_size)+",");
      styles.Append("0,lt|");
      labels.Append(IntegerToString(right_axe_index+1)+":|"+right_axe_label+"|");
      positions.Append(IntegerToString(right_axe_index+1)+","+IntegerToString(50)+"|");
     }
//---
   labels.TrimRight("|");
   positions.TrimRight("|");
   res.Append("&chxl=");
   res.Append(GetPointer(labels));
   res.Append("&chxp=");
   res.Append(GetPointer(positions));
   res.Append("&chxs=");
   res.Append(GetPointer(styles));
  }
//+------------------------------------------------------------------+
//\class CPieChart
//\brief Class for creating a circle diagram
//Notice: Inherited from CLineXYChart
//+------------------------------------------------------------------+
class CPieChart : public CLineXYChart
  {
protected:
   bool              ptype;
   bool              labels;
   double            rotation;
   void              ReqPieLabels(CString &res);
   void              ReqPieData(CString &res);
public:
   int               AddPieSlice(double Data,int dimensional,color color_,int digits,string label);
   //+------------------------------------------------------------------+
   //Sets type of the circle diagram
   // type - true-2D, false-3D
   //Notice: Doesn't affect ring diagrams, they are two-dimensional 
   //+------------------------------------------------------------------+
   void SetPieType(bool type) {ptype=type;}
   //+------------------------------------------------------------------+
   //Shows/hides extension labels of segments
   // set - true-show, false-hide
   //Notice: Labels are not fit to the size of the chart automatically That's why you should increase the diagram size if the labels fall out of the chart borders 
   //+------------------------------------------------------------------+   
   void SetPieLabels(bool set) {labels=set;}
   //+------------------------------------------------------------------+
   //Sets a rotation angle of the chart
   // val - rotation angle
   //+------------------------------------------------------------------+   
   void SetPieRotation(double val) {rotation=val;}
   string            CreateRequest();
   //+------------------------------------------------------------------+
   //Virtual function of identification of a diagram type
   //Returned value: Diagram type      
   //+------------------------------------------------------------------+         
   DIAGRAM_TYPE Type() {return DIAGRAM_TYPE_PIE;}
   //+------------------------------------------------------------------+ 
  };
//+------------------------------------------------------------------+
//Adds a segment to the diagram
// Data - Size of the segment
// dimensional - A dimension the segment will be located in
// color_ - Color
// digits - Number of digits for truncation
// legend - Legend
//Notice: To get additional information about an error, call the GetLastError() function
//Returned value: Segment identifier in case of success, otherwise -1
//+------------------------------------------------------------------+
int CPieChart::AddPieSlice(double Data,int dimensional=0,color color_=Red,int digits=8,string legend="")
  {
   double Y_data[1];
   Y_data[0]=Data;
   int id=AddLine(Y_data,color_,digits,legend);
   if(id>-1)
     {
      int pos=GetLinePos(id);
      Lines[pos].dimensional=dimensional;
     }
   return id;
  }
//+------------------------------------------------------------------+
//Creates request for the Google server
//Returned value: Request line
//+------------------------------------------------------------------+
string CPieChart::CreateRequest()
  {
   CString res;
   GetChartMaxMin();
   res.Assign("http://chart.apis.google.com/chart?cht=");
   bool flag=true;
   for(int i=0;i<100;i++)
             if(Lines[i].ID>-1 && Lines[i].dimensional>0) {flag=false; break;}
   if(flag)
     {
      if(ptype) res.Append("p");
      else res.Append("p3");
     }
   else res.Append("pc");
   res.Append("&chp="+DoubleToString(rotation));
   ReqSize(res);
   ReqFill(res);
   ReqTitle(res);
   ReqMargins(res);
   ReqPieData(res);
   ReqLineColor(res);
   ReqLineLegend(res);
   ReqPieLabels(res);
   return res.Str();
  }
//+------------------------------------------------------------------+
void CPieChart::ReqPieLabels(CString &res)
  {
   if(total_lines>0 && labels)
     {
      CString data;
      res.Append("&chl=");
      for(int i=0;i<100;i++)
         if(Lines[i].ID>-1)
            data.Append(Lines[i].legend+"|");
      data.TrimRight("|");
      res.Append(GetPointer(data));
     }
  }
//+------------------------------------------------------------------+
void CPieChart::ReqPieData(CString &res)
  {
   if(total_lines>0)
     {
      CString data[100],dat;
      res.Append("&chd=t:");
      for(int i=0;i<100;i++)
         if(Lines[i].ID>-1)
            data[Lines[i].dimensional].Append(DoubleToString(Lines[i].Y_data.At(0),Lines[i].digits)+",");
      for(int i=0;i<100;i++)
        {
         data[i].TrimRight(",");
         data[i].Append("|");
         dat.Append(GetPointer(data[i]));
        }
      dat.TrimRight("|");
      res.Append(GetPointer(dat));
     }
  }
//+------------------------------------------------------------------+
//\class CRadarChart
//\brief Class for creating a radar chart
//Notice: Inherited from CLineXYChart
//+------------------------------------------------------------------+
class CRadarChart : public CLineXYChart
  {

public:
   string            CreateRequest();
   //+------------------------------------------------------------------+
   //Virtual function of identification of a diagram type
   //Returned value: Diagram type      
   //+------------------------------------------------------------------+         
   DIAGRAM_TYPE Type() {return DIAGRAM_TYPE_RADAR;}
   //+------------------------------------------------------------------+  
  };
//+------------------------------------------------------------------+
//Creates request for the Google server
//Returned value: Request line
//+------------------------------------------------------------------+
string CRadarChart::CreateRequest()
  {
   GetChartMaxMin();
   CString res;
   res.Assign("http://chart.apis.google.com/chart?cht=rs");
   ReqSize(res);
   ReqGrid(res);
   ReqAxisLabels(res);
   ReqFill(res);
   ReqTitle(res);
   ReqMargins(res);
   ReqLineData(res);
   ReqLineColor(res);
   ReqLineStyle(res);
   ReqLineMarkers(res);
   ReqLineFilling(res);
   ReqLineLabels(res);
   ReqLineLegend(res);
   ReqLineScaling(res);
   return res.Str();
  }
//+------------------------------------------------------------------+
//\class CCandleChart
//\brief Class for creating candlestick charts
//Notice: Inherited from CLineXYChart
//+------------------------------------------------------------------+
class CCandleChart : public CLineXYChart
  {
protected:
   int               candle_size;
   void              ReqData(CString &res);
   void              ReqParams(CString &res);
   void              GetChartMaxMin();
   void              Transform(CArrayDouble &data);
public:
   int               AddCandles(double &Open[],double &Close[],double &High[],double &Low[],
                                color ColorBear=Black);
   int               AddCandles(CArrayDouble &Open,CArrayDouble &Close,CArrayDouble &High,CArrayDouble &Low,
                                color ColorBear=Black);
   //+------------------------------------------------------------------+
   //Sets size of a candlestick in pixels
   //size - size of a cnadlestick
   //Notice: Set size=0 for automatic adjustment of size
   //+------------------------------------------------------------------+            
   void SetCandleSize(int size) {candle_size=size;}
   string            CreateRequest();
   //+------------------------------------------------------------------+
   //Virtual function of identification of a diagram type
   //Returned value: Diagram type      
   //+------------------------------------------------------------------+         
   DIAGRAM_TYPE Type() {return DIAGRAM_TYPE_CANDLESTRICK;}
   //+------------------------------------------------------------------+   
  };
//+------------------------------------------------------------------+
//Creates request for the Google server
//Returned value: Request line
//+------------------------------------------------------------------+
string CCandleChart::CreateRequest()
  {
   GetChartMaxMin();
   CString res;
   res.Assign("http://chart.apis.google.com/chart?cht=lc");
   ReqSize(res);
   ReqGrid(res);
   ReqAxisLabels(res);
   ReqFill(res);
   ReqTitle(res);
   ReqMargins(res);
   ReqData(res);
   ReqParams(res);
   return res.Str();
  }
//+------------------------------------------------------------------+
void CCandleChart::ReqParams(CString &res)
  {
   if(total_lines>0)
     {
      res.Append("&chm=F,"+GetColorStr(Lines[0].color_)+",0,-1,");
      if(candle_size>0) res.Append(IntegerToString(candle_size));
      else
        {
         int total_size=margin_left+margin_right+X;
         res.Append(IntegerToString(total_size/(Lines[0].data_amount+4)));
        }
     }
  }
//+------------------------------------------------------------------+
//Adds candlesticks to a chart
// Open - Open prices
// Close - Close prices
// High - Highest price
// Low - Lowest price
// ColorBear - Color of bullish candlesticks
//Notice: To get additional information about an error, call the GetLastError() function
//Notice: Color of bullish candlesticks is automatically set to white
//Returned value: Array identifier in case of success, otherwise -1
//+------------------------------------------------------------------+
int CCandleChart::AddCandles(double &Open[],double &Close[],double &High[],double &Low[],
                             color ColorBear=Black)
  {
   int o_range=ArrayRange(Open, 0);
   int c_range=ArrayRange(Close, 0);
   int h_range=ArrayRange(High, 0);
   int l_range=ArrayRange(Low, 0);
   if(o_range!=c_range || o_range!=h_range || o_range!=l_range)
     {SetUserError(DIAGRAM_ERR_INCORRECT_DATA_AMOUNT); return -1;}
   Lines[0].X_data.AssignArray(Open);
   Lines[0].Y_data.AssignArray(Close);
   Lines[0].Z_data.AssignArray(Low);
   Lines[0].H_data.AssignArray(High);
   Lines[0].ID=0;
   Lines[0].color_=ColorBear;
   Lines[0].digits=4;
   Lines[0].legend="";
   Lines[0].data_amount=o_range;
   total_lines=1;
   return 0;
  }
//+------------------------------------------------------------------+
//Adds candlesticks to a chart
// Open - Open prices
// Close - Close prices
// High - Highest price
// Low - Lowest price
// ColorBear - Color of bullish candlesticks
//Notice: To get additional information about an error, call the GetLastError() function
//Notice: Color of bullish candlesticks is automatically set to white
//Returned value: Array identifier in case of success, otherwise -1
//+------------------------------------------------------------------+
int CCandleChart::AddCandles(CArrayDouble &Open,CArrayDouble &Close,CArrayDouble &High,CArrayDouble &Low,
                             color ColorBear=Black)
  {
   int o_range=Open.Total();
   int c_range=Close.Total();
   int h_range=High.Total();
   int l_range=Low.Total();
   if(o_range!=c_range || o_range!=h_range || o_range!=l_range)
     {SetUserError(DIAGRAM_ERR_INCORRECT_DATA_AMOUNT); return -1;}
   Lines[0].X_data.AssignArray(GetPointer(Open));
   Lines[0].Y_data.AssignArray(GetPointer(Close));
   Lines[0].Z_data.AssignArray(GetPointer(Low));
   Lines[0].H_data.AssignArray(GetPointer(High));
   Lines[0].ID=0;
   Lines[0].color_=ColorBear;
   Lines[0].digits=8;
   Lines[0].legend="";
   Lines[0].data_amount=o_range;
   total_lines=1;
   return 0;
  }
//+------------------------------------------------------------------+   
void CCandleChart::Transform(CArrayDouble &data)
  {
   if(data.Total()==0) return;
   for(int i=0;i<data.Total();i++)
      data.Update(i,((data.At(i)-ymin_val)/(ymax_val-ymin_val))*100.0);
  }
//+------------------------------------------------------------------+
void CCandleChart::ReqData(CString &res)
  {
   if(total_lines>0)
     {
      //****************
      CArrayDouble tmpO,tmpC,tmpH,tmpL;
      tmpO.AssignArray(GetPointer(Lines[0].X_data));
      tmpC.AssignArray(GetPointer(Lines[0].Y_data));
      tmpH.AssignArray(GetPointer(Lines[0].Z_data));
      tmpL.AssignArray(GetPointer(Lines[0].H_data));
      Transform(tmpO);
      Transform(tmpC);
      Transform(tmpH);
      Transform(tmpL);
      //****************
      CString data_str;
      CString odata_str,cdata_str,hdata_str,ldata_str;
      odata_str.Assign("-1,");
      cdata_str.Assign("-1,");
      hdata_str.Assign("-1,");
      ldata_str.Assign("-1,");
      for(int j=0;j<Lines[0].data_amount;j++)
        {
         odata_str.Append(DoubleToString(tmpO.At(j),3)+",");
         cdata_str.Append(DoubleToString(tmpC.At(j),3)+",");
         ldata_str.Append(DoubleToString(tmpL.At(j),3)+",");
         hdata_str.Append(DoubleToString(tmpH.At(j),3)+",");
        }
      odata_str.Append("-1");
      cdata_str.Append("-1");
      hdata_str.Append("-1");
      ldata_str.Append("-1");
      data_str.Append(GetPointer(ldata_str));
      data_str.Append("|");
      data_str.Append(GetPointer(odata_str));
      data_str.Append("|");
      data_str.Append(GetPointer(cdata_str));
      data_str.Append("|");
      data_str.Append(GetPointer(hdata_str));
      data_str.Append("|");
      res.Append("&chd=t0:");
      data_str.TrimRight("|");
      res.Append(GetPointer(data_str));
     }

  }
//+------------------------------------------------------------------+
void CCandleChart::GetChartMaxMin()
  {
   if(total_lines>0)
     {
      Lines[0].x_max=Lines[0].data_amount+2; Lines[0].x_min=1;
      Lines[0].y_max=DBL_MIN; Lines[0].y_min=DBL_MAX;
      for(int j=0;j<Lines[0].Z_data.Total();j++)
        {
         if(Lines[0].Z_data.At(j)<Lines[0].y_min) Lines[0].y_min=Lines[0].Z_data.At(j);
        }
      for(int j=0;j<Lines[0].H_data.Total();j++)
        {
         if(Lines[0].H_data.At(j)>Lines[0].y_max) Lines[0].y_max=Lines[0].H_data.At(j);
        }
     }
   Lines[0].y_max=Lines[0].y_max+(Lines[0].y_max-Lines[0].y_min)*Lines[0].stretch/100.0;
   Lines[0].y_min=Lines[0].y_min-(Lines[0].y_max-Lines[0].y_min)*Lines[0].stretch/100.0;
   xmax_val=Lines[0].x_max; xmin_val=Lines[0].x_min;
   ymax_val=Lines[0].y_max; ymin_val=Lines[0].y_min;
  }
//+------------------------------------------------------------------+
//\class CMapChart
//\brief Class for creating maps
//Notice: Inherited from CLineXYChart
//+------------------------------------------------------------------+
class CMapChart : public CLineXYChart
  {
protected:
   int               zoom;
   string            data;
   color             color1,color2,color3;
   void              ReqColor(CString &res);
public:
   //+------------------------------------------------------------------+
   //Setting of area (continent) of the map
   // zoom_area - area
   //+------------------------------------------------------------------+
   void SetZoomArea(DIAGRAM_MAP_AREA zoom_area) {zoom=zoom_area;}
   //+------------------------------------------------------------------+
   //Setting of codes that are necessary for displaying countries and states of USA in the ISO 3166-1-alpha-2 format
   // country_codes - codes without spaces
   //Notcie: You can find more detailed information about codes of countries <a href="http://www.iso.org/iso/english_country_names_and_code_elements">here</a>
   //+------------------------------------------------------------------+   
   void SetCountries(string country_codes) {data=country_codes;}
   //+------------------------------------------------------------------+
   //Setting a color for filling selected countries
   // default_ - Color of filling of unselected countries
   // start_gradient - Starting color in the filling gradient
   // end_gradient - End color in the filling gradient
   //+------------------------------------------------------------------+
   void SetColors(color default_=White,color start_gradient=Blue,color end_gradient=Blue)
     {color1=default_; color2=start_gradient; color3=end_gradient;}
   string            CreateRequest();
   //+------------------------------------------------------------------+
   //Virtual function of identification of a diagram type
   //Returned value: Diagram type      
   //+------------------------------------------------------------------+         
   DIAGRAM_TYPE Type() {return DIAGRAM_TYPE_MAP;}
   void              CMapChart();
  };
//+------------------------------------------------------------------+
void CMapChart::CMapChart()
  {
   color1=White;
   color2=Blue;
   color3=Red;
   data="";
   zoom=0;
  }
//+------------------------------------------------------------------+
void CMapChart::ReqColor(CString &res)
  {
   res.Append("&chco="+GetColorStr(color1)+","+GetColorStr(color2)+","+
              GetColorStr(color3));
  }
//+------------------------------------------------------------------+
//Creates request for the Google server
//Returned value: Request line
//+------------------------------------------------------------------+
string CMapChart::CreateRequest()
  {
   CString res;
   res.Assign("http://chart.apis.google.com/chart?cht=t&chld="+data);
   res.Append("&chtm=");
   switch(zoom)
     {
      case DIAGRAM_MAP_AREA_WORLD: {res.Append("world"); break;}
      case DIAGRAM_MAP_AREA_AFRICA: {res.Append("africa"); break;}
      case DIAGRAM_MAP_AREA_ASIA: {res.Append("asia"); break;}
      case DIAGRAM_MAP_AREA_EUROPE: {res.Append("europe"); break;}
      case DIAGRAM_MAP_AREA_MIDDLE_EAST: {res.Append("middle_east"); break;}
      case DIAGRAM_MAP_AREA_SOUTH_AMERICA: {res.Append("south_america"); break;}
      case DIAGRAM_MAP_AREA_USA: {res.Append("usa"); break;}
      default: res.Append("world");
     }
   res.Append("&chd=t:");
   int w=100/(StringLen(data)/2);
   int w0=w;
   for(int i=0;i<StringLen(data)/2;i++)
     {res.Append(IntegerToString(w0)+",");w0+=w;}
   res.TrimRight(",");
   if(X>440) X=440;//check the chart size
   if(Y>220) Y=220;//no to be too large
   ReqSize(res);
   ReqFill(res);
   ReqTitle(res);
   ReqColor(res);
   ReqMargins(res);
   return res.Str();
  }
//+------------------------------------------------------------------+
