//+------------------------------------------------------------------+
//|                                                    incMatrix.mq4 |
//|                                                                * |
//|                                                                * |
//+------------------------------------------------------------------+
#property copyright "Integer"
#property link      "https://login.mql5.com/ru/users/Integer"
//+------------------------------------------------------------------+
//|   Class with a set of functions to solve matrices                |
//+------------------------------------------------------------------+
/*

   Matrix is created in one dimensional array, in sequence: elements of first row, second and so on. 
   The last two elements represent the size of matrix: number of columns and rows.
   
   Example:
   
   double m[]={1,2,3,
               4,5,6,
               2,3}; // Matrix of two rows and three columns.
*/

class CIntMatrix
  {

public:
   //+------------------------------------------------------------------+
   //|   Sets the size of matrix aA                                     |
   //|   aRows - number of rows, aCols - number of columns              |
   //+------------------------------------------------------------------+
   void SetSize(double  &aA[],int aRows,int aCols)
     {
      int               m_sz=aRows*aCols+2;
      ArrayResize(aA,m_sz);
      ArrayInitialize(aA,0);
      aA[m_sz-2]=aRows;
      aA[m_sz-1]=aCols;
     }
   //+------------------------------------------------------------------+
   //|   Sets the value (Value) of matrix (aA) element,                 |
   //|   located in row aRow, column aCol                               |
   //+------------------------------------------------------------------+
   void SetValue(double  &aA[],int aRow,int aCol,double aValue)
     {
      aA[aRow*GetCols(aA)+aCol]=aValue;
     }
   //+------------------------------------------------------------------+
   //|   Returns the number of matrix aA elements.                      |
   //|   By reference returns:                                          |
   //|   aRows - number of rows, aCols - number of columns              |
   //+------------------------------------------------------------------+
   int GetSize(double  &aA[],int  &aRows,int  &aCols)
     {
      int               m_sz=ArraySize(aA);
      m_sz-=2;
      aRows=(int)aA[m_sz];
      aCols=(int)aA[m_sz+1];
      return(m_sz);
     }
   //+------------------------------------------------------------------+
   //|   Returns the number of rows in matrix aA                        |
   //+------------------------------------------------------------------+
   int GetRows(double  &aA[])
     {
      return((int)aA[ArraySize(aA)-2]);
     }
   //+------------------------------------------------------------------+
   //|   Returns the number of columns in matrix aA                     |
   //+------------------------------------------------------------------+
   int GetCols(double  &aA[])
     {
      return((int)aA[ArraySize(aA)-1]);
     }
   //+------------------------------------------------------------------+
   //|   Gets the value of element in matrix aA located                 |
   //|   in row aRow, column aCol                                       |
   //+------------------------------------------------------------------+
   double GetValue(double  &aA[],int aRow,int aCol)
     {
      return(aA[aRow*GetCols(aA)+aCol]);
     }
   //+------------------------------------------------------------------+
   //|   Copies matrix from array aFrom to array aTo                    |
   //+------------------------------------------------------------------+
   void Copy(double  &aFrom[],double  &aTo[])
     {
      ArrayResize(aTo,ArraySize(aFrom));
      ArrayCopy(aTo,aFrom,0,0,ArraySize(aFrom));
     }
   //+------------------------------------------------------------------+
   //|   Checks if two matrices match by size for addition              | 
   //|   (fully equivalent by height and width)                         |
   //+------------------------------------------------------------------+
   bool CheckForAdd(double  &aA[],double  &aB[])
     {
      int               m_ARows,m_ACols,m_BRows,m_BCols;
      GetSize(aA,m_ARows,m_ACols);
      GetSize(aB,m_BRows,m_BCols);
      return(m_ARows==m_BRows && m_ACols==m_BCols);
     }
   //+------------------------------------------------------------------+
   //|   Checks if two matrices match by size for multiplication        | 
   //|   (fully equivalent by height and width)                         |
   //+------------------------------------------------------------------+
   bool CheckForMult(double  &aA[],double  &aB[])
     {
      return(GetCols(aA)==GetRows(aB));
     }
   //+------------------------------------------------------------------+
   //|   Checks if matrix is square                                     |
   //+------------------------------------------------------------------+
   bool CheckIsSq(double  &aA[])
     {
      int               m_ARows,m_ACols;
      GetSize(aA,m_ARows,m_ACols);
      return(m_ARows==m_ACols);
     }
   //+------------------------------------------------------------------+
   //|   Adds number aNum to matrix aA.                                 |
   //|   Resulting matrix is returned by reference in array aR.         |
   //+------------------------------------------------------------------+
   void AddNum(double  &aA[],double aNum,double  &aR[])
     {
      int               m_sz=ArraySize(aA);
      ArrayResize(aR,m_sz);
      aR[m_sz-2]=aA[m_sz-2];
      aR[m_sz-1]=aA[m_sz-1];
      for(int m_i=ArraySize(aA)-3;m_i>=0;m_i--)
        {
         aR[m_i]=aA[m_i]+aNum;
        }
     }
   //+------------------------------------------------------------------+
   //|   Multiplies matrix aA by number aNum.                             | 
   //|   Resulting matrix is returned by reference in array aR.         |
   //+------------------------------------------------------------------+
   void MultNum(double  &aA[],double aNum,double  &aR[])
     {
      int               m_sz=ArraySize(aA);
      ArrayResize(aR,m_sz);
      aR[m_sz-2]=aA[m_sz-2];
      aR[m_sz-1]=aA[m_sz-1];
      for(int m_i=ArraySize(aA)-3;m_i>=0;m_i--)
        {
         aR[m_i]=aA[m_i]*aNum;
        }
     }
   //+------------------------------------------------------------------+
   //|   Adds matrix aA to matrix aB.                                    |
   //|   Resulting matrix is returned by reference in array aAB.        |
   //+------------------------------------------------------------------+
   void AddMx(double  &aA[],double  &aB[],double  &aAB[])
     {
      int               m_ARows,m_ACols,m_BRows,m_BCols,m_ABRows,m_ABCols;
      GetSize(aA,m_ARows,m_ACols);
      GetSize(aB,m_BRows,m_BCols);
      m_ABRows=MathMin(m_ARows,m_BRows);
      m_ABCols=MathMin(m_ACols,m_BCols);
      double            m_tmp[];
      SetSize(m_tmp,m_ABRows,m_ABCols);
      for(int m_r=0;m_r<m_ABRows;m_r++)
        {
         for(int m_c=0;m_c<m_ABCols;m_c++)
           {
            int               m_i=m_r*m_ABCols+m_c;
            m_tmp[m_i]=aA[m_i]+aB[m_i];
           }
        }
      Copy(m_tmp,aAB);
     }
   //+------------------------------------------------------------------+
   //|   Multiplies matrix aA by matrix aB.                             |
   //|   Resulting matrix is returned by reference in array aAB.        |
   //+------------------------------------------------------------------+
   void MultMx(double  &aA[],double  &aB[],double  &aAB[])
     {
      int               m_ARows,m_ACols,m_BRows,m_BCols,m_ABRows,m_ABCols;
      GetSize(aA,m_ARows,m_ACols);
      GetSize(aB,m_BRows,m_BCols);
      m_ABRows=m_ARows;
      m_ABCols=m_BCols;
      int               m_N=MathMin(m_ACols,m_BRows);
      double            m_tmp[];
      SetSize(m_tmp,m_ABRows,m_ABCols);
      for(int m_r=0;m_r<m_ABRows;m_r++)
        {
         for(int m_c=0;m_c<m_ABCols;m_c++)
           {
            double            m_sum=0;
            for(int m_n=0;m_n<m_N;m_n++)
              {
               m_sum+=GetValue(aA,m_r,m_n)*GetValue(aB,m_n,m_c);
              }
            SetValue(m_tmp,m_r,m_c,m_sum);
           }
        }
      Copy(m_tmp,aAB);
     }
   //+------------------------------------------------------------------+
   //|   Transposes matrix aA. Resulting matrix                     |
   //|   is returned by reference in array aT                           |
   //+------------------------------------------------------------------+
   void Transpose(double  &aA[],double  &aT[])
     {
      int               m_ARows,m_ACols;
      GetSize(aA,m_ARows,m_ACols);
      double            m_tmp[];
      SetSize(m_tmp,m_ACols,m_ARows);
      for(int m_r=0;m_r<m_ARows;m_r++)
        {
         for(int m_c=0;m_c<m_ACols;m_c++)
           {
            double            m_Val=GetValue(aA,m_r,m_c);
            SetValue(m_tmp,m_c,m_r,m_Val);
           }
        }
      Copy(m_tmp,aT);
     }
   //+------------------------------------------------------------------+
   //|   Gets cofactor matrix.                                          |
   //|   aA - source matrix,                                            |
   //|   aAA - cofactor (returned by reference)                         |
   //+------------------------------------------------------------------+
   void AlgAdd(double  &aA[],double  &aAA[])
     {
      int               m_ARows,m_ACols;
      GetSize(aA,m_ARows,m_ACols);
      double            m_tmp[];
      Copy(aA,m_tmp);
      for(int m_r=0;m_r<m_ARows;m_r++)
        {
         for(int m_c=0;m_c<m_ACols;m_c++)
           {
            if((m_r+m_c)%2==1)
              {
               double            m_Val=GetValue(m_tmp,m_r,m_c);
               m_Val*=-1;
               SetValue(m_tmp,m_r,m_c,m_Val);
              }
           }
        }
      Copy(m_tmp,aAA);
     }
   //+------------------------------------------------------------------+
   //|   Returns the inverse matrix aR of matrix aA by reference.        |
   //|   The method returns true if inverse matrix exists               |
   //|   or false, if inverse matrix does not exist                     |
   //+------------------------------------------------------------------+
   bool Invert(double  &aA[],double  &aR[])
     {
      double            m_D=Def(aA);
      if(m_D==0)return(false);
      m_D=1.0/m_D;
      double            m_tmp[];
      MinorDefMx(aA,m_tmp);
      AlgAdd(m_tmp,m_tmp);
      Transpose(m_tmp,m_tmp);
      MultNum(m_tmp,m_D,m_tmp);
      Copy(m_tmp,aR);
      return(true);
     }
   //+------------------------------------------------------------------+
   //|   Returns triangular matrix aT from matrix aA by reference       |
   //+------------------------------------------------------------------+
   void Triangle(double  &aA[],double  &aT[])
     {
      double            m_A[];
      Copy(aA,m_A);
      int               m_Rows,m_Cols;
      GetSize(m_A,m_Rows,m_Cols);
      for(int m_r=0;m_r<m_Rows;m_r++)
        {
         double            m_vrr=GetValue(m_A,m_r,m_r);
         if(m_vrr==0)
           {
            for(int m_r2=m_r+1;m_r2<m_Rows;m_r2++)
              {
               if(GetValue(m_A,m_r2,m_r)!=0)
                 {
                  for(int m_c=0;m_c<m_Cols;m_c++)
                    {
                     double            m_v=GetValue(m_A,m_r,m_c);
                     SetValue(m_A,m_r,m_c,GetValue(m_A,m_r2,m_c));
                     SetValue(m_A,m_r2,m_c,m_v);
                    }
                 }
              }
            m_vrr=GetValue(m_A,m_r,m_r);
           }
         if(m_vrr!=0)
           {
            for(int m_r2=m_r+1;m_r2<m_Rows;m_r2++)
              {
               double            m_vr0=GetValue(m_A,m_r2,m_r);
               double            m_d=(-1)*m_vr0/m_vrr;
               for(int m_c=0;m_c<m_Cols;m_c++)
                 {
                  double            m_v0=GetValue(m_A,m_r,m_c);
                  double            m_vr=GetValue(m_A,m_r2,m_c);
                  m_vr=m_vr+m_v0*m_d;
                  SetValue(m_A,m_r2,m_c,m_vr);
                 }
              }
           }
        }
      Copy(m_A,aT);
     }
   //+------------------------------------------------------------------+
   //|   Gets the minor of matrix aA by row aRow and column aCol.       |
   //|   Minor is returned by reference in array aM                     |
   //+------------------------------------------------------------------+
   void Minor(double  &aA[],int aRow,int aCol,double  &aM[])
     {
      int               m_ARows,m_ACols;
      GetSize(aA,m_ARows,m_ACols);
      int               m_MRows=m_ARows-1;
      int               m_MCols=m_ACols-1;
      double            m_tmp[];
      SetSize(m_tmp,m_MRows,m_MCols);
      int               m_r,m_c;
      double            m_Val;
      for(m_r=0;m_r<aRow;m_r++)
        {
         for(m_c=0;m_c<aCol;m_c++)
           {
            m_Val=GetValue(aA,m_r,m_c);
            SetValue(m_tmp,m_r,m_c,m_Val);
           }
         for(m_c=aCol;m_c<m_MCols;m_c++)
           {
            m_Val=GetValue(aA,m_r,m_c+1);
            SetValue(m_tmp,m_r,m_c,m_Val);
           }
        }
      for(m_r=aRow;m_r<m_MRows;m_r++)
        {
         for(m_c=0;m_c<aCol;m_c++)
           {
            m_Val=GetValue(aA,m_r+1,m_c);
            SetValue(m_tmp,m_r,m_c,m_Val);
           }
         for(m_c=aCol;m_c<m_MCols;m_c++)
           {
            m_Val=GetValue(aA,m_r+1,m_c+1);
            SetValue(m_tmp,m_r,m_c,m_Val);
           }
        }
      Copy(m_tmp,aM);
     }
   //+------------------------------------------------------------------+
   //|   Returns the determinant value of the matrix aA minor           |
   //|   by row aRow and column aCol                                    |
   //+------------------------------------------------------------------+
   double MinorDef(double  &aA[],int aRow,int aCol)
     {
      double            m_tmp[];
      Minor(aA,aRow,aCol,m_tmp);
      return(Def(m_tmp));
     }
   //+------------------------------------------------------------------+
   //|   Gets minors matrix (matrix with values                         |
   //|   of the minors determinants).                                        |
   //|   aA - source matrix,                                            |
   //|   aM - matrix with minors determinants (returned by reference)   |
   //+------------------------------------------------------------------+
   void MinorDefMx(double  &aA[],double  &aM[])
     {
      double            m_tmp[];
      int               m_ARows,m_ACols;
      GetSize(aA,m_ARows,m_ACols);
      SetSize(m_tmp,m_ARows,m_ACols);
      for(int m_r=0;m_r<m_ARows;m_r++)
        {
         for(int m_c=0;m_c<m_ACols;m_c++)
           {
            double            m_mdef=MinorDef(aA,m_r,m_c);
            SetValue(m_tmp,m_r,m_c,m_mdef);
           }
        }
      Copy(m_tmp,aM);
     }
   //+------------------------------------------------------------------+
   //|   Returns the determinant value of matrix aA                     |
   //+------------------------------------------------------------------+
   double Def(double  &aA[])
     {
      double            m_A[];
      Copy(aA,m_A);
      int               m_Rows,m_Cols;
      GetSize(m_A,m_Rows,m_Cols);
      int               m_MSize=MathMin(m_Rows,m_Cols);
      double            m_Sn[];
      ArrayResize(m_Sn,m_MSize);
      ArrayInitialize(m_Sn,1);
      for(int m_s=0;m_s<m_MSize;m_s++)
        {
         double            m_v00=GetValue(m_A,m_s,m_s);
         if(m_v00==0)
           {
            for(int m_r=m_s+1;m_r<m_MSize;m_r++)
              {
               double            m_vr0=GetValue(m_A,m_r,m_s);
               if(m_vr0!=0)
                 {
                  for(int m_c=m_s;m_c<m_MSize;m_c++)
                    {
                     double            m_v1=GetValue(m_A,m_s,m_c);
                     double            m_v2=GetValue(m_A,m_r,m_c);
                     SetValue(m_A,m_s,m_c,m_v2);
                     SetValue(m_A,m_r,m_c,m_v1);
                    }
                  m_Sn[m_s]=-1;
                  break;
                 }
              }
            m_v00=GetValue(m_A,m_s,m_s);
            if(m_v00==0)
              {
               return(0);
              }
           }
         for(int m_r=m_s+1;m_r<m_MSize;m_r++)
           {
            double            m_vr=GetValue(m_A,m_r,m_s);
            if(m_vr==0)continue;
            double            m_d=(-1)*m_vr/m_v00;
            for(int m_c=0;m_c<m_MSize;m_c++)
              {
               double            m_v0=GetValue(m_A,m_s,m_c);
               m_vr=GetValue(m_A,m_r,m_c);
               m_vr=m_vr+m_v0*m_d;
               SetValue(m_A,m_r,m_c,m_vr);
              }
           }
        }
      double            m_def=1;
      for(int m_s=0;m_s<m_MSize;m_s++)
        {
         m_def*=m_Sn[m_s]*GetValue(m_A,m_s,m_s);
        }
      return(m_def);
     }
   //+------------------------------------------------------------------+
   //|   Returns rank of matrix aA                                      |
   //+------------------------------------------------------------------+
   int Rank(double  &aA[])
     {
      double            m_d;
      int               m_r,m_c;
      return(RankDRC(aA,m_d,m_r,m_c));
     }
   //+------------------------------------------------------------------+
   //|   Returns rank of matrix aA and by reference returns:            |
   //|   aDef - the determinant value,                                  |
   //|   aRow - row of minor with determinant not equal to 0            |
   //|   aCol - column of minor with determinant not equal to 0         |
   //+------------------------------------------------------------------+
   int RankDRC(double  &aA[],double  &aDef,int  &aRow,int  &aCol)
     {
      int               m_Rows,m_Cols;
      GetSize(aA,m_Rows,m_Cols);
      int               m_MSize=MathMin(m_Rows,m_Cols);
      double            m_A[];
      for(int m_s=m_MSize;m_s>0;m_s--)
        {
         SetSize(m_A,m_s,m_s);
         for(int m_r=0;m_r<=m_Rows-m_s;m_r++)
           {
            for(int m_c=0;m_c<=m_Cols-m_s;m_c++)
              {
               for(int m_r2=0;m_r2<m_s;m_r2++)
                 {
                  for(int m_c2=0;m_c2<m_s;m_c2++)
                    {
                     double            m_v=GetValue(aA,m_r+m_r2,m_c+m_c2);
                     SetValue(m_A,m_r2,m_c2,m_v);
                    }
                 }
               aDef=Def(m_A);
               if(aDef!=0)
                 {
                  aRow=m_r;
                  aCol=m_c;
                  return(m_s);
                 }
              }
           }
        }
      return(0);
     }
   //+------------------------------------------------------------------+
   //|   Copies column with index aFromCol from matrix aFrom            |
   //|   to matrix aTo to column with index aToCol.                     |
   //|   Result is returned by reference in array aR                    |
   //+------------------------------------------------------------------+
   void CopyCol(double  &aFrom[],double  &aTo[],int aFromCol,int aToCol,double  &aR[])
     {
      int               m_FromRows=GetRows(aFrom);
      int               m_ToRows=GetRows(aTo);
      int               m_Rows=MathMin(m_FromRows,m_ToRows);
      double            m_tmp[];
      Copy(aTo,m_tmp);
      for(int m_r=0;m_r<m_Rows;m_r++)
        {
         double            m_Val=GetValue(aFrom,m_r,aFromCol);
         SetValue(m_tmp,m_r,aToCol,m_Val);
        }
      Copy(m_tmp,aR);
     }
   //+------------------------------------------------------------------+
   //|   Copies row with index aFromRow from matrix aFrom               |
   //|   to matrix aTo to row with index aToRow.                        |
   //|   Result is returned by reference in array aR                    |
   //+------------------------------------------------------------------+
   void CopyRow(double  &aFrom[],double  &aTo[],int aFromRow,int aToRow,double  &aR[])
     {
      int               m_FromCols=GetCols(aFrom);
      int               m_ToCols=GetCols(aTo);
      int               m_Cols=MathMin(m_FromCols,m_ToCols);
      double            m_tmp[];
      Copy(aTo,m_tmp);
      for(int m_c=0;m_c<m_Cols;m_c++)
        {
         double            m_Val=GetValue(aFrom,aFromRow,m_c);
         SetValue(m_tmp,aToRow,m_c,m_Val);
        }
      Copy(m_tmp,aR);
     }
   //+------------------------------------------------------------------+
   //|   Extends matrix aA by adding column aC to it.                   |
   //|   Result is returned by reference in array aF                    |
   //+------------------------------------------------------------------+
   void AppendCol(double  &aA[],double  &aC[],double  &aF[])
     {
      int               m_Rows,m_Cols;
      GetSize(aA,m_Rows,m_Cols);
      double            m_tmp[];
      SetSize(m_tmp,m_Rows,m_Cols+1);
      for(int m_r=0;m_r<m_Rows;m_r++)
        {
         for(int m_c=0;m_c<m_Cols;m_c++)
           {
            SetValue(m_tmp,m_r,m_c,GetValue(aA,m_r,m_c));
           }
        }
      m_Rows=MathMin(m_Rows,GetRows(aC));
      for(int m_r=0;m_r<m_Rows;m_r++)
        {
         SetValue(m_tmp,m_r,m_Cols,GetValue(aC,m_r,0));
        }
      Copy(m_tmp,aF);
     }
   //+------------------------------------------------------------------+
   //|   Extends matrix aA by adding row aR to it.                      |
   //|   Result is returned by reference in array aF                    |
   //+------------------------------------------------------------------+
   void AppendRow(double  &aA[],double  &aR[],double  &aF[])
     {
      int               m_Rows,m_Cols;
      GetSize(aA,m_Rows,m_Cols);
      double            m_tmp[];
      SetSize(m_tmp,m_Rows+1,m_Cols);
      for(int m_r=0;m_r<m_Rows;m_r++)
        {
         for(int m_c=0;m_c<m_Cols;m_c++)
           {
            SetValue(m_tmp,m_r,m_c,GetValue(aA,m_r,m_c));
           }
        }
      m_Cols=MathMin(m_Cols,GetCols(aR));
      for(int m_c=0;m_c<m_Cols;m_c++)
        {
         SetValue(m_tmp,m_Rows,m_c,GetValue(aR,0,m_c));
        }
      Copy(m_tmp,aF);
     }
   //+------------------------------------------------------------------+
   //|   Solves system of linear equations using Cramer's rule.                      |
   //|   aK - coefficient matrix (square),                              |
   //|   aY - column of values,                                         |
   //|   aX - row of results                                            |
   //+------------------------------------------------------------------+
   bool SystemKramer(double  &aK[],double  &aY[],double  &aX[])
     {
      double            m_def=Def(aK);
      if(m_def==0)return(false);
      int               m_KRows,m_KCols;
      GetSize(aK,m_KRows,m_KCols);
      int               m_Size=MathMin(m_KRows,m_KCols);
      int               m_YRows=GetRows(aY);
      m_Size=MathMin(m_Size,m_YRows);
      SetSize(aX,m_Size,1);
      double            m_tmp[];
      for(int m_c=0;m_c<m_Size;m_c++)
        {
         CopyCol(aY,aK,0,m_c,m_tmp);
         aX[m_c]=Def(m_tmp)/m_def;
        }
      return(true);
     }
   //+------------------------------------------------------------------+
   //|   Solves system of linear equations using invertible matrix.               |
   //|   aK - coefficient matrix (square),                              |
   //|   aY - column of values,                                         |
   //|   aX - row of results                                            |
   //+------------------------------------------------------------------+
   bool SystemInverse(double  &aK[],double  &aY[],double  &aX[])
     {
      double            m_B[];
      if(!Invert(aK,m_B))return(false);
      MultMx(m_B,aY,aX);
      return(true);
     }
   //+------------------------------------------------------------------+
   //|   Solves system of linear equations using Gaussian elimination.                       |
   //|   aK - coefficient matrix (square),                              |
   //|   aY - column of values,                                         |
   //|   aX - row of results                                            |
   //+------------------------------------------------------------------+
   bool SystemGauss(double  &aK[],double  &aY[],double  &aX[])
     {
      double            m_K[];
      AppendCol(aK,aY,m_K);
      Triangle(m_K,m_K);
      int               m_Rows,m_Cols;
      GetSize(m_K,m_Rows,m_Cols);
      SetSize(aX,m_Rows,1);
      for(int m_r=m_Rows-1;m_r>=0;m_r--)
        {
         double            m_y=GetValue(m_K,m_r,m_Cols-1);
         double            m_k=GetValue(m_K,m_r,m_r);
         if(m_k==0)
           {
            return(false);
           }
         double            m_x=m_y/m_k;
         SetValue(aX,m_r,0,m_x);
         for(int m_r2=m_r-1;m_r2>=0;m_r2--)
           {
            m_k=GetValue(m_K,m_r2,m_r);
            m_y=GetValue(m_K,m_r2,m_Cols-1);
            m_y-=m_k*m_x;
            SetValue(m_K,m_r2,m_Cols-1,m_y);
           }
        }
      return(true);
     }
   //+------------------------------------------------------------------+
   //|  Checks system of equations.                                     |
   //|   aK - coefficient matrix (square),                              |
   //|   aY - column of values.                                         |
   //|   Returned value:                                                |
   //|     -1 - no solutions,                                           |
   //|      0 - one solution,                                           |
   //|      1 - infinite number of solutions                            |
   //+------------------------------------------------------------------+
   int SystemCheck(double  &aK[],double  &aY[])
     {
      int               m_r0=Rank(aK);
      double            m_K[];
      AppendCol(aK,aY,m_K);
      int               m_r1=Rank(m_K);
      if(m_r0!=m_r1)
        {
         return(-1); // no solutions
        }
      int               m_Rows,m_Cols;
      GetSize(aK,m_Rows,m_Cols);
      int               m_r3=MathMin(m_Rows,m_Cols);
      if(m_r1!=m_r3)
        {
         return(1); // multiple solutions         
        }
      return(0); // one solution      

     }
   //+------------------------------------------------------------------+
   //|   Displays the entire matrix in one alert box.                         |
   //|      aA       - matrix,                                          |
   //|      aDigits  - number of digits after decimal point,            |
   //|      aCaption - message title                                    |
   //+------------------------------------------------------------------+
   void Alert(double  &aA[],int aDigits=2,string aCaption="")
     {
      int               m_ARows,m_ACols;
      GetSize(aA,m_ARows,m_ACols);
      string            m_str="=== === "+aCaption+" === ===\n";
      for(int m_r=0;m_r<m_ARows;m_r++)
        {
         for(int m_c=0;m_c<m_ACols;m_c++)
           {
            string            m_tstr=DoubleToString(GetValue(aA,m_r,m_c),aDigits);
            m_str=m_str+m_tstr+", ";
           }
         m_str=StringSubstr(m_str,0,StringLen(m_str)-2);
         m_str=m_str+"\n";
        }
      Alert(m_str);
     }
   //+------------------------------------------------------------------+
   //|  Displays matrix in alert box line by line, rows displayed from  |
   //|   bottom to top, then title, i.e. in alert box matrix is oriented |
   //|   normally: title at the top, then rows in order.                |
   //+------------------------------------------------------------------+
   void Alert2(double  &aA[],int aDigits=2,string aCaption="")
     {
      int               m_ARows,m_ACols;
      GetSize(aA,m_ARows,m_ACols);
      for(int m_r=m_ARows-1;m_r>=0;m_r--)
        {
         string            m_str="";
         for(int m_c=0;m_c<m_ACols;m_c++)
           {
            m_str=m_str+DoubleToString(GetValue(aA,m_r,m_c),aDigits)+", ";
           }
         m_str=StringSubstr(m_str,0,StringLen(m_str)-2);
         Alert(m_str);
        }
      Alert("=== === "+aCaption+" === ===");
     }
   //+------------------------------------------------------------------+
   //|   Displays array of matrix as a string in alert box              |
   //+------------------------------------------------------------------+
   void Alert1Str(double  &aA[],int aDigits=2)
     {
      string            m_str="";
      for(int m_i=0;m_i<ArraySize(aA);m_i++)
        {
         m_str=m_str+DoubleToString(aA[m_i],aDigits)+", ";
        }
      if(m_str!="")
        {
         m_str=StringSubstr(m_str,StringLen(m_str)-2);
        }
      Alert(m_str);
     }

  };
//+------------------------------------------------------------------+