//+------------------------------------------------------------------+
//|                                                          mql.mq5 |
//|                                    Copyright (c) 2019, Marketeer |
//|                          https://www.mql5.com/en/users/marketeer |
//|                            https://www.mql5.com/ru/articles/5638 |
//|                                                   rev.2020.02.13 |
//+------------------------------------------------------------------+

#property script_show_inputs

enum TEST_GRAMMAR {Expression, MQL};

input TEST_GRAMMAR TestMode = Expression;;
//Sources/Experts/Examples/MACD/MACD Sample.mq5
//Sources/Experts/Examples/Layouts/SlidingPuzzle2.mq5
input string SourceFile = "calc.txt";;
input string IncludesFolder = "Sources/Include/";;
input bool LoadIncludes = true;
input bool PrintCST = false;


//#define PRINTX Print   // print out a lot of traces how the grammar unwindes, use for small inputs only!
#define MAX_LEVELS 1000  // maximal recursion level, blunt check for possible looping in grammar
//#define CHECK_FOR_LOOP // more sophisticated check for looping, debug for incorrect grammars, but it's slow


#include <mql5/Scanner.mqh>
#include <mql5/Prsr.mqh>


class ProductionCallback: public Callback
{
  public:
    virtual void produce(const Parser *parser, const Terminal *p, const int index, const Token *t, const Source *context, const int level) override
    {
      // Print("*** ", index, " ", p.toString(), " ", t.toString(context), " ", context);
      // parser.printStack();
    }
    virtual void backtrack(const int index) override {}
};

ProductionCallback pc; // use it `R(node)[&pc]` to intercept specific node production result

  class Class
  {
    private:
      BaseArray<Class *> subclasses;
      Class *superclass;
      string name;

    public:
      Class(const string n): name(n), superclass(NULL)
      {
      }
      
      ~Class()
      {
        subclasses.clear();
      }
      
      void addSubclass(Class *derived)
      {
        // TODO: check for duplicates and redefinitions
        derived.superclass = &this;
        subclasses.add(derived);
      }
      
      bool hasParent() const
      {
        return superclass != NULL;
      }
      
      Class *operator[](int i) const
      {
        return subclasses[i];
      }
      
      int size() const
      {
        return subclasses.size();
      }
      
      void print(string &gap, const bool last = false) const
      {
        if(StringLen(gap) > 0 && gap[StringLen(gap) - 1] != '-')
        {
          gap = StringSubstr(gap, 0, StringLen(gap) - 3);
          gap += "+--";
        }
        Print(gap, name);
        if(subclasses.size() > 0)
        {
          if(last)
          {
            gap = StringSubstr(gap, 0, StringLen(gap) - 3);
            gap += "   ";
          }
          else
          {
            StringReplace(gap, "+--", "|  ");
          }
          Print(gap, "  ^");
          gap += "  +--";
          for(int i = 0; i < subclasses.size(); i++)
          {
            subclasses[i].print(gap, i == subclasses.size() - 1);
          }
          gap = StringSubstr(gap, 0, StringLen(gap) - 5);
        }
      }
  };

class MyCallback: public Callback
{
  Map<string,Class *> map;

  public:
    ~MyCallback()
    {
      reset();
    }
    
    void reset()
    {
      for(int i = 0; i < map.getSize(); i++)
      {
        if(CheckPointer(map[i]) == POINTER_DYNAMIC) delete map[i];
      }
      map.reset();
    }
    
    void print()
    {
      Print("Class hierarchy:");
      string gap = "";
      for(int i = 0; i < map.getSize(); i++)
      {
        Class *current = map[i];
        if(!current.hasParent())
        {
          Print("");
          current.print(gap);
        }
      }
    }
    
    virtual void produce(const Parser *parser, const Terminal *p, const int index, const Token *t, const Source *context, const int level) override
    {
      // TODO: for methods resolution, classes should be stored in a stack because classes can be nested
      static string templName = "";
      static string templBaseName = "";
      static string className = "";
      static string baseName = "";
      static string method = "";
      
      // collect all tokens from `method` nonterminal
      if(p.getName() == "method")
      {
        method += t.content(context) + " ";
      }
      // as soon as other [non]terminal is detected and string is filled, signature is ready
      else if(method != "")
      {
        Print(className + " :: " + method);
        method = "";
      }
      
      if(p.getName() == "template_decl" && t.getType() == IDENTIFIER)
      {
        if(templName != "") templName += ",";
        templName += t.content(context);
      }
      
      if(p.getName() == "class_name" && t.getType() == IDENTIFIER)
      {
        className = t.content(context);
        if(templName != "")
        {
          className += "<" + templName + ">";
          templName = "";
        }
      }

      if(p.getName() == "derived_clause" && t.getType() == IDENTIFIER)
      {
        if(baseName == "") baseName = t.content(context);
        else
        {
          if(templBaseName != "") templBaseName += ",";
          templBaseName += t.content(context);
        }
      }
      
      if(p.getName() == "class_decl") // finalization
      {
        if(className != "")
        {
          if(map[className] == NULL)
          {
            map.put(className, new Class(className));
          }
          else
          {
            // Class already defined, maybe forward declaration
          }
        }
      
        if(baseName != "")
        {
          if(templBaseName != "")
          {
            baseName += "<" + templBaseName + ">";
          }
          Class *base = map[baseName];
          if(base == NULL)
          {
            // Unknown class, maybe not included, but strange
            base = new Class(baseName);
            map.put(baseName, base);
          }
          
          if(map[className] == NULL)
          {
            Print("Error: base name `", baseName, "` resolved before declaration of the class: ", className);
          }
          else
          {
            base.addSubclass(map[className]);
          }
          
          baseName = "";
        }
        className = "";
        templName = "";
        templBaseName = "";
      }
    }
    
    virtual void backtrack(const int index) override
    {
    }
};

void OnStart()
{
  Print("\n");
  
  Preprocessor loader(SourceFile, IncludesFolder, LoadIncludes);
  
  Print("Scanning...");
  
  if(!loader.run())
  {
    Print("Loader failed");
    return;
  }
  
  Print("Files processed: ", loader.scannedFilesCount());
  Print("Source length: ", loader.totalCharactersCount());
  if(LoadIncludes)
  {
    Print("File map:");
    loader.text().printFiles();
  }

  Scanner scanner(loader.text(), false);
  List<Token *> *tokens = scanner.scanTokens();
  
  Print("Lines: ", scanner.getLines());
  Print("Tokens: ", tokens.size());
  
  if(!scanner.isSuccess())
  {
    Print("Tokenizer failed");
    // scanner.dump("dump.txt"); // output all sources
    delete tokens;
    return;
  }
  
  MyCallback myc;
  Parser parser(tokens, loader.text(), PrintCST);
  parser.setCallback(&myc);
  
  Print("Defining grammar...");
  if(TestMode == Expression)
  {
    testExpressionGrammar(&parser);
  }
  else
  {
    testMQLgrammar(&parser);
  }
  
  delete tokens;
  
  myc.print();
}

void testExpressionGrammar(Parser *p)
{
  _FORWARD(expression);
  _DECLARE(value) = T(CONST_INTEGER) | T(LEFT_PAREN) + R(expression) + T(RIGHT_PAREN);
  _DECLARE(operation) = (T(PLUS) | T(STAR) | T(MINUS) | T(SLASH)) + R(expression);
  expression = R(value) + R(operation)*0;
  
  Print("Parsing...");
  while(p.match(&expression) && !p.isAtEnd())
  {
    Print("", "Unexpected end");
    break;
  }

  if(p.isAtEnd())
  {
    Print("Success");
  }
  else
  {
    Print("Failed");
    p.printState();
  }

  if(PrintCST)
  {
    Print("Concrete Syntax Tree:");
    TreePrinter printer(p);
    printer.printTree();
  }

  Comment("");
}

void testMQLgrammar(Parser *p)
{
  _DECLARE(simple_type) = T(CHAR) | T(SHORT) | T(INT) | T(LONG) | T(UCHAR) | T(USHORT) | T(UINT) | T(ULONG) | T(BOOL) | T(STRING) | T(DOUBLE) | T(FLOAT) | T(COLOR) | T(DATETIME) | T(UNSIGNED) | T(VOID);
  _FORWARD(type);
  _DECLARE(next_template_types) = (T(COMMA) + R(type))*1;
  _DECLARE(template_types) = R(type) + R(next_template_types)*0;
  _DECLARE(template_specifier) = T(LESS) + R(template_types) + T(GREATER);
  _DECLARE(object_type) = T(IDENTIFIER);
  type = T(CONST)*0 + (R(simple_type) + (T(STAR) | T(BIT_AND))*0 | object_type + R(template_specifier)*0 + (T(STAR) + T(BIT_AND)*0 | T(BIT_AND))*0);

  _FORWARD(expression);
  
  _DECLARE(template_parameter) = T(TYPENAME) + T(IDENTIFIER);
  _DECLARE(next_template_parameters) = (T(COMMA) + R(template_parameter))*1;
  _DECLARE(template_parameters) = R(template_parameter) + R(next_template_parameters)*0;
  _DECLARE(template_decl) = T(TEMPLATE) + T(LESS) + R(template_parameters) + T(GREATER);
  _DECLARE(class_name) = T(IDENTIFIER);
  _DECLARE(class_name_decl) = R(template_decl)*0 + (T(CLASS) | T(STRUCT) | T(INTERFACE)) + class_name;
  _DECLARE(derived) = T(COLON) + (T(PUBLIC) | T(PROTECTED) | T(PRIVATE));
  _DECLARE(access) = T(PUBLIC) | T(PROTECTED) | T(PRIVATE);
  _DECLARE(typename_get) = T(TYPENAME) + T(LEFT_PAREN) + T(IDENTIFIER) + T(RIGHT_PAREN);
  
  _DECLARE(value) = T(CONST_STRING) | T(CONST_NUMBER) | T(CONST_INTEGER) | T(CONST_DATETIME) | T(LITERAL) | T(NIL) | T(TRUE) | T(FALSE) | R(typename_get);

  _FORWARD(array_index);
  _FORWARD(initializer);
  
  _DECLARE(argument) = R(type) + T(IDENTIFIER) + R(array_index)*0 + R(initializer)*0;
  _DECLARE(nextarguments) = (T(COMMA) + R(argument))*1;
  _DECLARE(arguments) = R(argument) + R(nextarguments)*0 | T(VOID);
  
  _FORWARD(conditional_expression);

  _DECLARE(parameter) = R(expression) | R(type);
  _DECLARE(nextparameters) = (T(COMMA) + R(parameter))*1;
  _DECLARE(parameters) = R(parameter) + nextparameters*0;
  _DECLARE(invocation) = T(IDENTIFIER) + T(LEFT_PAREN) + R(parameters) + T(RIGHT_PAREN);
  _DECLARE(macro) = R(invocation) + T(SEMICOLON)*0;
  
  _FORWARD(name);
  _FORWARD(qualified_class_name);
  name = R(qualified_class_name);
  
  _DECLARE(more_qualified_class_names) = (T(COLON) + T(COLON) + R(qualified_class_name))*1;
  qualified_class_name = T(TILDE)*0 + T(IDENTIFIER) + R(more_qualified_class_names)*0;

  _DECLARE(primary_expression) = R(value) | T(THIS) | R(more_qualified_class_names) | T(LEFT_PAREN) + R(expression) + T(RIGHT_PAREN) | R(name);
  
  _FORWARD(assignment_expression);
  
  _DECLARE(more_assignment_expressions) = (T(COMMA) + R(assignment_expression))*1;
  expression = R(assignment_expression) + R(more_assignment_expressions)*0;

  _DECLARE(assignment_operator) = T(EQUAL) | T(STAR_EQUAL) | T(SLASH_EQUAL) | T(DIV_EQUAL) | T(PLUS_EQUAL) | T(MINUS_EQUAL) | T(GREATER_EQUAL) | T(LESS_EQUAL) | T(BIT_AND_EQUAL) | T(BIT_XOR_EQUAL) | T(BIT_OR_EQUAL);

  _FORWARD(unary_expression);
  assignment_expression = R(unary_expression) + R(assignment_operator) + R(assignment_expression) | R(conditional_expression);

  _DECLARE(logical_or_expression_suffix) = T(QMARK) + R(expression) + T(COLON) + R(conditional_expression);
  _FORWARD(logical_or_expression);
  conditional_expression = R(logical_or_expression) + R(logical_or_expression_suffix)*0;
  _FORWARD(logical_and_expression);
  _DECLARE(more_logical_and_expressions) = (T(OR) + R(logical_and_expression))*1;
  logical_or_expression = R(logical_and_expression) + R(more_logical_and_expressions)*0;

  _FORWARD(inclusive_or_expression);
  _DECLARE(more_inclusive_or_expressions) = (T(AND) + R(inclusive_or_expression))*1;
  logical_and_expression = R(inclusive_or_expression) + R(more_inclusive_or_expressions)*0;

  _FORWARD(exclusive_or_expression);
  _DECLARE(more_exclusive_or_expressions) = (T(BIT_OR) + R(exclusive_or_expression))*1;
  inclusive_or_expression = R(exclusive_or_expression) + R(more_exclusive_or_expressions)*0;

  _FORWARD(and_expression);
  _DECLARE(more_and_expressions) = (T(BIT_XOR) + R(and_expression))*1;
  exclusive_or_expression = R(and_expression) + R(more_and_expressions)*0;

  _FORWARD(equality_expression);
  _DECLARE(more_equality_expressions) = (T(BIT_AND) + R(equality_expression))*1;
  and_expression = R(equality_expression) + R(more_equality_expressions)*0;

  _FORWARD(relational_expression);
  _DECLARE(more_relational_expressions) = ((T(EQUAL_EQUAL) | T(BANG_EQUAL)) + R(relational_expression))*1;
  equality_expression = R(relational_expression) + R(more_relational_expressions)*0;

  _FORWARD(shift_expression);
  _DECLARE(more_shift_expressions) = ((T(LESS) | T(GREATER) | T(LESS_EQUAL) | T(GREATER_EQUAL)) + R(shift_expression))*1;
  relational_expression = R(shift_expression) + R(more_shift_expressions)*0;

  _FORWARD(additive_expression);
  _DECLARE(more_additive_expressions) = ((T(LESS) + T(LESS) | T(GREATER) + T(GREATER)) + R(additive_expression))*1;
  shift_expression = R(additive_expression) + R(more_additive_expressions)*0;

  _FORWARD(multiplicative_expression);
  _DECLARE(more_multiplicative_expressions) = ((T(PLUS) | T(MINUS)) + R(multiplicative_expression))*1;
  additive_expression = R(multiplicative_expression) + R(more_multiplicative_expressions)*0;

  _FORWARD(cast_expression);
  _DECLARE(more_cast_expressions) = ((T(STAR) | T(SLASH) | T(DIV)) + R(cast_expression))*1;
  multiplicative_expression = R(cast_expression) + R(more_cast_expressions)*0;

  // casting can be made by functional notation (type(expr)) or c-like cast notation ((type)expr)
  cast_expression = R(simple_type) + T(LEFT_PAREN) + R(expression) + T(RIGHT_PAREN) | T(LEFT_PAREN) + R(type) + T(RIGHT_PAREN) + R(cast_expression) | R(unary_expression) | T(LEFT_PAREN) + R(macro) + T(RIGHT_PAREN) + R(cast_expression);

  _DECLARE(unary_operator) = T(STAR) | T(BIT_AND) | T(PLUS) | T(MINUS) | T(BANG) | T(TILDE);

  _FORWARD(postfix_expression);
  _FORWARD(allocation_expression);
  _FORWARD(deallocation_expression);
  unary_expression = R(postfix_expression) | R(unary_operator) + R(cast_expression) | T(INC) + R(unary_expression) | T(DEC) + R(unary_expression) | T(SIZEOF) + T(LEFT_PAREN) + R(type) + T(RIGHT_PAREN) | R(allocation_expression) | R(deallocation_expression);

  _DECLARE(more_expressions) = (T(COMMA) + R(expression))*1;
  _FORWARD(initializer_list);
  _DECLARE(more_initializer_list) = (T(COMMA) + R(initializer_list))*1;
  initializer_list = R(expression) + R(more_expressions)*0 | T(LEFT_BRACE) + R(initializer_list) + R(more_initializer_list)*0 + T(RIGHT_BRACE);
  _DECLARE(new_initializer) = T(LEFT_PAREN) + R(initializer_list)*0 + T(RIGHT_PAREN);
  allocation_expression = T(NEW) + R(type) + R(new_initializer)*0;
  deallocation_expression = T(DELETE) + R(cast_expression);

  _FORWARD(expression_list);
  _DECLARE(postfix_expression_suffix)
     = T(LEFT_BRACKET) + R(expression) + T(RIGHT_BRACKET)
     | T(LEFT_PAREN) + R(expression_list)*0 + T(RIGHT_PAREN)
     | T(DOT) + R(name) | T(INC) | T(DEC);
  _DECLARE(postfix_expression_list) = R(postfix_expression_suffix)*1;
    
  _DECLARE(initializer_clause) = R(assignment_expression);
  _DECLARE(more_initializer_clauses) = (T(COMMA) + R(initializer_clause))*1;
  expression_list = R(initializer_clause) + R(more_initializer_clauses)*0;

  _DECLARE(ptr_operator) = T(STAR);
  postfix_expression
    = R(primary_expression) + R(postfix_expression_list)*0
    | T(DYNAMIC_CAST) + T(LESS) + R(type) + T(GREATER) + T(LEFT_PAREN) + R(expression) + T(RIGHT_PAREN);

  FORWARD(statement);
  _FORWARD(codeblock);
  _DECLARE(expression_statement) = R(expression)*0 + T(SEMICOLON);
  _DECLARE(else_clause) = T(ELSE) + R(statement);
  _DECLARE(if_statement) = T(IF) + T(LEFT_PAREN) + R(expression) + T(RIGHT_PAREN) + R(statement) + R(else_clause)*0;
  _DECLARE(labeled_statement) = (T(DEFAULT) | T(CASE) + R(conditional_expression)) + T(COLON) + R(statement);
  _DECLARE(switch_statement) = T(SWITCH) + T(LEFT_PAREN) + R(expression) + T(RIGHT_PAREN) + R(statement);
  _DECLARE(selection_statement) = R(if_statement) | R(switch_statement);
  _DECLARE(while_statement) = T(WHILE) + T(LEFT_PAREN) + R(expression) + T(RIGHT_PAREN) + R(statement);
  _DECLARE(do_statement) = T(DO) + R(statement) + T(WHILE) + T(LEFT_PAREN) + R(expression) + T(RIGHT_PAREN) + T(SEMICOLON);
  
  _DECLARE(constant_expression) = R(conditional_expression);
  _DECLARE(enumerator) = T(IDENTIFIER) + (T(EQUAL) + R(constant_expression))*0;
  _DECLARE(more_enumerators) = (T(COMMA) + R(enumerator))*1;
  _DECLARE(enum_list) = R(enumerator) + R(more_enumerators)*0;
  _DECLARE(enum_type) = T(ENUM) + T(IDENTIFIER) + T(LEFT_BRACE) + R(enum_list) + T(RIGHT_BRACE) + T(SEMICOLON);
  _DECLARE(decl_specifier) = T(STATIC) | T(EXTERN) | T(CONST) | T(INPUT) | T(SINPUT) | R(template_decl);
  _DECLARE(more_decl_specifiers) = R(decl_specifier)*1;
  _DECLARE(decl_specifiers) = R(more_decl_specifiers)*0 + R(type);
  _FORWARD(declarator);
  array_index = T(LEFT_BRACKET) + R(constant_expression)*0 + T(RIGHT_BRACKET);
  _DECLARE(declarator_suffix) = (R(array_index) | R(new_initializer))*1;
  declarator = (R(name) | R(ptr_operator) + R(declarator)) + R(declarator_suffix)*0;
  
  initializer = (T(EQUAL) + (R(expression) | T(LEFT_BRACE) + R(initializer_list)*0 + T(RIGHT_BRACE))) | T(LEFT_PAREN) + R(expression_list) + T(RIGHT_PAREN);
  _DECLARE(init_declarator) = R(declarator) + R(initializer)*0;
  _DECLARE(more_init_declarators) = (T(COMMA) + R(init_declarator))*1;
  _DECLARE(declarator_list) = R(init_declarator) + R(more_init_declarators)*0;
  
  _DECLARE(plain_type) = R(decl_specifiers)*0 + R(declarator_list)*0 + T(SEMICOLON);
  DECLARE(declaration_statement) = R(plain_type) | R(enum_type);
  
  _DECLARE(for_statement) = T(FOR) + T(LEFT_PAREN) + (R(expression_statement) | R(declaration_statement)) + R(expression)*0 + T(SEMICOLON) + R(expression)*0 + T(RIGHT_PAREN) + R(statement);
  
  _DECLARE(iteration_statement) = R(while_statement) | R(do_statement) | R(for_statement);
  
  _DECLARE(result_clause) = T(LEFT_PAREN) + R(expression) + T(RIGHT_PAREN) | R(expression);
  _DECLARE(jump_statement) = (T(BREAK) | T(CONTINUE) | T(RETURN) + R(result_clause)*0) + T(SEMICOLON);
  statement = R(expression_statement) | R(codeblock) | R(selection_statement) | R(labeled_statement) | R(iteration_statement) | R(jump_statement);
  
  _DECLARE(sharp_conditional) = T(_DEFINE) | T(_INCLUDE) | T(_ELSE) | T(_ENDIF) | T(_IFDEF) | T(_IFNDEF) | T(_UNDEF);
  DECLARE(sharp) = T(_PROPERTY) | T(_IMPORT) | T(_RESOURCE) | R(sharp_conditional);
  
  _FORWARD(class_decl);
  _DECLARE(operator1) = R(declaration_statement) | R(class_decl) | R(statement) | R(sharp_conditional);
  _DECLARE(operators) = R(operator1)*1;
  codeblock = T(LEFT_BRACE) + R(operators)*0 + T(RIGHT_BRACE);
  
  _DECLARE(overloads) = T(SLASH) | T(DIV) | T(INC) | T(DEC) | T(BANG_EQUAL) | T(EQUAL_EQUAL) | T(GREATER) | T(GREATER_EQUAL) | T(LESS) | T(LESS_EQUAL) | T(AND) | T(OR) | T(LEFT_BRACKET) + T(RIGHT_BRACKET); // TODO: add more
  
  _DECLARE(overload) = T(OPERATOR) + (R(assignment_operator) | R(unary_operator) | R(overloads)) + T(LEFT_PAREN) + (T(INT) & R(arguments))*0 + T(RIGHT_PAREN);
  _DECLARE(name_with_arg_list) = T(IDENTIFIER) + T(LEFT_PAREN) + R(arguments)*0 + T(RIGHT_PAREN) | R(overload);
  
  _DECLARE(pure) = T(EQUAL) + (TC(CONST_INTEGER, "0") | T(NIL));
  _DECLARE(modifier) = T(CONST) | T(OVERRIDE) | T(FINAL) | R(pure);
  _DECLARE(modifiers) = R(modifier)*1;
  
  _DECLARE(method_specifiers) = (T(STATIC) | T(VIRTUAL))*1;
  _DECLARE(method) = R(template_decl)*0 + R(method_specifiers)*0 + R(type) + R(name_with_arg_list) + R(modifiers)*0;
  _DECLARE(ctor_initializer) = T(IDENTIFIER) + T(LEFT_PAREN) + R(expression) + T(RIGHT_PAREN);
  _DECLARE(more_ctor_initializers) = (T(COMMA) + R(ctor_initializer))*1;
  _DECLARE(ctor_initializers) = T(COLON) + R(ctor_initializer) + R(more_ctor_initializers)*0;
  _DECLARE(cdtors) = (R(template_decl) | T(TILDE))*0 + R(name_with_arg_list) + R(ctor_initializers)*0;
  _DECLARE(fullmethod) = (R(method) | R(cdtors)) + (T(SEMICOLON) | R(codeblock));
  _DECLARE(function_type_name) = R(type) + R(name) | R(name);
  _DECLARE(function)
    = R(template_decl)*0
    + R(function_type_name)
    + T(LEFT_PAREN) + R(arguments)*0 + T(RIGHT_PAREN)
    +  (R(modifiers)
      | R(ctor_initializers))*0
    + R(codeblock);
  
  _FORWARD(subclass);

  DECLARE(section) = R(access) + T(COLON);
  _DECLARE(member) = R(section) | R(subclass) | R(declaration_statement) | R(fullmethod) | R(macro) | R(sharp_conditional);
  _DECLARE(members) = member*1;
  
  _DECLARE(base_name) = R(type);
  _DECLARE(derived_clause) = derived + base_name;
  _DECLARE(body) = members*0;
  
  _DECLARE(class_body) = T(LEFT_BRACE) + body*0 + T(RIGHT_BRACE);
  class_decl = class_name_decl + derived_clause*0 + R(class_body)*0 + T(SEMICOLON);
  
  _FORWARD(union_decl);
  subclass = R(class_decl) | R(union_decl);
  
  union_decl = T(UNION) + T(IDENTIFIER) + R(class_body)*0 + T(IDENTIFIER)*0 + T(SEMICOLON);

  _DECLARE(element) = R(class_decl) | R(declaration_statement) | R(function) | R(sharp) | R(macro) | R(union_decl);
  _DECLARE(program) = R(element)*1;

  Print("Parsing...");

  while(p.match(&program) && !p.isAtEnd())
  {
    Print("", "Unexpected end");
    break;
  }
  
  if(p.isAtEnd())
  {
    Print("Success");
  }
  else
  {
    Print("Failed");
    p.printState();
  }

  if(PrintCST)
  {
    Print("Concrete Syntax Tree:");
    TreePrinter printer(p);
    printer.printTree();
  }
  
  Comment("");
}
