//+------------------------------------------------------------------+
//|                                                          cot.mqh |
//|                                           Copyright 2010, Sergan |
//|                                                                  |
//+------------------------------------------------------------------+
#property copyright "2010, Sergan"
#property link      "serganmt@hotbox.ru"

#include <common.mqh>

#define COT_FIRST_YEAR_DATA  1999  // starting year for COT reports
#define CIT_FIRST_YEAR_DATA  2006  // starting year for CIT reports
#define DCOT_FIRST_YEAR_DATA  2009 // starting year for DCOT reports
#define COT_STORE_FOLDER     "deacot" // folder name should be: deacotxxxx
#define COT_MAIN_URL         "http://www.cftc.gov/files/dea/history/"
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
enum cot_type_report
  { // type of report
   COT,DCOT,CIT
  };
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
enum cot_type_traders
  {// types of traders
   noncommercial,      // Non-commercial 
   commercial,         // Commercial
   nonreportable,      // Non-reportable
   index,              // Index
   producer,           // Producers
   swapdealers,        // Swap Dealers
   managedmoney,       // Managers
   otherreportables    // Other reportable
  };
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
enum cot_type_data
  {   // types of data
   netlongs,           // Net longs
   netlongsrate,       // Net longs rate in open interest
   willams_index,      // Williams Index
   openinterest        // Open Interest
  };
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
class cot_record
  {     // data string from cot report
public:
   datetime          COT_DATE;                  //Date 
   double            COT_OPENINTEREST;          //Open Interest
   double            COT_LONGNONCOM;            //Longs of non-commercial traders
   double            COT_SHORTNONCOM;           //Shorts of non-commercial trader
   double            COT_LONGCOM;               //Longs of commercial traders
   double            COT_SHORTCOM;              //Shorts of commercial traders
   double            COT_LONGNONREPORTABLE;     //Longs of non-reportable traders
   double            COT_SHORTNONREPORTABLE;    //Shorts of non-reportable traders
   double            COT_LONGINDEX;             //Longs of index traders
   double            COT_SHORTINDEX;            //Shorts of index traders
   double            COT_LONGPRODUCER;          //Producer/Merchant/Processor/User
   double            COT_SHORTPRODUCER;
   double            COT_LONGSWAPDEALERS;       //Swap Dealers
   double            COT_SHORTSWAPDEALERS;
   double            COT_LONGMANAGEDMONEY;      //Managed Money
   double            COT_SHORTMANAGEDMONEY;
   double            COT_LONGOTHERREPORTABLES;  //Other Reportables
   double            COT_SHORTOTHERREPORTABLES;
   string            COT_ID;                    //Symbol indentifier
   string            COT_NAME;                  //Symbol name


   void copyvar(const cot_record &from)
     {  //copying contents (values of the variables) from one class to another
      COT_ID                    = from.COT_ID;
      COT_NAME                  = from.COT_NAME;
      COT_DATE                  = from.COT_DATE;
      COT_OPENINTEREST          = from.COT_OPENINTEREST;
      COT_LONGNONCOM            = from.COT_LONGNONCOM;
      COT_SHORTNONCOM           = from.COT_SHORTNONCOM;
      COT_LONGCOM               = from.COT_LONGCOM;
      COT_SHORTCOM              = from.COT_SHORTCOM;
      COT_LONGNONREPORTABLE     = from.COT_LONGNONREPORTABLE;
      COT_SHORTNONREPORTABLE    = from.COT_SHORTNONREPORTABLE;
      COT_LONGINDEX             = from.COT_LONGINDEX;
      COT_SHORTINDEX            = from.COT_SHORTINDEX;
      COT_LONGPRODUCER          = from.COT_LONGPRODUCER;
      COT_SHORTPRODUCER         = from.COT_SHORTPRODUCER;
      COT_LONGSWAPDEALERS       = from.COT_LONGSWAPDEALERS;
      COT_SHORTSWAPDEALERS      = from.COT_SHORTSWAPDEALERS;
      COT_LONGMANAGEDMONEY      = from.COT_LONGMANAGEDMONEY;
      COT_SHORTMANAGEDMONEY     = from.COT_SHORTMANAGEDMONEY;
      COT_LONGOTHERREPORTABLES  = from.COT_LONGOTHERREPORTABLES;
      COT_SHORTOTHERREPORTABLES = from.COT_SHORTOTHERREPORTABLES;
     };
  };
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
class CCFTCReport
  {        // cot report
private:
   cot_type_report   type;          //type of current report
   cot_record        data[];        //cot report data
   int               sizedata;      //number of records of the current report
   string            idsymbol;      //symbol identifier in cot, field "CFTC Contract Market Code"
   string            terminalsymbol;//symbol name in the client terminal
public:
   void ~CCFTCReport(){ sizedata=0; };  //
   bool              Init( cot_type_report passedtype, string sym="" ); //initialization of class instance
   bool              LoadFile( string filename );                       //load data from file
   string            Name();                                            //returns short report name
   bool              Type(cot_type_report passedtype);                  //sets type of report
   cot_type_report Type(){return(type);};                               //returns report type
   bool              TestGroup( cot_type_traders type_trader );         //check for the presence of specified group of traders
   bool              DefineIdSymbol();                                  //definition of id in the cot report
   bool              GetString( int handle, cot_record& arr );          //gets string (record) from csv file
   string            GetFieldCommaDelimited(string &str );              //gets field from csv string
   double            At(datetime time,cot_type_traders type_trader,cot_type_data typedata=netlongs);
   //gets data from report
  };
//+------------------------------------------------------------------+
//| Returns the specified value of COT report                        |
//| Input : time - date/time of the data requested                   |
//|         type_trader - type of group of traders                   |
//|         typedata - type of data requested                        |
//| Output: the value of COT report                                  |
//| Note  : it doesn't check for the presence of group of traders    | 
//+------------------------------------------------------------------+
double CCFTCReport::At(datetime time,cot_type_traders type_trader,cot_type_data typedata=netlongs)
  {
   if(sizedata==0)return(0); //report is empty
   int lastfoundindata=sizedata-1;
   while((lastfoundindata>0) && (time>data[lastfoundindata-1].COT_DATE))lastfoundindata--;
// the array element with the necessary data was found after the cycle
   switch(typedata)
     {
      case netlongs:
         if( type_trader == noncommercial) return( data[lastfoundindata].COT_LONGNONCOM-data[lastfoundindata].COT_SHORTNONCOM );
         if( type_trader == commercial   ) return( data[lastfoundindata].COT_LONGCOM-data[lastfoundindata].COT_SHORTCOM );
         if( type_trader == nonreportable) return( data[lastfoundindata].COT_LONGNONREPORTABLE-data[lastfoundindata].COT_SHORTNONREPORTABLE);
         if( type_trader == producer     ) return( data[lastfoundindata].COT_LONGPRODUCER-data[lastfoundindata].COT_SHORTPRODUCER );
         if( type_trader == swapdealers  ) return( data[lastfoundindata].COT_LONGSWAPDEALERS-data[lastfoundindata].COT_SHORTSWAPDEALERS );
         if( type_trader == managedmoney ) return( data[lastfoundindata].COT_LONGMANAGEDMONEY-data[lastfoundindata].COT_SHORTMANAGEDMONEY );
         if(type_trader==otherreportables) return(data[lastfoundindata].COT_LONGOTHERREPORTABLES-data[lastfoundindata].COT_SHORTOTHERREPORTABLES);
      case openinterest:
         return(data[lastfoundindata].COT_OPENINTEREST);
     }
   return(0);
  }
//+------------------------------------------------------------------+
//| Returns the field (string) from the csv string                   |
//| Input  : str - the string with fields                            |
//| Output : the string with field                                   |
//| Note   : The parameter is passed by reference, after the reading |
//|          after the reading it will be deleted                    |
//+------------------------------------------------------------------+
string CCFTCReport::GetFieldCommaDelimited(string &str)
  {
   string ret;
   int pos=StringFind(str,",",0);
   if(pos==-1)
     {
      ret = "";
      str = "";
        }else{
      ret = StringSubstr(str, 0, pos );
      str = StringSubstr(str, pos+1 );
     }
   return(ret);
  }
//+------------------------------------------------------------------+
//| Get the string (record) from the .csv file                       |
//| Input : handle - handle of the opened file                       |
//|         arr - target variable for the fields of record           |
//| Output: true - if successful                                     |
//|         false - in the case of error                             |
//| Note  : the function reads the text string from the file, parses |
//|         the string fields and places them to the variable arr    |
//+------------------------------------------------------------------+
bool CCFTCReport::GetString(int handle,cot_record &arr)
  {
   string allstring;
   allstring=FileReadString(handle);
   if(allstring=="")return(true);
   string s1=GetFieldCommaDelimited(allstring);  ///symbol name in form "EURO FX - CHICAGO MERCANTILE EXCHANGE"
   if(s1==("\""+"GOLD"))
     {
      s1=s1+","+GetFieldCommaDelimited(allstring);
     }
   if(s1==("\""+"CRUDE OIL"))
     {
      s1=s1+","+GetFieldCommaDelimited(allstring);
     }
   arr.COT_NAME=s1;
   s1 = GetFieldCommaDelimited(allstring);  // date format: 080108
   s1 = GetFieldCommaDelimited(allstring);  // date format: 2008-01-08 yyyy-mm-dd
   arr.COT_DATE=StringToTime(StringSubstr(s1,0,4)+"."+StringSubstr(s1,5,2)+"."+StringSubstr(s1,8,2));
//conversion to format yyyy.mm.dd
   s1=GetFieldCommaDelimited(allstring); //symbol identifier, format 099741
   string symb=CharToString('"');
//replace "
   if(StringSubstr(s1,0,1)==symb)
     {
      s1=StringSubstr(s1,1);
      int pos=  StringFind(s1,symb,0);
      if(pos!=-1)
        {
         s1=StringSubstr(s1,0,pos);
        }
     }
   arr.COT_ID=s1;  //ID 
   s1 = GetFieldCommaDelimited(allstring);  // CME
   s1 = GetFieldCommaDelimited(allstring);  // 00
   s1 = GetFieldCommaDelimited(allstring);  // 099
   s1 = GetFieldCommaDelimited(allstring);

   arr.COT_OPENINTEREST=StringToDouble(s1); //open interest

   if(Type()==COT || Type()==CIT)
     {
      arr.COT_LONGNONCOM = StringToDouble( GetFieldCommaDelimited(allstring) ); //longs of non-commercial
      arr.COT_SHORTNONCOM = StringToDouble( GetFieldCommaDelimited(allstring)); //shorts of non-commercial
      s1=GetFieldCommaDelimited(allstring);  // Noncommercial Positions-Spreading (All)
      arr.COT_LONGCOM=StringToDouble(GetFieldCommaDelimited(allstring)); //Commercial Positions-Long (All)
      arr.COT_SHORTCOM=StringToDouble(GetFieldCommaDelimited(allstring)); //Commercial Positions-Short (All)
      s1 = GetFieldCommaDelimited(allstring);  //  Total Reportable Positions-Long (All)
      s1 = GetFieldCommaDelimited(allstring);  //  Total Reportable Positions-Short (All)
      s1 = GetFieldCommaDelimited(allstring);  //  Nonreportable Positions-Long (All)
      arr.COT_LONGNONREPORTABLE=StringToDouble(s1);
      s1=GetFieldCommaDelimited(allstring);  //  Nonreportable Positions-Short (All)
      arr.COT_SHORTNONREPORTABLE=StringToDouble(s1);
      // if( type_report == CIT ){ //not implemented yet
      //   string s14 = GetFieldCommaDelimited(allstring); //CIT_Positions_Long_All
      //   arr.COT_LONGINDEX = StringToDouble( s14 );
      //   string s15 = GetFieldCommaDelimited(allstring); //CIT_Positions_Short_All
      //   arr.COT_SHORTINDEX = StringToDouble( s15 ); 
      // break;
     }
   if(Type()==DCOT)
     {
      arr.COT_LONGPRODUCER  = StringToDouble( GetFieldCommaDelimited(allstring) );
      arr.COT_SHORTPRODUCER = StringToDouble( GetFieldCommaDelimited(allstring) );
      arr.COT_LONGSWAPDEALERS=StringToDouble(GetFieldCommaDelimited(allstring));
      arr.COT_SHORTSWAPDEALERS=StringToDouble(GetFieldCommaDelimited(allstring));
      s1=GetFieldCommaDelimited(allstring); //Swap__Positions_Spread_All"
      arr.COT_LONGMANAGEDMONEY=StringToDouble(GetFieldCommaDelimited(allstring));
      arr.COT_SHORTMANAGEDMONEY=StringToDouble(GetFieldCommaDelimited(allstring));
      s1=GetFieldCommaDelimited(allstring); //M_Money_Positions_Spread_All
      arr.COT_LONGOTHERREPORTABLES=StringToDouble(GetFieldCommaDelimited(allstring));
      arr.COT_SHORTOTHERREPORTABLES=StringToDouble(GetFieldCommaDelimited(allstring));
      s1 = GetFieldCommaDelimited(allstring); //Other_Rept_Positions_Spread_All","
      s1 = GetFieldCommaDelimited(allstring); //Tot_Rept_Positions_Long_All
      s1 = GetFieldCommaDelimited(allstring); //Tot_Rept_Positions_Short_All
      arr.COT_LONGNONREPORTABLE=StringToDouble(GetFieldCommaDelimited(allstring));//
      arr.COT_SHORTNONREPORTABLE=StringToDouble(GetFieldCommaDelimited(allstring));//
     }
   return(true);
  }
//+------------------------------------------------------------------+
//| Loads the data of the report from file                           |
//| Input : filename - file name                                     |
//| Output: true if successful, false in the case of an error        |
//| Note  : None                                                     |
//+------------------------------------------------------------------+
bool CCFTCReport::LoadFile(string filename)
  {
   int handle;
   handle=FileOpen(filename,FILE_READ|FILE_ANSI);
   if(handle<1)
     {
      Print("Error opening file",filename," Error : ",ErrorDescription(GetLastError()));
      //SetUserError( COT_ERROR_LOAD_FILE );
      //may be error - file not found
      return(true);
     }
// the data is sorted in reverse order
// skipping all until the current symbol
   bool foundsymbol=false; bool flagexit=false;
   int cnt=0;
   while(!FileIsEnding(handle) && !flagexit)
     {  //proceed strings of the file
      cot_record res;
      GetString(handle,res); // get string
      if(StringFind(res.COT_ID,idsymbol)!=-1)
        {  //found the string with the symbol identifier
         sizedata++;
         ArrayResize(data,sizedata);
         data[sizedata-1].copyvar(res);
         foundsymbol=true;
           }else{
         if(foundsymbol)
           {
            flagexit=true;
           }
        }
     }
   FileClose(handle);
   return(true);
  }
//+------------------------------------------------------------------+
//| Returns short name of report                                     |
//| Input : None                                                     |
//| Output: Name of report                                           |
//| Note  : None                                                     |
//+------------------------------------------------------------------+
string CCFTCReport::Name()
  {
   if( Type() == COT )return("COT");
   if( Type() == DCOT)return("DCOT");
   if( Type() == CIT )return("CIT");
   return("Error in type of report");

  }
//+------------------------------------------------------------------+
//| Sets type of report                                              |
//| Input : passedtype - type of report                              |
//| Output: true - if successful                                     |
//|         false - in the case of an error                          |
//| Note  : None                                                     |
//+------------------------------------------------------------------+
bool CCFTCReport::Type(cot_type_report passedtype)
  {
   if(passedtype==CIT)
     {
      SetUserError(COT_ERROR_CIT_NOT_SUPPORTED);
      return(false);
     }
   type=passedtype;
   return(true);
  }
//+------------------------------------------------------------------+
//| Determines the identifier in COT report                          |
//| Input : symb - the current symbol name in the client terminal    |
//| Output: true - is the identification was successful              |
//|         false- error in identification                           |
//| Note  : the function is used for the identification of symbol    |
//|         of the client terminal and symbol of COT report          |
//|         (field "CFTC Contract Market Code").                     |
//|         The error code is set by the function SetUserError()     |
//+------------------------------------------------------------------+
bool CCFTCReport::DefineIdSymbol()
  {
   idsymbol="";
   if(terminalsymbol=="USDLFX" || terminalsymbol=="DX_CONT" || StringFind(terminalsymbol,"DX")==0)idsymbol="098662";
//U.S. DOLLAR INDEX - CE FUTURES U.S.
   if( terminalsymbol == "FC_CONT" || StringFind(  terminalsymbol,  "FC")== 0)idsymbol = "061641";  //FEEDER CATTLE 
   if( terminalsymbol == "LC_CONT" || StringFind(  terminalsymbol,  "LC")== 0)idsymbol = "057642";  //LIVE CATTLE 
   if( terminalsymbol == "QM_CONT" || StringFind(  terminalsymbol, "QM")==0  )idsymbol = "067651";
//CRUDE OIL, LIGHT SWEET - NEW YORK MERCANTILE EXCHANGE
   if( terminalsymbol == "KC_CONT" || StringFind( terminalsymbol, "KC")  == 0 )idsymbol = "083731"; //coffee
   if( terminalsymbol == "ZW_CONT" || StringFind( terminalsymbol, "ZW") == 0  )idsymbol = "001602"; //wheat
   if( terminalsymbol == "ZG_CONT" || StringFind( terminalsymbol, "ZG") == 0  )idsymbol = "088606"; //gold
   if( terminalsymbol == "ES_CONT" || StringFind( terminalsymbol, "ES") == 0 || terminalsymbol == "SPX")idsymbol = "13874A";
// E-MINI S&P 500 
   if(terminalsymbol=="GC_CONT" || StringFind(terminalsymbol,"GC")==0 || terminalsymbol=="XAUUSD" || terminalsymbol=="GOLD")idsymbol="088691";
//  comex
   if(terminalsymbol=="XAGUSD" || StringFind(terminalsymbol,"SI")==0)idsymbol="084691"; //  comex
   if( StringFind( terminalsymbol,"EURUSD" ) >=0 || StringFind( terminalsymbol,"6E") == 0  )idsymbol = "099741";
   if( StringFind( terminalsymbol,"USDCHF" ) >=0 || StringFind( terminalsymbol,"6S") == 0  )idsymbol = "092741";
   if( StringFind( terminalsymbol,"USDCAD")  >=0 || StringFind( terminalsymbol,"6C") == 0  )idsymbol = "090741";
   if( StringFind( terminalsymbol,"GBPUSD")  >=0 || StringFind( terminalsymbol,"6B") == 0  )idsymbol = "096742";
   if( StringFind( terminalsymbol,"USDJPY")  >=0 || StringFind( terminalsymbol,"6J") == 0  )idsymbol = "097741";
   if( StringFind( terminalsymbol,"NZDUSD" ) >=0 || StringFind( terminalsymbol,"6N") == 0  )idsymbol = "112741";
   if( StringFind( terminalsymbol,"AUDUSD" ) >=0 || StringFind( terminalsymbol,"6A") == 0  )idsymbol = "232741";
   if(terminalsymbol=="HE_CONT" || StringFind(terminalsymbol,"HE")==0)idsymbol="054642"; // 
   if(StringFind(terminalsymbol,"ZC")==0)idsymbol="002602";
//CORN - CHICAGO BOARD OF TRADE
   if(StringFind(terminalsymbol,"ZS")==0)idsymbol="005602";
//SOYBEANS - CHICAGO BOARD OF TRADE
   if(StringFind(terminalsymbol,"ZL")==0)idsymbol="007601";
//SOYBEAN OIL - CHICAGO BOARD OF TRADE" 
   if(StringFind(terminalsymbol,"ZM")==0)idsymbol="026603";
//SOYBEAN MEAL - CHICAGO BOARD OF TRADE
   if(StringFind(terminalsymbol,"SB")==0)idsymbol="080732";
//SUGAR NO. 11 - ICE FUTURES U.S.
   if(StringFind(terminalsymbol,"LB")==0)idsymbol="058643";
//RANDOM LENGTH LUMBER - CHICAGO MERCANTILE EXCHANGE
   if(StringFind(terminalsymbol,"QG")==0)idsymbol="023651";
//NATURAL GAS - NEW YORK MERCANTILE EXCHANGE
   if(StringFind(terminalsymbol,"ZN")==0)idsymbol="043602";
//"U.S. TREASURY Notes 10 years
   if(StringFind(terminalsymbol,"ZB")==0)idsymbol="020601";
//"U.S. TREASURY BONDS - CHICAGO BOARD OF TRADE" ,0
   if(StringFind(terminalsymbol,"ZI")==0)idsymbol="084691";
//SILVER - COMMODITY EXCHANGE INC
   if(terminalsymbol=="USDRUB" || terminalsymbol=="USDRUR")idsymbol="089741";
//RUSSIAN RUBLE - CHICAGO MERCANTILE EXCHANGE
   if(StringFind(terminalsymbol,"HG")==0)idsymbol="085692";
//COPPER-GRADE #1 - COMMODITY EXCHANGE INC.
   if(StringFind(terminalsymbol,"PA")==0)idsymbol="075651";
//PALLADIUM - NEW YORK MERCANTILE EXCHANGE
   if(StringFind(terminalsymbol,"PL")==0)idsymbol="076651";
//PLATINUM - NEW YORK MERCANTILE EXCHANGE
   if(StringFind(terminalsymbol,"ZO")==0)idsymbol="004603";
//OATS - CHICAGO BOARD OF TRADE
   if(StringFind(terminalsymbol,"CT")==0)idsymbol="033661";
//COTTON NO. 2 - ICE FUTURES U.S.
   if(StringFind(terminalsymbol,"JO")==0)idsymbol="040701";
//FRZN CONCENTRATED ORANGE JUICE - ICE FUTURES U.S." 
   if(StringFind(terminalsymbol,"CC")==0)idsymbol="073732";
//COCOA - ICE FUTURES U.S.
   if( StringFind( terminalsymbol, "NQ") == 0  )idsymbol = "209741";
   if( StringFind( terminalsymbol, "ZR") ==0   )idsymbol = "039601";
//ROUGH RICE - CHICAGO BOARD OF TRADE" 
   if(StringFind(terminalsymbol,"ZQ")==0)idsymbol="045601";
//30-DAY FEDERAL FUNDS - CHICAGO BOARD OF TRADE
   if(StringFind(terminalsymbol,"ES")==0)idsymbol="13874A";
//E-MINI S&P 500 STOCK INDEX - CHICAGO MERCANTLE EXCHANGE

   if(StringFind(terminalsymbol,"XRB")==0)idsymbol="111659";

   if(idsymbol=="")
     {
      SetUserError(COT_ERROR_SYMBOL_NOT_SUPPORTED);
      return(false);
     }
   return(true);
  }
//+---------------------------------------------------------------------+
//| Initialization of class instance                                    |
//| Input : None                                                        |
//| Output: true - if successful                                        |
//|         false- in the case of an error                              |
//| Note  : It loads the data from the report files to the array data[] |
//|         The error code is set by the function SetUserError()        |
//+------------------------------------------------------------------+
bool CCFTCReport::Init(cot_type_report passedtype,string sym="")
  {
   if(!Type(passedtype))
     {
      return(false);
     }
   if(sym=="")terminalsymbol=Symbol();
   else  terminalsymbol=sym;
   sizedata=0;
   if(!DefineIdSymbol())
     { // return if symbol hasn't been determined
      return(false);
     }

   int beginyear;

   beginyear=COT_FIRST_YEAR_DATA;

   if( Type() == DCOT )beginyear= DCOT_FIRST_YEAR_DATA;
   if( Type() == CIT  )beginyear=CIT_FIRST_YEAR_DATA;
   int endyear=TimeYear(GetWindowsLocalTime());  //2010 may be
   for(int cury=endyear; cury>=beginyear; cury--)
     {
      string pref=COT_STORE_FOLDER+string(cury);
      string filename=pref+"\\annual.txt";
      if(Type()==COT && cury==2007)
        {
         filename=pref+"\\annual_2007.txt";
        }
      //if( Type()==CIT ){
      //  filename = "deacot"+cury+"\annualci.txt"; 
      //  if( cury == 2006 )filename = "deacot"+cury+"\annualci2006.txt";  
      //}
      if(Type()==DCOT)filename=pref+"\\f_year.txt";
      LoadFile(filename);
     }
   return(true);
  }
//+------------------------------------------------------------------+
//| Check for the presence of group of traders in the report         |
//| Input : type_trader - type of group of traders                   |
//| Output: true - if group is present in the report                 |
//|         false- if group is absent in the report                  |
//| Note  : None                                                     |
//+------------------------------------------------------------------+
bool CCFTCReport::TestGroup(cot_type_traders type_trader)
  {
   if(Type()==COT && (
      type_trader==noncommercial || type_trader==commercial || type_trader==nonreportable)
      )
     {
      return(true);
     }
   if(Type()==CIT && (
      type_trader==noncommercial || type_trader==commercial || type_trader==nonreportable || type_trader==index)
      )
     {
      return(true);
     }

   if(Type()==DCOT && (
      type_trader == producer         || type_trader == swapdealers || type_trader == managedmoney ||
      type_trader == otherreportables || type_trader == nonreportable )

      )
     {
      return(true);
     }
   return(false);
  }
//+------------------------------------------------------------------+
//| Returns the string description of group of traders               |
//| Input  : type_trader - type of group of traders                  |
//| Output : Description of group of traders                         |
//| Note   : None                                                    |
//+------------------------------------------------------------------+
string GetStringTypeTrader(cot_type_traders type_trader)
  {
   if( type_trader == noncommercial    )return( "Non-commercial"   );
   if( type_trader == commercial       )return( "Commercial"       );
   if( type_trader == nonreportable    )return( "Non-reportable"   );
   if( type_trader == producer         )return( "Producers"        );
   if( type_trader == swapdealers      )return( "Swap Dealers"     );
   if( type_trader == managedmoney     )return( "Managed Money"   );
   if( type_trader == otherreportables )return( "Other reportable" );
   return("Unknown group of traders");
  }
//+------------------------------------------------------------------+
//| Returns string description of the type of the indicator          |
//| Input : type_data - type of the indicator                        |
//| Output: string description                                       |
//| Note  : None                                                     |
//+------------------------------------------------------------------+
string GetCotIndicatorName(cot_type_data type_data)
  {
   string indname="Error in type of the indicator";
   if( type_data   == netlongs     )indname = "Net longs";
   if( type_data   == netlongsrate )indname = "Net longs rate";
   if( type_data   == willams_index )indname = "Williams Index";
   if( type_data   == openinterest  )indname = "Open Interest";
   return(indname);
  }
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
enum cot_type_statistics
  {  //types of statistics
   speculantsign,           //Sign of net longs of non-commercial traders
   speculantchange,         //Volume changes in longs of non-commercial traders
   willamszones             //Overbought/oversold zones of Willams Index
  };
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
class CCOTStatistic
  {       //class for the calculation of statisical data of COT signals
private:

   int               bars;                          // number of weekly bars in the history of the symbol,
   string            symbol;                        // symbol name in the client terminal
   double            netlongsspeculantsdata[];      // data for net longs of the speculators
   double            willamsindexspeculantsdata[];  // data for Willams index of net longs of the speculators
   cot_type_statistics liststatistic[];             // list of statistics
public:
   //Initialization and setup
   void              Init(string symbol_passed,cot_type_statistics &listpassed[]);
   //Get the forecast for the direction
   int               GetCotSignalDirection(int ibar,cot_type_statistics type_stat);
   bool              Get(double &probably[]);       // Returns statistics
  };
//+------------------------------------------------------------------+
//| Initialization and setting                                       |
//| Input : symbol_passed - symbol in the terminal                   |
//|         listpassed[] - list of signals for the statistics        |
//| Output : None                                                    |
//| Note   : None                                                    |
//+------------------------------------------------------------------+
void CCOTStatistic::Init(string symbol_passed,cot_type_statistics &listpassed[])
  {
   ArrayResize(liststatistic,0);
   ArrayCopy(liststatistic,listpassed);
   bars=150;
   symbol=symbol_passed;

   ArraySetAsSeries( netlongsspeculantsdata    , true ); // set as timeseries
   ArraySetAsSeries( willamsindexspeculantsdata, true ); // set as timeseries
  }
//+------------------------------------------------------------------+
//| Returns forecast based on the signal of cot indicator            |
//| Input : ibar - number of weekly bat                              |
//|         type_stat - type of a cot signal                         |
//| Output: >0 - bull forecast, <0 - bear forecast, 0 - undefined    |
//| Note  : None                                                     |
//+------------------------------------------------------------------+
int CCOTStatistic::GetCotSignalDirection(int ibar,cot_type_statistics type_stat)
  {
   int zoneup=75; int zonedown=25;
   switch(type_stat)
     {
      case speculantsign  : return( (netlongsspeculantsdata[ibar] > 0)?1:-1 );
      case speculantchange: return( (netlongsspeculantsdata[ibar] > netlongsspeculantsdata[ibar+1])>0?1:-1);
      case willamszones   :
         if(willamsindexspeculantsdata[ibar]>zonedown && willamsindexspeculantsdata[ibar]<zoneup)return(0);
         // forecast absent
         if(willamsindexspeculantsdata[ibar]<zonedown)return(1);
         return(-1);
     }
   return(0);
  }
//+------------------------------------------------------------------+
//| Calculates statistics of COT reports                             |
//| Input : probably - target array for the probabilities            |
//| Output: true if successful/false in the case of error            |
//| Note  : None                                                     |
//+------------------------------------------------------------------+
bool CCOTStatistic::Get(double &probably[])
  {

// creation of necessary indicators, request for the data, calculate of the statistics
   ArrayResize(probably,ArraySize(liststatistic));
   ArrayInitialize(probably,0.);

   if(!LoadDataFromCOTIndicator(symbol,PERIOD_W1,noncommercial,netlongs,COT,netlongsspeculantsdata,bars))return(false);
// we have got the data - net longs of the speculators
   if(!LoadDataFromCOTIndicator(symbol,PERIOD_W1,noncommercial,willams_index,COT,willamsindexspeculantsdata,bars))return(false);
// we have got the data - Williams Index of the speculators
   Print("Calculation of statistics for the symbol ",symbol);

   for(int istat=0; istat<ArraySize(liststatistic); istat++)
     {
      int cntsignals = 0;
      int cntsucsess = 0;
      for(int i=bars-1; i>=0; i--)
        {
         double colorcandle=iClose(symbol,PERIOD_W1,i)-iOpen(symbol,PERIOD_W1,i);
         if(symbol=="USDCHF" || symbol=="USDCAD" || symbol=="USDJPY") colorcandle=-colorcandle;
         double cotdirection=GetCotSignalDirection(i,liststatistic[istat]);
         if(cotdirection==0)continue; //no signal
         cntsignals++;
         if(cotdirection*colorcandle>0) cntsucsess++; //color and direction are same
        }

      if(cntsignals!=0) probably[istat]=1.*cntsucsess/cntsignals;
      Print("Calculation complete for ",symbol,GetStringTypeStatistic(liststatistic[istat])," the correct forecast probability=",DoubleToString(probably[istat],2));
     }
   return(true);
  }
//+------------------------------------------------------------------+
//| The function loads the data from COT indicator                   |
//| Input : see comments in the parameters of the function           |
//| Output: true/false, depending on success                         |
//| Note  : None                                                     |
//+------------------------------------------------------------------+
bool LoadDataFromCOTIndicator( string           symbol,       //symbol name
                              ENUM_TIMEFRAMES  timeframe,     //timeframe
                              cot_type_traders type_trader,   //type of group of traders
                              cot_type_data    type_data  ,   //type of the indicator
                              cot_type_report  type_report,   //type of report
                              double&          loadto[],      //target array for output data
                              int              nbars   )      //number of bars requested
  {

   int cothandle;
   nbars+=10; //for reserve
   cothandle=iCustom(symbol,timeframe,"cot",type_trader,type_data,type_report);
   if(cothandle==INVALID_HANDLE)
     {
      Print("Error in indicator creation for the symbol ",symbol);
      return(false);
     }

   int n=BarsSinh(symbol,timeframe);
   if(n<nbars)
     {
      datetime tm=GetWindowsLocalTime()-nbars*7*24*60*60;
      Print("Loading history for ",symbol," on the date ",tm);
      CheckLoadHistory(symbol,timeframe,tm);
      n=BarsSinh(symbol,timeframe);
      if(n<nbars)
        {
         Print("Not enough history for "+symbol," total ",n," bars");
         return(false);
        }
     }
   if(!WaitForCalculcateIndicator(cothandle))
     {
      Print("Timeout in the calculation of COT indicator ");
      return(false);
     }
   int res= CopyBuffer(cothandle,0,0,nbars,loadto);
   if(res!=nbars)
     {
      Print("Error in obtaining the data, ",MyErrorDescription(_LastError));
      return(false);
     }
   return(true);
  }
//+------------------------------------------------------------------+
//| Returns string description of the type of statistics             |
//| Input : type of statistics                                       |
//| Output: String description                                       |
//| Note  : None                                                     |
//+------------------------------------------------------------------+
string GetStringTypeStatistic(cot_type_statistics type)
  {
   if( type == speculantsign   )return("Sign non-comm" );
   if( type == speculantchange )return("Vol. changes"  );
   if( type == willamszones    )return("Williams zones");
   return("unknown type of statisitcs");
  }
//+------------------------------------------------------------------+
//| Wait for complete of the indicator's calculations                |
//| Input  : handle of the indicator                                 |
//| Output : true - calculation complete, false - timeout 40 seconds |
//| Note   : None                                                    |
//+------------------------------------------------------------------+
bool WaitForCalculcateIndicator(int handle)
  {
   int cnt=0;
   while(true)
     {
      int clc=BarsCalculated(handle);
      //Print("bars calculated=", clc );
      if(clc<=0)
        {
         Sleep(500);
        }
      else break;
      cnt++;
      if(cnt>500*80/1000)
        {//timeout 40 seconds
         //Print("time out calculating");
         return(false);
        }
     }
   return(true);
  }
//+------------------------------------------------------------------+
