//+------------------------------------------------------------------+
//|                                                     PipeLine.mq5 |
//|                                  Copyright 2025, MetaQuotes Ltd. |
//|                                             https://www.mql5.com |
//+------------------------------------------------------------------+
#property copyright "Copyright 2025, MetaQuotes Ltd."
#property link      "https://www.mql5.com"
#property version   "1.00"

//+------------------------------------------------------------------+
//| Preprocessing Pipeline for MQL5                                  |
//+------------------------------------------------------------------+

// Constants
const float EPSILON = 1e-5f;

// Enum for preprocessing step types
enum ENUM_PREPROCESSOR_TYPE
{  PREPROCESSOR_IMPUTE_MEDIAN,
   PREPROCESSOR_IMPUTE_MODE,
   PREPROCESSOR_STANDARD_SCALER,
   PREPROCESSOR_ROBUST_SCALER,
   PREPROCESSOR_MINMAX_SCALER,
   PREPROCESSOR_ONEHOT_ENCODER
};

// Structure to store preprocessing step
struct SPreprocessorStep
{  ENUM_PREPROCESSOR_TYPE type;
   int                    column; // For column-specific preprocessors
};

// Base interface for preprocessors
interface IPreprocessor
{  bool Fit(matrixf &data);
   bool Transform(matrixf &data, matrixf &out);
   bool FitTransform(matrixf &data, matrixf &out);
};

// Utility class for array operations
class CVectorUtils
{
public:
   //+------------------------------------------------------------------+
   //| Find index of Key in sorted Sranges array using binary search    |
   //+------------------------------------------------------------------+
   static int BinarySearch(vectorf &V, float Key)
   {  int left = 0;
      int right = int(V.Size()) - 1;
      int _i = -1; // Return -1 if not found
      while(left <= right)
      {  int mid = (left + right) >> 1; // Equivalent to (left + right) / 2
         if(V[mid] == Key)
         {  _i = mid;
            break;
         }
         else if(V[mid] < Key)
            left = mid + 1;
         else
            right = mid - 1;
      }
      return _i;
   }

   // Custom quantile-based mode function
   static float QuantileMode(vectorf &data, float quantile)
   {  // Input validation
      if(data.Size() == 0 || quantile <= 0.0f || quantile > 1.0f)
         return FLT_MIN;
      // Filter out invalid values
      if(data.HasNan())
         return FLT_MIN;
      // Sort data for quantile computation
      Sort(data);
      // Calculate number of bins and edges
      int num_bins = (int)(1.0 / quantile);
      if(num_bins < 1)
         num_bins = 1;
      vectorf edges(num_bins + 1);
      for(int i = 0; i <= num_bins; i++)
         edges[i] = data.Quantile((float)i / num_bins);
      // Count frequencies in bins
      vectorf counts(num_bins);
      counts.Fill(0.0);
      for(ulong i = 0; i < data.Size(); i++)
      {  float value = data[i];
         // Find bin: value in [edges[j], edges[j+1])
         for(int j = 0; j < num_bins; j++)
         {  if(value >= edges[j] && (j == num_bins - 1 || value < edges[j + 1]))
            {  counts[j]++;
               break;
            }
         }
      }
      // Find bin with maximum count
      int max_count = int(counts.Max());
      int mode_bin = -1;
      for(int j = 0; j < num_bins; j++)
      {  if(counts[j] == max_count)
         {  mode_bin = j;
            break;
         }
      }
      // Return midpoint of mode bin
      if(mode_bin < 0)
         return FLT_MIN;
      return (edges[mode_bin] + edges[mode_bin + 1]) / 2.0f;
   }

// Sorts an input vectorf in ascending order, returning a new sorted vectorf
   static void Sort(vectorf &data)
   {  // Input validation
      if(data.Size() == 0)
         return; // Return empty vectorf for empty input
      // Create a copy to avoid modifying the input
      if(data.HasNan())
         return; // Return empty vectorf if all values are invalid
      // Sort using MQL5's built-in vectorf.Sort() (ascending)
      float sorter[];
      data.Swap(sorter);
      ArraySort(sorter);
      data.Swap(sorter);
   }
};

// Standard Scaler
class CStandardScaler : public IPreprocessor
{
private:
   float m_means[];
   float m_stds[];
   bool   m_is_fitted;

public:
   CStandardScaler() : m_is_fitted(false) {}

   bool Fit(matrixf &data)
   {  int rows = int(data.Rows());
      int cols = int(data.Cols());
      ArrayResize(m_means, cols);
      ArrayResize(m_stds, cols);
      for(int j = 0; j < cols; j++)
      {  vectorf column(rows);
         for(int i = 0; i < rows; i++) column[i] = data[i][j];
         m_means[j] = column.Mean();
         m_stds[j] = column.Std();
         if(m_stds[j] == 0.0) m_stds[j] = EPSILON;
      }
      m_is_fitted = true;
      return true;
   }

   bool Transform(matrixf &data, matrixf &out)
   {  if(!m_is_fitted) return false;
      int rows = int(data.Rows());
      int cols = int(data.Cols());
      out.Init(rows, cols);
      for(int j = 0; j < cols; j++)
         for(int i = 0; i < rows; i++)
            out[i][j] = (!MathIsValidNumber(data[i][j]) ? FLT_MIN : (data[i][j] - m_means[j]) / fmax(EPSILON, m_stds[j]));
      return true;
   }

   bool FitTransform(matrixf &data, matrixf &out)
   {  if(!Fit(data)) return false;
      return Transform(data, out);
   }
};

// Robust Scaler
class CRobustScaler : public IPreprocessor
{
private:
   float m_medians[];
   float m_iqrs[];
   bool   m_is_fitted;

public:
   CRobustScaler() : m_is_fitted(false) {}

   bool Fit(matrixf &data)
   {  int rows = int(data.Rows());
      int cols = int(data.Cols());
      ArrayResize(m_medians, cols);
      ArrayResize(m_iqrs, cols);
      for(int j = 0; j < cols; j++)
      {  vectorf column(rows);
         for(int i = 0; i < rows; i++) column[i] = data[i][j];
         m_medians[j] = column.Median();
         float q25 = column.Quantile(0.25);
         float q75 = column.Quantile(0.75);
         m_iqrs[j] = q75 - q25;
         if(m_iqrs[j] == 0.0) m_iqrs[j] = EPSILON;
      }
      m_is_fitted = true;
      return true;
   }

   bool Transform(matrixf &data, matrixf &out)
   {  if(!m_is_fitted) return false;
      int rows = int(data.Rows());
      int cols = int(data.Cols());
      out.Init(rows, cols);
      for(int j = 0; j < cols; j++)
         for(int i = 0; i < rows; i++)
            out[i][j] = (!MathIsValidNumber(data[i][j]) ? FLT_MIN : (data[i][j] - m_medians[j]) / fmax(EPSILON, m_iqrs[j]));
      return true;
   }

   bool FitTransform(matrixf &data, matrixf &out)
   {  if(!Fit(data)) return false;
      return Transform(data, out);
   }
};

// MinMax Scaler
class CMinMaxScaler : public IPreprocessor
{
private:
   vectorf m_mins;
   vectorf m_maxs;
   float m_new_min;
   float m_new_max;
   bool   m_is_fitted;

public:
   CMinMaxScaler(float new_min = 0.0, float new_max = 1.0) : m_new_min(new_min), m_new_max(new_max), m_is_fitted(false) {}

   bool Fit(matrixf &data)
   {  int rows = int(data.Rows());
      int cols = int(data.Cols());
      m_mins.Init(cols);
      m_maxs.Init(cols);
      for(int j = 0; j < cols; j++)
      {  vectorf column(rows);
         for(int i = 0; i < rows; i++) column[i] = data[i][j];
         m_mins[j] = column.Min();
         m_maxs[j] = column.Max();
         if(m_maxs[j] - m_mins[j] == 0.0) m_maxs[j] += EPSILON;
      }
      m_is_fitted = true;
      return true;
   }

   bool Transform(matrixf &data, matrixf &out)
   {  if(!m_is_fitted) return false;
      int rows = int(data.Rows());
      int cols = int(data.Cols());
      out.Init(rows, cols);
      for(int j = 0; j < cols; j++)
         for(int i = 0; i < rows; i++)
         {  if(!MathIsValidNumber(data[i][j])) out[i][j] = FLT_MIN;
            else
            {  float scale = (m_new_max - m_new_min) / fmax(EPSILON, m_maxs[j] - m_mins[j]);
               out[i][j] = (data[i][j] - m_mins[j]) * scale + m_new_min;
            }
         }
      return true;
   }

   bool FitTransform(matrixf &data, matrixf &out)
   {  if(!Fit(data)) return false;
      return Transform(data, out);
   }
};

// One-Hot Encoder
class COneHotEncoder : public IPreprocessor
{
private:
   int    m_column;
   vectorf m_categories;
   bool   m_is_fitted;

public:
   COneHotEncoder(int column) : m_column(column), m_is_fitted(false) {}

   bool Fit(matrixf &data)
   {  int rows = int(data.Rows());
      vectorf values;
      int unique = 0;
      for(int i = 0; i < rows; i++)
      {  if(!MathIsValidNumber(data[i][m_column])) continue;
         int idx = CVectorUtils::BinarySearch(values, data[i][m_column]);
         if(idx == -1)
         {  values.Resize(unique + 1);
            values[unique] = data[i][m_column];
            unique++;
         }
      }
      values.Swap(m_categories);
      //ArrayCopy(m_categories, values);
      m_is_fitted = true;
      return true;
   }

   bool Transform(matrixf &data, matrixf &out)
   {  if(!m_is_fitted) return false;
      int rows = int(data.Rows());
      int cols = int(data.Cols());
      int cat_count = int(m_categories.Size());
      if(data.Cols() == cols - 1 + cat_count) return false;
      out.Resize( rows, cols - 1 + cat_count);
      out.Fill(0.0);
      for(int i = 0; i < rows; i++)
      {  int out_col = 0;
         for(int j = 0; j < cols; j++)
         {  if(j == m_column) continue;
            out[i][out_col] = data[i][j];
            out_col++;
         }
         for(int k = 0; k < cat_count; k++)
            if(data[i][m_column] == m_categories[k])
            {  out[i][out_col + k] = 1.0;
               break;
            }
      }
      m_is_fitted = true;
      return true;
   }

   bool FitTransform(matrixf &data, matrixf &out)
   {  if(!Fit(data)) return false;
      return Transform(data, out);
   }
};

// Impute Median
class CImputeMedian : public IPreprocessor
{
private:
   int    m_column;
   float m_median;
   bool   m_is_fitted;

public:
   CImputeMedian(int column) : m_column(column), m_is_fitted(false) {}

   bool Fit(matrixf &data)
   {  int rows = int(data.Rows());
      vectorf column(rows);
      for(int i = 0; i < rows; i++) column[i] = data[i][m_column];
      m_median = column.Median();
      m_is_fitted = true;
      return true;
   }

   bool Transform(matrixf &data, matrixf &out)
   {  if(!m_is_fitted) return false;
      int rows = int(data.Rows());
      int cols = int(data.Cols());
      out.Init(rows, cols);
      for(int i = 0; i < rows; i++)
         for(int j = 0; j < cols; j++)
            out[i][j] = (j == m_column && !MathIsValidNumber(data[i][j])) ? m_median : data[i][j];
      return true;
   }

   bool FitTransform(matrixf &data, matrixf &out)
   {  if(!Fit(data)) return false;
      return Transform(data, out);
   }
};

// Impute Mode
class CImputeMode : public IPreprocessor
{
private:
   int    m_column;
   float m_mode;
   bool   m_is_fitted;

public:
   CImputeMode(int column) : m_column(column), m_is_fitted(false) {}

   bool Fit(matrixf &data)
   {  int rows = int(data.Rows());
      vectorf column(rows);
      for(int i = 0; i < rows; i++) column[i] = data[i][m_column];
      m_mode = CVectorUtils::QuantileMode(column, 0.02f);
      m_is_fitted = true;
      return true;
   }

   bool Transform(matrixf &data, matrixf &out)
   {  if(!m_is_fitted) return false;
      int rows = int(data.Rows());
      int cols = int(data.Cols());
      out.Init(rows, cols);
      for(int i = 0; i < rows; i++)
         for(int j = 0; j < cols; j++)
            out[i][j] = (j == m_column && !MathIsValidNumber(data[i][j])) ? m_mode : data[i][j];
      return true;
   }

   bool FitTransform(matrixf &data, matrixf &out)
   {  if(!Fit(data)) return false;
      return Transform(data, out);
   }
};

// Preprocessing Pipeline
class CPreprocessingPipeline
{
private:
   SPreprocessorStep m_steps[];
   IPreprocessor    *m_preprocessors[];
   int              m_step_count;

public:
   CPreprocessingPipeline() : m_step_count(0) {}

   ~CPreprocessingPipeline()
   {  for(int i = 0; i < m_step_count; i++)
         delete m_preprocessors[i];
   }

   void AddImputeMedian(int column)
   {  ArrayResize(m_steps, m_step_count + 1);
      ArrayResize(m_preprocessors, m_step_count + 1);
      m_steps[m_step_count].type = PREPROCESSOR_IMPUTE_MEDIAN;
      m_steps[m_step_count].column = column;
      m_preprocessors[m_step_count] = new CImputeMedian(column);
      m_step_count++;
   }

   void AddImputeMode(int column)
   {  ArrayResize(m_steps, m_step_count + 1);
      ArrayResize(m_preprocessors, m_step_count + 1);
      m_steps[m_step_count].type = PREPROCESSOR_IMPUTE_MODE;
      m_steps[m_step_count].column = column;
      m_preprocessors[m_step_count] = new CImputeMode(column);
      m_step_count++;
   }

   void AddStandardScaler()
   {  ArrayResize(m_steps, m_step_count + 1);
      ArrayResize(m_preprocessors, m_step_count + 1);
      m_steps[m_step_count].type = PREPROCESSOR_STANDARD_SCALER;
      m_steps[m_step_count].column = -1;
      m_preprocessors[m_step_count] = new CStandardScaler();
      m_step_count++;
   }

   void AddRobustScaler()
   {  ArrayResize(m_steps, m_step_count + 1);
      ArrayResize(m_preprocessors, m_step_count + 1);
      m_steps[m_step_count].type = PREPROCESSOR_ROBUST_SCALER;
      m_steps[m_step_count].column = -1;
      m_preprocessors[m_step_count] = new CRobustScaler();
      m_step_count++;
   }

   void AddMinMaxScaler(float new_min = 0.0, float new_max = 1.0)
   {  ArrayResize(m_steps, m_step_count + 1);
      ArrayResize(m_preprocessors, m_step_count + 1);
      m_steps[m_step_count].type = PREPROCESSOR_MINMAX_SCALER;
      m_steps[m_step_count].column = -1;
      m_preprocessors[m_step_count] = new CMinMaxScaler(new_min, new_max);
      m_step_count++;
   }

   void AddOneHotEncoder(int column)
   {  ArrayResize(m_steps, m_step_count + 1);
      ArrayResize(m_preprocessors, m_step_count + 1);
      m_steps[m_step_count].type = PREPROCESSOR_ONEHOT_ENCODER;
      m_steps[m_step_count].column = column;
      m_preprocessors[m_step_count] = new COneHotEncoder(column);
      m_step_count++;
   }

   bool FitPipeline(matrixf &data)
   {  matrixf temp;
      temp.Copy(data);
      for(int i = 0; i < m_step_count; i++)
      {  matrixf out;
         if(!m_preprocessors[i].Fit(temp)) return false;
         if(!m_preprocessors[i].Transform(temp, out)) return false;
         temp.Copy(out);
      }
      return true;
   }

   bool TransformPipeline(matrixf &data, matrixf &out)
   {  out.Copy(data);
      for(int i = 0; i < m_step_count; i++)
      {  matrixf temp;
         if(!m_preprocessors[i].Transform(out, temp)) return false;
         out.Copy(temp);
      }
      return true;
   }

   bool FitTransformPipeline(matrixf &data, matrixf &out)
   {  if(!FitPipeline(data)) return false;
      return TransformPipeline(data, out);
   }
};
//+------------------------------------------------------------------+
