//+------------------------------------------------------------------+
//|                                            Nasdaq_NVDA_Coint.mq5 |
//|                                  Copyright 2025, MetaQuotes Ltd. |
//|                                             https://www.mql5.com |
//+------------------------------------------------------------------+
//+------------------------------------------------------------------+
//| Expert Advisor - Cointegration Statistical Arbitrage             |
//| Assets: MU, NVDA, MPWR, MCHP                                     |
//| Strategy: Mean-reversion on Johansen cointegration portfolio      |
//+------------------------------------------------------------------+
#include <Trade\Trade.mqh>
#include <Math\Stat\Math.mqh> // For statistical functions
//#include <Trade\SymbolInfo.mqh> // For SymbolInfoTick

CTrade trade;

// Input parameters
input double EntryThreshold = 2.0;    // Entry threshold (standard deviations)
input double ExitThreshold = 0.3;     // Exit threshold (standard deviations)
input double LotSize = 10.0;           // Fixed lot size per leg
input int LookbackPeriod = 252;       // Lookback for moving average/standard deviation
input int Slippage = 3;               // Max allowed slippage

// Global variables
string symbols[] = {"MU", "NVDA", "MPWR", "MCHP"}; // Asset symbols
double weights[] = {2.699439, 1.000000, -1.877447, -2.505294}; // Johansen eigenvector
double spreadBuffer[];                // Array to store historical spread values
double spreadMean = 0.0;              // Mean of the spread
double spreadStdDev = 0.0;            // Std dev of the spread
bool tradeOpen = false;               // Flag to track if a trade is open
double currentSpread = 0.0;            // Current spread

//+------------------------------------------------------------------+
//| Custom Moving Average Calculation                                |
//+------------------------------------------------------------------+
double CalculateMA(const double &array[], int period)
  {
   double sum = 0.0;
   for(int i = 0; i < period; i++)
      sum += array[i];
   return sum / period;
  }

//+------------------------------------------------------------------+
//| Custom Standard Deviation Calculation                            |
//+------------------------------------------------------------------+
double CalculateStdDev(const double &array[], int period, double mean)
  {
   double variance = 0.0;
   for(int i = 0; i < period; i++)
      variance += MathPow(array[i] - mean, 2);
   return MathSqrt(variance / period);
  }

//+------------------------------------------------------------------+
//| Expert initialization function                                   |
//+------------------------------------------------------------------+
int OnInit()
  {
// Check if all symbols are available
   for(int i = 0; i < ArraySize(symbols); i++)
     {
      if(!SymbolSelect(symbols[i], true))
        {
         Print("Error: Symbol ", symbols[i], " not found!");
         return(INIT_FAILED);
        }
     }
// Initialize spread buffer
   ArrayResize(spreadBuffer, LookbackPeriod);
// Set a timer for spread, mean and stdev calculations
   EventSetTimer(1); // one second
   return(INIT_SUCCEEDED);
  }
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
void OnTimer(void)
  {
// Calculate current spread value
   currentSpread = CalculateSpread();
// Update spread buffer (rolling window)
   static int barCount = 0;
   if(barCount < LookbackPeriod)
     {
      spreadBuffer[barCount] = currentSpread;
      barCount++;
      return; // Wait until buffer is filled
     }
// Shift buffer (remove oldest value, add newest)
   for(int i = 0; i < LookbackPeriod - 1; i++)
      spreadBuffer[i] = spreadBuffer[i + 1];
   spreadBuffer[LookbackPeriod - 1] = currentSpread;
// Calculate mean and standard deviation using custom functions
   spreadMean = CalculateMA(spreadBuffer, LookbackPeriod);
   spreadStdDev = CalculateStdDev(spreadBuffer, LookbackPeriod, spreadMean);
  }

//+------------------------------------------------------------------+
//| Expert tick function                                             |
//+------------------------------------------------------------------+
void OnTick()
  {
// Trading logic
   if(!tradeOpen)
     {
      // Check for entry signal (spread deviates from mean)
      if(currentSpread > spreadMean + EntryThreshold * spreadStdDev)
        {
         // Short spread (sell MU/NVDA, buy MPWR/MCHP)
         ExecuteTrade(ORDER_TYPE_SELL);
         tradeOpen = true;
        }
      else
         if(currentSpread < spreadMean - EntryThreshold * spreadStdDev)
           {
            // Buy spread (buy MU/NVDA, sell MPWR/MCHP)
            ExecuteTrade(ORDER_TYPE_BUY);
            tradeOpen = true;
           }
     }
   else
     {
      // Check for exit signal (spread reverts to mean)
      if((currentSpread <= spreadMean + ExitThreshold * spreadStdDev) &&
         (currentSpread >= spreadMean - ExitThreshold * spreadStdDev))
        {
         CloseAllTrades();
         tradeOpen = false;
        }
     }
// Optional: Display spread in chart comment
   Comment(StringFormat("Spread: %.2f | Mean: %.2f | StdDev: %.2f", currentSpread, spreadMean, spreadStdDev));
  }

//+------------------------------------------------------------------+
//| Calculate current spread value                                   |
//+------------------------------------------------------------------+
double CalculateSpread()
  {
   double spread = 0.0;
   for(int i = 0; i < ArraySize(symbols); i++)
     {
      double price = iClose(symbols[i], PERIOD_D1, 0); // Use latest closing price
      spread += weights[i] * price;
     }
   return spread;
  }

//+------------------------------------------------------------------+
//| Execute trade with normalized integer lots                       |
//+------------------------------------------------------------------+
void ExecuteTrade(ENUM_ORDER_TYPE orderType)
  {
   double volumeArray[];
   ArrayResize(volumeArray, ArraySize(symbols));
   if(!NormalizeVolumeToIntegerLots(volumeArray, symbols, weights, LotSize))
     {
      Print("Volume normalization failed!");
      return;
     }
   for(int i = 0; i < ArraySize(symbols); i++)
     {
      ENUM_ORDER_TYPE legType = (weights[i] > 0) ? orderType :
                                (orderType == ORDER_TYPE_BUY ? ORDER_TYPE_SELL : ORDER_TYPE_BUY);
      trade.PositionOpen(symbols[i], legType, volumeArray[i], 0, 0, 0, "NVDA Coint");
     }
  }
//+------------------------------------------------------------------+
//| Close all open positions                                         |
//+------------------------------------------------------------------+
void CloseAllTrades()
  {
   for(int i = 0; i < PositionsTotal(); i++)
     {
      ulong ticket = PositionGetTicket(i);
      trade.PositionClose(ticket, Slippage);
     }
  }
//+------------------------------------------------------------------+
//| Custom clamping function (replaces MathClamp)                    |
//+------------------------------------------------------------------+
double Clamp(double value, double min, double max)
  {
   return MathMin(MathMax(value, min), max);
  }
//+------------------------------------------------------------------+
//| Normalize floating-point volumes to integer lots                 |
//| while preserving hedge ratios.                                   |
//| Returns true if successful, false if normalization fails.        |
//+------------------------------------------------------------------+
//+------------------------------------------------------------------+
//| Normalize volumes to integer lots                                |
//+------------------------------------------------------------------+
bool NormalizeVolumeToIntegerLots(double &volumeArray[], const string &symbols_arr[], const double &weights_arr[], double baseLotSize)
  {
   MqlTick tick; // Structure to store bid/ask prices
   double totalDollarExposure = 0.0;
   double dollarExposures[];
   ArrayResize(dollarExposures, ArraySize(symbols_arr));
// Step 1: Calculate dollar exposure for each leg
   for(int i = 0; i < ArraySize(symbols_arr); i++)
     {
      if(!SymbolInfoTick(symbols_arr[i], tick)) // Get latest bid/ask
        {
         Print("Failed to get price for ", symbols_arr[i]);
         return false;
        }
      // Use bid price for short legs, ask for long legs
      double price = (weights_arr[i] > 0) ? tick.ask : tick.bid;
      dollarExposures[i] = MathAbs(weights_arr[i]) * price * baseLotSize;
      totalDollarExposure += dollarExposures[i];
     }
// Step 2: Convert dollar exposure to integer lots
   for(int i = 0; i < ArraySize(symbols_arr); i++)
     {
      double ratio = dollarExposures[i] / totalDollarExposure;
      double targetDollarExposure = ratio * totalDollarExposure;
      // Get min/max lot size and step for the symbol
      double minLot = SymbolInfoDouble(symbols_arr[i], SYMBOL_VOLUME_MIN);
      double maxLot = SymbolInfoDouble(symbols_arr[i], SYMBOL_VOLUME_MAX);
      double lotStep = SymbolInfoDouble(symbols_arr[i], SYMBOL_VOLUME_STEP);
      // Get current price again (for lot calculation)
      if(!SymbolInfoTick(symbols_arr[i], tick))
         return false;
      double price = (weights_arr[i] > 0) ? tick.ask : tick.bid;
      double lots = targetDollarExposure / price;
      lots = MathFloor(lots / lotStep) * lotStep; // Round down to nearest step
      // Clamp to broker constraints using custom Clamp()
      volumeArray[i] = Clamp(lots, minLot, maxLot);
     }
   return true;
  }
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
void OnDeinit(const int32_t reason)
  {
   EventKillTimer();
//--- The first way to get a deinitialization reason code
   Print(__FUNCTION__, " Deinitialization reason code = ", reason);
//--- The second way to get a deinitialization reason code
   Print(__FUNCTION__, " _UninitReason = ", getUninitReasonText(_UninitReason));
//--- The third way to get a deinitialization reason code
   Print(__FUNCTION__, " UninitializeReason() = ", getUninitReasonText(UninitializeReason()));
  }
//+------------------------------------------------------------------+
// from MQL5 ref
//+------------------------------------------------------------------+
//| Return a textual description of the deinitialization reason code |
//+------------------------------------------------------------------+
string getUninitReasonText(int reasonCode)
  {
   string text = "";
//---
   switch(reasonCode)
     {
      case REASON_ACCOUNT:
         text = "Account was changed";
         break;
      case REASON_CHARTCHANGE:
         text = "Symbol or timeframe was changed";
         break;
      case REASON_CHARTCLOSE:
         text = "Chart was closed";
         break;
      case REASON_PARAMETERS:
         text = "Input-parameter was changed";
         break;
      case REASON_RECOMPILE:
         text = "Program " + __FILE__ + " was recompiled";
         break;
      case REASON_REMOVE:
         text = "Program " + __FILE__ + " was removed from chart";
         break;
      case REASON_TEMPLATE:
         text = "New template was applied to chart";
         break;
      default:
         text = "Another reason";
     }
//---
   return text;
  }
//+------------------------------------------------------------------+
