//+------------------------------------------------------------------+
//|                                                          ssa.mqh |
//|                                  Copyright 2024, MetaQuotes Ltd. |
//|                                             https://www.mql5.com |
//+------------------------------------------------------------------+
#property copyright "Copyright 2024, MetaQuotes Ltd."
#property link      "https://www.mql5.com"
#include <Math\Alglib\dataanalysis.mqh>
//+------------------------------------------------------------------+
//| component correlations                                           |
//+------------------------------------------------------------------+
matrix component_corr(ulong windowlen,matrix &components)
  {
   double w[];
   ulong fsize = components.Cols();
   ulong r = fsize - windowlen + 1;
  
   for(ulong i = 0; i<fsize; i++)
     {
      if(i>=0 && i<=windowlen-1)
         w.Push(i+1);
      else
         if(i>=windowlen && i<=r-1)
            w.Push(windowlen);
         else
            if(i>=r && i<=fsize-1)
               w.Push(fsize-i);
     }

   vector weights;
   weights.Assign(w);
   ulong d = windowlen;

   vector norms(d);
   matrix out = matrix::Identity(d,d);

   for(ulong i = 0; i<d; i++)
     {
      norms[i] = weights.Dot(pow(components.Row(i),2.0));
      norms[i] = pow(norms[i],-0.5);
     }

   for(ulong i = 0; i<d; i++)
     {
      for(ulong j = i+1; j<d; j++)
        {
         out[i][j] = MathAbs(weights.Dot(components.Row(i)*components.Row(j))*norms[i]*norms[j]);
         out[j][i] = out[i][j];
        }
     }

   return out;

  }
//+------------------------------------------------------------------+
//|  Wrapper class for Alglib's SSA implementation                   |
//+------------------------------------------------------------------+
class CSingularSpecAnalysis
  {
private:
   CSSAModel         m_model;
   ulong             m_windowlen,m_components;
   double            m_its;
   int               m_lastsize;
   bool              m_realtime_mode;
   void              reset(void);
   

public:
                     CSingularSpecAnalysis();
                    ~CSingularSpecAnalysis(void);

   vector            relative_contributions(void);
   vector            cumulative_contributions(void);
   void              analyze(vector &data,ulong windowlen,ulong num_components,bool last_window_only,ulong histlen,vector &trend_component,vector &noise_component);
   vector            forecast(vector &data,ulong windowlen,ulong num_components,ulong num_steps, bool avg_mode,ulong sliding_win);
   void              init_realtime_mode(vector &series,ulong windowlen,ulong num_components,bool set_power_up,double its=1.0, ulong rnd_seed=0);
   void              init_realtime_mode(double &series[],ulong windowlen,ulong num_components,bool set_power_up,double its=1.0, ulong rnd_seed=0);
   void              real_time_analysis(double new_sequence_value,bool last_window_only,ulong histlen,vector &trend_component,vector &noise_component);
   vector            real_time_forecast(double new_sequence_value,ulong num_steps, bool avg_mode, ulong sliding_win);
   vector            real_time_forecast(ulong num_steps, bool avg_mode, ulong sliding_win);
   void              real_time_analysis(bool last_window_only,ulong histlen,vector &trend_component,vector &noise_component);

  };
//+------------------------------------------------------------------+
//| constructor                                                      |
//+------------------------------------------------------------------+
CSingularSpecAnalysis::CSingularSpecAnalysis(void)
  {
   CSSA::SSACreate(m_model);
   m_realtime_mode = false;
  }
//+------------------------------------------------------------------+
//| destructor                                                       |
//+------------------------------------------------------------------+
CSingularSpecAnalysis::~CSingularSpecAnalysis(void)
  {
  }
//+------------------------------------------------------------------+
//| relative contributions                                           |
//+------------------------------------------------------------------+
vector CSingularSpecAnalysis::relative_contributions(void)
  {
   vector ps;
   CMatrixDouble ls;
   CRowDouble sv;
   int nbasis;
   
   CSSA::SSAGetBasis(m_model,ls,sv,m_lastsize,nbasis);
   
   if(sv.Size()==0)
    {
     Print(__FUNCTION__, " error ", GetLastError());
     return sv.ToVector();
    }
   
   ps = pow(sv.ToVector(),2.0);
   vector relcontribution = (ps/ps.Sum())*100.0;
   return relcontribution;
  }
//+------------------------------------------------------------------+
//| cumulative contributions                                         |
//+------------------------------------------------------------------+
vector CSingularSpecAnalysis::cumulative_contributions(void)
  {
   vector ps;
   CMatrixDouble ls;
   CRowDouble sv;
   int nbasis;
   
   CSSA::SSAGetBasis(m_model,ls,sv,m_lastsize,nbasis);
   
   if(sv.Size()==0)
    {
     Print(__FUNCTION__, " error ", GetLastError());
     return sv.ToVector();
    }
   
   ps = pow(sv.ToVector(),2.0);
   vector cumcontribution = (ps.CumSum()/ps.Sum())*100.0;
   return cumcontribution;
  }
//+------------------------------------------------------------------+
//| get the trend component                                          |
//+------------------------------------------------------------------+
void CSingularSpecAnalysis::analyze(vector &data,ulong windowlen,ulong num_components,bool last_window_only,ulong histlen,vector &trend_component,vector &noise_component)
  {
   reset();
   CSSA::SSASetAlgoTopKDirect(m_model,int(num_components));
   CSSA::SSASetWindow(m_model,int(windowlen));
   m_realtime_mode = false;

   CRowDouble rdata = data;
   CSSA::SSAAddSequence(m_model,rdata,rdata.Size());

   CRowDouble trendcomp;
   CRowDouble noisecomp;
   if(!last_window_only)
      CSSA::SSAAnalyzeLast(m_model,int(histlen),trendcomp,noisecomp);
   else
      CSSA::SSAAnalyzeLastWindow(m_model,trendcomp,noisecomp,m_lastsize);

   trend_component = trendcomp.ToVector();
   noise_component = noisecomp.ToVector();
  }
//+------------------------------------------------------------------+
//| make a forecast                                                  |
//+------------------------------------------------------------------+
vector CSingularSpecAnalysis::forecast(vector &data,ulong windowlen,ulong num_components,ulong num_steps,bool avg_mode,ulong sliding_win)
  {
   reset();
   CSSA::SSASetAlgoTopKDirect(m_model,int(num_components));
   CSSA::SSASetWindow(m_model,int(windowlen));
   m_realtime_mode = false;
   
   CRowDouble rdata = data;
   CSSA::SSAAddSequence(m_model,rdata,rdata.Size());
   CRowDouble trendcomp;
   if(!avg_mode)
      CSSA::SSAForecastLast(m_model,int(num_steps),trendcomp);
   else
      CSSA::SSAForecastAvgLast(m_model,int(sliding_win),int(num_steps),trendcomp);

   return trendcomp.ToVector();
  }
//+------------------------------------------------------------------+
//| reset and clear internal buffers                                 |
//+------------------------------------------------------------------+
void CSingularSpecAnalysis::reset(void)
  {
   CSSA::SSAClearData(m_model);
  }
//+------------------------------------------------------------------+
//| intialize object for realtime forecasting operations             |
//+------------------------------------------------------------------+
void CSingularSpecAnalysis::init_realtime_mode(vector &series,ulong windowlen,ulong num_components, bool power_up,double its=1.0,ulong rnd_seed=0)
  {
   m_windowlen = windowlen;
   m_components = num_components;
   m_realtime_mode = true;
   m_its = its;
   reset();
   CSSA::SSASetAlgoTopKRealtime(m_model,int(m_components));
   CSSA::SSASetWindow(m_model,int(m_windowlen));
   if(series.Size())
    {
     CRowDouble seriesdata = series;
     CSSA::SSAAddSequence(m_model,seriesdata,seriesdata.Size());
    }
   if(power_up)
    CSSA::SSASetPowerUpLength(m_model,3);
   if(rnd_seed)
    CSSA::SSASetSeed(m_model,int(rnd_seed));
   return;
  }
//+------------------------------------------------------------------+
//| intialize object for realtime forecasting operations             |
//+------------------------------------------------------------------+
void CSingularSpecAnalysis::init_realtime_mode(double &series[],ulong windowlen,ulong num_components, bool power_up,double its=1.0,ulong rnd_seed=0)
  {
   vector nv;
   nv.Assign(series);
   init_realtime_mode(nv,windowlen,num_components,power_up,its,rnd_seed);
   return;
  }
//+------------------------------------------------------------------+
//| realtime forecasting                                             |
//+--------------------------- --------------------------------------+
vector CSingularSpecAnalysis::real_time_forecast(double new_sequence_value,ulong num_steps,bool avg_mode,ulong sliding_win)
  {
   if(!m_realtime_mode)
    {
     Print(__FUNCTION__, " real time mode has not been initialized ");
     return vector::Zeros(0);
    }
    
   CSSA::SSAAppendPointAndUpdate(m_model,new_sequence_value,m_its);

   CRowDouble trendcomp;
   if(!avg_mode)
      CSSA::SSAForecastLast(m_model,int(num_steps),trendcomp);
   else
      CSSA::SSAForecastAvgLast(m_model,int(sliding_win),int(num_steps),trendcomp);

   return trendcomp.ToVector();
  }
//+------------------------------------------------------------------+
//| realtime analysis                                                |
//+------------------------------------------------------------------+
void CSingularSpecAnalysis::real_time_analysis(double new_sequence_value,bool last_window_only,ulong histlen,vector &trend_component,vector &noise_component)
  {
   if(!m_realtime_mode)
    {
     Print(__FUNCTION__, " real time mode has not been initialized ");
     return;
    }
    
   CSSA::SSAAppendPointAndUpdate(m_model,new_sequence_value,m_its);

   CRowDouble trendcomp;
   CRowDouble noisecomp;
   if(!last_window_only)
      CSSA::SSAAnalyzeLast(m_model,int(histlen),trendcomp,noisecomp);
   else
      CSSA::SSAAnalyzeLastWindow(m_model,trendcomp,noisecomp,m_lastsize);

   trend_component = trendcomp.ToVector();
   noise_component = noisecomp.ToVector();
  }
//+------------------------------------------------------------------+
//| realtime forecasting                                             |
//+--------------------------- --------------------------------------+
vector CSingularSpecAnalysis::real_time_forecast(ulong num_steps,bool avg_mode,ulong sliding_win)
  {
   if(!m_realtime_mode)
    {
     Print(__FUNCTION__, " real time mode has not been initialized ");
     return vector::Zeros(0);
    }
    
   CRowDouble trendcomp;
   if(!avg_mode)
      CSSA::SSAForecastLast(m_model,int(num_steps),trendcomp);
   else
      CSSA::SSAForecastAvgLast(m_model,int(sliding_win),int(num_steps),trendcomp);

   return trendcomp.ToVector();
  }
//+------------------------------------------------------------------+
//| realtime analysis                                                |
//+------------------------------------------------------------------+
void CSingularSpecAnalysis::real_time_analysis(bool last_window_only,ulong histlen,vector &trend_component,vector &noise_component)
  {
   if(!m_realtime_mode)
    {
     Print(__FUNCTION__, " real time mode has not been initialized ");
     return;
    }

   CRowDouble trendcomp;
   CRowDouble noisecomp;
   if(!last_window_only)
      CSSA::SSAAnalyzeLast(m_model,int(histlen),trendcomp,noisecomp);
   else
      CSSA::SSAAnalyzeLastWindow(m_model,trendcomp,noisecomp,m_lastsize);

   trend_component = trendcomp.ToVector();
   noise_component = noisecomp.ToVector();
  }
//+------------------------------------------------------------------+
