//+------------------------------------------------------------------+
//|                                       SSA_PriceDecomposition.mq5 |
//|                                  Copyright 2024, MetaQuotes Ltd. |
//|                                             https://www.mql5.com |
//+------------------------------------------------------------------+
#property copyright "Copyright 2024, MetaQuotes Ltd."
#property link      "https://www.mql5.com"
#property version   "1.00"
#include<ssd.mqh>
#include<EasyAndFastGUI\WndCreate.mqh>
#include<TimeOptions.mqh>
//+------------------------------------------------------------------+
//|  price type                                                      |
//+------------------------------------------------------------------+
enum ENUM_RATES_TYPE
  {
   RATES_OPEN = 1,//OPEN
   RATES_HIGH = 2,//HIGH
   RATES_LOW  = 4,//LOW
   RATES_CLOSE = 8//CLOSE
  };
//+------------------------------------------------------------------+
//| Gui application class                                            |
//+------------------------------------------------------------------+
class CApp:public CWndCreate
  {
protected:

   CWindow           m_window;                   // main window

   CTextEdit         m_hist_size_inc;            // history length

   CComboBox         m_timeframe;                // timeframe

   CDropCalendar     m_hist_start;               // history start date

   CComboBox         m_pricetype;                // applied price
   
   CCheckBox         m_detrend_check;             //detrend option 

   //------------------------------------------------------------


   CTextEdit         m_windowlen_inc;            // ssa window length

   CTextEdit         m_symboledit;               // symbol specification

   //-------------------------------------------------------------

   CGraph            m_graph,m_graph1;           //graph plot

   CCurve       *m_curve,
                *m_curve1;                 // individual curves on graph plot
   CButton           m_next;
   CButton           m_show;

   double       m_x[],                      //x axis values for series plots
                m_x1[],                     //x axis for contribution plots
                m_output[],                 //output for first plot
                m_output1[],                //output for second plot
                m_output2[];                //output for contributions plot

   vector            m_buffer;                   // buffer vector
   matrix            m_components;               // components matrix
   vector            m_contributions;            // contributions

   string            m_symbol;
   datetime          m_start;
   ENUM_TIMEFRAMES   m_tf;
   ENUM_RATES_TYPE   m_rates;
   ulong             m_histlen;
   ulong             m_winlen;

   long              m_increment;
   bool              m_changed,m_detrend;
   int               m_mode;
   CSingularSpecDecomp m_ssa;

public:
                     CApp(void);                 //constructor
                    ~CApp(void);                 //destructor

   void              OnInitEvent(void);
   void              OnDeinitEvent(const int reason);

   virtual void      OnEvent(const int id, const long &lparam, const double &dparam,const string &sparam);

   bool              CreateGUI(void);

protected:
   bool              CreatePlot1(const int x_gap, const int y_gap,const int ysize);
   bool              CreatePlot2(const int x_gap,const int y_gap);

private:

   void              RecalculatingSeries(void);
   void              RecalculateDisplaySeries(void);
   bool              SetBufferPlot1(void);
   bool              SetBufferPlot2(void);
   bool              SetBufferPlot3(void);
  };
//+------------------------------------------------------------------+
//| Recalcualte buffers                                              |
//+------------------------------------------------------------------+
void CApp::RecalculatingSeries(void)
  {
//---reset the internal buffers
   if(!SetBufferPlot1())
      return;
//---get graph plot
   CGraphic *graph=m_graph.GetGraphicPointer();
//---
   m_curve=graph.CurveGetByIndex(0);
//---update graph plot
   m_curve.Update(m_x,m_output);
//---
   graph.YAxis().Name(m_symbol);
//---redraw graph plot
   graph.Redraw(true);
//---redraw and recalculate
   graph.Update(true);
//---
  }
//+------------------------------------------------------------------+
//|  Redraw second plot                                              |
//+------------------------------------------------------------------+
void CApp::RecalculateDisplaySeries(void)
  {
//---
   if(!m_mode)
     {
      if(!SetBufferPlot2())
         return;
     }
   else
     {
      if(!SetBufferPlot3())
         return;
     }
//---
   CGraphic *graph=m_graph1.GetGraphicPointer();
//---
   m_curve1=graph.CurveGetByIndex(0);
//---
   if(!m_mode)
     {
      m_curve1.Update(m_x,m_output1);
      m_curve1.Type(CURVE_LINES);
      graph.YAxis().Name(m_symbol+" Component "+string(m_increment));
      graph.XAxis().Name("Time");
     }
   else
     {
      m_curve1.Update(m_x1,m_output2);
      m_curve1.Type(CURVE_POINTS_AND_LINES);
      graph.YAxis().Name(m_symbol+" Component Contributions ");
      graph.XAxis().Name("Components");
     }
//---redraw graph plot
   graph.Redraw(true);
//---redraw and recalculate
   graph.Update(true);
  }
//+------------------------------------------------------------------+
//| Constructor                                                      |
//+------------------------------------------------------------------+
CApp::CApp(void)
  {
   m_curve=m_curve1=NULL;
   m_symbol = NULL;
   m_winlen = 0;
   m_start = NULL;
   m_histlen = 0;
   m_increment = 0;
   m_changed = false;
   m_rates = WRONG_VALUE;
   m_tf = WRONG_VALUE;
   m_mode = 0;
   m_detrend = false;
  }
//+------------------------------------------------------------------+
//| Destructor                                                       |
//+------------------------------------------------------------------+
CApp::~CApp(void)
  {

  }

//+------------------------------------------------------------------+
//| On initialization                                                |
//+------------------------------------------------------------------+
void CApp::OnInitEvent(void)
  {
  }
//+------------------------------------------------------------------+
//| On DeInitilization                                               |
//+------------------------------------------------------------------+
void CApp::OnDeinitEvent(const int reason)
  {
   CWndEvents::Destroy();
  }
//+------------------------------------------------------------------+
//| Event handler                                                    |
//+------------------------------------------------------------------+
void CApp::OnEvent(const int id,const long &lparam,const double &dparam,const string &sparam)
  {
   m_winlen = (ulong)m_windowlen_inc.GetValue();
   m_symbol = m_symboledit.GetValue();
   m_histlen =(uint)StringToInteger(m_hist_size_inc.GetValue());
   m_tf = (ENUM_TIMEFRAMES)m_timeframe.GetValue();
   m_start = m_hist_start.SelectedDate();
   m_rates = (ENUM_RATES_TYPE)m_pricetype.GetListViewPointer().SelectedItemIndex();
   m_detrend = m_detrend_check.IsPressed();
   
   if(id==CHARTEVENT_CUSTOM+ON_CLICK_CHECKBOX)
     {
      if(lparam==m_detrend_check.Id())
        {
         if(m_winlen && m_symbol!=NULL)
         {
          m_mode = 0;
          m_increment = 0;
          m_changed = true;
          RecalculatingSeries();
          RecalculateDisplaySeries();
          return;
         }
        }
      return;
     }

   if(id==CHARTEVENT_CUSTOM+ON_CLICK_BUTTON)
     {
      if(lparam==m_next.Id())
        {
         if(m_winlen && m_symbol!=NULL)
           {
            m_increment++;
            m_mode = 0;
            if(ulong(m_increment)>=m_components.Cols())
               m_increment = 0;
            RecalculateDisplaySeries();
            return;
           }
        }
      if(lparam==m_show.Id())
        {
         if(m_winlen && m_symbol!=NULL)
           {
            m_mode++;
            if(m_mode>2)
               m_mode = 1;
            RecalculateDisplaySeries();
            return;
           }
        }
     }
   if(id==CHARTEVENT_CUSTOM+ON_CLICK_COMBOBOX_ITEM)
     {
      if(lparam==m_timeframe.Id() || lparam==m_hist_start.Id() || lparam==m_pricetype.Id())
        {
         if(m_symbol!=NULL)
            RecalculatingSeries();
         if(m_winlen)
           {
            m_mode = 0;
            m_increment = 0;
            m_changed = true;
            RecalculateDisplaySeries();
           }
         return;
        }

      return;
     }
   if(id==CHARTEVENT_CUSTOM+ON_END_EDIT||id==CHARTEVENT_CUSTOM+ON_CLICK_INC || id==CHARTEVENT_CUSTOM+ON_CLICK_DEC)
     {
      if(lparam==m_hist_size_inc.Id() || lparam==m_symboledit.Id())
        {
         if(m_symbol!=NULL)
           {
            m_mode = 0;
            RecalculatingSeries();
            m_increment = 0;
            m_changed = true;
            RecalculateDisplaySeries();
           }
         return;
        }
      if(lparam==m_windowlen_inc.Id())
        {
         if(m_winlen && m_symbol!=NULL)
           {
            m_mode = 0;
            m_increment = 0;
            m_changed = true;
            RecalculateDisplaySeries();
           }
         return;
        }
     }
  }
//+------------------------------------------------------------------+
//|  Create the graphical user interface                             |
//+------------------------------------------------------------------+
bool CApp::CreateGUI(void)
  {
//---initialize window creation
   if(!CWndCreate::CreateWindow(m_window,"SSA Components Viewer",1,1,750,600,true,true,true,false))
      return(false);
//---create text edit for length of series
   if(!CWndCreate::CreateTextEdit(m_hist_size_inc,"HistorySize",m_window,0,false,7,25,150,70,1000000,100,1,0,100))
      return(false);
//---create text edit for Seed of random values
   if(!CWndCreate::CreateCombobox(m_timeframe,"Period: ",m_window,0,false,175,25,150,100,tfitems,93,19))
      return(false);
//---create text edit for frequency
   if(!CWndCreate::CreateDropCalendar(m_hist_start,"Date: ",m_window,0,7,50,150,D'2025.01.01'))
      return(false);
//---create text edit for width in frequency units
   if(!CWndCreate::CreateTextEdit(m_symboledit,"Set Symbol",m_window,0,false,175,50,150,70,""))
      return(false);
//---create text edit for amount of padding
   if(!CWndCreate::CreateTextEdit(m_windowlen_inc,"WindowLen",m_window,0,false,350,50,150,70,1000000,2,1,0,40))
      return(false);
//---create combobox for applied filtering
   string items1[]= {"OPEN","HIGH","LOW","CLOSE"};
   if(!CWndCreate::CreateCombobox(m_pricetype,"Price: ",m_window,0,false,350,25,150,100,items1,93,3))
      return(false);
//---create checkbox
   if(!CWndCreate::CreateButton(m_next,"Next Component",m_window,0,540,23,100))
      return(false);
//---create checkbox
   if(!CWndCreate::CreateButton(m_show,"Show Contributions",m_window,0,540,53,100))
      return(false);
//---
   if(!CWndCreate::CreateCheckbox(m_detrend_check,"Detrend",m_window,0,650,53,70))
      return(false);
//---initialize graph plot1
   if(!CreatePlot1(2,75,280))
      return(false);
//---initialize graph plot2
   if(!CreatePlot2(2,350))
      return(false);
//---init events
   CWndEvents::CompletedGUI();
//---
   return(true);
  }
//+------------------------------------------------------------------+
//| Draw the equity curves                                           |
//+------------------------------------------------------------------+
bool CApp::CreatePlot1(const int x_gap,const int y_gap,const int ysize)
  {
//---attach graph object
   m_graph.MainPointer(m_window);
//---set resize properties of graph
   m_graph.AutoXResizeMode(true);
   m_graph.YSize(ysize);
//m_graph.AutoYResizeMode(true);
   m_graph.AutoXResizeRightOffset(2);
//m_graph.AutoYResizeBottomOffset(23);
//---create graph
   if(!m_graph.CreateGraph(x_gap,y_gap))
      return(false);
//---set background color
   CGraphic *graph=m_graph.GetGraphicPointer();
   graph.BackgroundColor(ColorToARGB(clrWhiteSmoke));
   graph.HistoryNameWidth(0);
   graph.HistorySymbolSize(0);
//---
   m_buffer = np::arange(100);
   m_contributions = m_buffer;
   m_contributions.Random(0.0,1.0);
//---
   if(!np::vecAsArray(m_buffer,m_x) || !np::vecAsArray(m_contributions,m_output))
      return false;
//---
   m_curve=graph.CurveAdd(m_x,m_output,ColorToARGB(clrBlue),CURVE_LINES,NULL);
//---set axis properties
   graph.YAxis().Name("series values");
   graph.YAxis().NameSize(20);
   graph.XAxis().Name("time");
   graph.XAxis().NameSize(20);
//---set line width for curves
   m_curve.LinesWidth(3);
//---plot the curves
   graph.CurvePlotAll();
//---add graph to window container
   CWndContainer::AddToElementsArray(0,m_graph);
//---
   return true;
  }
//+------------------------------------------------------------------+
//| Draw the equity curves                                           |
//+------------------------------------------------------------------+
bool CApp::CreatePlot2(const int x_gap,const int y_gap)
  {
//---attach graph object
   m_graph1.MainPointer(m_window);
//---set resize properties of graph
   m_graph1.AutoXResizeMode(true);
   m_graph1.AutoYResizeMode(true);
   m_graph1.AutoXResizeRightOffset(2);
   m_graph1.AutoYResizeBottomOffset(23);
//---create graph
   if(!m_graph1.CreateGraph(x_gap,y_gap))
      return(false);
//---set background color
   CGraphic *graph=m_graph1.GetGraphicPointer();
   graph.BackgroundColor(ColorToARGB(clrWhiteSmoke));
   graph.HistoryNameWidth(0);
   graph.HistorySymbolSize(0);
//---
   if(!np::vecAsArray(m_contributions,m_output1))
      return false;
//---
   m_curve1=graph.CurveAdd(m_x,m_output1,ColorToARGB(clrRosyBrown),CURVE_LINES,NULL);
//---set axis properties
   graph.YAxis().Name("Y values");
   graph.YAxis().NameSize(20);
   graph.XAxis().Name("X values");
   graph.XAxis().NameSize(20);
//---set line width for curves
   m_curve1.LinesWidth(3);
//---plot the curves
   graph.CurvePlotAll();
//---add graph to window container
   CWndContainer::AddToElementsArray(0,m_graph1);
//---
   return true;
  }
//+------------------------------------------------------------------+
//|set or reset arrays for first plot                                |
//+------------------------------------------------------------------+
bool CApp::SetBufferPlot1(void)
  {
//---
   if(!m_buffer.CopyRates(m_symbol,m_tf,ulong(pow(2,m_rates)),m_start,ulong(m_histlen)))
     {
      Print(__FUNCTION__, " copy rates error ", GetLastError());
      return false;
     }
//---
   if(m_detrend)
    {
     vector reg = m_buffer.LinearRegression();
     m_buffer -= reg;
    }
//---set all curves
   if(!np::vecAsArray(m_buffer,m_output))
      return false;
//---
   ArrayResize(m_x,ArraySize(m_output));
//---
   for(int i=0;i<ArraySize(m_output);i++)
      m_x[i]=double(i+1);
//---
   if(!m_ssa.decompose(m_buffer,m_winlen))
     {
      Print(__FUNCTION__, " ssa error ", GetLastError());
      return false;
     }
//---
   m_components = m_ssa.components();
//---
   return true;
  }
//+------------------------------------------------------------------+
//|set or reset arrays for second plot                               |
//+------------------------------------------------------------------+
bool CApp::SetBufferPlot2(void)
  {
   if(!m_buffer.Size() || !m_components.Cols())
      return false;
//---
   ArrayResize(m_x,ArraySize(m_output));
//---
   for(int i=0;i<ArraySize(m_output);i++)
      m_x[i]=double(i+1);
//---get selected metric from gui control
   if(m_changed && m_winlen>0)
     {
      m_components = m_ssa.components();
      m_changed = false;
      m_increment = 0;
     }
//---
   return np::vecAsArray(m_components.Col(m_increment),m_output1);
  }
//+------------------------------------------------------------------+
//|  set or reset arrays for contribution plots                      |
//+------------------------------------------------------------------+
bool CApp::SetBufferPlot3(void)
  {
   if(!m_buffer.Size() || !m_components.Cols())
      return false;
//---get selected metric from gui control
   if(m_mode == 1)
      m_contributions = m_ssa.relative_contributions();
   else
      m_contributions = m_ssa.cumulative_contributions();
//---
   ArrayResize(m_x1,int(m_contributions.Size()));
//---
   for(int i=0;i<ArraySize(m_x1);i++)
      m_x1[i]=double(i+1);
//---
   return np::vecAsArray(m_contributions,m_output2);
  }
CApp app;
//+------------------------------------------------------------------+
//| Expert initialization function                                   |
//+------------------------------------------------------------------+
int OnInit(void)
  {
   ulong tick_counter=::GetTickCount();
//---
   app.OnInitEvent();
//---
   if(!app.CreateGUI())
     {
      ::Print(__FUNCTION__," > error");
      return(INIT_FAILED);
     }

   return(INIT_SUCCEEDED);
  }
//+------------------------------------------------------------------+
//| Expert deinitialization function                                 |
//+------------------------------------------------------------------+
void OnDeinit(const int reason)
  {
   app.OnDeinitEvent(reason);
  }
//+------------------------------------------------------------------+
//| Expert tick function                                             |
//+------------------------------------------------------------------+
void OnTick(void)
  {
  }
//+------------------------------------------------------------------+
//| Timer function                                                   |
//+------------------------------------------------------------------+
void OnTimer(void)
  {
   app.OnTimerEvent();
  }
//+------------------------------------------------------------------+
//| Trade function                                                   |
//+------------------------------------------------------------------+
void OnTrade(void)
  {
  }
//+------------------------------------------------------------------+
//| ChartEvent function                                              |
//+------------------------------------------------------------------+
void OnChartEvent(const int    id,
                  const long   &lparam,
                  const double &dparam,
                  const string &sparam)
  {
   app.ChartEvent(id,lparam,dparam,sparam);
  }
//+------------------------------------------------------------------+
