//+------------------------------------------------------------------+
//|                                                     gaussian.mqh |
//|                                  Copyright 2025, MetaQuotes Ltd. |
//|                                             https://www.mql5.com |
//+------------------------------------------------------------------+
#property copyright "Copyright 2025, MetaQuotes Ltd."
#property link      "https://www.mql5.com"
#ifndef _NP_
  #include<np.mqh>
#endif 
#include "base.mqh"
//+------------------------------------------------------------------+
//| Bivariate Gaussian Copula.                                       |
//+------------------------------------------------------------------+
class CGaussian: public CBivariateCopula
  {
private:
   virtual double    theta_hat(const double tau) override
     {
      return sin(tau*M_PI/2.0);
     }
   matrix            generate_corr_gaussian(ulong num, matrix& cov)
     {
      vector means = vector::Zeros(2);
      return np::multivariate_normal(means,cov,num);
     }
protected:
   virtual double    pdf(double u,double v) override
     {
      double inv_u = CAlglib::InvNormalCDF(u);
      double inv_v = CAlglib::InvNormalCDF(v);

      double exp_ker = (m_rho*(-2.0*inv_u*inv_v+pow(inv_u,2.0)*m_rho+pow(inv_v,2.0)*m_rho)/(2.0*(pow(m_rho,2.0)-1.0)));
      return exp(exp_ker)/sqrt(1.0 - pow(m_rho,2.0));
     }
   virtual double    cdf(double u,double v) override
     {
      double inv_u = CAlglib::InvNormalCDF(u);
      double inv_v = CAlglib::InvNormalCDF(v);
      return CAlglib::BivariateNormalCDF(inv_u,inv_v,m_rho);
     }
   virtual double    condi_cdf(double u,double v) override
     {
      double inv_u = CAlglib::InvNormalCDF(u);
      double inv_v = CAlglib::InvNormalCDF(v);
      double sqrtdetcorr = sqrt(1.0 - m_rho*m_rho);
      return CAlglib::NormalCDF(((inv_u-m_rho*inv_v)/sqrtdetcorr));
     }
   virtual vector    pdf(vector &u,vector &v) override
     {
      vector inv_u(u.Size());
      vector inv_v(v.Size());
      
      for(ulong i = 0; i<u.Size(); ++i)
        {
         inv_u[i] = CAlglib::InvNormalCDF(u[i]);
         inv_v[i] = CAlglib::InvNormalCDF(v[i]);
        }
        
      vector exp_ker = (m_rho*(-2.0*inv_u*inv_v+pow(inv_u,2.0)*m_rho+pow(inv_v,2.0)*m_rho)/(2.0*(pow(m_rho,2.0)-1.0)));
      return exp(exp_ker)/sqrt(1.0 - pow(m_rho,2.0));
     }
   virtual vector    cdf(vector &u,vector &v) override
     {
      vector out(u.Size());
      for(ulong i = 0; i<u.Size(); ++i)
         out[i] = cdf(u[i],v[i]);
      return out;   
     }
   virtual vector    condi_cdf(vector &u,vector &v) override
     {
      vector out(u.Size());
      for(ulong i = 0; i<u.Size(); ++i)
         out[i] = condi_cdf(u[i],v[i]);
      return out; 
     }
public:
                     CGaussian(void)
     {
      m_cov = matrix::Zeros(0,0);
      m_copula_type = GAUSSIAN_COPULA;
     }
                    ~CGaussian(void)
     {
     }

   virtual matrix    Sample(ulong num_samples) override
     {
      matrix gpairs = generate_corr_gaussian(num_samples,m_cov);
      matrix ncdf = gpairs;
      for(ulong i = 0; i<gpairs.Rows(); i++)
        {
         for(ulong j = 0; j<gpairs.Cols(); j++)
            ncdf[i,j] = CAlglib::NormalCDF(gpairs[i,j]);
        }
      return ncdf;
     }
   virtual void Set_covariance(matrix& cov) override
     {
      if(cov.Rows()!=2 || cov.Cols()!=2)
       {
        Print(__FUNCTION__," invalid input: expecting 2x2 matrix ");
        return;
       }
      m_cov = cov;
      m_rho = m_cov[0][1] / (sqrt(m_cov[0][0]) * sqrt(m_cov[1][1]));
      m_tau = (2.0/M_PI) * MathArcsin(m_rho);
      m_theta = theta_hat(m_tau);
     }
   virtual double    Fit(vector &u, vector&v) override
     {
      if(u.Max()>1.0 || v.Max()>1.0 || v.Min()<0.0 ||u.Min()<0.0)
       {
        Print(__FUNCTION__, " Invalid input variable(s) ");
        return EMPTY_VALUE;
       }
      m_tau = kendalTau(u,v);
      m_theta = theta_hat(m_tau);
      matrix vals = matrix::Zeros(u.Size(),2);
      if(!vals.Col(u,0) || !vals.Col(v,1))
        {
         Print(__FUNCTION__, " error ", GetLastError());
         return EMPTY_VALUE;
        }
      for(ulong i = 0; i<vals.Rows(); i++)
        {
         for(ulong j = 0; j<vals.Cols(); j++)
            vals[i,j] = CAlglib::InvNormalCDF(vals[i,j]);
        }

      m_cov = vals.Cov(false);
      m_rho = m_cov[0,1]/(sqrt(m_cov[0,0])*sqrt(m_cov[1,1]));

      return m_rho;
     }
  };
