#property strict

#include "ILogHandler.mqh"
#include "LogLevels.mqh"

//+------------------------------------------------------------------+
//| Class: FileLogHandler                                            |
//| Description: Implements ILogHandler to output log messages to    |
//|              files with rotation capabilities.                   |
//+------------------------------------------------------------------+
class FileLogHandler : public ILogHandler
  {
private:
   LogLevel          m_min_level;       // Minimum level to log
   string            m_format;          // Log message format string
   string            m_file_path;       // Base path for log files
   string            m_file_prefix;     // Prefix for log file names
   int               m_file_handle;     // Current file handle
   datetime          m_current_day;     // Current day for rotation
   int               m_max_size_kb;     // Maximum file size in KB before rotation
   int               m_max_files;       // Maximum number of log files to keep
   
   //--- Helper to format the log message
   string            FormatMessage(const datetime time, const LogLevel level, const string origin, const string message);
   //--- Helper to get string representation of LogLevel
   string            LogLevelToString(const LogLevel level);
   //--- Helper to create or rotate log file
   bool              EnsureFileOpen();
   //--- Helper to generate file name based on date
   string            GenerateFileName(const datetime time);
   //--- Helper to perform log rotation
   void              RotateLogFiles();
   //--- Helper to check if file size exceeds limit
   bool              IsFileSizeExceeded();
   // Add custom helper function to sort string arrays
   void              SortStringArray(string &arr[]);
   //--- New helper to clean file paths
   string CleanPath(const string path);

public:
   FileLogHandler(const string file_path="MQL5\\Logs", 
                  const string file_prefix="EA_Log", 
                  const LogLevel min_level=LOG_LEVEL_INFO, 
                  const string format="[{time}] {level}: {origin} - {message}",
                  const int max_size_kb=1024,
                  const int max_files=5);
   virtual ~FileLogHandler();
   //--- ILogHandler implementation
   virtual bool      Setup(const string settings="") override;
   virtual void      Log(const datetime time, const LogLevel level, const string origin, const string message, const long expert_id=0) override;
   virtual void      Shutdown() override;

   //--- Setters
   void SetFilePath(const string path)    { m_file_path = CleanPath(path); }
   void              SetMinLevel(const LogLevel level) { m_min_level = level; }
   void              SetFormat(const string format)    { m_format = format; }
   void              SetFilePrefix(const string prefix){ m_file_prefix = prefix; }
   void              SetMaxSizeKB(const int size)      { m_max_size_kb = size; }
   void              SetMaxFiles(const int count)      { m_max_files = count; }
  };
//+------------------------------------------------------------------+
//| Constructor                                                      |
//+------------------------------------------------------------------+
FileLogHandler::FileLogHandler(const string file_path, 
                               const string file_prefix, 
                               const LogLevel min_level, 
                               const string format,
                               const int max_size_kb,
                               const int max_files)
  {
   m_min_level = min_level;
   m_format = format;
   m_file_path = CleanPath(file_path);
   m_file_prefix = file_prefix;
   m_file_handle = INVALID_HANDLE;
   m_current_day = 0;
   m_max_size_kb = max_size_kb;
   m_max_files = max_files;
   
   // Create directory if it doesn't exist
   if(!FolderCreate(m_file_path))
     {
      if(GetLastError() != 0)
         Print("FileLogHandler: Failed to create directory: ", m_file_path, ", error: ", GetLastError());
     }
  }
//+------------------------------------------------------------------+
//| Destructor                                                       |
//+------------------------------------------------------------------+
FileLogHandler::~FileLogHandler()
  {
   Shutdown();
  }
//+------------------------------------------------------------------+
//| Setup                                                            |
//+------------------------------------------------------------------+
bool FileLogHandler::Setup(const string settings)
  {
   // Parse settings if provided
   // Format could be: "path=MQL5/Logs;prefix=MyEA;min_level=INFO;max_size=2048;max_files=10"
   if(settings != "")
     {
      string parts[];
      int count = StringSplit(settings, ';', parts);
      
      for(int i = 0; i < count; i++)
        {
         string key_value[];
         if(StringSplit(parts[i], '=', key_value) == 2)
           {
            string key = key_value[0];
            StringTrimLeft(key);
            StringTrimRight(key);
            string value = key_value[1];
            StringTrimLeft(value);
            StringTrimRight(value);
            
            if(key == "path")
               m_file_path = CleanPath(value);
            else if(key == "prefix")
               m_file_prefix = value;
            else if(key == "min_level")
              {
               if(value == "DEBUG")
                  m_min_level = LOG_LEVEL_DEBUG;
               else if(value == "INFO")
                  m_min_level = LOG_LEVEL_INFO;
               else if(value == "WARN")
                  m_min_level = LOG_LEVEL_WARN;
               else if(value == "ERROR")
                  m_min_level = LOG_LEVEL_ERROR;
               else if(value == "FATAL")
                  m_min_level = LOG_LEVEL_FATAL;
              }
            else if(key == "max_size")
               m_max_size_kb = (int)StringToInteger(value);
            else if(key == "max_files")
               m_max_files = (int)StringToInteger(value);
           }
        }
     }
   
   return true;
  }
//+------------------------------------------------------------------+
//| Log                                                              |
//+------------------------------------------------------------------+
void FileLogHandler::Log(const datetime time, const LogLevel level, const string origin, const string message, const long expert_id=0)
  {
   // Check if the message level meets the minimum requirement
   if(level >= m_min_level && level < LOG_LEVEL_OFF)
     {
      // Ensure file is open and ready for writing
      if(EnsureFileOpen())
        {
         // Format the message
         string formatted_message = FormatMessage(time, level, origin, message);
         
         // Write to file
         FileWriteString(m_file_handle, formatted_message + "\r\n");
         
         // Flush to ensure data is written immediately
         FileFlush(m_file_handle);
         
         // Check if rotation is needed
         if(IsFileSizeExceeded())
           {
            FileClose(m_file_handle);
            m_file_handle = INVALID_HANDLE;
            RotateLogFiles();
            EnsureFileOpen();
           }
        }
     }
  }
//+------------------------------------------------------------------+
//| Shutdown                                                         |
//+------------------------------------------------------------------+
void FileLogHandler::Shutdown()
  {
   if(m_file_handle != INVALID_HANDLE)
     {
      FileClose(m_file_handle);
      m_file_handle = INVALID_HANDLE;
     }
  }
//+------------------------------------------------------------------+
//| FormatMessage                                                    |
//+------------------------------------------------------------------+
string FileLogHandler::FormatMessage(const datetime time, const LogLevel level, const string origin, const string message)
  {
   string formatted_message = m_format;

   // Replace placeholders
   StringReplace(formatted_message, "{time}", TimeToString(time, TIME_DATE | TIME_SECONDS));
   StringReplace(formatted_message, "{level}", LogLevelToString(level));
   StringReplace(formatted_message, "{origin}", origin);
   StringReplace(formatted_message, "{message}", message);

   return formatted_message;
  }
//+------------------------------------------------------------------+
//| LogLevelToString                                                 |
//+------------------------------------------------------------------+
string FileLogHandler::LogLevelToString(const LogLevel level)
  {
   switch(level)
     {
      case LOG_LEVEL_DEBUG: return "DEBUG";
      case LOG_LEVEL_INFO:  return "INFO";
      case LOG_LEVEL_WARN:  return "WARN";
      case LOG_LEVEL_ERROR: return "ERROR";
      case LOG_LEVEL_FATAL: return "FATAL";
      default:              return "UNKNOWN";
     }
  }
//+------------------------------------------------------------------+
//| EnsureFileOpen                                                   |
//+------------------------------------------------------------------+
bool FileLogHandler::EnsureFileOpen()
  {
   datetime current_time = TimeCurrent();
   MqlDateTime time_struct;
   TimeToStruct(current_time, time_struct);
   
   // Create a datetime that represents just the current day (time set to 00:00:00)
   MqlDateTime day_struct;
   day_struct.year = time_struct.year;
   day_struct.mon = time_struct.mon;
   day_struct.day = time_struct.day;
   day_struct.hour = 0;
   day_struct.min = 0;
   day_struct.sec = 0;
   datetime current_day = StructToTime(day_struct);
   
   // Check if we need to open a new file (either first time or new day)
   if(m_file_handle == INVALID_HANDLE || m_current_day != current_day)
     {
      // Close existing file if open
      if(m_file_handle != INVALID_HANDLE)
        {
         FileClose(m_file_handle);
         m_file_handle = INVALID_HANDLE;
        }
      
      // Update current day
      m_current_day = current_day;
      
      // Generate new file name
      string file_name = GenerateFileName(current_time);
      
      // Open file for writing (append if exists)
      m_file_handle = FileOpen(file_name, FILE_WRITE | FILE_READ | FILE_TXT);
      
      if(m_file_handle == INVALID_HANDLE)
        {
         Print("FileLogHandler: Failed to open log file: ", file_name, ", error: ", GetLastError());
         return false;
        }
      
      // Move to end of file for appending
      FileSeek(m_file_handle, 0, SEEK_END);
     }
   
   return true;
  }
//+------------------------------------------------------------------+
//| GenerateFileName                                                 |
//+------------------------------------------------------------------+
string FileLogHandler::GenerateFileName(const datetime time)
  {
   MqlDateTime time_struct;
   TimeToStruct(time, time_struct);
   
   string date_str = StringFormat("%04d%02d%02d", 
                                 time_struct.year, 
                                 time_struct.mon, 
                                 time_struct.day);
   
   return m_file_path + "\\" + m_file_prefix + "_" + date_str + ".log";
  }
//+------------------------------------------------------------------+
//| IsFileSizeExceeded                                               |
//+------------------------------------------------------------------+
bool FileLogHandler::IsFileSizeExceeded()
  {
   if(m_file_handle != INVALID_HANDLE)
     {
      // Get current position (file size)
      ulong size = FileSize(m_file_handle);
      
      // Check if size exceeds limit (convert KB to bytes)
      return (size > (ulong)m_max_size_kb * 1024);
     }
   
   return false;
  }
//+------------------------------------------------------------------+
//| RotateLogFiles                                                   |
//+------------------------------------------------------------------+
void FileLogHandler::RotateLogFiles()
  {
   // Get list of log files
   string terminal_path = TerminalInfoString(TERMINAL_DATA_PATH);
   string full_path = terminal_path + "\\" + m_file_path;
   string file_pattern = m_file_prefix + "_*.log";
   
   string files[];
   int file_count = 0;
   
   long search_handle = FileFindFirst(full_path + "\\" + file_pattern, files[file_count]);
   if(search_handle != INVALID_HANDLE)
     {
      file_count++;
      
      // Find all matching files
      while(FileFindNext(search_handle, files[file_count]))
        {
         file_count++;
         ArrayResize(files, file_count + 1);
        }
      
      // Close search handle
      FileFindClose(search_handle);
     }
   
   // Resize array to actual number of found files before sorting
   ArrayResize(files, file_count);
   // Sort the string array using the custom sorter
   SortStringArray(files);
   
   // Delete oldest files if we have too many
   int files_to_delete = file_count - m_max_files + 1; // +1 for the new file we'll create
   
   if(files_to_delete > 0)
     {
      for(int i = 0; i < files_to_delete; i++)
        {
         if(!FileDelete(m_file_path + "\\" + files[i]))
            Print("FileLogHandler: Failed to delete old log file: ", files[i], ", error: ", GetLastError());
        }
     }
  }
//+------------------------------------------------------------------+
//| SortStringArray                                                  |
//+------------------------------------------------------------------+
void FileLogHandler::SortStringArray(string &arr[])
  {
   int n = ArraySize(arr);
   for(int i = 0; i < n - 1; i++)
     {
      for(int j = i + 1; j < n; j++)
        {
         if(arr[i] > arr[j])
           {
            string temp = arr[i];
            arr[i] = arr[j];
            arr[j] = temp;
           }
        }
     }
  }
//+------------------------------------------------------------------+
//| New implementation: CleanPath                                    |
//+------------------------------------------------------------------+
string FileLogHandler::CleanPath(const string path)
  {
   string result = path;
   // Replace all "/" with "\\"
   StringReplace(result, "/", "\\");
   return result;
  }
//+------------------------------------------------------------------+
