//+------------------------------------------------------------------+
//|                                             Sha256Algorithm.mqh  |
//|                                  Copyright 2024, MetaQuotes Ltd. |
//|                                             https://www.mql5.com |
//+------------------------------------------------------------------+
#property copyright "Copyright 2024, MetaQuotes Ltd."
#property link      "kadiabdulkudus@gmail.com"


class CSha256Class
  {
private:

   uint              total_message[];
   uint              paddedMessage[64];
   void              Initialize_H();//This function initializes the values of h0-h7
   uint              RawMessage[];//Keeps track of the raw message sent in
public:
   //hash values from h0 - h7
   uint              h0;
   uint              h1;
   uint              h2;
   uint              h3;
   uint              h4;
   uint              h5;
   uint              h6;
   uint              h7;

   uint              K[64];
   uint              W[64];

                     CSha256Class(void);
                    ~CSha256Class() {};

   void              PreProcessMessage(uint &message[], int messageLength);
   void              CreateMessageSchedule();
   void              Compression();
   void              UpdateHash(uint &message[], int message_len);
   void              GetDigest(uint &digest[]);
   string            GetHex();
  };

//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
CSha256Class::CSha256Class(void)
  {
//initialize the round constants
   K[0] = 0x428a2f98;
   K[1] = 0x71374491;
   K[2] = 0xb5c0fbcf;
   K[3] = 0xe9b5dba5;
   K[4] = 0x3956c25b;
   K[5] = 0x59f111f1;
   K[6] = 0x923f82a4;
   K[7] = 0xab1c5ed5;
   K[8] = 0xd807aa98;
   K[9] = 0x12835b01;
   K[10] = 0x243185be;
   K[11] = 0x550c7dc3;
   K[12] = 0x72be5d74;
   K[13] = 0x80deb1fe;
   K[14] = 0x9bdc06a7;
   K[15] = 0xc19bf174;
   K[16] = 0xe49b69c1;
   K[17] = 0xefbe4786;
   K[18] = 0x0fc19dc6;
   K[19] = 0x240ca1cc;
   K[20] = 0x2de92c6f;
   K[21] = 0x4a7484aa;
   K[22] = 0x5cb0a9dc;
   K[23] = 0x76f988da;
   K[24] = 0x983e5152;
   K[25] = 0xa831c66d;
   K[26] = 0xb00327c8;
   K[27] = 0xbf597fc7;
   K[28] = 0xc6e00bf3;
   K[29] = 0xd5a79147;
   K[30] = 0x06ca6351;
   K[31] = 0x14292967;
   K[32] = 0x27b70a85;
   K[33] = 0x2e1b2138;
   K[34] = 0x4d2c6dfc;
   K[35] = 0x53380d13;
   K[36] = 0x650a7354;
   K[37] = 0x766a0abb;
   K[38] = 0x81c2c92e;
   K[39] = 0x92722c85;
   K[40] = 0xa2bfe8a1;
   K[41] = 0xa81a664b;
   K[42] = 0xc24b8b70;
   K[43] = 0xc76c51a3;
   K[44] = 0xd192e819;
   K[45] = 0xd6990624;
   K[46] = 0xf40e3585;
   K[47] = 0x106aa070;
   K[48] = 0x19a4c116;
   K[49] = 0x1e376c08;
   K[50] = 0x2748774c;
   K[51] = 0x34b0bcb5;
   K[52] = 0x391c0cb3;
   K[53] = 0x4ed8aa4a;
   K[54] = 0x5b9cca4f;
   K[55] = 0x682e6ff3;
   K[56] = 0x748f82ee;
   K[57] = 0x78a5636f;
   K[58] = 0x84c87814;
   K[59] = 0x8cc70208;
   K[60] = 0x90befffa;
   K[61] = 0xa4506ceb;
   K[62] = 0xbef9a3f7;
   K[63] = 0xc67178f2;
  }

//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
void CSha256Class::Initialize_H(void)
  {
   h0 = 0x6a09e667;
   h1 = 0xbb67ae85;
   h2 = 0x3c6ef372;
   h3 = 0xa54ff53a;
   h4 = 0x510e527f;
   h5 = 0x9b05688c;
   h6 = 0x1f83d9ab;
   h7 = 0x5be0cd19;

  }
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
void CSha256Class::PreProcessMessage(uint &message[],int messageLength)
  {
   ArrayInitialize(paddedMessage, 0);
   for(int i=0; i < messageLength; i++)
      paddedMessage[i] = message[i];
  }

//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
void CSha256Class::CreateMessageSchedule()
  {
   ArrayInitialize(W, 0);

   int counts = 0;
   for(int i=0; i<ArraySize(paddedMessage); i+=4)
     {
      //32 bits is equivalent to 4 bytes from message
      uint byte1 = paddedMessage[i];
      uint byte2 = paddedMessage[i+1];
      uint byte3 = paddedMessage[i+2];
      uint byte4 = paddedMessage[i+3];

      uint combined = ((byte1 << 24) | (byte2 << 16) | (byte3 << 8) | byte4);
      W[counts] = combined & 0xFFFFFFFF;

      counts += 1;
     }
   for(int i=counts; i<64; i++)
      W[i] = 0x00000000;
//preserve previous counts

   int prev_counts = counts;
   int left_count = 64-counts;
   for(int i=counts; i<64; i++)
     {
      uint s0 = (RightRotate(W[i-15], 7)) ^ (RightRotate(W[i-15],18)) ^ (W[i-15] >> 3);
      uint s1 = (RightRotate(W[i-2], 17)) ^ (RightRotate(W[i-2],19)) ^ (W[i-2] >> 10);

      W[i] = (W[i-16] + s0 + W[i-7] + s1) & 0xFFFFFFFF;
     }
  }

//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
void CSha256Class::Compression(void)
  {
   uint a = h0;
   uint b = h1;
   uint c = h2;
   uint d = h3;
   uint e = h4;
   uint f = h5;
   uint g = h6;
   uint h = h7;

   for(int i=0; i<64; i++)
     {
      uint S1 = (RightRotate(e, 6) ^ RightRotate(e,11) ^ RightRotate(e,25)) & 0xFFFFFFFF;
      uint ch = ((e & f) ^ ((~e) & g))& 0xFFFFFFFF;
      uint temp1 = (h + S1 + ch + K[i] + W[i]) & 0xFFFFFFFF;
      uint S0 = (RightRotate(a, 2) ^ RightRotate(a, 13) ^ RightRotate(a, 22)) & 0xFFFFFFFF;
      uint maj = ((a & b) ^ (a & c) ^ (b & c)) & 0xFFFFFFFF;
      uint temp2 = (S0 + maj) & 0xFFFFFFFF;
      h = g & 0xFFFFFFFF;
      g = f & 0xFFFFFFFF;
      f = e & 0xFFFFFFFF;
      e = (d + temp1) & 0xFFFFFFFF;
      d = c & 0xFFFFFFFF;
      c = b & 0xFFFFFFFF;
      b = a & 0xFFFFFFFF;
      a = (temp1 + temp2)&0xFFFFFFFF;
     }

   h0 = h0 + a;
   h1 = h1 + b;
   h2 = h2 + c;
   h3 = h3 + d;
   h4 = h4 + e;
   h5 = h5 + f;
   h6 = h6 + g;
   h7 = h7 + h;

  }
//+------------------------------------------------------------------+
//| Modified UpdateHash function with proper array initialization       |
//+------------------------------------------------------------------+
void CSha256Class::UpdateHash(uint &message[], int message_len)
  {
   int prev_tot_length = ArraySize(RawMessage);
   int counts = prev_tot_length;

// Initialize new h values
   Initialize_H();

// Clear and initialize total_message array
   ArrayFree(total_message);
   ArrayResize(total_message, 0);

// Clear and resize RawMessage with proper initialization
   ArrayFree(RawMessage);
   ArrayResize(RawMessage, counts + message_len);
   ArrayInitialize(RawMessage, 0);

// Copy message to RawMessage
   for(int i=0; i<message_len; i++)
     {
      RawMessage[counts] = message[i];
      counts += 1;
     }

// Initialize total_message with proper size and values
   ArrayResize(total_message, ArraySize(RawMessage));
   ArrayInitialize(total_message, 0);

   for(int i=0; i<ArraySize(RawMessage); i++)
     {
      total_message[i] = RawMessage[i];
     }

// Padding operations
   int messageBitLength = message_len * 8;
   int paddingSize = (64 - (message_len % 64));

   if(paddingSize < 9)
      paddingSize += 64;

// Initialize padding array properly
   uint paddingvals[];
   ArrayResize(paddingvals, paddingSize);
   ArrayInitialize(paddingvals, 0);  // Initialize all padding values to 0
   paddingvals[0] = 0x80;

// Calculate the length bits (big-endian)
   int endian_bits_arr[];
   ArrayResize(endian_bits_arr, 8);
   ArrayInitialize(endian_bits_arr, 0);
   GetEndianBigValueArray(messageBitLength, endian_bits_arr);

   int copiedcounts = 0;
   for(int i=paddingSize-8; i<paddingSize; i++)
     {
      paddingvals[i] = endian_bits_arr[copiedcounts];
      copiedcounts += 1;
     }

// Append the padding and message length with proper initialization
   int new_size = message_len + paddingSize;
   ArrayResize(total_message, new_size);

   for(int i=message_len; i<new_size; i++)
     {
      total_message[i] = paddingvals[i - message_len];
     }

//Add a new size of 8 to the total_message
   ArrayResize(total_message, new_size+8);
   ArrayFill(total_message, new_size, 8, 0);

   int chunks_count = (int)MathFloor(ArraySize(total_message)/64.0);
   int copied = 0;

   for(int i=0; i<chunks_count; i++)
     {
      uint newChunk[];
      ArrayResize(newChunk, 64);
      ArrayInitialize(newChunk, 0);  // Initialize chunk array

      for(int j=0; j<64; j++)
        {
         newChunk[j] = total_message[copied];
         copied += 1;
        }

      PreProcessMessage(newChunk, ArraySize(newChunk));
      CreateMessageSchedule();
      Compression();
     }
  }

//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
string CSha256Class::GetHex(void)
  {
   string result  = "";

   result += UintToHex(h0);//StringFormat("%.2X",h0);
   result += UintToHex(h1);//StringFormat("%.2X",h1);
   result += UintToHex(h2);//StringFormat("%.2X",h2);
   result += UintToHex(h3);//StringFormat("%.2X",h3);
   result += UintToHex(h4);//StringFormat("%.2X",h4);
   result += UintToHex(h5);//StringFormat("%.2X",h5);
   result += UintToHex(h6);//StringFormat("%.2X",h6);
   result += UintToHex(h7);//StringFormat("%.2X",h7);

   return (result);
  }

//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
string UintToHex(uint value)
  {
   string hexDigits = "0123456789ABCDEF";
   string result = "";

   if(value == 0)
     {
      return "00000000";
     }

   while(value > 0)
     {
      uint remainder = value % 16;
      result = StringSubstr(hexDigits, remainder, 1) + result;
      value /= 16;
     }

   int padding = 8 - StringLen(result);
   string paddedResult = "";

   for(int i = 0; i < padding; i++)
     {
      paddedResult += "0";
     }

   return paddedResult + result;
  }
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
void CSha256Class::GetDigest(uint &digest[])
  {
   ArrayResize(digest,32);

   string hexval = GetHex();

   int copied_counts = 0;
   for(int i=0; i<64;i+=2)
     {
      digest[copied_counts] = convertHexStringToUint(StringSubstr(hexval, i, 2));
      copied_counts+=1;
     }
  }

//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
uint convertHexStringToUint(string hexString)
  {
//i will be converting pairs of hex string

   uint firstval = CharToInteger(StringGetCharacter(hexString,0))*16;
   uint secondval = CharToInteger(StringGetCharacter(hexString,1));

   return firstval+secondval;
  }

//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
int CharToInteger(ushort c)
  {
   if(c >= '0' && c <= '9')
      return c - '0';
   if(c >= 'A' && c <= 'F')
      return c - 'A' + 10;
   if(c >= 'a' && c <= 'f')
      return c - 'a' + 10;
   return 0; // Return 0 for invalid characters
  }

//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
void ConvertDecimaltoBinary(int decimalValue, int &binary_arr[])
  {
   int value = decimalValue;
   ArrayResize(binary_arr, 0);
   while(value>0)
     {
      int prev_bin_size = ArraySize(binary_arr);
      ArrayResize(binary_arr, prev_bin_size+1);
      binary_arr[prev_bin_size] = value%2;

      value /=2;
     }
  }

//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
int ConvertBinaryToDecimal(int &binary_arr[])
  {
   int total = 0;

   for(int i=0;i<ArraySize(binary_arr);i++)
     {
      total += binary_arr[i] * (int)pow(2, i);
     }
   return (total);
  }

//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
void GetEndianBigValueArray(int value, uint &bitIntegerValueArray[])
  {
   int binary_values[];
   ConvertDecimaltoBinary(value, binary_values);
   int binary_size = ArraySize(binary_values);
   ArrayResize(bitIntegerValueArray, 8);

   int buffers_count = (int)floor(binary_size/8);
   if(binary_size%8>0)
      buffers_count+=1;

   int copied = 0;
   int array_update_index = 7;//used to update each bytes
   for(int i=0;i<buffers_count;i++)
     {
      int new_buffer_binaries[];
      for(int j=0;j<8;j++)
        {
         if(copied>=binary_size)
            break;
         int prev_bin_size = ArraySize(new_buffer_binaries);
         ArrayResize(new_buffer_binaries, prev_bin_size+1);
         new_buffer_binaries[prev_bin_size] = binary_values[copied];
         copied += 1;
        }
      int decimalval = ConvertBinaryToDecimal(new_buffer_binaries);
      bitIntegerValueArray[array_update_index] = decimalval;
      array_update_index -=1;
     }

  }
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
uint RightRotate(uint value, int positions)
  {
// Ensure positions is within 0-31 range
   positions = positions % 32;

// Perform the right rotate operation
   uint rotated = (value >> positions) | (value << (32 - positions));

// Return the rotated result
   return rotated;
  }

//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
string charToHex(uint &characters[])
  {
   string results = "";
   for(int i=0; i<ArraySize(characters); i++)
      results += StringFormat("%.2X", characters[i]);
   return (results);
  }

//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
class HMacSha256
  {
private:

public:
   uint              k_ipad[64];

   uint              k_opad[64];
   uint              K[];
   string            hexval;
                     HMacSha256(string key, string message);
                    ~HMacSha256() {};
   CSha256Class      myshaclass;
   void              ProcessKey(string key);
  };

//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
HMacSha256::HMacSha256(string key,string message)
  {
//process key and add zeros to complete n bytes of 64
   ProcessKey(key);

   for(int i=0;i<64;i++)
     {
      uint keyval = K[i];
      k_ipad[i] = 0x36 ^ keyval;
      k_opad[i] = 0x5c ^ keyval;
     }

//text chars
   uchar messageCharacters[];
   StringToCharArray(message, messageCharacters);
   int innerPaddingLength = 64+StringLen(message);

   uint innerPadding[];
   ArrayResize(innerPadding, innerPaddingLength);

   for(int i=0;i<64;i++)
      innerPadding[i] = k_ipad[i];

   int msg_counts = 0;
   for(int i=64;i<innerPaddingLength;i++)
     {
      innerPadding[i] = (uint)messageCharacters[msg_counts];
      msg_counts +=1;
     }

//send inner padding for hashing
   CSha256Class innerpaddHasher;
   innerpaddHasher.UpdateHash(innerPadding, ArraySize(innerPadding));


   uint ipad_digest_result[];
   innerpaddHasher.GetDigest(ipad_digest_result);

//   merge digest with outer padding
   uint outerpadding[];
   int outerpaddSize = 64 + ArraySize(ipad_digest_result);

   ArrayResize(outerpadding, outerpaddSize);
   for(int i=0;i<64;i++)
      outerpadding[i] = k_opad[i];

   int inner_counts = 0;
   for(int i=64;i<outerpaddSize;i++)
     {
      outerpadding[i] = ipad_digest_result[inner_counts];
      inner_counts+=1;
     }

   CSha256Class outerpaddHash;
   outerpaddHash.UpdateHash(outerpadding, ArraySize(outerpadding));
   hexval = outerpaddHash.GetHex();
  }

//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
void HMacSha256::ProcessKey(string key)
  {
   int keyLength = StringLen(key);

   if(keyLength>64)
     {
      uchar keyCharacters[];

      StringToCharArray(key, keyCharacters);
      uint KeyCharuint[];
      ArrayResize(KeyCharuint, keyLength);

      for(int i=0;i<keyLength;i++)
         KeyCharuint[i] = (uint)keyCharacters[i];

      CSha256Class keyhasher;
      keyhasher.UpdateHash(KeyCharuint, keyLength);
      uint digestValue[];
      keyhasher.GetDigest(digestValue);
      ArrayResize(K, 64);

      for(int i=0;i<ArraySize(digestValue);i++)
         K[i] = digestValue[i];

      for(int i=ArraySize(digestValue);i<64;i++)
         K[i] = 0x00;

     }
   else
     {
      uchar keyCharacters[];

      StringToCharArray(key, keyCharacters);
      ArrayResize(K, 64);

      for(int i=0;i<keyLength;i++)
         K[i] = (uint)keyCharacters[i];

      for(int i=keyLength;i<64;i++)
         K[i] = 0x00;
     }
  }
//+------------------------------------------------------------------+
