//+------------------------------------------------------------------+
//|                                           MoneyBAYES.mqh |
//|                             Copyright 2000-2024, MetaQuotes Ltd. |
//|                                             https://www.mql5.com |
//+------------------------------------------------------------------+
#include <Expert\ExpertMoney.mqh>
#include <Trade\DealInfo.mqh>
// wizard description start
//+------------------------------------------------------------------+
//| Description of the class                                         |
//| Title=Trading with Gaussian Process/ATR optimized trade volume   |
//| Type=Money                                                       |
//| Name=GAUSS                                                       |
//| Class=CMoneyGAUSS                                                |
//| Page=                                                            |
//| Parameter=DecreaseFactor,double,3.0,Decrease factor              |
//| Parameter=Percent,double,10.0,Percent                            |
//| Parameter=kernel,int,1,kernel Type                               |
//| Parameter=Past,int,65,Past                                       |
//| Parameter=Next,int,5,Next                                        |
//| Parameter=Constant,double,0.1,Linear Constant                    |
//+------------------------------------------------------------------+
// wizard description end
//+------------------------------------------------------------------+
//| Class CMoneyGAUSS.                                               |
//| Purpose: Class of money management with Gaussian Process/ATR optimized size. |
//|              Derives from class CExpertMoney.                    |
//+------------------------------------------------------------------+
class CMoneyGAUSS : public CExpertMoney
{
protected:
   
   CiATR             m_atr;
   
   int               m_past;
   int               m_next;
   double            m_constant;
   int               m_kernel;

   double            m_decrease_factor;

public:
                     CMoneyGAUSS(void);
                    ~CMoneyGAUSS(void);

   //--- methods of setting adjustable parameters
   void              Past(int value)
   {  m_past = value;
   }
   void              Next(int value)
   {  m_next = value;
   }
   void              Constant(double value)
   {  m_constant = value;
   }
   void              kernel(int value)
   {  m_kernel = value;
   }

   //---
   void              DecreaseFactor(double decrease_factor)
   {  m_decrease_factor = decrease_factor;
   }
   virtual bool      ValidationSettings(void);
   //--- method of creating the indicator and timeseries
   virtual bool      InitIndicators(CIndicators *indicators);
   //---
   virtual double    CheckOpenLong(double price, double sl);
   virtual double    CheckOpenShort(double price, double sl);

protected:
   double            Optimize(double lots);
   void              GetOutput(double BasisMean, vector &Output);
   void              SetOutput(vector &Output);
   matrix            Linear_Kernel(vector &Rows, vector &Cols);
   matrix            Matern_Kernel(vector &Rows, vector &Cols);
};
//+------------------------------------------------------------------+
//| Constructor                                                      |
//+------------------------------------------------------------------+
void CMoneyGAUSS::CMoneyGAUSS(void) :  m_decrease_factor(3.0),
   m_past(65),
   m_next(5),
   m_constant(0.1),
   m_kernel(0)
{
//--- initialization of protected data
   m_used_series = USE_SERIES_HIGH + USE_SERIES_LOW;
}
//+------------------------------------------------------------------+
//| Destructor                                                       |
//+------------------------------------------------------------------+
void CMoneyGAUSS::~CMoneyGAUSS(void)
{
}
//+------------------------------------------------------------------+
//| Validation settings protected data.                              |
//+------------------------------------------------------------------+
bool CMoneyGAUSS::ValidationSettings(void)
{  if(!CExpertMoney::ValidationSettings())
      return(false);
//--- initial data checks
   if(m_decrease_factor <= 0.0)
   {  printf(__FUNCTION__ + ": decrease factor must be greater then 0");
      return(false);
   }
//--- ok
   return(true);
}
//+------------------------------------------------------------------+
//| Create indicators.                                               |
//+------------------------------------------------------------------+
bool CMoneyGAUSS::InitIndicators(CIndicators *indicators)
{
//--- check pointer
   if(indicators == NULL)
      return(false);
//--- initialization of indicators and timeseries of additional filters
   if(!CMoneyGAUSS::InitIndicators(indicators))
      return(false);
//--- initialization of indicators and timeseries of additional filters
   if(!m_atr.Create(m_symbol.Name(), m_period, m_past))
   {  printf(__FUNCSIG__+" failed to create atr indicator! ");
      return(false);
   }
//--- ok
   return(true);
}
//+------------------------------------------------------------------+
//| Getting lot size for open long position.                         |
//+------------------------------------------------------------------+
double CMoneyGAUSS::CheckOpenLong(double price, double sl)
{  if(m_symbol == NULL)
      return(0.0);
//--- select lot size
   double lot;
   if(price == 0.0)
      lot = m_account.MaxLotCheck(m_symbol.Name(), ORDER_TYPE_BUY, m_symbol.Ask(), m_percent);
   else
      lot = m_account.MaxLotCheck(m_symbol.Name(), ORDER_TYPE_BUY, price, m_percent);
//--- return trading volume
   return(Optimize(lot));
}
//+------------------------------------------------------------------+
//| Getting lot size for open short position.                        |
//+------------------------------------------------------------------+
double CMoneyGAUSS::CheckOpenShort(double price, double sl)
{  if(m_symbol == NULL)
      return(0.0);
//--- select lot size
   double lot;
   if(price == 0.0)
      lot = m_account.MaxLotCheck(m_symbol.Name(), ORDER_TYPE_SELL, m_symbol.Bid(), m_percent);
   else
      lot = m_account.MaxLotCheck(m_symbol.Name(), ORDER_TYPE_SELL, price, m_percent);
//--- return trading volume
   return(Optimize(lot));
}
//+------------------------------------------------------------------+
//| Optimizing lot size for open.                                    |
//+------------------------------------------------------------------+
double CMoneyGAUSS::Optimize(double lots)
{  double lot = lots;
//--- calculate number of losses orders without a break
   if(m_decrease_factor > 0)
   {  //--- select history for access
      HistorySelect(0, TimeCurrent());
      //---
      int       orders = HistoryDealsTotal(); // total history deals
      int       losses = 0;                  // number of consequent losing orders
      //--
      int      size = 0;
      //--
      CDealInfo deal;
      //---
      for(int i = orders - 1; i >= 0; i--)
      {  deal.Ticket(HistoryDealGetTicket(i));
         if(deal.Ticket() == 0)
         {  Print("CMoneySizeOptimized::Optimize: HistoryDealGetTicket failed, no trade history");
            break;
         }
         //--- check symbol
         if(deal.Symbol() != m_symbol.Name())
            continue;
         //--- check profit
         double profit = deal.Profit();
         //--
         if(profit < 0.0)
            losses++;
      }
      //--
      double _cond = 0.0;
      //--
      vector _o;
      GetOutput(0.0, _o);
      //---
      //decrease lots on rising ATR
      if(_o[_o.Size()-1] > _o[0])
         lot = NormalizeDouble(lot - lot * losses / m_decrease_factor, 2);
   }
//--- normalize and check limits
   double stepvol = m_symbol.LotsStep();
   lot = stepvol * NormalizeDouble(lot / stepvol, 0);
//---
   double minvol = m_symbol.LotsMin();
   if(lot < minvol)
      lot = minvol;
//---
   double maxvol = m_symbol.LotsMax();
   if(lot > maxvol)
      lot = maxvol;
//---
   return(lot);
}
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
void CMoneyGAUSS::GetOutput(double BasisMean, vector &Output)
{  vector _past_old, _past_new, _past, _past_time;
   _past_old.CopyIndicatorBuffer(m_atr.Handle(), 0, 0, m_past);
   _past_new.CopyIndicatorBuffer(m_atr.Handle(), 0, 1, m_past);
   _past = _past_new - _past_old;
   SetOutput(_past);
   _past_time.Init(m_past);
   for(int i = 0; i < m_past; i++)
   {  _past_time[i] = i;
   }
   matrix _k = Linear_Kernel(_past_time, _past_time);
   matrix _noise;
   _noise.Init(_k.Rows(), _k.Cols());
   _noise.Fill(fabs(0.05 * _k.Min()));
   _k += _noise;
   matrix _norm;
   _norm.Init(_k.Rows(), _k.Cols());
   _norm.Identity();
   _norm *= 0.0005;
   _k += _norm;
   vector _next_time;
   _next_time.Init(m_next);
   for(int i = 0; i < m_next; i++)
   {  _next_time[i] = _past_time[_past_time.Size() - 1] + i + 1;
   }
   matrix _k_s;
   matrix _k_ss;
   _k_s.Init(_next_time.Size(), _past_time.Size());
   _k_ss.Init(_next_time.Size(), _next_time.Size());
   if(m_kernel == 0)
   {  _k_s = Linear_Kernel(_next_time, _past_time);
      _k_ss = Linear_Kernel(_next_time, _next_time);
   }
   else if(m_kernel == 1)
   {  _k_s = Matern_Kernel(_next_time, _past_time);
      _k_ss = Matern_Kernel(_next_time, _next_time);
   }
// Compute K^-1 * y
   matrix _k_inv = _k.Inv();
   if(_k_inv.Rows() > 0 && _k_inv.Cols() > 0)
   {  vector _alpha = _k_inv.MatMul(_past);
      // Compute mean predictions: mu_* = K_s * alpha
      vector _mu_star = _k_s.MatMul(_alpha);
      vector _mean;
      _mean.Init(_mu_star.Size());
      _mean.Fill(BasisMean);
      _mu_star += _mean;
      // Compute covariance: Sigma_* = K_ss - K_s * K_inv * K_s^T
      matrix _v = _k_s.MatMul(_k_inv);
      matrix _sigma_star = _k_ss - (_v.MatMul(_k_s.Transpose()));
      vector _variances = _sigma_star.Diag();
      Output = _mu_star;
      SetOutput(Output);
      SetOutput(_variances);
   }
}
//+------------------------------------------------------------------+
//| Normalizing to 0 - 1 range
//+------------------------------------------------------------------+
void CMoneyGAUSS::SetOutput(vector &Output)
{  vector _copy;
   _copy.Copy(Output);
   if(Output.HasNan() == 0 && _copy.Max() - _copy.Min() > 0.0)
   {  for (int i = 0; i < int(Output.Size()); i++)
      {  if(_copy[i] >= 0.0)
         {  Output[i] = 0.5 + (0.5 * ((_copy[i] - _copy.Min()) / (_copy.Max() - _copy.Min())));
         }
         else if(_copy[i] < 0.0)
         {  Output[i] = (0.5 * ((_copy[i] - _copy.Min()) / (_copy.Max() - _copy.Min())));
         }
      }
   }
   else
   {  Output.Fill(0.5);
   }
}
//+------------------------------------------------------------------+
// Linear Kernel Function
//+------------------------------------------------------------------+
matrix CMoneyGAUSS::Linear_Kernel(vector &Rows, vector &Cols)
{  matrix _linear, _c;
   _linear.Init(Rows.Size(), Cols.Size());
   _c.Init(Rows.Size(), Cols.Size());
   for(int i = 0; i < int(Rows.Size()); i++)
   {  for(int ii = 0; ii < int(Cols.Size()); ii++)
      {  _linear[i][ii] = Rows[i] * Cols[ii];
      }
   }
   _c.Fill(m_constant);
   _linear += _c;
   return(_linear);
}
//+------------------------------------------------------------------+
// Matern Kernel Function
//+------------------------------------------------------------------+
matrix CMoneyGAUSS::Matern_Kernel(vector &Rows, vector &Cols)
{  matrix _matern;
   _matern.Init(Rows.Size(), Cols.Size());
   for(int i = 0; i < int(Rows.Size()); i++)
   {  for(int ii = 0; ii < int(Cols.Size()); ii++)
      {  _matern[i][ii] = (1.0 + (sqrt(3.0) * fabs(Rows[i] - Cols[ii]) / m_next)) * exp(-1.0 * sqrt(3.0) * fabs(Rows[i] - Cols[ii]) / m_next);
      }
   }
   return(_matern);
}
//+------------------------------------------------------------------+
