//+------------------------------------------------------------------+
//|                                                  CBaseSymbol.mqh |
//|                                           Copyright 2022, denkir |
//|                             https://www.mql5.com/ru/users/denkir |
//+------------------------------------------------------------------+
#property copyright "Copyright 2022, denkir"
#property link      "https://www.mql5.com/ru/users/denkir"
//--- include
#include <Trade\SymbolInfo.mqh>
//+------------------------------------------------------------------+
//| Class CBaseSymbol                                                |
//+------------------------------------------------------------------+
class CBaseSymbol : public CObject
  {
      //--- === Data members === ---
   private:
      CSymbolInfo    m_symbol;
      ENUM_TIMEFRAMES m_tf;
      matrix         m_ticks_mx;
      datetime       m_start_date;
      ulong          m_last_idx;
      //--- === Methods === ---
   public:
      //--- constructor/destructor
      void           CBaseSymbol(void);
      void          ~CBaseSymbol(void) {};
      //---
      bool           Init(const string _symbol, const ENUM_TIMEFRAMES _tf, datetime start_date);
      int            CheckLoadHistory(void);
      bool           LoadTicks(const datetime _stop_date, const uint _flags);
      matrix         GetTicks(void) const
        {
         return m_ticks_mx;
        };
      bool           SearchTickLessOrEqual(const double _dbl_time, vector &_res_row);
      bool           CopyLastTick(vector &_res_row);
  };
//+------------------------------------------------------------------+
//| Constructor                                                      |
//+------------------------------------------------------------------+
void CBaseSymbol::CBaseSymbol(void)
  {
   m_ticks_mx.Init(0, 0);
   m_start_date=0;
   m_last_idx=0;
  };
//+------------------------------------------------------------------+
//| Initialization                                                   |
//+------------------------------------------------------------------+
bool CBaseSymbol::Init(const string _symbol, const ENUM_TIMEFRAMES _tf, datetime _start_date)
  {
   if(m_symbol.Name(_symbol))
      if(m_symbol.Select(true))
        {
         m_tf=_tf;
         m_start_date=_start_date;
         return true;
        }
//---
   return false;
  }
//+------------------------------------------------------------------+
//| Check and load quotes history                                    |
//+------------------------------------------------------------------+
int CBaseSymbol::CheckLoadHistory(void)
  {
   datetime first_date=0;
   datetime times[100];
   string curr_symbol=m_symbol.Name();
//ENUM_TIMEFRAMES period=m_tf;
//--- check if symbol is selected in the Market Watch
   if(!m_symbol.Select())
     {
      if(::GetLastError()==ERR_MARKET_UNKNOWN_SYMBOL)
         return -1;
      m_symbol.Select(true);
     }
//--- check if data is present
   ::SeriesInfoInteger(curr_symbol, m_tf, SERIES_FIRSTDATE, first_date);
   if(first_date>0 && first_date<=m_start_date) return 1;
//--- don't ask for load of its own data if it is an indicator
   if(::MQLInfoInteger(MQL_PROGRAM_TYPE)==PROGRAM_INDICATOR &&
         ::Period()==m_tf &&::Symbol()==curr_symbol)
      return -4;
//--- second attempt
   if(::SeriesInfoInteger(curr_symbol, PERIOD_M1, SERIES_TERMINAL_FIRSTDATE, first_date))
     {
      //--- there is loaded data to build timeseries
      if(first_date>0)
        {
         //--- force timeseries build
         ::CopyTime(curr_symbol, m_tf, first_date+::PeriodSeconds(m_tf), 1, times);
         //--- check date
         if(::SeriesInfoInteger(curr_symbol, m_tf, SERIES_FIRSTDATE, first_date))
            if(first_date>0 && first_date<=m_start_date) return 2;
        }
     }
//--- max bars in chart from terminal options
   int max_bars=::TerminalInfoInteger(TERMINAL_MAXBARS);
//--- load symbol history info
   datetime first_server_date=0;
   while(!::SeriesInfoInteger(curr_symbol, PERIOD_M1, SERIES_SERVER_FIRSTDATE,
                              first_server_date) && !::IsStopped())
      ::Sleep(5);
//--- fix start date for loading
   if(first_server_date>m_start_date) m_start_date=first_server_date;
   if(first_date>0 && first_date<first_server_date)
      ::Print("Warning: first server date ", first_server_date, " for ", curr_symbol,
              " does not match to first series date ", first_date);
//--- load data step by step
   int fail_cnt=0;
   while(!::IsStopped())
     {
      //--- wait for timeseries build
      while(!::SeriesInfoInteger(curr_symbol, m_tf, SERIES_SYNCHRONIZED) && !::IsStopped())
         ::Sleep(5);
      //--- ask for built bars
      int bars=::Bars(curr_symbol, m_tf);
      if(bars>0)
        {
         if(bars>=max_bars) return(-2);
         //--- ask for first date
         if(::SeriesInfoInteger(curr_symbol, m_tf, SERIES_FIRSTDATE, first_date))
            if(first_date>0 && first_date<=m_start_date) return 0;
        }
      //--- copying of next part forces data loading
      int copied=::CopyTime(curr_symbol, m_tf, bars, 100, times);
      if(copied>0)
        {
         //--- check for data
         if(times[0]<=m_start_date)  return 0;
         if(bars+copied>=max_bars) return -2;
         fail_cnt=0;
        }
      else
        {
         //--- no more than 100 failed attempts
         fail_cnt++;
         if(fail_cnt>=100) return -5;
         ::Sleep(10);
        }
     }
//--- stopped
   return -3;
  }
//+------------------------------------------------------------------+
//| Load ticks                                                       |
//+------------------------------------------------------------------+
bool CBaseSymbol::LoadTicks(const datetime _stop_date, const uint _flags)
  {
   string curr_symbol=m_symbol.Name();
   ulong from_msc, to_msc;
   from_msc=1000*m_start_date;
   to_msc=1000*_stop_date;
   if(m_ticks_mx.CopyTicksRange(curr_symbol, _flags, from_msc, to_msc))
     {
      m_ticks_mx=m_ticks_mx.Transpose();
      return true;
     }
//---
   return false;
  }
//+------------------------------------------------------------------+
//| Search a tick less or equal                                      |
//+------------------------------------------------------------------+
bool CBaseSymbol::SearchTickLessOrEqual(const double _dbl_time, vector &_res_row)
  {
   ulong ticks_size=m_ticks_mx.Rows();
   if(_res_row.Resize(2))
     {
      _res_row.Fill(0.);
      double prev_dbl_time, curr_dbl_time;
      prev_dbl_time=curr_dbl_time=0.;
      for(ulong idx=m_last_idx; idx<ticks_size && prev_dbl_time<_dbl_time; idx++)
        {
         curr_dbl_time=m_ticks_mx[idx][0];
         if(idx>0)
            prev_dbl_time=m_ticks_mx[idx-1][0];;
         ulong idx_for_tick=WRONG_VALUE;
         if(curr_dbl_time==_dbl_time)
           {
            idx_for_tick=idx;
           }
         else if(prev_dbl_time<_dbl_time && curr_dbl_time>_dbl_time)
           {
            if(prev_dbl_time>0.)
               idx_for_tick=idx-1;
           }
         if(idx_for_tick!=WRONG_VALUE)
           {
            _res_row[0]=m_ticks_mx[idx_for_tick][1];
            _res_row[1]=m_ticks_mx[idx_for_tick][2];
            m_last_idx=idx_for_tick; // todo
            return true;
           }
        }
     }
//---
   return false;
  }
//+------------------------------------------------------------------+
//| Copy the last tick                                               |
//+------------------------------------------------------------------+
bool CBaseSymbol::CopyLastTick(vector &_res_row)
  {
   if(_res_row.Resize(2))
     {
      _res_row.Fill(0.);
      if(m_symbol.RefreshRates())
        {
         _res_row[0]=m_symbol.Bid();
         _res_row[1]=m_symbol.Ask();
         return true;
        }
     }
//---
   return false;
  }
//+------------------------------------------------------------------+
