//+------------------------------------------------------------------+
//|                            BinaryCustomSymbolWithTickHistory.mq5 |
//|                                  Copyright 2021, MetaQuotes Ltd. |
//|                                             https://www.mql5.com |
//+------------------------------------------------------------------+
#property copyright "Copyright 2021, MetaQuotes Ltd."
#property link      "https://www.mql5.com"
#property version   "1.00"
#include<websocket.mqh>
#include<JAson.mqh>
#include<Files/FileTxt.mqh>

#define BINARY_URL "ws.binaryws.com/websockets/v3?app_id="
#define BINARY_SYMBOL_SETTINGS "binarysymbolset.json"
#define BINARY_SYMBOL_BASE_PATH "Binary.com\\"

enum ENUM_BINARY_SYMBOL
{
 BINARY_1HZ10V=0,//Volatility 10 (1s)
 BINARY_1HZ25V,//Volatility 25 (1s)
 BINARY_1HZ50V,//Volatility 50 (1s)
 BINARY_1HZ75V,//Volatility 75 (1s)
 BINARY_1HZ100V,//Volatility 100 (1s)
 BINARY_1HZ200V,//Volatility 200 (1s)
 BINARY_1HZ300V,//Volatility 300 (1s)
 BINARY_BOOM300N,//BOOM 300
 BINARY_BOOM500,//BOOM 500
 BINARY_BOOM1000,//BOOM 1000
 BINARY_CRASH300N,//CRASH 300
 BINARY_CRASH500,//CRASH 500
 BINARY_CRASH1000,//CRASH 1000
 BINARY_cryBTCUSD,//BTCUSD
 BINARY_cryETHUSD,//ETHUSD
 BINARY_frxAUDCAD,//AUDCAD
 BINARY_frxAUDCHF,//AUDCHF
 BINARY_frxAUDJPY,//AUDJPY
 BINARY_frxAUDNZD,//AUDNZD
 BINARY_frxAUDUSD,//AUDUSD
 BINARY_frxBROUSD,//BROUSD
 BINARY_frxEURAUD,//EURAUD
 BINARY_frxEURCAD,//EURCAD
 BINARY_frxEURCHF,//EURCHF
 BINARY_frxEURGBP,//EURGBP
 BINARY_frxEURJPY,//EURJPY
 BINARY_frxEURNZD,//EURNZD
 BINARY_frxEURUSD,//EURUSD
 BINARY_frxGBPAUD,//GBPAUD
 BINARY_frxGBPCAD,//GBPCAD
 BINARY_frxGBPCHF,//GBPCHF
 BINARY_frxGBPJPY,//GBPJPY
 BINARY_frxGBPNOK,//GBPNOK
 BINARY_frxGBPNZD,//GBPNZD
 BINARY_frxGBPUSD,//GBPUSD
 BINARY_frxNZDJPY,//NZDJPY
 BINARY_frxNZDUSD,//NZDUSD
 BINARY_frxUSDCAD,//USDCAD
 BINARY_frxUSDCHF,//USDCHF
 BINARY_frxUSDJPY,//USDJPY
 BINARY_frxUSDMXN,//USDMXN
 BINARY_frxUSDNOK,//USDNOK
 BINARY_frxUSDPLN,//USDPLN
 BINARY_frxUSDSEK,//USDSEK
 BINARY_frxXAUUSD,//XAUUSD
 BINARY_frxXAGUSD,//XAGUSD
 BINARY_frxXPDUSD,//XPDUSD
 BINARY_frxXPTUSD,//XPTUSD
 BINARY_JD10,//Jump 10 Index
 BINARY_JD25,//Jump 25 Index
 BINARY_JD50,//Jump 50 Index
 BINARY_JD75,//Jump 75 Index
 BINARY_JD100,//Jump 100 Index
 BINARY_OTC_AEX,//Dutch Index
 BINARY_OTC_AS51,//Australian Index
 BINARY_OTC_DJI,//Wall Street Index
 BINARY_OTC_FCHI,//French Index 
 BINARY_OTC_FTSE,//UK Index
 BINARY_OTC_GDAXI,//German Index
 BINARY_OTC_HSI,//Hong Kong Index
 BINARY_OTC_N225,//Japanese Index
 BINARY_OTC_NDX,//US Tech Index
 BINARY_OTC_SPC,//US Index
 BINARY_OTC_SSMI,//Swiss Index 
 BINARY_OTC_SX5E,//Euro 50 Index
 BINARY_R_10,//Volatility 10 Index
 BINARY_R_25,//Volatility 25 Index
 BINARY_R_50,//Volatility 50 Index
 BINARY_R_75,//Volatility 75 Index
 BINARY_R_100,//Volatility 100 Index
 BINARY_RDBEAR,//Bear Market Index
 BINARY_RDBULL,//Bull Market Index
 BINARY_stpRNG,//Step Index
 BINARY_WLDAUD,//AUD Index
 BINARY_WLDEUR,//EUR Index
 BINARY_WLDGBP,//GBP Index
 BINARY_WLDUSD,//USD Index
 BINARY_WLDXAU//Gold Index
};


input string binary_appid="";//Binary.com registered application ID
input ENUM_BINARY_SYMBOL binary_symbol=BINARY_cryBTCUSD;//Binary.com symbol
input ENUM_TIMEFRAMES binary_timeframe=PERIOD_M1;//Chart period


//+------------------------------------------------------------------+
//|General class for creating custom symbols from external source    |
//+------------------------------------------------------------------+
class CCustomSymbol
  {
protected:
   string            m_symbol_name;       //symbol name
   datetime          m_history_start;     //existing tick history start date
   datetime          m_history_end;       //existing tick history end date
   bool              m_new;               //flag specifying whether a symbol has just been created or already exists in the terminal
   ENUM_TIMEFRAMES   m_chart_tf;          //chart timeframe
public:
   //constructor
                     CCustomSymbol(void)
     {
      m_symbol_name=NULL;
      m_chart_tf=PERIOD_M1;
      m_history_start=0;
      m_history_end=0;
      m_new=false;
     }
   //destructor
                    ~CCustomSymbol(void)
     {

     }
   //method for initializing symbol, sets the symbol name and chart timeframe properties
   virtual bool      Initialize(const string sy,string sy_path=NULL, ENUM_TIMEFRAMES chart_tf=PERIOD_M1)
     {
      m_symbol_name=sy;
      m_chart_tf=chart_tf;
      return(InitSymbol(sy_path));
     }
   //gets the symbol name
   string            Name(void) const
     {
      return(m_symbol_name);
     }
   //sets the history start date
   bool              SetHistoryStartDate(const datetime startime)
     {
      if(startime>=TimeLocal())
        {
         Print("Invalid history start time");
         return(false);
        }

      m_history_start=startime;

      return(true);
     }
   //gets the history start date
   datetime          GetHistoryStartDate(void)
     {
      return(m_history_start);
     }
   //general methods for setting the properties of the custom symbol
   bool              SetProperty(const ENUM_SYMBOL_INFO_DOUBLE Property, double Value) const
     {
      return(::CustomSymbolSetDouble(m_symbol_name, Property, Value));
     }

   bool              SetProperty(const ENUM_SYMBOL_INFO_INTEGER Property, long Value) const
     {
      return(::CustomSymbolSetInteger(m_symbol_name, Property, Value));
     }

   bool              SetProperty(const ENUM_SYMBOL_INFO_STRING Property, string Value) const
     {
      return(::CustomSymbolSetString(m_symbol_name, Property, Value));
     }
   //general methods for getting the symbol properties of the custom symbol
   long              GetProperty(const ENUM_SYMBOL_INFO_INTEGER Property) const
     {
      return(::SymbolInfoInteger(m_symbol_name, Property));
     }

   double            GetProperty(const ENUM_SYMBOL_INFO_DOUBLE Property) const
     {
      return(::SymbolInfoDouble(m_symbol_name, Property));
     }

   string            GetProperty(const ENUM_SYMBOL_INFO_STRING Property) const
     {
      return(::SymbolInfoString(m_symbol_name, Property));
     }
   //method for deleting a custom symbol
   bool              Delete(void)
     {
      return((bool)(GetProperty(SYMBOL_CUSTOM)) && DeleteAllCharts()  && ::CustomSymbolDelete(m_symbol_name) && SymbolSelect(m_symbol_name,false));
     }
   //unimplemented virtual method for adding new ticks
   virtual void      AddTick(void)
     {
      return;
     }
   //unimplemented virtual method for aquiring  either ticks or candle history from an external source
   virtual bool      UpdateHistory(void)
     {
      //OpenChart();
      return(false);
     }

protected:
   //checks if the symbol already exists or not
   bool              SymbolExists(void)
     {
      return(SymbolSelect(m_symbol_name,true));
     }
   //method that opens a new chart according to the m_chart_tf property
   void              OpenChart(void)
     {
      long Chart = ::ChartFirst();

      bool opened=false;

      while(Chart != -1)
        {
         if((::ChartSymbol(Chart) == m_symbol_name))
           {
            ChartRedraw(Chart);
            if(ChartPeriod(Chart)==m_chart_tf)
               opened=true;
           }
         Chart = ::ChartNext(Chart);
        }

      if(!opened)
        {
         long id = ChartOpen(m_symbol_name,m_chart_tf);
         if(id == 0)
           {
            Print("Can't open new chart for " + m_symbol_name + ", code: " + (string)GetLastError());
            return;
           }
         else
           {
            Sleep(1000);
            ChartSetSymbolPeriod(id, m_symbol_name, m_chart_tf);
            ChartSetInteger(id, CHART_MODE,CHART_CANDLES);
           }
        }
     }
   //deletes all charts for the specified symbol
   bool              DeleteAllCharts(void)
     {
      long Chart = ::ChartFirst();

      while(Chart != -1)
        {
         if((Chart != ::ChartID()) && (::ChartSymbol(Chart) == m_symbol_name))
            if(!ChartClose(Chart))
              {
               Print("Error closing chart id ", Chart, m_symbol_name, ChartPeriod(Chart));
               return(false);
              }

         Chart = ::ChartNext(Chart);
        }

      return(true);
     }
   //helper method that initializes a custom symbol
   bool              InitSymbol(const string _path=NULL)
     {
      if(!SymbolExists())
        {
         if(!CustomSymbolCreate(m_symbol_name,_path))
           {
            Print("error creating custom symbol ", ::GetLastError());
            return(false);
           }

         if(!SetProperty(SYMBOL_CHART_MODE,SYMBOL_CHART_MODE_BID)   ||
            !SetProperty(SYMBOL_SWAP_MODE,SYMBOL_SWAP_MODE_DISABLED) ||
            !SetProperty(SYMBOL_TRADE_MODE,SYMBOL_TRADE_MODE_DISABLED))
           {
            Print("error setting symbol properties");
            return(false);
           }

         if(!SymbolSelect(m_symbol_name,true))
           {
            Print("error adding symbol to market watch",::GetLastError());
            return(false);
           }

         m_new=true;

         return(true);
        }
      else
        {
         long custom=GetProperty(SYMBOL_CUSTOM);

         if(!custom)
           {
            Print("Error, symbol is not custom ",m_symbol_name,::GetLastError());
            return(false);
           }

         m_history_end=GetLastBarTime();
         m_history_start=GetFirstBarTime();
         m_new=false;

         return(true);
        }
     }

   //gets the last tick time for an existing custom symbol

   datetime          GetLastTickTime(void)
     {
      MqlTick tick;

      ZeroMemory(tick);

      if(!SymbolInfoTick(m_symbol_name,tick))
        {
         Print("symbol info tick failure ", ::GetLastError());
         return(0);
        }
      else
         return(tick.time);
     }


   //gets the last bar time of the one minute timeframe in candle history
   datetime          GetLastBarTime(void)
     {
      MqlRates candle[1];

      ZeroMemory(candle);

      int bars=iBars(m_symbol_name,PERIOD_M1);

      if(bars<=0)
         return(0);

      if(CopyRates(m_symbol_name,PERIOD_M1,0,1,candle)>0)
         return(candle[0].time);
      else
         return(0);
     }
   //gets the first bar time of the one minute timeframe  in candle  history
   datetime          GetFirstBarTime(void)
     {
      MqlRates candle[1];

      ZeroMemory(candle);

      int bars=iBars(m_symbol_name,PERIOD_M1);

      if(bars<=0)
         return(0);

      if(CopyRates(m_symbol_name,PERIOD_M1,bars-1,1,candle)>0)
         return(candle[0].time);
      else
         return(0);

     }


  };

//+------------------------------------------------------------------+
//|Class for creating custom Binary.com specific symbols             |
//+------------------------------------------------------------------+
class CBinarySymbol:public CCustomSymbol
  {
private:
   //private properties
   string            m_appID;      //app id string issued by Binary.com
   string            m_url;        //final url
   string            m_stream_id;  //stream identifier for a symbol
   int               m_index;      //array index
   int               m_max_ticks;  //max number of ticks downloadable from Binary.com
   CWebsocket*       websocket;    //websocket client
   CJAVal*           json;         //utility json object
   CJAVal*           symbolSpecs;  //json object storing symbol specification
   //private methods
   bool              CheckBinaryError(CJAVal &j);
   bool              GetSymbolSettings(void);
public:
   //Constructor
                     CBinarySymbol(void):m_appID(NULL),
                     m_url(NULL),
                     m_stream_id(NULL),
                     m_index(-1),
                     m_max_ticks(86400)
     {
      json=new CJAVal();
      symbolSpecs=new CJAVal();
      websocket=new CWebsocket();
     }
   //Destructor
                    ~CBinarySymbol(void)
     {
      if(CheckPointer(websocket)==POINTER_DYNAMIC)
        {
         if(m_stream_id!="")
            StopTicksStream();
         delete websocket;
        }

      if(CheckPointer(json)==POINTER_DYNAMIC)
         delete json;

      if(CheckPointer(symbolSpecs)==POINTER_DYNAMIC)
         delete symbolSpecs;

      Comment("");

     }
   //public methods
   virtual void      AddTick(void) override;
   virtual bool      Initialize(const string sy,string sy_path=NULL, ENUM_TIMEFRAMES chart_tf=PERIOD_M1) override;
   virtual bool      UpdateHistory(void) override;
   void              SetMaximumHistoryTicks(const int max) { m_max_ticks=(max>=5000)?max:86400; }
   void              SetAppID(const string id);
   bool              StartTicksStream(void);
   bool              StopTicksStream(void);
  };

//+------------------------------------------------------------------+
//|sets the the application id used to consume binary.com api        |
//+------------------------------------------------------------------+
void CBinarySymbol::SetAppID(const string id)
  {
   if(m_appID!=NULL && StringCompare(id,m_appID,false))
      websocket.Abort();

   m_appID=id;
   m_url=BINARY_URL+m_appID;

  }
//+------------------------------------------------------------------+
//|Begins process of creating custom symbol                          |
//+------------------------------------------------------------------+
bool CBinarySymbol::Initialize(const string sy,string sy_path=NULL, ENUM_TIMEFRAMES chart_tf=PERIOD_M1)
  {

   if(CheckPointer(websocket)==POINTER_INVALID || CheckPointer(json)==POINTER_INVALID || CheckPointer(symbolSpecs)==POINTER_INVALID)
     {
      Print("Invalid pointer found ");
      return(false);
     }
     
     
   if(m_appID=="")
     {
      Alert("Application ID has not been set, It is required for the program to work");
      return(false);
     }

   m_symbol_name=(StringFind(sy,"BINARY_")>=0)?StringSubstr(sy,7):sy;
   m_chart_tf=chart_tf;

   Comment("Initializing Symbol "+m_symbol_name+".......");

   if(!GetSymbolSettings())
      return(false);

   string s_path=BINARY_SYMBOL_BASE_PATH+symbolSpecs["active_symbols"][m_index]["market_display_name"].ToStr();
   string symbol_description=symbolSpecs["active_symbols"][m_index]["display_name"].ToStr();
   double s_point=symbolSpecs["active_symbols"][m_index]["pip"].ToDbl();
   int s_digits=(int)MathAbs(MathLog10(s_point));

   if(!InitSymbol(s_path))
      return(false);

   if(m_new)
     {
      if(!SetProperty(SYMBOL_DESCRIPTION,symbol_description) ||
         !SetProperty(SYMBOL_POINT,s_point)                 ||
         !SetProperty(SYMBOL_DIGITS,s_digits))
        {
         Print("error setting symbol properties ", ::GetLastError());
         return(false);
        }
     }

   Comment("Symbol "+m_symbol_name+" initialized.......");

           return(true);
  }

//+------------------------------------------------------------------+
//|Overridden method that handles new ticks streamed from binary.com |
//+------------------------------------------------------------------+
void CBinarySymbol::AddTick(void)
  {
   string str_tick;

   MqlTick current_tick[1];

   json.Clear();

   if(websocket.ReadString(str_tick))
     {
      json.Deserialize(str_tick);
      ZeroMemory(current_tick);

      if(CheckBinaryError(json))
         return;

      if(!json["tick"]["ask"].ToDbl())
         return;

      current_tick[0].ask=json["tick"]["ask"].ToDbl();
      current_tick[0].bid=json["tick"]["bid"].ToDbl();
      current_tick[0].last=0;
      current_tick[0].time=(datetime)json["tick"]["epoch"].ToInt();
      current_tick[0].time_msc=(long)((json["tick"]["epoch"].ToInt())*1000);
      current_tick[0].volume=0;
      current_tick[0].volume_real=0;

      if(current_tick[0].ask)
         current_tick[0].flags|=TICK_FLAG_ASK;
      if(current_tick[0].bid)
         current_tick[0].flags|=TICK_FLAG_BID;

      if(m_stream_id==NULL)
         m_stream_id=json["tick"]["id"].ToStr();

      if(CustomTicksAdd(m_symbol_name,current_tick)<0)
        {
         Print("failed to add new tick ", ::GetLastError());
         return;
        }
      Comment("New ticks for  "+m_symbol_name+".......");
     }
   else
     {
      Print("read error ",websocket.LastError(), websocket.LastErrorMessage());

      websocket.ResetLastError();

      if(websocket.ClientState()!=CONNECTED && websocket.Connect(m_url))
        {
         if(m_stream_id!=NULL)
            if(StopTicksStream())
              {
               if(InitSymbol())
                  if(UpdateHistory())
                    {
                     StartTicksStream();
                     return;
                    }
              }
        }
     }
//---
  }

//+------------------------------------------------------------------+
//|method for updating the tick history for a particular symbol      |
//+------------------------------------------------------------------+
bool CBinarySymbol::UpdateHistory(void)
  {
   if(websocket.ClientState()!=CONNECTED && !websocket.Connect(m_url))
     {
      Print(websocket.LastErrorMessage()," : ",websocket.LastError());
      return(false);
     }

   Comment("Updating history for "+m_symbol_name+".......");

   MqlTick history_ticks[];
   string history=NULL;
   json.Clear();

   json["ticks_history"]=m_symbol_name;

   if(m_new)
     {
      if(m_history_start>0)
        {
         json["start"]=(int)(m_history_start);
        }
     }
   else
      if(m_history_end!=0)
        {
         json["start"]=(int)(m_history_start);
        }


   json["count"]=m_max_ticks;
   json["end"]="latest";
   json["style"]="ticks";

   if(!websocket.SendString(json.Serialize()))
     {
      Print(websocket.LastErrorMessage());
      return(false);
     }

   if(websocket.ReadString(history))
     {
      json.Deserialize(history);

      if(CheckBinaryError(json))
         return(false);

      int i=0;


      int z=i;
      int diff=0;


      while(json["history"]["prices"][i].ToDbl()!=0.0)
        {

         diff=(i>0)?(int)(json["history"]["times"][i].ToInt() - json["history"]["times"][i-1].ToInt()):0;//((m_history_end>0)?(json["history"]["times"][i].ToInt() - (int)(m_history_end)):0);

         if(diff > 1)
           {
            int k=z+diff;
            int p=1;

            if(ArrayResize(history_ticks,k,100)!=k)
              {
               Print("Memory allocation error,  "+IntegerToString(::GetLastError()));
               return(false);
              }

            while(z<(k-1))
              {
               history_ticks[z].bid=json["history"]["prices"][i-1].ToDbl();
               history_ticks[z].ask=0;
               history_ticks[z].time=(datetime)(json["history"]["times"][i-1].ToInt()+p);
               history_ticks[z].time_msc=(long)((json["history"]["times"][i-1].ToInt()+p)*1000);
               history_ticks[z].last=0;
               history_ticks[z].volume=0;
               history_ticks[z].volume_real=0;
               history_ticks[z].flags=TICK_FLAG_BID;
               z++;
               p++;
              }

            history_ticks[z].bid=json["history"]["prices"][i].ToDbl();
            history_ticks[z].ask=0;
            history_ticks[z].time=(datetime)(json["history"]["times"][i].ToInt());
            history_ticks[z].time_msc=(long)((json["history"]["times"][i].ToInt())*1000);
            history_ticks[z].last=0;
            history_ticks[z].volume=0;
            history_ticks[z].volume_real=0;
            history_ticks[z].flags=TICK_FLAG_BID;

            i++;
            z++;
           }
         else
           {
            if(ArrayResize(history_ticks,z+1,100)==(z+1))
              {
               history_ticks[z].bid=json["history"]["prices"][i].ToDbl();
               history_ticks[z].ask=0;
               history_ticks[z].time=(datetime)json["history"]["times"][i].ToInt();
               history_ticks[z].time_msc=(long)(json["history"]["times"][i].ToInt()*1000);
               history_ticks[z].last=0;
               history_ticks[z].volume=0;
               history_ticks[z].volume_real=0;
               history_ticks[z].flags=TICK_FLAG_BID;
              }
            else
              {
               Print("Memory allocation error,  "+IntegerToString(::GetLastError()));
               return(false);
              }

            i++;
            z++;
           }
        }

      //Print("z is ",z,". Arraysize is ",ArraySize(history_ticks));

      if(m_history_end>0 && z>0)
        {
         DeleteAllCharts();

         if(CustomTicksDelete(m_symbol_name,int(m_history_start)*1000,(history_ticks[0].time_msc-1000))<0)
           {
            Print("error deleting ticks ", ::GetLastError());
            return(false);
           }
         else
           {
            m_history_end=history_ticks[z-1].time;
            m_history_start=history_ticks[0].time;
           }
        }


      if(ArraySize(history_ticks)>0)
        {
         //ArrayPrint(history_ticks);
         if(CustomTicksAdd(m_symbol_name,history_ticks)<0)//CustomTicksReplace(m_symbol_name,history_ticks[0].time_msc,history_ticks[z-1].time_msc,history_ticks)
           {
            Print("Error adding history "+IntegerToString(::GetLastError()));
            return(false);
           }
        }
      else
        {
         Print("Received unexpected response from server ",IntegerToString(::GetLastError()), " "+history);
         return(false);
        }
     }
   else
     {
      Print("error reading "," error: ",websocket.LastError(), websocket.LastErrorMessage());
      return(false);
     }

   OpenChart();

   return(true);

  }

//+---------------------------------------------------------------------+
//|method that enables the reciept of new ticks as they become available|
//+---------------------------------------------------------------------+
bool CBinarySymbol::StartTicksStream(void)
  {
   Comment("Starting live ticks stream for "+m_symbol_name+".......");

   if(m_stream_id!="")
      StopTicksStream();

   json.Clear();
   json["subscribe"]=1;
   json["ticks"]=m_symbol_name;

   return(websocket.SendString(json.Serialize()));
  }

//+------------------------------------------------------------------+
//|Used to cancel all tick streams that may have been initiated      |
//+------------------------------------------------------------------+
bool CBinarySymbol::StopTicksStream(void)
  {
   
   json.Clear();
   json["forget_all"]="ticks";

   if(websocket.SendString(json.Serialize()))
     {
      m_stream_id=NULL;
      if(websocket.ReadString(m_stream_id)>0)
        {
         m_stream_id=NULL;
         Comment("Stopping live ticks stream for  "+m_symbol_name+".......");
         return(true);
        }
     }

   return(false);
  }

//+------------------------------------------------------------------+
//|gets the symbol specification details.                            |
//+------------------------------------------------------------------+
bool CBinarySymbol::GetSymbolSettings(void)
  {
   bool fileopened=false;
   bool specs_read=false;

   Comment("Reading Symbol Specification for "+m_symbol_name+".......");

   if(symbolSpecs["active_symbols"][0]["symbol"].ToStr()!="")
      specs_read=true;

   CFileTxt symbolSet;

   if(!specs_read)
     {
      if(!symbolSet.IsExist(BINARY_SYMBOL_SETTINGS,FILE_COMMON))
        {
         if(symbolSet.Open(BINARY_SYMBOL_SETTINGS,FILE_WRITE|FILE_UNICODE|FILE_COMMON,0)==INVALID_HANDLE)
            Print("failed to open file ",BINARY_SYMBOL_SETTINGS," .Error - ",::GetLastError());
         else
            fileopened=true;
        }
      else
        {
         if(symbolSet.Open(BINARY_SYMBOL_SETTINGS,FILE_READ|FILE_UNICODE|FILE_COMMON,0)==INVALID_HANDLE)
            Print("failed to open file ",BINARY_SYMBOL_SETTINGS," .Error - ",::GetLastError());
         else
           {
            if(symbolSpecs.Deserialize(symbolSet.ReadString()))
              {
               symbolSet.Close();
               specs_read=true;
              }
            else
              {
               Print("failed to read from ",BINARY_SYMBOL_SETTINGS,".Error -",::GetLastError());
               symbolSet.Close();
              }
           }
        }
     }


   if(!specs_read)
     {
      if(!websocket.Connect(m_url))
        {
         Print(websocket.LastErrorMessage()," : ",websocket.LastError());

         if(fileopened)
            symbolSet.Close();

         return(false);
        }

      json.Clear();
      json["active_symbols"]="brief";
      json["product_type"]="basic";

      if(!websocket.SendString(json.Serialize()))
        {
         Print(websocket.LastErrorMessage()," : ",websocket.LastError());

         if(fileopened)
            symbolSet.Close();

         return(false);
        }

      string response;
      if(websocket.ReadString(response))
        {
         if(symbolSpecs.Deserialize(response))
            if(!CheckBinaryError(symbolSpecs) && symbolSpecs["active_symbols"][0]["symbol"].ToStr()!="")
               specs_read=true;

         if(fileopened && specs_read)
            if(symbolSet.WriteString(response)==0)
               Print("failed to write to ",BINARY_SYMBOL_SETTINGS,".Error -",::GetLastError());

         if(fileopened)
            symbolSet.Close();

         if(!specs_read)
            return(false);
        }
      else
        {
         Print(websocket.LastErrorMessage()," : ",websocket.LastError());

         if(fileopened)
            symbolSet.Close();

         return(false);
        }
     }

   if(specs_read)
     {
      int i=0;
      while(symbolSpecs["active_symbols"][i]["symbol"].ToStr()!="")
        {
         string cs=symbolSpecs["active_symbols"][i]["symbol"].ToStr();
         if(StringFind(cs,m_symbol_name)>=0)
           {
            m_index=i;
            break;
           }
         i++;
        }
      if(m_index<0)
        {
         Print("Matching symbol name not found, Check the symbol name on Binary.com");
         return(false);
        }
     }

   return(specs_read);

  }

//+-----------------------------------------------------------------------------+
//|takes as input a json object and checks for an error message contained in it.|
//+-----------------------------------------------------------------------------+
bool CBinarySymbol::CheckBinaryError(CJAVal &j)
  {
   string error=j["error"]["message"].ToStr();

   if(error!="")
     {
      Print("Binary.com error: ",error);
      return(true);
     }

   return(false);
  }


CBinarySymbol b_symbol;
//+------------------------------------------------------------------+
//| Expert initialization function                                   |
//+------------------------------------------------------------------+
int OnInit()
  {
   b_symbol.SetAppID(binary_appid);
//---
   if(!b_symbol.Initialize(EnumToString(binary_symbol)))
      return(INIT_FAILED);
//---
   if(!b_symbol.UpdateHistory())
      return(INIT_FAILED);
//---
   if(!b_symbol.StartTicksStream())
      return(INIT_FAILED);
//--- create timer
   EventSetMillisecondTimer(500);
//---
   return(INIT_SUCCEEDED);
  }
//+------------------------------------------------------------------+
//| Expert deinitialization function                                 |
//+------------------------------------------------------------------+
void OnDeinit(const int reason)
  {
//--- destroy timer
   EventKillTimer();
//--- stop the ticks stream
   b_symbol.StopTicksStream();

  }
//+------------------------------------------------------------------+
//| Expert tick function                                             |
//+------------------------------------------------------------------+
void OnTick()
  {
//---

  }
//+------------------------------------------------------------------+
//| Timer function                                                   |
//+------------------------------------------------------------------+
void OnTimer()
  {
//---
   b_symbol.AddTick();
  }
//+------------------------------------------------------------------+
